/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.processor.internals;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.HashMap;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.streams.errors.ProcessorStateException;
import org.apache.kafka.streams.processor.TaskId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StateDirectory {
    static final String LOCK_FILE_NAME = ".lock";
    private static final Logger log = LoggerFactory.getLogger(StateDirectory.class);
    private final File stateDir;
    private final HashMap<TaskId, FileChannel> channels = new HashMap();
    private final HashMap<TaskId, FileLock> locks = new HashMap();

    public StateDirectory(String applicationId, String stateDirConfig) {
        File baseDir = new File(stateDirConfig);
        if (!baseDir.exists() && !baseDir.mkdirs()) {
            throw new ProcessorStateException(String.format("state directory [%s] doesn't exist and couldn't be created", stateDirConfig));
        }
        this.stateDir = new File(baseDir, applicationId);
        if (!this.stateDir.exists() && !this.stateDir.mkdir()) {
            throw new ProcessorStateException(String.format("state directory [%s] doesn't exist and couldn't be created", this.stateDir.getPath()));
        }
    }

    public File directoryForTask(TaskId taskId) {
        File taskDir = new File(this.stateDir, taskId.toString());
        if (!taskDir.exists() && !taskDir.mkdir()) {
            throw new ProcessorStateException(String.format("task directory [%s] doesn't exist and couldn't be created", taskDir.getPath()));
        }
        return taskDir;
    }

    public boolean lock(TaskId taskId, int retry) throws IOException {
        if (this.locks.containsKey(taskId)) {
            return true;
        }
        File lockFile = new File(this.directoryForTask(taskId), LOCK_FILE_NAME);
        FileChannel channel = this.getOrCreateFileChannel(taskId, lockFile.toPath());
        FileLock lock = this.tryAcquireLock(channel);
        while (lock == null && retry > 0) {
            try {
                Thread.sleep(200L);
            }
            catch (Exception exception) {
                // empty catch block
            }
            --retry;
            lock = this.tryAcquireLock(channel);
        }
        if (lock != null) {
            this.locks.put(taskId, lock);
        }
        return lock != null;
    }

    public void unlock(TaskId taskId) throws IOException {
        FileLock lock = this.locks.remove(taskId);
        if (lock != null) {
            lock.release();
            FileChannel fileChannel = this.channels.remove(taskId);
            if (fileChannel != null) {
                fileChannel.close();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void cleanRemovedTasks() {
        File[] taskDirs = this.listTaskDirectories();
        if (taskDirs == null || taskDirs.length == 0) {
            return;
        }
        for (File taskDir : taskDirs) {
            String dirName = taskDir.getName();
            TaskId id = TaskId.parse(dirName);
            if (this.locks.containsKey(id)) continue;
            try {
                if (!this.lock(id, 0)) continue;
                log.info("Deleting obsolete state directory {} for task {}", (Object)dirName, (Object)id);
                Utils.delete((File)taskDir);
            }
            catch (OverlappingFileLockException e) {
            }
            catch (IOException e) {
                log.error("Failed to lock the state directory due to an unexpected exception", (Throwable)e);
            }
            finally {
                try {
                    this.unlock(id);
                }
                catch (IOException e) {
                    log.error("Failed to release the state directory lock");
                }
            }
        }
    }

    public File[] listTaskDirectories() {
        return this.stateDir.listFiles(new FileFilter(){

            @Override
            public boolean accept(File pathname) {
                String name = pathname.getName();
                return pathname.isDirectory() && name.matches("\\d+_\\d+");
            }
        });
    }

    private FileChannel getOrCreateFileChannel(TaskId taskId, Path lockPath) throws IOException {
        if (!this.channels.containsKey(taskId)) {
            this.channels.put(taskId, FileChannel.open(lockPath, StandardOpenOption.CREATE, StandardOpenOption.WRITE));
        }
        return this.channels.get(taskId);
    }

    private FileLock tryAcquireLock(FileChannel channel) throws IOException {
        try {
            return channel.tryLock();
        }
        catch (OverlappingFileLockException e) {
            return null;
        }
    }
}

