/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.store;

import com.google.api.client.http.HttpResponseException;
import com.google.api.gax.paging.Page;
import com.google.auth.Credentials;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.ReadChannel;
import com.google.cloud.WriteChannel;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.Bucket;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageException;
import com.google.cloud.storage.StorageOptions;
import com.google.common.collect.Lists;
import com.google.crypto.tink.Aead;
import com.google.crypto.tink.KeyTemplate;
import com.google.crypto.tink.KeysetHandle;
import com.google.crypto.tink.aead.AeadConfig;
import com.google.crypto.tink.aead.KmsAeadKeyManager;
import com.google.crypto.tink.integration.gcpkms.GcpKmsClient;
import io.confluent.kafka.storage.checksum.E2EChecksumProtectedObjectType;
import io.confluent.kafka.storage.checksum.E2EChecksumStore;
import io.confluent.kafka.storage.checksum.E2EChecksumUtils;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.StandardOpenOption;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import kafka.tier.exceptions.E2EChecksumInvalidException;
import kafka.tier.exceptions.TierObjectStoreFatalException;
import kafka.tier.exceptions.TierObjectStoreRetriableException;
import kafka.tier.store.BucketHealthResult;
import kafka.tier.store.CombinedObjectStream;
import kafka.tier.store.GcsTierObjectStoreConfig;
import kafka.tier.store.OpaqueData;
import kafka.tier.store.TierObjectAttribute;
import kafka.tier.store.TierObjectStore;
import kafka.tier.store.TierObjectStoreResponse;
import kafka.tier.store.TierObjectStoreUtils;
import kafka.tier.store.VersionInformation;
import kafka.tier.store.encryption.EncryptionKeyManager;
import kafka.tier.store.encryption.KeyContext;
import kafka.tier.store.encryption.KeySha;
import kafka.tier.store.objects.FragmentType;
import kafka.tier.store.objects.ObjectType;
import kafka.tier.store.objects.TierSegmentUpload;
import kafka.tier.store.objects.metadata.HealthMetadata;
import kafka.tier.store.objects.metadata.ObjectMetadata;
import kafka.tier.store.objects.metadata.ObjectStoreMetadata;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.storage.internals.utils.Throttler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GcsTierObjectStore
extends TierObjectStore {
    private static final Logger log = LoggerFactory.getLogger(GcsTierObjectStore.class);
    private static final int UNKNOWN_END_RANGE_CHUNK_SIZE = 1000000;
    private final Optional<String> clusterIdOpt;
    private final Optional<Integer> brokerIdOpt;
    private final String bucket;
    private final String prefix;
    private final int writeChunkSize;
    private final Storage storage;
    private final EncryptionKeyManager encryptionKeyManager;
    private final Optional<E2EChecksumStore> checksumStoreOpt;
    private static final int DEFAULT_GCS_DELETE_BATCH_SIZE = 1000;
    private static final int ERROR_CODE_INVALID_REQ = 400;
    private static final String CRC32C_MISMATCH_STR = "calculated CRC32C";
    public static final String CRC32C_METADATA_KEY = "crc32c";

    public GcsTierObjectStore(Time time, GcsTierObjectStoreConfig config, Optional<E2EChecksumStore> checksumStoreOpt) {
        this(GcsTierObjectStore.storage(config), GcsTierObjectStore.encryptionKeyManager(config, time, null), config, checksumStoreOpt);
    }

    public GcsTierObjectStore(Time time, Metrics metrics, GcsTierObjectStoreConfig config, Optional<E2EChecksumStore> checksumStoreOpt) {
        this(GcsTierObjectStore.storage(config), GcsTierObjectStore.encryptionKeyManager(config, time, metrics), config, checksumStoreOpt);
    }

    GcsTierObjectStore(Storage storage, EncryptionKeyManager encryptionKeyManager, GcsTierObjectStoreConfig config, Optional<E2EChecksumStore> checksumStoreOpt) {
        this.clusterIdOpt = config.clusterIdOpt;
        this.brokerIdOpt = config.brokerIdOpt;
        this.storage = storage;
        this.bucket = config.gcsBucket;
        this.prefix = config.gcsPrefix;
        this.writeChunkSize = config.gcsWriteChunkSize;
        this.expectBucket(this.bucket, config.gcsRegion);
        this.encryptionKeyManager = encryptionKeyManager;
        if (this.encryptionKeyManager != null) {
            this.encryptionKeyManager.bindHook(new EncryptionKeyManagerHook());
        }
        this.checksumStoreOpt = checksumStoreOpt;
    }

    @Override
    public String keyPrefix() {
        return this.prefix;
    }

    @Override
    public TierObjectStore.Backend getBackend() {
        return TierObjectStore.Backend.GCS;
    }

    @Override
    public BucketHealthResult checkBucketHealth() {
        try {
            ByteBuffer payload = TierObjectStoreUtils.timeHealthPayload();
            HealthMetadata metadata = new HealthMetadata(this.clusterIdOpt, this.brokerIdOpt);
            String key = metadata.toFragmentLocation(this.prefix, FragmentType.HEALTH_CHECK).get().objectPath();
            this.putBuf(key, metadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt), payload);
            try (InputStream inputStream = this.getObjectStoreFragment(metadata, FragmentType.HEALTH_CHECK).getInputStream();){
                int read;
                while ((read = inputStream.read()) > 0) {
                    log.trace("Bucket probe read {} bytes", (Object)read);
                }
            }
            BlobId blobId = BlobId.of((String)this.bucket, (String)key);
            if (!this.storage.delete(blobId)) {
                throw new Exception("Error deleting health key " + key);
            }
            return BucketHealthResult.HEALTHY;
        }
        catch (Exception e) {
            if (this.encryptionKeyManager != null && e instanceof StorageException && ((StorageException)e).getCode() == 400 && (((StorageException)e).getReason().startsWith("customerEncryption") || ((StorageException)e).getReason().startsWith("cloudKms"))) {
                log.error("Bucket health checker resulted in a BYOK related error with error code: {}, status code: {}", new Object[]{((StorageException)e).getCode(), ((StorageException)e).getReason(), e});
                return BucketHealthResult.BYOK;
            }
            if (e instanceof StorageException && ((StorageException)e).getCode() == 403 && Objects.equals(((StorageException)e).getReason(), "UserProjectAccessDenied")) {
                log.error("Bucket health checker resulted in a permission error for customer key");
                if (this.encryptionKeyManager != null) {
                    return BucketHealthResult.BYOK;
                }
                return BucketHealthResult.PERMISSION;
            }
            log.error("Bucket health checker returned unclassified error", (Throwable)e);
            return BucketHealthResult.UNCLASSIFIED;
        }
    }

    @Override
    public Map<String, List<VersionInformation>> listObject(String keyPrefix, boolean getVersionInfo) {
        HashMap<String, List<VersionInformation>> results = new HashMap<String, List<VersionInformation>>();
        try {
            Page blobs = this.storage.list(this.bucket, new Storage.BlobListOption[]{Storage.BlobListOption.prefix((String)keyPrefix), Storage.BlobListOption.versions((boolean)getVersionInfo)});
            for (Blob blob : blobs.iterateAll()) {
                results.putIfAbsent(blob.getName(), new ArrayList());
                ((List)results.get(blob.getName())).add(new VersionInformation(blob.getGeneration().toString()));
            }
            if (log.isDebugEnabled()) {
                StringBuilder allBlobs = new StringBuilder();
                results.forEach((key, versions) -> allBlobs.append("[").append((String)key).append("->").append(Arrays.toString(versions.toArray())).append("] "));
                log.debug("TierObjectStore listObjects versions: " + getVersionInfo + " prefix: " + keyPrefix + " " + allBlobs);
            }
        }
        catch (StorageException e) {
            log.info("Google storage client returned exception while listing " + keyPrefix + " " + (Object)((Object)e));
            throw new TierObjectStoreRetriableException(e.getMessage(), e);
        }
        catch (Exception e) {
            log.info("Unknown exception while listing " + keyPrefix + " " + e);
            throw new TierObjectStoreFatalException(e.getMessage(), e);
        }
        return results;
    }

    @Override
    public TierObjectStoreResponse getObject(ObjectStoreMetadata objectMetadata, ObjectType objectType, Long byteOffsetStart, Long byteOffsetEndExclusive, VersionInformation versionInformation) {
        BlobId blobId;
        String key = this.keyPath(objectMetadata, objectType);
        BlobId blobId2 = blobId = versionInformation != null ? BlobId.of((String)this.bucket, (String)key, (Long)Long.parseLong(versionInformation.getVersionId())) : BlobId.of((String)this.bucket, (String)key);
        if (byteOffsetStart != null && byteOffsetEndExclusive != null && byteOffsetStart > byteOffsetEndExclusive) {
            throw new IllegalStateException(String.format("Invalid range of byteOffsetStart %d and byteOffsetEndExclusive %d", byteOffsetStart, byteOffsetEndExclusive));
        }
        if (byteOffsetStart == null && byteOffsetEndExclusive != null) {
            throw new IllegalStateException(String.format("Cannot specify a byteOffsetEndExclusive=%d without specifying a byteOffsetStart", byteOffsetEndExclusive));
        }
        if (byteOffsetEndExclusive != null && byteOffsetEndExclusive - byteOffsetStart > Integer.MAX_VALUE) {
            throw new IllegalArgumentException(String.format("GcsTierObjectStore does not support fetch size over Integer.MAX_VALUE bytes, byteOffsetStart=%d, byteOffsetEnd=%d", byteOffsetStart, byteOffsetEndExclusive));
        }
        try {
            ReadChannel reader = this.getReader(objectMetadata, objectType, blobId);
            long byteOffsetStartLong = byteOffsetStart != null ? byteOffsetStart : 0L;
            OptionalInt chunkSize = byteOffsetEndExclusive != null ? OptionalInt.of((int)(byteOffsetEndExclusive - byteOffsetStartLong)) : OptionalInt.empty();
            return new GcsTierObjectStoreResponse(reader, byteOffsetStartLong, chunkSize);
        }
        catch (TierObjectStoreRetriableException e) {
            throw e;
        }
        catch (StorageException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to fetch object, blobId: %s metadata: %s type: %s range %s-%s", new Object[]{blobId, objectMetadata, objectType, byteOffsetStart, byteOffsetEndExclusive}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Unknown exception when fetching object, blobId: %s metadata: %s type: %s range %s-%s", new Object[]{blobId, objectMetadata, objectType, byteOffsetStart, byteOffsetEndExclusive}), e);
        }
    }

    @Override
    public ByteBuffer getSnapshot(ObjectStoreMetadata metadata, FragmentType fragmentType, int estimatedBufferSize) {
        ByteBuffer buffer;
        if (FragmentType.checkGetSnapshotSupported(fragmentType)) {
            throw new IllegalArgumentException("getSnapshot does not support the given fragmentType: " + (Object)((Object)fragmentType));
        }
        try (TierObjectStoreResponse response = this.getObjectStoreFragment(metadata, fragmentType);){
            buffer = ByteBuffer.wrap(Utils.readFullyToArray((InputStream)response.getInputStream(), (int)estimatedBufferSize));
        }
        catch (Exception e) {
            for (Throwable cause = e.getCause(); cause != null && cause != cause.getCause(); cause = cause.getCause()) {
                if (!(cause instanceof StorageException) || ((StorageException)cause).getCode() != 404) continue;
                throw new TierObjectStoreFatalException("Snapshot object not found in object store.", cause);
            }
            throw new TierObjectStoreRetriableException("Encountered an exception when fetching snapshot from object store.", e);
        }
        return buffer;
    }

    private ReadChannel getReader(ObjectStoreMetadata objectMetadata, ObjectType objectType, BlobId blobId) {
        if (!objectMetadata.opaqueData().isEmpty() && (objectType.equals((Object)ObjectType.SEGMENT) || objectType.equals((Object)ObjectType.SEGMENT_WITH_METADATA))) {
            KeyContext keyContext = this.getKeyContext(objectMetadata.opaqueData(), blobId);
            return this.storage.reader(blobId, new Storage.BlobSourceOption[]{Storage.BlobSourceOption.decryptionKey((String)keyContext.cleartextDataKey.base64Encoded())});
        }
        return this.storage.reader(blobId, new Storage.BlobSourceOption[0]);
    }

    KeyContext getKeyContext(OpaqueData opaqueData, BlobId blobId) {
        if (this.encryptionKeyManager == null) {
            throw new TierObjectStoreFatalException("EncryptionKeyManager is not configured");
        }
        KeySha keySha = KeySha.fromRawBytes(opaqueData.intoByteArray());
        KeyContext keyContext = this.encryptionKeyManager.keyContext(keySha);
        if (keyContext == null) {
            log.info("EncryptionKeyManager cache miss while downloading object with KeySha {}, fetching object metadata for cache hydration", (Object)keySha);
            Blob blobMetadata = this.storage.get(blobId, new Storage.BlobGetOption[]{Storage.BlobGetOption.fields((Storage.BlobField[])new Storage.BlobField[]{Storage.BlobField.METADATA})});
            Map extractedMetadata = blobMetadata.getMetadata();
            KeySha restoredKeySha = this.encryptionKeyManager.registerKeyFromObjectMetadata(extractedMetadata);
            if (!restoredKeySha.equals(keySha)) {
                throw new TierObjectStoreFatalException(String.format("KeySha of key material restored from object metadata %s does not match KeySha provided via OpaqueData %s", restoredKeySha, keySha));
            }
            keyContext = this.encryptionKeyManager.keyContext(keySha);
        }
        return keyContext;
    }

    @Override
    public OpaqueData prepPutSegment() throws TierObjectStoreRetriableException, IOException {
        if (this.encryptionKeyManager != null) {
            KeySha active = this.encryptionKeyManager.activeKeySha();
            return OpaqueData.fromByteArray(active.toRawBytes());
        }
        return OpaqueData.ZEROED;
    }

    private void handlePutSegmentStorageException(StorageException e, TierSegmentUpload<?> tierSegmentUpload, Map<String, String> metadata, E2EChecksumProtectedObjectType currentPutObjectType) {
        log.warn("Deleting partially uploaded files due to failed to upload segment: " + tierSegmentUpload.objectMetadata(), (Throwable)e);
        this.deleteObjects(this.objectsForSegment(tierSegmentUpload.objectMetadata()));
        if (currentPutObjectType == E2EChecksumProtectedObjectType.SEGMENT_WITH_METADATA) {
            throw new TierObjectStoreRetriableException("Failed to upload segment: " + tierSegmentUpload.objectMetadata(), e);
        }
        Optional<File> file = TierObjectStoreUtils.getCurrentPutObjectFile(tierSegmentUpload, currentPutObjectType);
        if (!file.isPresent()) {
            throw new TierObjectStoreRetriableException("Failed to upload segment: " + tierSegmentUpload.objectMetadata(), e);
        }
        String filePath = file.get().getAbsolutePath();
        boolean isChecksumMismatchFromPureStorageException = e.getCode() == 400 && e.getMessage().contains(CRC32C_MISMATCH_STR);
        boolean isChecksumMismatchFromHttpResponseException = false;
        if (e.getCause() != null && e.getCause() instanceof HttpResponseException) {
            HttpResponseException causalException = (HttpResponseException)e.getCause();
            boolean bl = isChecksumMismatchFromHttpResponseException = causalException.getStatusCode() == 400 && causalException.getStatusMessage().contains(CRC32C_MISMATCH_STR);
        }
        if (isChecksumMismatchFromPureStorageException || isChecksumMismatchFromHttpResponseException) {
            Optional crcFromChecksumStore = Optional.empty();
            if (this.checksumStoreOpt.isPresent() && this.checksumStoreOpt.get().checksumProtectionEnabled(currentPutObjectType)) {
                crcFromChecksumStore = E2EChecksumUtils.getBase64CrcFromStore((E2EChecksumStore)this.checksumStoreOpt.get(), (File)file.get(), metadata);
            }
            if (crcFromChecksumStore.isPresent()) {
                Optional recalculatedCrcFromFile = E2EChecksumUtils.compute32BitBase64Crc32c((File)file.get());
                if (crcFromChecksumStore.equals(recalculatedCrcFromFile)) {
                    log.info("Network Error: On-network corruption of a file during upload: {} with expected CRC value: {} and recalculated CRC value: {}", new Object[]{filePath, crcFromChecksumStore.get(), recalculatedCrcFromFile.get()});
                    throw new TierObjectStoreRetriableException("Failed to upload object after an on-network corruption of file: " + filePath, e);
                }
                throw new E2EChecksumInvalidException("Failed to upload object due to an on-disk corruption of file: " + filePath, (Throwable)e);
            }
        }
        throw new TierObjectStoreRetriableException("Failed to upload segment: " + tierSegmentUpload.objectMetadata() + " due to file: " + filePath, e);
    }

    private void handlePutSegmentStorageExceptionMaskedAsIOException(IOException e, TierSegmentUpload<?> tierSegmentUpload, Map<String, String> metadata, E2EChecksumProtectedObjectType currentPutObjectType) {
        Throwable[] suppressedExceptions;
        for (Throwable cause = e.getCause(); cause != null && cause != cause.getCause(); cause = cause.getCause()) {
            if (!(cause instanceof StorageException)) continue;
            this.handlePutSegmentStorageException((StorageException)cause, tierSegmentUpload, metadata, currentPutObjectType);
        }
        for (Throwable suppressedException : suppressedExceptions = e.getSuppressed()) {
            if (!(suppressedException instanceof StorageException)) continue;
            this.handlePutSegmentStorageException((StorageException)suppressedException, tierSegmentUpload, metadata, currentPutObjectType);
        }
    }

    private void putSegmentAsMultiObject(TierSegmentUpload<?> tierSegmentUpload, Map<String, String> metadata) throws StorageException {
        E2EChecksumProtectedObjectType currentPutObjectType = E2EChecksumProtectedObjectType.SEGMENT;
        try {
            this.putSegmentFile(metadata, tierSegmentUpload.objectMetadata(), tierSegmentUpload.segment(), tierSegmentUpload.throttlerOpt());
            currentPutObjectType = E2EChecksumProtectedObjectType.OFFSET_INDEX;
            this.putFileUnencrypted(this.keyPath(tierSegmentUpload.objectMetadata(), ObjectType.OFFSET_INDEX), metadata, tierSegmentUpload.offsetIdx(), E2EChecksumProtectedObjectType.OFFSET_INDEX, tierSegmentUpload.throttlerOpt());
            currentPutObjectType = E2EChecksumProtectedObjectType.TIMESTAMP_INDEX;
            this.putFileUnencrypted(this.keyPath(tierSegmentUpload.objectMetadata(), ObjectType.TIMESTAMP_INDEX), metadata, tierSegmentUpload.timestampIdx(), E2EChecksumProtectedObjectType.TIMESTAMP_INDEX, tierSegmentUpload.throttlerOpt());
            if (tierSegmentUpload.producerStateSnapshotOpt().isPresent()) {
                currentPutObjectType = E2EChecksumProtectedObjectType.PRODUCER_STATE;
                Object producerState = tierSegmentUpload.producerStateSnapshotOpt().get();
                if (producerState instanceof File) {
                    this.putFileUnencrypted(this.keyPath(tierSegmentUpload.objectMetadata(), ObjectType.PRODUCER_STATE), metadata, (File)producerState, E2EChecksumProtectedObjectType.PRODUCER_STATE, tierSegmentUpload.throttlerOpt());
                } else if (producerState instanceof ByteBuffer) {
                    this.putBuf(this.keyPath(tierSegmentUpload.objectMetadata(), ObjectType.PRODUCER_STATE), metadata, (ByteBuffer)producerState);
                }
            }
            if (tierSegmentUpload.txnIdxOpt().isPresent()) {
                currentPutObjectType = E2EChecksumProtectedObjectType.TRANSACTION_INDEX;
                this.putBuf(this.keyPath(tierSegmentUpload.objectMetadata(), ObjectType.TRANSACTION_INDEX), metadata, tierSegmentUpload.txnIdxOpt().get());
            }
            if (tierSegmentUpload.epochStateOpt().isPresent()) {
                currentPutObjectType = E2EChecksumProtectedObjectType.EPOCH_STATE;
                this.putBuf(this.keyPath(tierSegmentUpload.objectMetadata(), ObjectType.EPOCH_STATE), metadata, tierSegmentUpload.epochStateOpt().get());
            }
            this.checksumStoreOpt.ifPresent(tierSegmentUpload::postPutSegmentCleanup);
        }
        catch (IOException e) {
            this.handlePutSegmentStorageExceptionMaskedAsIOException(e, tierSegmentUpload, metadata, currentPutObjectType);
            throw new TierObjectStoreFatalException("Unknown exception when uploading segment: " + tierSegmentUpload.objectMetadata(), e);
        }
        catch (StorageException e) {
            this.handlePutSegmentStorageException(e, tierSegmentUpload, metadata, currentPutObjectType);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when uploading segment: " + tierSegmentUpload.objectMetadata(), e);
        }
    }

    private void putSegmentAsCombinedObject(TierSegmentUpload<?> tierSegmentUpload, Map<String, String> metadata) throws StorageException {
        try (CombinedObjectStream combinedObjectStream = tierSegmentUpload.makeCombinedObjectStream();){
            Optional<String> checksumOpt = this.checksumStoreOpt.flatMap(checksumStore -> {
                if (!checksumStore.checksumProtectionEnabled(E2EChecksumProtectedObjectType.SEGMENT_WITH_METADATA)) {
                    return Optional.empty();
                }
                return Optional.of(tierSegmentUpload.getChecksumForCombinedObject());
            });
            String combinedObjectKeyPath = this.keyPath(tierSegmentUpload.objectMetadata(), ObjectType.SEGMENT_WITH_METADATA);
            if (this.encryptionKeyManager != null) {
                this.putInputStreamEncrypted(combinedObjectKeyPath, metadata, tierSegmentUpload.objectMetadata().opaqueData(), combinedObjectStream, checksumOpt);
            } else {
                if (!tierSegmentUpload.objectMetadata().opaqueData().isEmpty()) {
                    throw new TierObjectStoreFatalException(String.format("Attempted to upload a CombinedObject segment-with-metadata with OpaqueData %s, but encryption is not configured", tierSegmentUpload.objectMetadata().opaqueData()));
                }
                this.putInputStream(combinedObjectKeyPath, metadata, combinedObjectStream, checksumOpt);
            }
            this.checksumStoreOpt.ifPresent(tierSegmentUpload::postPutSegmentCleanup);
        }
        catch (IOException e) {
            this.handlePutSegmentStorageExceptionMaskedAsIOException(e, tierSegmentUpload, metadata, E2EChecksumProtectedObjectType.SEGMENT_WITH_METADATA);
            throw new TierObjectStoreFatalException("Unknown exception when uploading segment: " + tierSegmentUpload.objectMetadata(), e);
        }
        catch (StorageException e) {
            this.handlePutSegmentStorageException(e, tierSegmentUpload, metadata, E2EChecksumProtectedObjectType.SEGMENT_WITH_METADATA);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when uploading segment: " + tierSegmentUpload.objectMetadata(), e);
        }
    }

    private void putInputStreamEncrypted(String key, Map<String, String> metadata, OpaqueData opaqueData, InputStream stream, Optional<String> checksumOpt) throws IOException {
        if (opaqueData == null || opaqueData.isEmpty()) {
            throw new TierObjectStoreFatalException("Encryption was enabled but no valid OpaqueData object was provided");
        }
        KeySha keySha = KeySha.fromRawBytes(opaqueData.intoByteArray());
        KeyContext keyContext = this.encryptionKeyManager.keyContext(keySha);
        if (keyContext == null) {
            throw new TierObjectStoreFatalException(String.format("No valid KeyContext for KeySha '%s'", keySha));
        }
        metadata.putAll(keyContext.metadata);
        Storage.BlobWriteOption encryption = Storage.BlobWriteOption.encryptionKey((String)keyContext.cleartextDataKey.base64Encoded());
        if (!stream.markSupported()) {
            log.warn("InputStream does not support mark/reset, wrapping in BufferedInputStream");
            stream = new BufferedInputStream(stream);
        }
        BlobId blobId = BlobId.of((String)this.bucket, (String)key);
        BlobInfo.Builder blobInfoBuilder = BlobInfo.newBuilder((BlobId)blobId);
        log.debug("Uploading encrypted object {} from inputStream", (Object)key);
        if (checksumOpt.isPresent()) {
            String checksum = checksumOpt.get();
            metadata.put(CRC32C_METADATA_KEY, checksum);
            BlobInfo blobInfo = blobInfoBuilder.setMetadata(metadata).setCrc32c(checksum).build();
            this.storage.createFrom(blobInfo, stream, new Storage.BlobWriteOption[]{encryption, Storage.BlobWriteOption.doesNotExist(), Storage.BlobWriteOption.crc32cMatch()});
            metadata.remove(CRC32C_METADATA_KEY);
        } else {
            BlobInfo blobInfo = blobInfoBuilder.setMetadata(metadata).build();
            this.storage.createFrom(blobInfo, stream, new Storage.BlobWriteOption[]{encryption, Storage.BlobWriteOption.doesNotExist()});
        }
    }

    private void putInputStream(String key, Map<String, String> metadata, InputStream stream, Optional<String> checksumOpt) throws IOException {
        if (!stream.markSupported()) {
            log.warn("InputStream does not support mark/reset, wrapping in BufferedInputStream");
            stream = new BufferedInputStream(stream);
        }
        BlobId blobId = BlobId.of((String)this.bucket, (String)key);
        BlobInfo.Builder blobInfoBuilder = BlobInfo.newBuilder((BlobId)blobId);
        log.debug("Uploading object {} from inputStream", (Object)key);
        if (checksumOpt.isPresent()) {
            String checksum = checksumOpt.get();
            metadata.put(CRC32C_METADATA_KEY, checksum);
            BlobInfo blobInfo = blobInfoBuilder.setMetadata(metadata).setCrc32c(checksum).build();
            this.storage.createFrom(blobInfo, stream, new Storage.BlobWriteOption[]{Storage.BlobWriteOption.doesNotExist(), Storage.BlobWriteOption.crc32cMatch()});
            metadata.remove(CRC32C_METADATA_KEY);
        } else {
            BlobInfo blobInfo = blobInfoBuilder.setMetadata(metadata).build();
            this.storage.createFrom(blobInfo, stream, new Storage.BlobWriteOption[]{Storage.BlobWriteOption.doesNotExist()});
        }
    }

    @Override
    public void putSegment(TierSegmentUpload<?> tierSegmentUpload) {
        Map<String, String> metadata = tierSegmentUpload.objectMetadata().objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        switch (tierSegmentUpload.putMode()) {
            case LegacyMultiObject: {
                this.putSegmentAsMultiObject(tierSegmentUpload, metadata);
                break;
            }
            case CombinedObject: {
                this.putSegmentAsCombinedObject(tierSegmentUpload, metadata);
                break;
            }
            default: {
                throw new UnsupportedOperationException("Unknown putMode: " + tierSegmentUpload.putMode());
            }
        }
    }

    private void putSegmentFile(Map<String, String> metadata, ObjectMetadata objectMetadata, File segmentData, Optional<Throttler> throttlerOpt) throws IOException {
        if (this.encryptionKeyManager != null) {
            this.putFileEncrypted(this.keyPath(objectMetadata, ObjectType.SEGMENT), metadata, segmentData, objectMetadata.opaqueData(), E2EChecksumProtectedObjectType.SEGMENT, throttlerOpt);
        } else {
            if (!objectMetadata.opaqueData().isEmpty()) {
                throw new TierObjectStoreFatalException(String.format("Attempted to upload a segment with OpaqueData %s, but encryption is not configured", objectMetadata.opaqueData()));
            }
            this.putFileUnencrypted(this.keyPath(objectMetadata, ObjectType.SEGMENT), metadata, segmentData, E2EChecksumProtectedObjectType.SEGMENT, throttlerOpt);
        }
    }

    @Override
    public String putObject(final ObjectStoreMetadata objectMetadata, File file, final ObjectType objectType) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            String key = this.keyPath(objectMetadata, objectType);
            this.putFileUnencrypted(key, metadata, file, objectType.toE2EChecksumProtectedObjectType(), Optional.empty());
            return key;
        }
        catch (StorageException e) {
            log.warn("Deleting partially uploaded files due to failed to upload segment: " + objectMetadata, (Throwable)e);
            this.deleteObjects((List<BlobId>)new ArrayList<BlobId>(){
                {
                    this.add(BlobId.of((String)GcsTierObjectStore.this.bucket, (String)GcsTierObjectStore.this.keyPath(objectMetadata, objectType)));
                }
            });
            if (e.getCode() == 400 && e.getMessage().contains(CRC32C_MISMATCH_STR)) {
                throw new E2EChecksumInvalidException("Checksum mismatch during object store upload", (Throwable)e);
            }
            throw new TierObjectStoreRetriableException(String.format("Failed to upload object %s, file %s, type %s", new Object[]{objectMetadata, file, objectType}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Failed to upload object %s, file %s, type %s", new Object[]{objectMetadata, file, objectType}), e);
        }
    }

    @Override
    public String putBuffer(final ObjectStoreMetadata objectMetadata, ByteBuffer buffer, final ObjectType objectType) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            String key = this.keyPath(objectMetadata, objectType);
            this.putBuf(key, metadata, buffer);
            return key;
        }
        catch (StorageException e) {
            log.warn("Deleting partially uploaded files due to failed to upload segment: " + objectMetadata, (Throwable)e);
            this.deleteObjects((List<BlobId>)new ArrayList<BlobId>(){
                {
                    this.add(BlobId.of((String)GcsTierObjectStore.this.bucket, (String)GcsTierObjectStore.this.keyPath(objectMetadata, objectType)));
                }
            });
            throw new TierObjectStoreRetriableException(String.format("Failed to upload object %s, buffer %s, type %s", new Object[]{objectMetadata, buffer, objectType}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Failed to upload object %s, buffer %s, type %s", new Object[]{objectMetadata, buffer, objectType}), e);
        }
    }

    @Override
    public void restoreObjectByCopy(ObjectMetadata objectMetadata, String key, VersionInformation lastLiveVersion) {
        String lastLiveVersionId = lastLiveVersion.getVersionId();
        try {
            Storage.CopyRequest.Builder copyRequestBuilder = Storage.CopyRequest.newBuilder();
            BlobId source = BlobId.of((String)this.bucket, (String)key, (Long)Long.parseLong(lastLiveVersionId));
            BlobId target = BlobId.of((String)this.bucket, (String)key);
            if (this.encryptionKeyManager != null && !objectMetadata.opaqueData().isEmpty() && (key.endsWith(ObjectType.SEGMENT.suffix()) || key.endsWith(ObjectType.SEGMENT_WITH_METADATA.suffix()))) {
                KeyContext keyContext = this.getKeyContext(objectMetadata.opaqueData(), source);
                if (keyContext == null) {
                    throw new TierObjectStoreFatalException(String.format("No valid KeyContext for copying object '%s'", key));
                }
                String dataKeyBase64 = keyContext.cleartextDataKey.base64Encoded();
                log.debug("Restore encrypted object by copying the last live version {} to gs://{}/{}", new Object[]{lastLiveVersionId, this.bucket, key});
                copyRequestBuilder.setSource(source).setSourceOptions(new Storage.BlobSourceOption[]{Storage.BlobSourceOption.decryptionKey((String)dataKeyBase64)}).setTarget(target, new Storage.BlobTargetOption[]{Storage.BlobTargetOption.encryptionKey((String)dataKeyBase64)});
            } else {
                log.debug("Restore unencrypted object by copying the last live version {} to gs://{}/{}", new Object[]{lastLiveVersionId, this.bucket, key});
                copyRequestBuilder.setSource(source).setTarget(target);
            }
            Storage.CopyRequest copyRequest = copyRequestBuilder.build();
            this.storage.copy(copyRequest);
        }
        catch (StorageException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to restore object %s (version: %s)", key, lastLiveVersionId), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Unknown exception when restoring object %s (version: %s)", key, lastLiveVersionId), e);
        }
    }

    private List<BlobId> objectsForSegment(ObjectMetadata objectMetadata) {
        if (objectMetadata.isCombinedObject(this.prefix)) {
            return Collections.singletonList(BlobId.of((String)this.bucket, (String)this.keyPath(objectMetadata, ObjectType.SEGMENT_WITH_METADATA)));
        }
        ArrayList<BlobId> blobIds = new ArrayList<BlobId>();
        block5: for (ObjectType objectType : TierObjectStore.getObjectTypesPerSegment()) {
            switch (objectType) {
                case TRANSACTION_INDEX: {
                    if (!objectMetadata.hasAbortedTxns()) continue block5;
                    blobIds.add(BlobId.of((String)this.bucket, (String)this.keyPath(objectMetadata, objectType)));
                    continue block5;
                }
                case EPOCH_STATE: {
                    if (!objectMetadata.hasEpochState()) continue block5;
                    blobIds.add(BlobId.of((String)this.bucket, (String)this.keyPath(objectMetadata, objectType)));
                    continue block5;
                }
                case PRODUCER_STATE: {
                    if (!objectMetadata.hasProducerState()) continue block5;
                    blobIds.add(BlobId.of((String)this.bucket, (String)this.keyPath(objectMetadata, objectType)));
                    continue block5;
                }
            }
            blobIds.add(BlobId.of((String)this.bucket, (String)this.keyPath(objectMetadata, objectType)));
        }
        return blobIds;
    }

    @Override
    public void deleteSegment(ObjectMetadata objectMetadata) {
        List<BlobId> blobIds = this.objectsForSegment(objectMetadata);
        log.debug("Deleting " + blobIds);
        ArrayList<BlobId> foundBlobIds = new ArrayList<BlobId>();
        try {
            List success = this.storage.delete(blobIds);
            log.debug("Deletion result " + success);
            for (int blobIndex = 0; blobIndex < success.size(); ++blobIndex) {
                Blob blob;
                if (((Boolean)success.get(blobIndex)).booleanValue() || (blob = this.storage.get(blobIds.get(blobIndex))) == null) continue;
                log.warn("Found object " + blob.getBlobId() + " that was expected to be deleted of " + objectMetadata);
                foundBlobIds.add(blob.getBlobId());
            }
        }
        catch (StorageException e) {
            throw new TierObjectStoreRetriableException("Failed to delete segment: " + objectMetadata, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when deleting segment: " + objectMetadata, e);
        }
        if (!foundBlobIds.isEmpty()) {
            throw new TierObjectStoreRetriableException("Deletion failed for " + objectMetadata + ". Blobs still exist in object storage with blob ids: " + foundBlobIds);
        }
    }

    @Override
    public void deleteVersions(List<TierObjectStore.KeyAndVersion> keys) {
        ArrayList<BlobId> blobsToDelete = new ArrayList<BlobId>();
        for (TierObjectStore.KeyAndVersion key : keys) {
            BlobId blobId = key.versionId() == null ? BlobId.of((String)this.bucket, (String)key.key()) : BlobId.of((String)this.bucket, (String)key.key(), (Long)Long.parseLong(key.versionId()));
            blobsToDelete.add(blobId);
            log.debug("TierObjectStore sending delete request for " + blobId.getName());
            if (blobsToDelete.size() < 1000) continue;
            this.deleteObjects(blobsToDelete);
            blobsToDelete.clear();
        }
        if (!blobsToDelete.isEmpty()) {
            this.deleteObjects(blobsToDelete);
        }
    }

    private void deleteObjects(List<BlobId> blobsToDelete) {
        List results;
        log.info("TierObjectStore sending batch delete request for " + blobsToDelete.size() + " objects");
        try {
            results = this.storage.delete(blobsToDelete);
        }
        catch (StorageException e) {
            log.error("StorageException while deleting versioned objects of size: " + blobsToDelete.size(), (Throwable)e);
            throw new TierObjectStoreRetriableException("StorageException while deleting versioned objects. " + (Object)((Object)e));
        }
        catch (Exception e) {
            log.error("Fatal exception while deleting versioned objects of size: " + blobsToDelete.size(), (Throwable)e);
            throw new TierObjectStoreFatalException("Fatal exception while deleting versioned objects." + e);
        }
        Iterator resultsIter = results.iterator();
        Iterator<BlobId> blobsIter = blobsToDelete.iterator();
        while (resultsIter.hasNext() && blobsIter.hasNext()) {
            BlobId blobId = blobsIter.next();
            Boolean deletionResult = (Boolean)resultsIter.next();
            if (!deletionResult.booleanValue()) {
                log.warn("Unable to delete blob " + blobId.toString() + ". Could be deletion failure or that blob is not found");
                continue;
            }
            log.info("Deleted blob " + blobId.toString());
        }
    }

    @Override
    public TierObjectAttribute objectExists(ObjectStoreMetadata objectMetadata, ObjectType objectType) throws TierObjectStoreRetriableException {
        TierObjectAttribute result = new TierObjectAttribute(false);
        try {
            String key = this.keyPath(objectMetadata, objectType);
            Blob blob = this.storage.get(this.bucket, key, new Storage.BlobGetOption[0]);
            if (blob != null) {
                result.exist = true;
                result.size = blob.getSize();
                log.trace("objectExists at {}/{} with size {}", new Object[]{this.bucket, key, result.size});
            }
        }
        catch (StorageException e) {
            throw new TierObjectStoreRetriableException("Failed to check object existence: " + objectMetadata + " type: " + (Object)((Object)objectType), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when checking object existence: " + objectMetadata + " type: " + (Object)((Object)objectType), e);
        }
        return result;
    }

    @Override
    public void close() {
        super.close();
        if (this.encryptionKeyManager != null) {
            this.encryptionKeyManager.close();
        }
    }

    private void putFileUnencrypted(String key, Map<String, String> metadata, File file, E2EChecksumProtectedObjectType objectType, Optional<Throttler> throttlerOpt) throws IOException {
        BlobId blobId = BlobId.of((String)this.bucket, (String)key);
        Optional crc = Optional.empty();
        if (this.checksumStoreOpt.isPresent() && this.checksumStoreOpt.get().checksumProtectionEnabled(objectType)) {
            crc = objectType.shouldCalculateBeforeUpload() ? E2EChecksumUtils.compute32BitBase64Crc32c((File)file) : E2EChecksumUtils.getBase64CrcFromStore((E2EChecksumStore)this.checksumStoreOpt.get(), (File)file, metadata);
        }
        if (crc.isPresent()) {
            String crcValue = (String)crc.get();
            metadata.put(CRC32C_METADATA_KEY, crcValue);
            BlobInfo blobInfo = BlobInfo.newBuilder((BlobId)blobId).setMetadata(metadata).setCrc32c(crcValue).build();
            log.debug("Uploading object to gs://{}/{} with crc {}, with throttling: {}", new Object[]{this.bucket, key, crcValue, throttlerOpt.isPresent()});
            this.doFileWrite(file, this.storage.writer(blobInfo, new Storage.BlobWriteOption[]{Storage.BlobWriteOption.doesNotExist(), Storage.BlobWriteOption.crc32cMatch()}), throttlerOpt);
            metadata.remove(CRC32C_METADATA_KEY);
        } else {
            BlobInfo blobInfo = BlobInfo.newBuilder((BlobId)blobId).setMetadata(metadata).build();
            log.debug("Uploading object to gs://{}/{}, with throttling: {}", new Object[]{this.bucket, key, throttlerOpt.isPresent()});
            this.doFileWrite(file, this.storage.writer(blobInfo, new Storage.BlobWriteOption[]{Storage.BlobWriteOption.doesNotExist()}), throttlerOpt);
        }
    }

    private void putFileEncrypted(String key, Map<String, String> metadata, File file, OpaqueData opaqueData, E2EChecksumProtectedObjectType objectType, Optional<Throttler> throttlerOpt) throws IOException {
        if (opaqueData == null || opaqueData.isEmpty()) {
            throw new TierObjectStoreFatalException("Encryption was enabled but no valid OpaqueData object was provided");
        }
        KeySha keySha = KeySha.fromRawBytes(opaqueData.intoByteArray());
        KeyContext keyContext = this.encryptionKeyManager.keyContext(keySha);
        if (keyContext == null) {
            throw new TierObjectStoreFatalException(String.format("No valid KeyContext for KeySha '%s'", keySha));
        }
        metadata.putAll(keyContext.metadata);
        Storage.BlobWriteOption encryption = Storage.BlobWriteOption.encryptionKey((String)keyContext.cleartextDataKey.base64Encoded());
        BlobId blobId = BlobId.of((String)this.bucket, (String)key);
        Optional crc = Optional.empty();
        if (this.checksumStoreOpt.isPresent() && this.checksumStoreOpt.get().checksumProtectionEnabled(objectType)) {
            crc = E2EChecksumUtils.getBase64CrcFromStore((E2EChecksumStore)this.checksumStoreOpt.get(), (File)file, metadata);
        }
        if (crc.isPresent()) {
            String crcValue = (String)crc.get();
            metadata.put(CRC32C_METADATA_KEY, crcValue);
            BlobInfo blobInfo = BlobInfo.newBuilder((BlobId)blobId).setMetadata(metadata).setCrc32c(crcValue).build();
            log.debug("Uploading encrypted object to gs://{}/{} with KeySha {} , crc {}, with throttling: {}", new Object[]{this.bucket, key, keyContext.keySha, crcValue, throttlerOpt.isPresent()});
            this.doFileWrite(file, this.storage.writer(blobInfo, new Storage.BlobWriteOption[]{encryption, Storage.BlobWriteOption.doesNotExist(), Storage.BlobWriteOption.crc32cMatch()}), throttlerOpt);
            metadata.remove(CRC32C_METADATA_KEY);
        } else {
            BlobInfo blobInfo = BlobInfo.newBuilder((BlobId)blobId).setMetadata(metadata).build();
            log.debug("Uploading encrypted object to gs://{}/{} with KeySha {}, with throttling: {}", new Object[]{this.bucket, key, keyContext.keySha, throttlerOpt.isPresent()});
            this.doFileWrite(file, this.storage.writer(blobInfo, new Storage.BlobWriteOption[]{encryption, Storage.BlobWriteOption.doesNotExist()}), throttlerOpt);
        }
    }

    private void doFileWrite(File file, WriteChannel channel, Optional<Throttler> throttlerOpt) throws IOException {
        try (WriteChannel writer = channel;
             FileChannel fileChannel = FileChannel.open(file.toPath(), StandardOpenOption.READ);){
            long n;
            if (this.writeChunkSize > 0) {
                writer.setChunkSize(this.writeChunkSize);
            }
            long fileLength = file.length();
            for (long position = 0L; position < fileLength; position += n) {
                n = fileChannel.transferTo(position, fileLength, (WritableByteChannel)writer);
                throttlerOpt.ifPresent(throttler -> {
                    if (n > 0L) {
                        throttler.maybeThrottle((double)n);
                    }
                });
            }
        }
    }

    public void putBuf(String key, Map<String, String> metadata, ByteBuffer buf) throws IOException {
        ByteBuffer dupBuf = buf.duplicate();
        BlobId blobId = BlobId.of((String)this.bucket, (String)key);
        BlobInfo blobInfo = BlobInfo.newBuilder((BlobId)blobId).setMetadata(metadata).setCrc32c(E2EChecksumUtils.compute32BitBase64Crc32c((ByteBuffer)buf)).build();
        log.debug("Uploading object {}", (Object)key);
        try (WriteChannel writer = this.storage.writer(blobInfo, new Storage.BlobWriteOption[]{Storage.BlobWriteOption.crc32cMatch()});){
            if (this.writeChunkSize > 0) {
                writer.setChunkSize(this.writeChunkSize);
            }
            while (dupBuf.hasRemaining()) {
                writer.write(dupBuf);
            }
        }
    }

    private static EncryptionKeyManager encryptionKeyManager(GcsTierObjectStoreConfig config, Time time, Metrics metrics) {
        if (config.gcsSseCustomerEncryptionKey != null && !config.gcsSseCustomerEncryptionKey.isEmpty()) {
            String uri = config.gcsSseCustomerEncryptionKey;
            String prefix = "gcp-kms://";
            if (!uri.startsWith(prefix)) {
                uri = prefix + uri;
            }
            log.info(String.format("Configuring EncryptionKeyManager using KMS key '%s'", uri));
            try {
                GcpKmsClient.register(Optional.of(uri), config.gcsCredFilePath);
                AeadConfig.register();
                KeysetHandle keysetHandle = KeysetHandle.generateNew((KeyTemplate)KmsAeadKeyManager.createKeyTemplate((String)uri));
                Aead masterKey = (Aead)keysetHandle.getPrimitive(Aead.class);
                return new EncryptionKeyManager(time, metrics, masterKey, config.encryptionKeyManagerKeyRotationInterval);
            }
            catch (GeneralSecurityException e) {
                throw new TierObjectStoreFatalException("Could not construct master key AEAD", e);
            }
        }
        return null;
    }

    private static Storage storage(GcsTierObjectStoreConfig config) {
        if (config.gcsCredFilePath.isPresent()) {
            try {
                GoogleCredentials credentials = GoogleCredentials.fromStream((InputStream)new FileInputStream(config.gcsCredFilePath.get())).createScoped((Collection)Lists.newArrayList((Object[])new String[]{"https://www.googleapis.com/auth/cloud-platform"}));
                return (Storage)((StorageOptions.Builder)StorageOptions.newBuilder().setCredentials((Credentials)credentials)).build().getService();
            }
            catch (IOException e) {
                throw new TierObjectStoreFatalException("Error in opening GCS credentials file", e);
            }
        }
        return (Storage)StorageOptions.getDefaultInstance().getService();
    }

    private void expectBucket(String bucket, String expectedRegion) throws TierObjectStoreFatalException {
        Bucket bucketObj;
        try {
            bucketObj = this.storage.get(bucket, new Storage.BucketGetOption[]{Storage.BucketGetOption.fields((Storage.BucketField[])new Storage.BucketField[]{Storage.BucketField.LOCATION})});
        }
        catch (StorageException e) {
            throw new TierObjectStoreFatalException("Unable to access bucket " + bucket, e);
        }
        if (bucketObj == null) {
            throw new TierObjectStoreFatalException("Configured bucket " + bucket + " does not exist or could not be found");
        }
        String actualRegion = bucketObj.getLocation();
        if (!expectedRegion.equalsIgnoreCase(actualRegion)) {
            log.warn("Bucket region {} does not match expected region {}", (Object)actualRegion, (Object)expectedRegion);
        }
    }

    private String lastActiveKeyPath() {
        return this.prefix + TierObjectStore.DataTypePathPrefix.LAST_ACTIVE_ENCRYPTION_KEY.prefix() + "last-active-key";
    }

    private String keyPath(ObjectStoreMetadata objectMetadata, ObjectType objectType) {
        return TierObjectStoreUtils.keyPath(this.prefix, objectMetadata, objectType);
    }

    private static class GcsTierObjectStoreResponse
    implements TierObjectStoreResponse {
        private final InputStream inputStream;

        GcsTierObjectStoreResponse(ReadChannel channel, long startOffset, OptionalInt chunkSizeOpt) throws IOException {
            int chunkSize = chunkSizeOpt.orElse(1000000);
            channel.seek(startOffset);
            channel.setChunkSize(chunkSize);
            this.inputStream = Channels.newInputStream((ReadableByteChannel)channel);
        }

        @Override
        public void close() throws IOException {
            this.inputStream.close();
        }

        @Override
        public InputStream getInputStream() {
            return this.inputStream;
        }
    }

    private class EncryptionKeyManagerHook
    implements EncryptionKeyManager.WellKnownKeypathHook {
        private EncryptionKeyManagerHook() {
        }

        @Override
        public void writeWellKnownPathMetadata(Map<String, String> metadata) {
            String path = GcsTierObjectStore.this.lastActiveKeyPath();
            log.info("Uploading newly generated key to path {}", (Object)path);
            BlobId blobId = BlobId.of((String)GcsTierObjectStore.this.bucket, (String)path);
            BlobInfo blobInfo = BlobInfo.newBuilder((BlobId)blobId).setMetadata(metadata).build();
            GcsTierObjectStore.this.storage.create(blobInfo, new Storage.BlobTargetOption[0]);
        }

        @Override
        public Map<String, String> fetchWellKnownPathMetadata() {
            String path = GcsTierObjectStore.this.lastActiveKeyPath();
            log.info("Downloading previously generated key from path {}", (Object)path);
            BlobId blobId = BlobId.of((String)GcsTierObjectStore.this.bucket, (String)path);
            Blob blob = GcsTierObjectStore.this.storage.get(blobId, new Storage.BlobGetOption[]{Storage.BlobGetOption.fields((Storage.BlobField[])new Storage.BlobField[]{Storage.BlobField.METADATA})});
            if (blob != null) {
                return blob.getMetadata();
            }
            return new HashMap<String, String>();
        }
    }
}

