/*
 * $Id$
 */

package org.apache.maven.settings.io.xpp3;

  //---------------------------------/
 //- Imported classes and packages -/
//---------------------------------/

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.text.DateFormat;
import java.util.Locale;
import org.apache.maven.settings.Activation;
import org.apache.maven.settings.ActivationFile;
import org.apache.maven.settings.ActivationOS;
import org.apache.maven.settings.ActivationProperty;
import org.apache.maven.settings.IdentifiableBase;
import org.apache.maven.settings.Mirror;
import org.apache.maven.settings.Profile;
import org.apache.maven.settings.Proxy;
import org.apache.maven.settings.Repository;
import org.apache.maven.settings.RepositoryBase;
import org.apache.maven.settings.RepositoryPolicy;
import org.apache.maven.settings.Server;
import org.apache.maven.settings.Settings;
import org.apache.maven.settings.TrackableBase;
import org.codehaus.plexus.util.ReaderFactory;
import org.codehaus.plexus.util.xml.Xpp3DomBuilder;
import org.codehaus.plexus.util.xml.pull.MXParser;
import org.codehaus.plexus.util.xml.pull.XmlPullParser;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;

/**
 * Class SettingsXpp3Reader.
 * 
 * @version $Revision$ $Date$
 */
public class SettingsXpp3Reader {


      //--------------------------/
     //- Class/Member Variables -/
    //--------------------------/

    /**
     * If set the parser will be loaded with all single characters
     * from the XHTML specification.
     * The entities used:
     * <ul>
     * <li>http://www.w3.org/TR/xhtml1/DTD/xhtml-lat1.ent</li>
     * <li>http://www.w3.org/TR/xhtml1/DTD/xhtml-special.ent</li>
     * <li>http://www.w3.org/TR/xhtml1/DTD/xhtml-symbol.ent</li>
     * </ul>
     */
    private boolean addDefaultEntities = true;


      //-----------/
     //- Methods -/
    //-----------/

    /**
     * Returns the state of the "add default entities" flag.
     * 
     * @return boolean
     */
    public boolean getAddDefaultEntities()
    {
        return addDefaultEntities;
    } //-- boolean getAddDefaultEntities() 

    /**
     * Method getBooleanValue.
     * 
     * @param s
     * @param parser
     * @param attribute
     * @throws XmlPullParserException
     * @return boolean
     */
    public boolean getBooleanValue( String s, String attribute, XmlPullParser parser )
        throws XmlPullParserException
    {
        return getBooleanValue( s, attribute, parser, null );
    } //-- boolean getBooleanValue( String, String, XmlPullParser ) 

    /**
     * Method getBooleanValue.
     * 
     * @param s
     * @param defaultValue
     * @param parser
     * @param attribute
     * @throws XmlPullParserException
     * @return boolean
     */
    public boolean getBooleanValue( String s, String attribute, XmlPullParser parser, String defaultValue )
        throws XmlPullParserException
    {
        if ( s != null && s.length() != 0 )
        {
            return Boolean.valueOf( s ).booleanValue();
        }
        if ( defaultValue != null )
        {
            return Boolean.valueOf( defaultValue ).booleanValue();
        }
        return false;
    } //-- boolean getBooleanValue( String, String, XmlPullParser, String ) 

    /**
     * Method getByteValue.
     * 
     * @param s
     * @param strict
     * @param parser
     * @param attribute
     * @throws XmlPullParserException
     * @return byte
     */
    public byte getByteValue( String s, String attribute, XmlPullParser parser, boolean strict )
        throws XmlPullParserException
    {
        if ( s != null )
        {
            try
            {
                return Byte.valueOf( s ).byteValue();
            }
            catch ( NumberFormatException e )
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unable to parse element '" + attribute + "', must be a byte", parser, null );
                }
            }
        }
        return 0;
    } //-- byte getByteValue( String, String, XmlPullParser, boolean ) 

    /**
     * Method getCharacterValue.
     * 
     * @param s
     * @param parser
     * @param attribute
     * @throws XmlPullParserException
     * @return char
     */
    public char getCharacterValue( String s, String attribute, XmlPullParser parser )
        throws XmlPullParserException
    {
        if ( s != null )
        {
            return s.charAt( 0 );
        }
        return 0;
    } //-- char getCharacterValue( String, String, XmlPullParser ) 

    /**
     * Method getDateValue.
     * 
     * @param s
     * @param parser
     * @param attribute
     * @throws XmlPullParserException
     * @return java.util.Date
     */
    public java.util.Date getDateValue( String s, String attribute, XmlPullParser parser )
        throws XmlPullParserException
    {
        return getDateValue( s, attribute, null, parser );
    } //-- java.util.Date getDateValue( String, String, XmlPullParser ) 

    /**
     * Method getDateValue.
     * 
     * @param s
     * @param parser
     * @param dateFormat
     * @param attribute
     * @throws XmlPullParserException
     * @return java.util.Date
     */
    public java.util.Date getDateValue( String s, String attribute, String dateFormat, XmlPullParser parser )
        throws XmlPullParserException
    {
        if ( s != null )
        {
            if ( dateFormat == null )
            {
                return new java.util.Date( Long.valueOf( s ).longValue() );
            }
            else
            {
                DateFormat dateParser = new java.text.SimpleDateFormat( dateFormat, Locale.US );
                try
                {
                    return dateParser.parse( s );
                }
                catch ( java.text.ParseException e )
                {
                    throw new XmlPullParserException( e.getMessage() );
                }
            }
        }
        return null;
    } //-- java.util.Date getDateValue( String, String, String, XmlPullParser ) 

    /**
     * Method getDoubleValue.
     * 
     * @param s
     * @param strict
     * @param parser
     * @param attribute
     * @throws XmlPullParserException
     * @return double
     */
    public double getDoubleValue( String s, String attribute, XmlPullParser parser, boolean strict )
        throws XmlPullParserException
    {
        if ( s != null )
        {
            try
            {
                return Double.valueOf( s ).doubleValue();
            }
            catch ( NumberFormatException e )
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unable to parse element '" + attribute + "', must be a floating point number", parser, null );
                }
            }
        }
        return 0;
    } //-- double getDoubleValue( String, String, XmlPullParser, boolean ) 

    /**
     * Method getFloatValue.
     * 
     * @param s
     * @param strict
     * @param parser
     * @param attribute
     * @throws XmlPullParserException
     * @return float
     */
    public float getFloatValue( String s, String attribute, XmlPullParser parser, boolean strict )
        throws XmlPullParserException
    {
        if ( s != null )
        {
            try
            {
                return Float.valueOf( s ).floatValue();
            }
            catch ( NumberFormatException e )
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unable to parse element '" + attribute + "', must be a floating point number", parser, null );
                }
            }
        }
        return 0;
    } //-- float getFloatValue( String, String, XmlPullParser, boolean ) 

    /**
     * Method getIntegerValue.
     * 
     * @param s
     * @param strict
     * @param parser
     * @param attribute
     * @throws XmlPullParserException
     * @return int
     */
    public int getIntegerValue( String s, String attribute, XmlPullParser parser, boolean strict )
        throws XmlPullParserException
    {
        if ( s != null )
        {
            try
            {
                return Integer.valueOf( s ).intValue();
            }
            catch ( NumberFormatException e )
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unable to parse element '" + attribute + "', must be an integer", parser, null );
                }
            }
        }
        return 0;
    } //-- int getIntegerValue( String, String, XmlPullParser, boolean ) 

    /**
     * Method getLongValue.
     * 
     * @param s
     * @param strict
     * @param parser
     * @param attribute
     * @throws XmlPullParserException
     * @return long
     */
    public long getLongValue( String s, String attribute, XmlPullParser parser, boolean strict )
        throws XmlPullParserException
    {
        if ( s != null )
        {
            try
            {
                return Long.valueOf( s ).longValue();
            }
            catch ( NumberFormatException e )
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unable to parse element '" + attribute + "', must be a long integer", parser, null );
                }
            }
        }
        return 0;
    } //-- long getLongValue( String, String, XmlPullParser, boolean ) 

    /**
     * Method getRequiredAttributeValue.
     * 
     * @param s
     * @param strict
     * @param parser
     * @param attribute
     * @throws XmlPullParserException
     * @return String
     */
    public String getRequiredAttributeValue( String s, String attribute, XmlPullParser parser, boolean strict )
        throws XmlPullParserException
    {
        if ( s == null )
        {
            if ( strict )
            {
                throw new XmlPullParserException( "Missing required value for attribute '" + attribute + "'", parser, null );
            }
        }
        return s;
    } //-- String getRequiredAttributeValue( String, String, XmlPullParser, boolean ) 

    /**
     * Method getShortValue.
     * 
     * @param s
     * @param strict
     * @param parser
     * @param attribute
     * @throws XmlPullParserException
     * @return short
     */
    public short getShortValue( String s, String attribute, XmlPullParser parser, boolean strict )
        throws XmlPullParserException
    {
        if ( s != null )
        {
            try
            {
                return Short.valueOf( s ).shortValue();
            }
            catch ( NumberFormatException e )
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unable to parse element '" + attribute + "', must be a short integer", parser, null );
                }
            }
        }
        return 0;
    } //-- short getShortValue( String, String, XmlPullParser, boolean ) 

    /**
     * Method getTrimmedValue.
     * 
     * @param s
     * @return String
     */
    public String getTrimmedValue( String s )
    {
        if ( s != null )
        {
            s = s.trim();
        }
        return s;
    } //-- String getTrimmedValue( String ) 

    /**
     * Method parseActivation.
     * 
     * @param tagName
     * @param strict
     * @param parser
     * @throws IOException
     * @throws XmlPullParserException
     * @return Activation
     */
    private Activation parseActivation( String tagName, XmlPullParser parser, boolean strict )
        throws IOException, XmlPullParserException
    {
        Activation activation = new Activation();
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "activeByDefault" )  )
            {
                if ( parsed.contains( "activeByDefault" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "activeByDefault" );
                activation.setActiveByDefault( getBooleanValue( getTrimmedValue( parser.nextText()), "activeByDefault", parser, "false" ) );
            }
            else if ( parser.getName().equals( "jdk" )  )
            {
                if ( parsed.contains( "jdk" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "jdk" );
                activation.setJdk( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "os" )  )
            {
                if ( parsed.contains( "os" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "os" );
                activation.setOs( parseActivationOS( "os", parser, strict ) );
            }
            else if ( parser.getName().equals( "property" )  )
            {
                if ( parsed.contains( "property" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "property" );
                activation.setProperty( parseActivationProperty( "property", parser, strict ) );
            }
            else if ( parser.getName().equals( "file" )  )
            {
                if ( parsed.contains( "file" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "file" );
                activation.setFile( parseActivationFile( "file", parser, strict ) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null );
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return activation;
    } //-- Activation parseActivation( String, XmlPullParser, boolean ) 

    /**
     * Method parseActivationFile.
     * 
     * @param tagName
     * @param strict
     * @param parser
     * @throws IOException
     * @throws XmlPullParserException
     * @return ActivationFile
     */
    private ActivationFile parseActivationFile( String tagName, XmlPullParser parser, boolean strict )
        throws IOException, XmlPullParserException
    {
        ActivationFile activationFile = new ActivationFile();
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "missing" )  )
            {
                if ( parsed.contains( "missing" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "missing" );
                activationFile.setMissing( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "exists" )  )
            {
                if ( parsed.contains( "exists" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "exists" );
                activationFile.setExists( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null );
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return activationFile;
    } //-- ActivationFile parseActivationFile( String, XmlPullParser, boolean ) 

    /**
     * Method parseActivationOS.
     * 
     * @param tagName
     * @param strict
     * @param parser
     * @throws IOException
     * @throws XmlPullParserException
     * @return ActivationOS
     */
    private ActivationOS parseActivationOS( String tagName, XmlPullParser parser, boolean strict )
        throws IOException, XmlPullParserException
    {
        ActivationOS activationOS = new ActivationOS();
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "name" );
                activationOS.setName( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "family" )  )
            {
                if ( parsed.contains( "family" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "family" );
                activationOS.setFamily( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "arch" )  )
            {
                if ( parsed.contains( "arch" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "arch" );
                activationOS.setArch( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "version" )  )
            {
                if ( parsed.contains( "version" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "version" );
                activationOS.setVersion( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null );
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return activationOS;
    } //-- ActivationOS parseActivationOS( String, XmlPullParser, boolean ) 

    /**
     * Method parseActivationProperty.
     * 
     * @param tagName
     * @param strict
     * @param parser
     * @throws IOException
     * @throws XmlPullParserException
     * @return ActivationProperty
     */
    private ActivationProperty parseActivationProperty( String tagName, XmlPullParser parser, boolean strict )
        throws IOException, XmlPullParserException
    {
        ActivationProperty activationProperty = new ActivationProperty();
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "name" );
                activationProperty.setName( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "value" )  )
            {
                if ( parsed.contains( "value" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "value" );
                activationProperty.setValue( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null );
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return activationProperty;
    } //-- ActivationProperty parseActivationProperty( String, XmlPullParser, boolean ) 

    /**
     * Method parseIdentifiableBase.
     * 
     * @param tagName
     * @param strict
     * @param parser
     * @throws IOException
     * @throws XmlPullParserException
     * @return IdentifiableBase
     */
    private IdentifiableBase parseIdentifiableBase( String tagName, XmlPullParser parser, boolean strict )
        throws IOException, XmlPullParserException
    {
        IdentifiableBase identifiableBase = new IdentifiableBase();
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "id" )  )
            {
                if ( parsed.contains( "id" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "id" );
                identifiableBase.setId( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null );
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return identifiableBase;
    } //-- IdentifiableBase parseIdentifiableBase( String, XmlPullParser, boolean ) 

    /**
     * Method parseMirror.
     * 
     * @param tagName
     * @param strict
     * @param parser
     * @throws IOException
     * @throws XmlPullParserException
     * @return Mirror
     */
    private Mirror parseMirror( String tagName, XmlPullParser parser, boolean strict )
        throws IOException, XmlPullParserException
    {
        Mirror mirror = new Mirror();
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "mirrorOf" )  )
            {
                if ( parsed.contains( "mirrorOf" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "mirrorOf" );
                mirror.setMirrorOf( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "name" );
                mirror.setName( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "url" )  )
            {
                if ( parsed.contains( "url" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "url" );
                mirror.setUrl( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "id" )  )
            {
                if ( parsed.contains( "id" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "id" );
                mirror.setId( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null );
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return mirror;
    } //-- Mirror parseMirror( String, XmlPullParser, boolean ) 

    /**
     * Method parseProfile.
     * 
     * @param tagName
     * @param strict
     * @param parser
     * @throws IOException
     * @throws XmlPullParserException
     * @return Profile
     */
    private Profile parseProfile( String tagName, XmlPullParser parser, boolean strict )
        throws IOException, XmlPullParserException
    {
        Profile profile = new Profile();
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "activation" )  )
            {
                if ( parsed.contains( "activation" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "activation" );
                profile.setActivation( parseActivation( "activation", parser, strict ) );
            }
            else if ( parser.getName().equals( "properties" )  )
            {
                if ( parsed.contains( "properties" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "properties" );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    String key = parser.getName();
                    String value = parser.nextText().trim();
                    profile.addProperty( key, value );
                }
            }
            else if ( parser.getName().equals( "repositories" )  )
            {
                if ( parsed.contains( "repositories" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "repositories" );
                java.util.List repositories = new java.util.ArrayList();
                profile.setRepositories( repositories );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "repository" ) )
                    {
                        repositories.add( parseRepository( "repository", parser, strict ) );
                    }
                    else if ( strict )
                    {
                        throw new XmlPullParserException( "Unrecognised association: '" + parser.getName() + "'", parser, null );
                    }
                    else
                    {
                        // swallow up to end tag since this is not valid
                        while ( parser.next() != XmlPullParser.END_TAG ) {}
                    }
                }
            }
            else if ( parser.getName().equals( "pluginRepositories" )  )
            {
                if ( parsed.contains( "pluginRepositories" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "pluginRepositories" );
                java.util.List pluginRepositories = new java.util.ArrayList();
                profile.setPluginRepositories( pluginRepositories );
                while ( parser.nextTag() == XmlPullParser.START_TAG )
                {
                    if ( parser.getName().equals( "pluginRepository" ) )
                    {
                        pluginRepositories.add( parseRepository( "pluginRepository", parser, strict ) );
                    }
                    else if ( strict )
                    {
                        throw new XmlPullParserException( "Unrecognised association: '" + parser.getName() + "'", parser, null );
                    }
                    else
                    {
                        // swallow up to end tag since this is not valid
                        while ( parser.next() != XmlPullParser.END_TAG ) {}
                    }
                }
            }
            else if ( parser.getName().equals( "id" )  )
            {
                if ( parsed.contains( "id" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "id" );
                profile.setId( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null );
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return profile;
    } //-- Profile parseProfile( String, XmlPullParser, boolean ) 

    /**
     * Method parseProxy.
     * 
     * @param tagName
     * @param strict
     * @param parser
     * @throws IOException
     * @throws XmlPullParserException
     * @return Proxy
     */
    private Proxy parseProxy( String tagName, XmlPullParser parser, boolean strict )
        throws IOException, XmlPullParserException
    {
        Proxy proxy = new Proxy();
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "active" )  )
            {
                if ( parsed.contains( "active" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "active" );
                proxy.setActive( getBooleanValue( getTrimmedValue( parser.nextText()), "active", parser, "true" ) );
            }
            else if ( parser.getName().equals( "protocol" )  )
            {
                if ( parsed.contains( "protocol" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "protocol" );
                proxy.setProtocol( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "username" )  )
            {
                if ( parsed.contains( "username" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "username" );
                proxy.setUsername( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "password" )  )
            {
                if ( parsed.contains( "password" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "password" );
                proxy.setPassword( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "port" )  )
            {
                if ( parsed.contains( "port" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "port" );
                proxy.setPort( getIntegerValue( getTrimmedValue( parser.nextText()), "port", parser, strict ) );
            }
            else if ( parser.getName().equals( "host" )  )
            {
                if ( parsed.contains( "host" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "host" );
                proxy.setHost( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "nonProxyHosts" )  )
            {
                if ( parsed.contains( "nonProxyHosts" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "nonProxyHosts" );
                proxy.setNonProxyHosts( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "id" )  )
            {
                if ( parsed.contains( "id" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "id" );
                proxy.setId( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null );
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return proxy;
    } //-- Proxy parseProxy( String, XmlPullParser, boolean ) 

    /**
     * Method parseRepository.
     * 
     * @param tagName
     * @param strict
     * @param parser
     * @throws IOException
     * @throws XmlPullParserException
     * @return Repository
     */
    private Repository parseRepository( String tagName, XmlPullParser parser, boolean strict )
        throws IOException, XmlPullParserException
    {
        Repository repository = new Repository();
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "releases" )  )
            {
                if ( parsed.contains( "releases" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "releases" );
                repository.setReleases( parseRepositoryPolicy( "releases", parser, strict ) );
            }
            else if ( parser.getName().equals( "snapshots" )  )
            {
                if ( parsed.contains( "snapshots" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "snapshots" );
                repository.setSnapshots( parseRepositoryPolicy( "snapshots", parser, strict ) );
            }
            else if ( parser.getName().equals( "id" )  )
            {
                if ( parsed.contains( "id" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "id" );
                repository.setId( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "name" );
                repository.setName( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "url" )  )
            {
                if ( parsed.contains( "url" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "url" );
                repository.setUrl( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "layout" )  )
            {
                if ( parsed.contains( "layout" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "layout" );
                repository.setLayout( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null );
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return repository;
    } //-- Repository parseRepository( String, XmlPullParser, boolean ) 

    /**
     * Method parseRepositoryBase.
     * 
     * @param tagName
     * @param strict
     * @param parser
     * @throws IOException
     * @throws XmlPullParserException
     * @return RepositoryBase
     */
    private RepositoryBase parseRepositoryBase( String tagName, XmlPullParser parser, boolean strict )
        throws IOException, XmlPullParserException
    {
        RepositoryBase repositoryBase = new RepositoryBase();
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "id" )  )
            {
                if ( parsed.contains( "id" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "id" );
                repositoryBase.setId( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "name" )  )
            {
                if ( parsed.contains( "name" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "name" );
                repositoryBase.setName( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "url" )  )
            {
                if ( parsed.contains( "url" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "url" );
                repositoryBase.setUrl( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "layout" )  )
            {
                if ( parsed.contains( "layout" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "layout" );
                repositoryBase.setLayout( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null );
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return repositoryBase;
    } //-- RepositoryBase parseRepositoryBase( String, XmlPullParser, boolean ) 

    /**
     * Method parseRepositoryPolicy.
     * 
     * @param tagName
     * @param strict
     * @param parser
     * @throws IOException
     * @throws XmlPullParserException
     * @return RepositoryPolicy
     */
    private RepositoryPolicy parseRepositoryPolicy( String tagName, XmlPullParser parser, boolean strict )
        throws IOException, XmlPullParserException
    {
        RepositoryPolicy repositoryPolicy = new RepositoryPolicy();
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "enabled" )  )
            {
                if ( parsed.contains( "enabled" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "enabled" );
                repositoryPolicy.setEnabled( getBooleanValue( getTrimmedValue( parser.nextText()), "enabled", parser, "true" ) );
            }
            else if ( parser.getName().equals( "updatePolicy" )  )
            {
                if ( parsed.contains( "updatePolicy" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "updatePolicy" );
                repositoryPolicy.setUpdatePolicy( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "checksumPolicy" )  )
            {
                if ( parsed.contains( "checksumPolicy" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "checksumPolicy" );
                repositoryPolicy.setChecksumPolicy( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null );
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return repositoryPolicy;
    } //-- RepositoryPolicy parseRepositoryPolicy( String, XmlPullParser, boolean ) 

    /**
     * Method parseServer.
     * 
     * @param tagName
     * @param strict
     * @param parser
     * @throws IOException
     * @throws XmlPullParserException
     * @return Server
     */
    private Server parseServer( String tagName, XmlPullParser parser, boolean strict )
        throws IOException, XmlPullParserException
    {
        Server server = new Server();
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( parser.getName().equals( "username" )  )
            {
                if ( parsed.contains( "username" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "username" );
                server.setUsername( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "password" )  )
            {
                if ( parsed.contains( "password" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "password" );
                server.setPassword( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "privateKey" )  )
            {
                if ( parsed.contains( "privateKey" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "privateKey" );
                server.setPrivateKey( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "passphrase" )  )
            {
                if ( parsed.contains( "passphrase" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "passphrase" );
                server.setPassphrase( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "filePermissions" )  )
            {
                if ( parsed.contains( "filePermissions" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "filePermissions" );
                server.setFilePermissions( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "directoryPermissions" )  )
            {
                if ( parsed.contains( "directoryPermissions" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "directoryPermissions" );
                server.setDirectoryPermissions( getTrimmedValue( parser.nextText()) );
            }
            else if ( parser.getName().equals( "configuration" )  )
            {
                if ( parsed.contains( "configuration" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "configuration" );
                server.setConfiguration( Xpp3DomBuilder.build( parser ) );
            }
            else if ( parser.getName().equals( "id" )  )
            {
                if ( parsed.contains( "id" ) )
                {
                    throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                }
                parsed.add( "id" );
                server.setId( getTrimmedValue( parser.nextText()) );
            }
            else
            {
                if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null );
                }
                else
                {
                    // swallow up to end tag since this is not valid
                    while ( parser.next() != XmlPullParser.END_TAG ) {}
                }
            }
        }
        return server;
    } //-- Server parseServer( String, XmlPullParser, boolean ) 

    /**
     * Method parseSettings.
     * 
     * @param tagName
     * @param strict
     * @param parser
     * @throws IOException
     * @throws XmlPullParserException
     * @return Settings
     */
    private Settings parseSettings( String tagName, XmlPullParser parser, boolean strict )
        throws IOException, XmlPullParserException
    {
        Settings settings = new Settings();
        java.util.Set parsed = new java.util.HashSet();
        int eventType = parser.getEventType();
        boolean foundRoot = false;
        while ( eventType != XmlPullParser.END_DOCUMENT )
        {
            if ( eventType == XmlPullParser.START_TAG )
            {
                if ( parser.getName().equals( tagName ) )
                {
                    foundRoot = true;
                }
                else if ( parser.getName().equals( "localRepository" )  )
                {
                    if ( parsed.contains( "localRepository" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                    }
                    parsed.add( "localRepository" );
                    settings.setLocalRepository( getTrimmedValue( parser.nextText()) );
                }
                else if ( parser.getName().equals( "interactiveMode" )  )
                {
                    if ( parsed.contains( "interactiveMode" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                    }
                    parsed.add( "interactiveMode" );
                    settings.setInteractiveMode( getBooleanValue( getTrimmedValue( parser.nextText()), "interactiveMode", parser, "true" ) );
                }
                else if ( parser.getName().equals( "usePluginRegistry" )  )
                {
                    if ( parsed.contains( "usePluginRegistry" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                    }
                    parsed.add( "usePluginRegistry" );
                    settings.setUsePluginRegistry( getBooleanValue( getTrimmedValue( parser.nextText()), "usePluginRegistry", parser, "false" ) );
                }
                else if ( parser.getName().equals( "offline" )  )
                {
                    if ( parsed.contains( "offline" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                    }
                    parsed.add( "offline" );
                    settings.setOffline( getBooleanValue( getTrimmedValue( parser.nextText()), "offline", parser, "false" ) );
                }
                else if ( parser.getName().equals( "proxies" )  )
                {
                    if ( parsed.contains( "proxies" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                    }
                    parsed.add( "proxies" );
                    java.util.List proxies = new java.util.ArrayList();
                    settings.setProxies( proxies );
                    while ( parser.nextTag() == XmlPullParser.START_TAG )
                    {
                        if ( parser.getName().equals( "proxy" ) )
                        {
                            proxies.add( parseProxy( "proxy", parser, strict ) );
                        }
                        else if ( strict )
                        {
                            throw new XmlPullParserException( "Unrecognised association: '" + parser.getName() + "'", parser, null );
                        }
                        else
                        {
                            // swallow up to end tag since this is not valid
                            while ( parser.next() != XmlPullParser.END_TAG ) {}
                        }
                    }
                }
                else if ( parser.getName().equals( "servers" )  )
                {
                    if ( parsed.contains( "servers" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                    }
                    parsed.add( "servers" );
                    java.util.List servers = new java.util.ArrayList();
                    settings.setServers( servers );
                    while ( parser.nextTag() == XmlPullParser.START_TAG )
                    {
                        if ( parser.getName().equals( "server" ) )
                        {
                            servers.add( parseServer( "server", parser, strict ) );
                        }
                        else if ( strict )
                        {
                            throw new XmlPullParserException( "Unrecognised association: '" + parser.getName() + "'", parser, null );
                        }
                        else
                        {
                            // swallow up to end tag since this is not valid
                            while ( parser.next() != XmlPullParser.END_TAG ) {}
                        }
                    }
                }
                else if ( parser.getName().equals( "mirrors" )  )
                {
                    if ( parsed.contains( "mirrors" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                    }
                    parsed.add( "mirrors" );
                    java.util.List mirrors = new java.util.ArrayList();
                    settings.setMirrors( mirrors );
                    while ( parser.nextTag() == XmlPullParser.START_TAG )
                    {
                        if ( parser.getName().equals( "mirror" ) )
                        {
                            mirrors.add( parseMirror( "mirror", parser, strict ) );
                        }
                        else if ( strict )
                        {
                            throw new XmlPullParserException( "Unrecognised association: '" + parser.getName() + "'", parser, null );
                        }
                        else
                        {
                            // swallow up to end tag since this is not valid
                            while ( parser.next() != XmlPullParser.END_TAG ) {}
                        }
                    }
                }
                else if ( parser.getName().equals( "profiles" )  )
                {
                    if ( parsed.contains( "profiles" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                    }
                    parsed.add( "profiles" );
                    java.util.List profiles = new java.util.ArrayList();
                    settings.setProfiles( profiles );
                    while ( parser.nextTag() == XmlPullParser.START_TAG )
                    {
                        if ( parser.getName().equals( "profile" ) )
                        {
                            profiles.add( parseProfile( "profile", parser, strict ) );
                        }
                        else if ( strict )
                        {
                            throw new XmlPullParserException( "Unrecognised association: '" + parser.getName() + "'", parser, null );
                        }
                        else
                        {
                            // swallow up to end tag since this is not valid
                            while ( parser.next() != XmlPullParser.END_TAG ) {}
                        }
                    }
                }
                else if ( parser.getName().equals( "activeProfiles" )  )
                {
                    if ( parsed.contains( "activeProfiles" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                    }
                    parsed.add( "activeProfiles" );
                    java.util.List activeProfiles = new java.util.ArrayList();
                    settings.setActiveProfiles( activeProfiles );
                    while ( parser.nextTag() == XmlPullParser.START_TAG )
                    {
                        if ( parser.getName().equals( "activeProfile" ) )
                        {
                            activeProfiles.add( getTrimmedValue( parser.nextText()) );
                        }
                        else if ( strict )
                        {
                            throw new XmlPullParserException( "Unrecognised association: '" + parser.getName() + "'", parser, null );
                        }
                        else
                        {
                            // swallow up to end tag since this is not valid
                            while ( parser.next() != XmlPullParser.END_TAG ) {}
                        }
                    }
                }
                else if ( parser.getName().equals( "pluginGroups" )  )
                {
                    if ( parsed.contains( "pluginGroups" ) )
                    {
                        throw new XmlPullParserException( "Duplicated tag: '" + parser.getName() + "'", parser, null );
                    }
                    parsed.add( "pluginGroups" );
                    java.util.List pluginGroups = new java.util.ArrayList();
                    settings.setPluginGroups( pluginGroups );
                    while ( parser.nextTag() == XmlPullParser.START_TAG )
                    {
                        if ( parser.getName().equals( "pluginGroup" ) )
                        {
                            pluginGroups.add( getTrimmedValue( parser.nextText()) );
                        }
                        else if ( strict )
                        {
                            throw new XmlPullParserException( "Unrecognised association: '" + parser.getName() + "'", parser, null );
                        }
                        else
                        {
                            // swallow up to end tag since this is not valid
                            while ( parser.next() != XmlPullParser.END_TAG ) {}
                        }
                    }
                }
                else if ( strict )
                {
                    throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null );
                }
            }
            eventType = parser.next();
        }
        return settings;
    } //-- Settings parseSettings( String, XmlPullParser, boolean ) 

    /**
     * Method parseTrackableBase.
     * 
     * @param tagName
     * @param strict
     * @param parser
     * @throws IOException
     * @throws XmlPullParserException
     * @return TrackableBase
     */
    private TrackableBase parseTrackableBase( String tagName, XmlPullParser parser, boolean strict )
        throws IOException, XmlPullParserException
    {
        TrackableBase trackableBase = new TrackableBase();
        java.util.Set parsed = new java.util.HashSet();
        while ( parser.nextTag() == XmlPullParser.START_TAG )
        {
            if ( strict )
            {
                throw new XmlPullParserException( "Unrecognised tag: '" + parser.getName() + "'", parser, null );
            }
            else
            {
                // swallow up to end tag since this is not valid
                while ( parser.next() != XmlPullParser.END_TAG ) {}
            }
        }
        return trackableBase;
    } //-- TrackableBase parseTrackableBase( String, XmlPullParser, boolean ) 

    /**
     * @see ReaderFactory#newXmlReader
     * 
     * @param reader
     * @param strict
     * @throws IOException
     * @throws XmlPullParserException
     * @return Settings
     */
    public Settings read( Reader reader, boolean strict )
        throws IOException, XmlPullParserException
    {
        XmlPullParser parser = new MXParser();
    
        parser.setInput( reader );
    
        if ( addDefaultEntities )
        {
            // ----------------------------------------------------------------------
            // Latin 1 entities
            // ----------------------------------------------------------------------
    
            parser.defineEntityReplacementText( "nbsp", "\u00a0" );
            parser.defineEntityReplacementText( "iexcl", "\u00a1" );
            parser.defineEntityReplacementText( "cent", "\u00a2" );
            parser.defineEntityReplacementText( "pound", "\u00a3" );
            parser.defineEntityReplacementText( "curren", "\u00a4" );
            parser.defineEntityReplacementText( "yen", "\u00a5" );
            parser.defineEntityReplacementText( "brvbar", "\u00a6" );
            parser.defineEntityReplacementText( "sect", "\u00a7" );
            parser.defineEntityReplacementText( "uml", "\u00a8" );
            parser.defineEntityReplacementText( "copy", "\u00a9" );
            parser.defineEntityReplacementText( "ordf", "\u00aa" );
            parser.defineEntityReplacementText( "laquo", "\u00ab" );
            parser.defineEntityReplacementText( "not", "\u00ac" );
            parser.defineEntityReplacementText( "shy", "\u00ad" );
            parser.defineEntityReplacementText( "reg", "\u00ae" );
            parser.defineEntityReplacementText( "macr", "\u00af" );
            parser.defineEntityReplacementText( "deg", "\u00b0" );
            parser.defineEntityReplacementText( "plusmn", "\u00b1" );
            parser.defineEntityReplacementText( "sup2", "\u00b2" );
            parser.defineEntityReplacementText( "sup3", "\u00b3" );
            parser.defineEntityReplacementText( "acute", "\u00b4" );
            parser.defineEntityReplacementText( "micro", "\u00b5" );
            parser.defineEntityReplacementText( "para", "\u00b6" );
            parser.defineEntityReplacementText( "middot", "\u00b7" );
            parser.defineEntityReplacementText( "cedil", "\u00b8" );
            parser.defineEntityReplacementText( "sup1", "\u00b9" );
            parser.defineEntityReplacementText( "ordm", "\u00ba" );
            parser.defineEntityReplacementText( "raquo", "\u00bb" );
            parser.defineEntityReplacementText( "frac14", "\u00bc" );
            parser.defineEntityReplacementText( "frac12", "\u00bd" );
            parser.defineEntityReplacementText( "frac34", "\u00be" );
            parser.defineEntityReplacementText( "iquest", "\u00bf" );
            parser.defineEntityReplacementText( "Agrave", "\u00c0" );
            parser.defineEntityReplacementText( "Aacute", "\u00c1" );
            parser.defineEntityReplacementText( "Acirc", "\u00c2" );
            parser.defineEntityReplacementText( "Atilde", "\u00c3" );
            parser.defineEntityReplacementText( "Auml", "\u00c4" );
            parser.defineEntityReplacementText( "Aring", "\u00c5" );
            parser.defineEntityReplacementText( "AElig", "\u00c6" );
            parser.defineEntityReplacementText( "Ccedil", "\u00c7" );
            parser.defineEntityReplacementText( "Egrave", "\u00c8" );
            parser.defineEntityReplacementText( "Eacute", "\u00c9" );
            parser.defineEntityReplacementText( "Ecirc", "\u00ca" );
            parser.defineEntityReplacementText( "Euml", "\u00cb" );
            parser.defineEntityReplacementText( "Igrave", "\u00cc" );
            parser.defineEntityReplacementText( "Iacute", "\u00cd" );
            parser.defineEntityReplacementText( "Icirc", "\u00ce" );
            parser.defineEntityReplacementText( "Iuml", "\u00cf" );
            parser.defineEntityReplacementText( "ETH", "\u00d0" );
            parser.defineEntityReplacementText( "Ntilde", "\u00d1" );
            parser.defineEntityReplacementText( "Ograve", "\u00d2" );
            parser.defineEntityReplacementText( "Oacute", "\u00d3" );
            parser.defineEntityReplacementText( "Ocirc", "\u00d4" );
            parser.defineEntityReplacementText( "Otilde", "\u00d5" );
            parser.defineEntityReplacementText( "Ouml", "\u00d6" );
            parser.defineEntityReplacementText( "times", "\u00d7" );
            parser.defineEntityReplacementText( "Oslash", "\u00d8" );
            parser.defineEntityReplacementText( "Ugrave", "\u00d9" );
            parser.defineEntityReplacementText( "Uacute", "\u00da" );
            parser.defineEntityReplacementText( "Ucirc", "\u00db" );
            parser.defineEntityReplacementText( "Uuml", "\u00dc" );
            parser.defineEntityReplacementText( "Yacute", "\u00dd" );
            parser.defineEntityReplacementText( "THORN", "\u00de" );
            parser.defineEntityReplacementText( "szlig", "\u00df" );
            parser.defineEntityReplacementText( "agrave", "\u00e0" );
            parser.defineEntityReplacementText( "aacute", "\u00e1" );
            parser.defineEntityReplacementText( "acirc", "\u00e2" );
            parser.defineEntityReplacementText( "atilde", "\u00e3" );
            parser.defineEntityReplacementText( "auml", "\u00e4" );
            parser.defineEntityReplacementText( "aring", "\u00e5" );
            parser.defineEntityReplacementText( "aelig", "\u00e6" );
            parser.defineEntityReplacementText( "ccedil", "\u00e7" );
            parser.defineEntityReplacementText( "egrave", "\u00e8" );
            parser.defineEntityReplacementText( "eacute", "\u00e9" );
            parser.defineEntityReplacementText( "ecirc", "\u00ea" );
            parser.defineEntityReplacementText( "euml", "\u00eb" );
            parser.defineEntityReplacementText( "igrave", "\u00ec" );
            parser.defineEntityReplacementText( "iacute", "\u00ed" );
            parser.defineEntityReplacementText( "icirc", "\u00ee" );
            parser.defineEntityReplacementText( "iuml", "\u00ef" );
            parser.defineEntityReplacementText( "eth", "\u00f0" );
            parser.defineEntityReplacementText( "ntilde", "\u00f1" );
            parser.defineEntityReplacementText( "ograve", "\u00f2" );
            parser.defineEntityReplacementText( "oacute", "\u00f3" );
            parser.defineEntityReplacementText( "ocirc", "\u00f4" );
            parser.defineEntityReplacementText( "otilde", "\u00f5" );
            parser.defineEntityReplacementText( "ouml", "\u00f6" );
            parser.defineEntityReplacementText( "divide", "\u00f7" );
            parser.defineEntityReplacementText( "oslash", "\u00f8" );
            parser.defineEntityReplacementText( "ugrave", "\u00f9" );
            parser.defineEntityReplacementText( "uacute", "\u00fa" );
            parser.defineEntityReplacementText( "ucirc", "\u00fb" );
            parser.defineEntityReplacementText( "uuml", "\u00fc" );
            parser.defineEntityReplacementText( "yacute", "\u00fd" );
            parser.defineEntityReplacementText( "thorn", "\u00fe" );
            parser.defineEntityReplacementText( "yuml", "\u00ff" );
    
            // ----------------------------------------------------------------------
            // Special entities
            // ----------------------------------------------------------------------
    
            parser.defineEntityReplacementText( "OElig", "\u0152" );
            parser.defineEntityReplacementText( "oelig", "\u0153" );
            parser.defineEntityReplacementText( "Scaron", "\u0160" );
            parser.defineEntityReplacementText( "scaron", "\u0161" );
            parser.defineEntityReplacementText( "Yuml", "\u0178" );
            parser.defineEntityReplacementText( "circ", "\u02c6" );
            parser.defineEntityReplacementText( "tilde", "\u02dc" );
            parser.defineEntityReplacementText( "ensp", "\u2002" );
            parser.defineEntityReplacementText( "emsp", "\u2003" );
            parser.defineEntityReplacementText( "thinsp", "\u2009" );
            parser.defineEntityReplacementText( "zwnj", "\u200c" );
            parser.defineEntityReplacementText( "zwj", "\u200d" );
            parser.defineEntityReplacementText( "lrm", "\u200e" );
            parser.defineEntityReplacementText( "rlm", "\u200f" );
            parser.defineEntityReplacementText( "ndash", "\u2013" );
            parser.defineEntityReplacementText( "mdash", "\u2014" );
            parser.defineEntityReplacementText( "lsquo", "\u2018" );
            parser.defineEntityReplacementText( "rsquo", "\u2019" );
            parser.defineEntityReplacementText( "sbquo", "\u201a" );
            parser.defineEntityReplacementText( "ldquo", "\u201c" );
            parser.defineEntityReplacementText( "rdquo", "\u201d" );
            parser.defineEntityReplacementText( "bdquo", "\u201e" );
            parser.defineEntityReplacementText( "dagger", "\u2020" );
            parser.defineEntityReplacementText( "Dagger", "\u2021" );
            parser.defineEntityReplacementText( "permil", "\u2030" );
            parser.defineEntityReplacementText( "lsaquo", "\u2039" );
            parser.defineEntityReplacementText( "rsaquo", "\u203a" );
            parser.defineEntityReplacementText( "euro", "\u20ac" );
    
            // ----------------------------------------------------------------------
            // Symbol entities
            // ----------------------------------------------------------------------
    
            parser.defineEntityReplacementText( "fnof", "\u0192" );
            parser.defineEntityReplacementText( "Alpha", "\u0391" );
            parser.defineEntityReplacementText( "Beta", "\u0392" );
            parser.defineEntityReplacementText( "Gamma", "\u0393" );
            parser.defineEntityReplacementText( "Delta", "\u0394" );
            parser.defineEntityReplacementText( "Epsilon", "\u0395" );
            parser.defineEntityReplacementText( "Zeta", "\u0396" );
            parser.defineEntityReplacementText( "Eta", "\u0397" );
            parser.defineEntityReplacementText( "Theta", "\u0398" );
            parser.defineEntityReplacementText( "Iota", "\u0399" );
            parser.defineEntityReplacementText( "Kappa", "\u039a" );
            parser.defineEntityReplacementText( "Lambda", "\u039b" );
            parser.defineEntityReplacementText( "Mu", "\u039c" );
            parser.defineEntityReplacementText( "Nu", "\u039d" );
            parser.defineEntityReplacementText( "Xi", "\u039e" );
            parser.defineEntityReplacementText( "Omicron", "\u039f" );
            parser.defineEntityReplacementText( "Pi", "\u03a0" );
            parser.defineEntityReplacementText( "Rho", "\u03a1" );
            parser.defineEntityReplacementText( "Sigma", "\u03a3" );
            parser.defineEntityReplacementText( "Tau", "\u03a4" );
            parser.defineEntityReplacementText( "Upsilon", "\u03a5" );
            parser.defineEntityReplacementText( "Phi", "\u03a6" );
            parser.defineEntityReplacementText( "Chi", "\u03a7" );
            parser.defineEntityReplacementText( "Psi", "\u03a8" );
            parser.defineEntityReplacementText( "Omega", "\u03a9" );
            parser.defineEntityReplacementText( "alpha", "\u03b1" );
            parser.defineEntityReplacementText( "beta", "\u03b2" );
            parser.defineEntityReplacementText( "gamma", "\u03b3" );
            parser.defineEntityReplacementText( "delta", "\u03b4" );
            parser.defineEntityReplacementText( "epsilon", "\u03b5" );
            parser.defineEntityReplacementText( "zeta", "\u03b6" );
            parser.defineEntityReplacementText( "eta", "\u03b7" );
            parser.defineEntityReplacementText( "theta", "\u03b8" );
            parser.defineEntityReplacementText( "iota", "\u03b9" );
            parser.defineEntityReplacementText( "kappa", "\u03ba" );
            parser.defineEntityReplacementText( "lambda", "\u03bb" );
            parser.defineEntityReplacementText( "mu", "\u03bc" );
            parser.defineEntityReplacementText( "nu", "\u03bd" );
            parser.defineEntityReplacementText( "xi", "\u03be" );
            parser.defineEntityReplacementText( "omicron", "\u03bf" );
            parser.defineEntityReplacementText( "pi", "\u03c0" );
            parser.defineEntityReplacementText( "rho", "\u03c1" );
            parser.defineEntityReplacementText( "sigmaf", "\u03c2" );
            parser.defineEntityReplacementText( "sigma", "\u03c3" );
            parser.defineEntityReplacementText( "tau", "\u03c4" );
            parser.defineEntityReplacementText( "upsilon", "\u03c5" );
            parser.defineEntityReplacementText( "phi", "\u03c6" );
            parser.defineEntityReplacementText( "chi", "\u03c7" );
            parser.defineEntityReplacementText( "psi", "\u03c8" );
            parser.defineEntityReplacementText( "omega", "\u03c9" );
            parser.defineEntityReplacementText( "thetasym", "\u03d1" );
            parser.defineEntityReplacementText( "upsih", "\u03d2" );
            parser.defineEntityReplacementText( "piv", "\u03d6" );
            parser.defineEntityReplacementText( "bull", "\u2022" );
            parser.defineEntityReplacementText( "hellip", "\u2026" );
            parser.defineEntityReplacementText( "prime", "\u2032" );
            parser.defineEntityReplacementText( "Prime", "\u2033" );
            parser.defineEntityReplacementText( "oline", "\u203e" );
            parser.defineEntityReplacementText( "frasl", "\u2044" );
            parser.defineEntityReplacementText( "weierp", "\u2118" );
            parser.defineEntityReplacementText( "image", "\u2111" );
            parser.defineEntityReplacementText( "real", "\u211c" );
            parser.defineEntityReplacementText( "trade", "\u2122" );
            parser.defineEntityReplacementText( "alefsym", "\u2135" );
            parser.defineEntityReplacementText( "larr", "\u2190" );
            parser.defineEntityReplacementText( "uarr", "\u2191" );
            parser.defineEntityReplacementText( "rarr", "\u2192" );
            parser.defineEntityReplacementText( "darr", "\u2193" );
            parser.defineEntityReplacementText( "harr", "\u2194" );
            parser.defineEntityReplacementText( "crarr", "\u21b5" );
            parser.defineEntityReplacementText( "lArr", "\u21d0" );
            parser.defineEntityReplacementText( "uArr", "\u21d1" );
            parser.defineEntityReplacementText( "rArr", "\u21d2" );
            parser.defineEntityReplacementText( "dArr", "\u21d3" );
            parser.defineEntityReplacementText( "hArr", "\u21d4" );
            parser.defineEntityReplacementText( "forall", "\u2200" );
            parser.defineEntityReplacementText( "part", "\u2202" );
            parser.defineEntityReplacementText( "exist", "\u2203" );
            parser.defineEntityReplacementText( "empty", "\u2205" );
            parser.defineEntityReplacementText( "nabla", "\u2207" );
            parser.defineEntityReplacementText( "isin", "\u2208" );
            parser.defineEntityReplacementText( "notin", "\u2209" );
            parser.defineEntityReplacementText( "ni", "\u220b" );
            parser.defineEntityReplacementText( "prod", "\u220f" );
            parser.defineEntityReplacementText( "sum", "\u2211" );
            parser.defineEntityReplacementText( "minus", "\u2212" );
            parser.defineEntityReplacementText( "lowast", "\u2217" );
            parser.defineEntityReplacementText( "radic", "\u221a" );
            parser.defineEntityReplacementText( "prop", "\u221d" );
            parser.defineEntityReplacementText( "infin", "\u221e" );
            parser.defineEntityReplacementText( "ang", "\u2220" );
            parser.defineEntityReplacementText( "and", "\u2227" );
            parser.defineEntityReplacementText( "or", "\u2228" );
            parser.defineEntityReplacementText( "cap", "\u2229" );
            parser.defineEntityReplacementText( "cup", "\u222a" );
            parser.defineEntityReplacementText( "int", "\u222b" );
            parser.defineEntityReplacementText( "there4", "\u2234" );
            parser.defineEntityReplacementText( "sim", "\u223c" );
            parser.defineEntityReplacementText( "cong", "\u2245" );
            parser.defineEntityReplacementText( "asymp", "\u2248" );
            parser.defineEntityReplacementText( "ne", "\u2260" );
            parser.defineEntityReplacementText( "equiv", "\u2261" );
            parser.defineEntityReplacementText( "le", "\u2264" );
            parser.defineEntityReplacementText( "ge", "\u2265" );
            parser.defineEntityReplacementText( "sub", "\u2282" );
            parser.defineEntityReplacementText( "sup", "\u2283" );
            parser.defineEntityReplacementText( "nsub", "\u2284" );
            parser.defineEntityReplacementText( "sube", "\u2286" );
            parser.defineEntityReplacementText( "supe", "\u2287" );
            parser.defineEntityReplacementText( "oplus", "\u2295" );
            parser.defineEntityReplacementText( "otimes", "\u2297" );
            parser.defineEntityReplacementText( "perp", "\u22a5" );
            parser.defineEntityReplacementText( "sdot", "\u22c5" );
            parser.defineEntityReplacementText( "lceil", "\u2308" );
            parser.defineEntityReplacementText( "rceil", "\u2309" );
            parser.defineEntityReplacementText( "lfloor", "\u230a" );
            parser.defineEntityReplacementText( "rfloor", "\u230b" );
            parser.defineEntityReplacementText( "lang", "\u2329" );
            parser.defineEntityReplacementText( "rang", "\u232a" );
            parser.defineEntityReplacementText( "loz", "\u25ca" );
            parser.defineEntityReplacementText( "spades", "\u2660" );
            parser.defineEntityReplacementText( "clubs", "\u2663" );
            parser.defineEntityReplacementText( "hearts", "\u2665" );
            parser.defineEntityReplacementText( "diams", "\u2666" );
    
        }
    
        parser.next();
        return parseSettings( "settings", parser, strict );
    } //-- Settings read( Reader, boolean ) 

    /**
     * @see ReaderFactory#newXmlReader
     * 
     * @param reader
     * @throws IOException
     * @throws XmlPullParserException
     * @return Settings
     */
    public Settings read( Reader reader )
        throws IOException, XmlPullParserException
    {
        return read( reader, true );
    } //-- Settings read( Reader ) 

    /**
     * Method read.
     * 
     * @param in
     * @param strict
     * @throws IOException
     * @throws XmlPullParserException
     * @return Settings
     */
    public Settings read( InputStream in, boolean strict )
        throws IOException, XmlPullParserException
    {
        Reader reader = ReaderFactory.newXmlReader( in );
    
        return read( reader, strict );
    } //-- Settings read( InputStream, boolean ) 

    /**
     * Method read.
     * 
     * @param in
     * @throws IOException
     * @throws XmlPullParserException
     * @return Settings
     */
    public Settings read( InputStream in )
        throws IOException, XmlPullParserException
    {
        Reader reader = ReaderFactory.newXmlReader( in );
    
        return read( reader );
    } //-- Settings read( InputStream ) 

    /**
     * Sets the state of the "add default entities" flag.
     * 
     * @param addDefaultEntities
     */
    public void setAddDefaultEntities( boolean addDefaultEntities )
    {
        this.addDefaultEntities = addDefaultEntities;
    } //-- void setAddDefaultEntities( boolean ) 


}
