/*
 * Decompiled with CFR 0.152.
 */
package org.apache.qpid.test.utils;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.Collection;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TCPTunneler
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(TCPTunneler.class);
    private static final int BUFFER_SIZE = 4096;
    private final TCPWorker _tcpWorker;
    private final ExecutorService _executor;
    private final int _bufferSize;

    public TCPTunneler(int localPort, String remoteHost, int remotePort, int numberOfConcurrentClients, int bufferSize) {
        this._executor = Executors.newFixedThreadPool(numberOfConcurrentClients * 2 + 1);
        this._tcpWorker = new TCPWorker(localPort, remoteHost, remotePort, bufferSize, this._executor);
        this._bufferSize = bufferSize;
    }

    public TCPTunneler(int localPort, String remoteHost, int remotePort, int numberOfConcurrentClients) {
        this(localPort, remoteHost, remotePort, numberOfConcurrentClients, 4096);
    }

    public void start() throws IOException {
        this._tcpWorker.start();
    }

    public void stopClientToServerForwarding(InetSocketAddress clientAddress) {
        this._tcpWorker.stopClientToServerForwarding(clientAddress);
    }

    public void stop() {
        try {
            this._tcpWorker.stop();
        }
        finally {
            this._executor.shutdown();
        }
    }

    public void addClientListener(TunnelListener listener) {
        this._tcpWorker.addClientListener(listener);
    }

    public void removeClientListener(TunnelListener listener) {
        this._tcpWorker.removeClientListener(listener);
    }

    public void disconnect(InetSocketAddress address) {
        LOGGER.info("Disconnecting {}", (Object)address);
        if (address != null) {
            this._tcpWorker.disconnect(address);
        }
    }

    public int getLocalPort() {
        return this._tcpWorker.getLocalPort();
    }

    @Override
    public void close() throws Exception {
        this.stop();
    }

    public static interface ForwardCallback {
        public void notify(int var1);
    }

    private static class StreamForwarder
    implements Runnable {
        private final int _bufferSize;
        private final InputStream _inputStream;
        private final OutputStream _outputStream;
        private final String _name;
        private final ForwardCallback _forwardCallback;
        private final AtomicBoolean _stopForwarding = new AtomicBoolean();

        public StreamForwarder(Socket input, Socket output, int bufferSize, ForwardCallback forwardCallback) throws IOException {
            this._inputStream = input.getInputStream();
            this._outputStream = output.getOutputStream();
            this._forwardCallback = forwardCallback == null ? numberOfBytesForwarded -> {} : forwardCallback;
            this._name = "Forwarder-" + input.getLocalSocketAddress() + "->" + output.getRemoteSocketAddress();
            this._bufferSize = bufferSize;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            byte[] buffer = new byte[this._bufferSize];
            try {
                int bytesRead;
                while ((bytesRead = this._inputStream.read(buffer)) != -1) {
                    if (!this._stopForwarding.get()) {
                        this._outputStream.write(buffer, 0, bytesRead);
                        this._outputStream.flush();
                        this._forwardCallback.notify(bytesRead);
                        LOGGER.debug("Forwarded {} byte(s)", (Object)bytesRead);
                        continue;
                    }
                    LOGGER.debug("Discarded {} byte(s)", (Object)bytesRead);
                }
            }
            catch (IOException e) {
                LOGGER.warn("Exception on forwarding data for {}: {}", (Object)this._name, (Object)e.getMessage());
            }
            finally {
                try {
                    this._inputStream.close();
                }
                catch (IOException iOException) {}
                try {
                    this._outputStream.close();
                }
                catch (IOException iOException) {}
            }
        }

        public String getName() {
            return this._name;
        }

        public void stopForwarding() {
            this._stopForwarding.set(true);
        }
    }

    private static class SocketTunnel {
        private final Socket _clientSocket;
        private final Socket _serverSocket;
        private final TunnelListener _tunnelListener;
        private final AtomicBoolean _closed;
        private final AutoClosingStreamForwarder _clientToServer;
        private final AutoClosingStreamForwarder _serverToClient;
        private final InetSocketAddress _clientSocketAddress;

        SocketTunnel(Socket clientSocket, Socket serverSocket, TunnelListener tunnelListener, int bufferSize) throws IOException {
            this._clientSocket = clientSocket;
            this._clientSocketAddress = new InetSocketAddress(clientSocket.getInetAddress().getHostName(), this._clientSocket.getPort());
            this._serverSocket = serverSocket;
            this._closed = new AtomicBoolean();
            this._tunnelListener = tunnelListener;
            this._clientSocket.setKeepAlive(true);
            this._serverSocket.setKeepAlive(true);
            this._clientToServer = new AutoClosingStreamForwarder(new StreamForwarder(this._clientSocket, this._serverSocket, bufferSize, numBytes -> this._tunnelListener.notifyClientToServerBytesDelivered(this._clientSocket.getInetAddress(), numBytes)));
            this._serverToClient = new AutoClosingStreamForwarder(new StreamForwarder(this._serverSocket, this._clientSocket, bufferSize, numBytes -> this._tunnelListener.notifyServerToClientBytesDelivered(this._serverSocket.getInetAddress(), numBytes)));
        }

        public void close() {
            if (this._closed.compareAndSet(false, true)) {
                try {
                    SocketTunnel.closeSocket(this._serverSocket);
                    SocketTunnel.closeSocket(this._clientSocket);
                }
                finally {
                    this._tunnelListener.clientDisconnected(this.getClientAddress());
                }
            }
        }

        public void start(Executor executor) throws IOException {
            executor.execute(this._clientToServer);
            executor.execute(this._serverToClient);
            this._tunnelListener.clientConnected(this.getClientAddress());
        }

        public void stopClientToServerForwarding() {
            this._clientToServer.stopForwarding();
        }

        public boolean isClosed() {
            return this._closed.get();
        }

        public boolean isClientAddress(InetSocketAddress clientAddress) {
            return this.getClientAddress().equals(clientAddress);
        }

        public InetSocketAddress getClientAddress() {
            return this._clientSocketAddress;
        }

        private static void closeSocket(Socket socket) {
            if (socket != null) {
                try {
                    socket.close();
                }
                catch (IOException e) {
                    LOGGER.warn("Exception on closing of socket {}", (Object)socket, (Object)e);
                }
            }
        }

        private class AutoClosingStreamForwarder
        implements Runnable {
            private StreamForwarder _streamForwarder;

            public AutoClosingStreamForwarder(StreamForwarder streamForwarder) {
                this._streamForwarder = streamForwarder;
            }

            @Override
            public void run() {
                Thread currentThread = Thread.currentThread();
                String originalThreadName = currentThread.getName();
                try {
                    currentThread.setName(this._streamForwarder.getName());
                    this._streamForwarder.run();
                }
                finally {
                    SocketTunnel.this.close();
                    currentThread.setName(originalThreadName);
                }
            }

            public void stopForwarding() {
                this._streamForwarder.stopForwarding();
            }
        }
    }

    private static class TCPWorker
    implements Runnable {
        private final String _remoteHost;
        private final int _remotePort;
        private final int _localPort;
        private final String _remoteHostPort;
        private final AtomicBoolean _closed;
        private final Collection<SocketTunnel> _tunnels;
        private final Collection<TunnelListener> _tunnelListeners;
        private final TunnelListener _notifyingListener;
        private final int _bufferSize;
        private volatile ServerSocket _serverSocket;
        private volatile ExecutorService _executor;
        private int _actualLocalPort;

        public TCPWorker(int localPort, String remoteHost, int remotePort, int bufferSize, ExecutorService executor) {
            this._bufferSize = bufferSize;
            this._closed = new AtomicBoolean();
            this._remoteHost = remoteHost;
            this._remotePort = remotePort;
            this._localPort = localPort;
            this._remoteHostPort = this._remoteHost + ":" + this._remotePort;
            this._executor = executor;
            this._tunnels = new CopyOnWriteArrayList<SocketTunnel>();
            this._tunnelListeners = new CopyOnWriteArrayList<TunnelListener>();
            this._notifyingListener = new TunnelListener(){

                @Override
                public void clientConnected(InetSocketAddress clientAddress) {
                    this.notifyClientConnected(clientAddress);
                }

                @Override
                public void clientDisconnected(InetSocketAddress clientAddress) {
                    try {
                        this.notifyClientDisconnected(clientAddress);
                    }
                    finally {
                        this.removeTunnel(clientAddress);
                    }
                }

                @Override
                public void notifyClientToServerBytesDelivered(InetAddress inetAddress, int numberOfBytesForwarded) {
                    for (TunnelListener listener : _tunnelListeners) {
                        listener.notifyClientToServerBytesDelivered(inetAddress, numberOfBytesForwarded);
                    }
                }

                @Override
                public void notifyServerToClientBytesDelivered(InetAddress inetAddress, int numberOfBytesForwarded) {
                    for (TunnelListener listener : _tunnelListeners) {
                        listener.notifyClientToServerBytesDelivered(inetAddress, numberOfBytesForwarded);
                    }
                }
            };
        }

        @Override
        public void run() {
            String threadName = Thread.currentThread().getName();
            try {
                Thread.currentThread().setName("TCPTunnelerAcceptingThread");
                while (!this._closed.get()) {
                    Socket acceptedSocket = this._serverSocket.accept();
                    LOGGER.debug("Client opened socket {}", (Object)acceptedSocket);
                    this.createTunnel(acceptedSocket);
                }
            }
            catch (IOException e) {
                if (!this._closed.get()) {
                    LOGGER.error("Exception in accepting thread", (Throwable)e);
                }
            }
            finally {
                this.closeServerSocket();
                this._closed.set(true);
                Thread.currentThread().setName(threadName);
            }
        }

        public void start() {
            this._actualLocalPort = this._localPort;
            try {
                this._serverSocket = new ServerSocket(this._localPort);
                this._actualLocalPort = this._serverSocket.getLocalPort();
                LOGGER.info("Starting TCPTunneler forwarding from port {} to {}", (Object)this._actualLocalPort, (Object)this._remoteHostPort);
                this._serverSocket.setReuseAddress(true);
            }
            catch (IOException e) {
                throw new RuntimeException("Cannot start TCPTunneler on port " + this._actualLocalPort, e);
            }
            if (this._serverSocket != null) {
                try {
                    this._executor.execute(this);
                }
                catch (Exception e) {
                    try {
                        this.closeServerSocket();
                    }
                    finally {
                        throw new RuntimeException("Cannot start acceptor thread for TCPTunneler on port " + this._actualLocalPort, e);
                    }
                }
            }
        }

        public void stop() {
            if (this._closed.compareAndSet(false, true)) {
                LOGGER.info("Stopping TCPTunneler forwarding from port {} to {}", (Object)this._actualLocalPort, (Object)this._remoteHostPort);
                try {
                    for (SocketTunnel tunnel : this._tunnels) {
                        tunnel.close();
                    }
                }
                finally {
                    this.closeServerSocket();
                }
                LOGGER.info("TCPTunneler forwarding from port {} to {} is stopped", (Object)this._actualLocalPort, (Object)this._remoteHostPort);
            }
        }

        public void addClientListener(TunnelListener listener) {
            this._tunnelListeners.add(listener);
            for (SocketTunnel socketTunnel : this._tunnels) {
                try {
                    listener.clientConnected(socketTunnel.getClientAddress());
                }
                catch (Exception e) {
                    LOGGER.warn("Exception on notifying client listener about connected client", (Throwable)e);
                }
            }
        }

        public void removeClientListener(TunnelListener listener) {
            this._tunnelListeners.remove(listener);
        }

        public void disconnect(InetSocketAddress address) {
            SocketTunnel client = this.removeTunnel(address);
            if (client != null && !client.isClosed()) {
                client.close();
                LOGGER.info("Tunnel for {} is disconnected", (Object)address);
            } else {
                LOGGER.info("Tunnel for {} not found", (Object)address);
            }
        }

        private void createTunnel(Socket localSocket) {
            Socket remoteSocket = null;
            try {
                LOGGER.debug("Opening socket to {} for {}", (Object)this._remoteHostPort, (Object)localSocket);
                remoteSocket = new Socket(this._remoteHost, this._remotePort);
                LOGGER.debug("Opened socket to {} for {}", (Object)remoteSocket, (Object)localSocket);
                SocketTunnel tunnel = new SocketTunnel(localSocket, remoteSocket, this._notifyingListener, this._bufferSize);
                LOGGER.debug("Socket tunnel is created from {} to {}", (Object)localSocket, (Object)remoteSocket);
                this._tunnels.add(tunnel);
                tunnel.start(this._executor);
            }
            catch (Exception e) {
                LOGGER.error("Cannot forward i/o traffic between {} and {}", new Object[]{localSocket, this._remoteHostPort, e});
                SocketTunnel.closeSocket(localSocket);
                SocketTunnel.closeSocket(remoteSocket);
            }
        }

        private void notifyClientConnected(InetSocketAddress clientAddress) {
            for (TunnelListener listener : this._tunnelListeners) {
                try {
                    listener.clientConnected(clientAddress);
                }
                catch (Exception e) {
                    LOGGER.warn("Exception on notifying client listener about connected client", (Throwable)e);
                }
            }
        }

        private void notifyClientDisconnected(InetSocketAddress clientAddress) {
            for (TunnelListener listener : this._tunnelListeners) {
                try {
                    listener.clientDisconnected(clientAddress);
                }
                catch (Exception e) {
                    LOGGER.warn("Exception on notifying client listener about disconnected client", (Throwable)e);
                }
            }
        }

        public void stopClientToServerForwarding(InetSocketAddress clientAddress) {
            SocketTunnel target = null;
            for (SocketTunnel tunnel : this._tunnels) {
                if (!tunnel.getClientAddress().equals(clientAddress)) continue;
                target = tunnel;
                break;
            }
            if (target == null) {
                throw new IllegalArgumentException("Could not find tunnel for address " + clientAddress);
            }
            LOGGER.debug("Stopping forwarding from client {} to server", (Object)clientAddress);
            target.stopClientToServerForwarding();
        }

        private void closeServerSocket() {
            if (this._serverSocket != null) {
                try {
                    this._serverSocket.close();
                }
                catch (IOException e) {
                    LOGGER.warn("Exception on closing of accepting socket", (Throwable)e);
                }
                finally {
                    this._serverSocket = null;
                }
            }
        }

        private SocketTunnel removeTunnel(InetSocketAddress clientAddress) {
            SocketTunnel client = null;
            for (SocketTunnel c : this._tunnels) {
                if (!c.isClientAddress(clientAddress)) continue;
                client = c;
                break;
            }
            if (client != null) {
                this._tunnels.remove(client);
            }
            return client;
        }

        public int getLocalPort() {
            if (this._serverSocket == null) {
                return -1;
            }
            return this._serverSocket.getLocalPort();
        }
    }

    public static interface TunnelListener {
        public void clientConnected(InetSocketAddress var1);

        public void clientDisconnected(InetSocketAddress var1);

        public void notifyClientToServerBytesDelivered(InetAddress var1, int var2);

        public void notifyServerToClientBytesDelivered(InetAddress var1, int var2);
    }
}

