/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss;

import eu.europa.esig.dss.ASiCContainerType;
import eu.europa.esig.dss.DSSASN1Utils;
import eu.europa.esig.dss.DSSDocument;
import eu.europa.esig.dss.DSSException;
import eu.europa.esig.dss.DigestAlgorithm;
import eu.europa.esig.dss.SignatureForm;
import eu.europa.esig.dss.SignatureLevel;
import eu.europa.esig.dss.SigningOperation;
import eu.europa.esig.dss.TokenIdentifier;
import eu.europa.esig.dss.client.http.DataLoader;
import eu.europa.esig.dss.utils.Utils;
import eu.europa.esig.dss.x509.CertificateToken;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLDecoder;
import java.nio.ByteBuffer;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.security.NoSuchProviderException;
import java.security.Provider;
import java.security.Security;
import java.security.cert.CRLException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TimeZone;
import java.util.concurrent.TimeUnit;
import javax.security.auth.x500.X500Principal;
import org.bouncycastle.cert.X509CRLHolder;
import org.bouncycastle.cert.jcajce.JcaX509CRLConverter;
import org.bouncycastle.cert.ocsp.BasicOCSPResp;
import org.bouncycastle.cert.ocsp.OCSPResp;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DSSUtils {
    private static final Logger LOG = LoggerFactory.getLogger(DSSUtils.class);
    public static final String CERT_BEGIN = "-----BEGIN CERTIFICATE-----";
    public static final String CERT_END = "-----END CERTIFICATE-----";
    public static final String CRL_BEGIN = "-----BEGIN X509 CRL-----";
    public static final String CRL_END = "-----END X509 CRL-----";
    private static final BouncyCastleProvider securityProvider = new BouncyCastleProvider();
    private static final CertificateFactory certificateFactory;
    public static final byte[] EMPTY_BYTE_ARRAY;
    public static final String DEFAULT_DATE_TIME_FORMAT = "yyyy-MM-dd'T'HH:mm:ss'Z'";
    public static final String DEFAULT_DATE_FORMAT = "yyyy-MM-dd";
    private static final String NEW_LINE = "\n";

    private DSSUtils() {
    }

    public static String formatInternal(Date date) {
        String formatedDate = date == null ? "N/A" : new SimpleDateFormat(DEFAULT_DATE_TIME_FORMAT).format(date);
        return formatedDate;
    }

    public static String toHex(byte[] value) {
        return value != null ? Utils.toHex((byte[])value) : null;
    }

    protected static int toDigit(char ch, int index) throws DSSException {
        int digit = Character.digit(ch, 16);
        if (digit == -1) {
            throw new DSSException("Illegal hexadecimal character " + ch + " at index " + index);
        }
        return digit;
    }

    private static String normalisePath(String path) {
        return path.replace('\\', '/');
    }

    public static boolean resourceExists(String path) {
        String path_ = DSSUtils.normalisePath(path);
        URL url = DSSUtils.class.getResource(path_);
        return url != null;
    }

    public static boolean fileExists(String path) {
        String path_ = DSSUtils.normalisePath(path);
        boolean exists = new File(path_).exists();
        return exists;
    }

    public static File getFile(String filePath) {
        String normalisedFolderFileName = DSSUtils.normalisePath(filePath);
        File file = new File(normalisedFolderFileName);
        return file;
    }

    public static String convertToPEM(CertificateToken cert) throws DSSException {
        byte[] derCert = cert.getEncoded();
        String pemCertPre = Utils.toBase64((byte[])derCert);
        String pemCert = "-----BEGIN CERTIFICATE-----\n" + pemCertPre + NEW_LINE + CERT_END;
        return pemCert;
    }

    public static String convertCrlToPEM(X509CRL crl) throws DSSException {
        try {
            byte[] derCrl = crl.getEncoded();
            String pemCrlPre = Utils.toBase64((byte[])derCrl);
            String pemCrl = "-----BEGIN X509 CRL-----\n" + pemCrlPre + NEW_LINE + CRL_END;
            return pemCrl;
        }
        catch (CRLException e) {
            throw new DSSException("Unable to convert CRL to PEM encoding : " + e.getMessage(), (Throwable)e);
        }
    }

    public static boolean isPEM(InputStream is) {
        try {
            String startPEM = "-----BEGIN";
            int headerLength = 100;
            byte[] preamble = new byte[headerLength];
            if (is.read(preamble, 0, headerLength) > 0) {
                String startArray = new String(preamble);
                return startArray.startsWith(startPEM);
            }
            return false;
        }
        catch (Exception e) {
            throw new DSSException("Unable to read InputStream", (Throwable)e);
        }
    }

    public static boolean isPEM(byte[] byteArray) {
        try {
            String startPEM = "-----BEGIN";
            int headerLength = 100;
            byte[] preamble = new byte[headerLength];
            System.arraycopy(byteArray, 0, preamble, 0, headerLength);
            String startArray = new String(preamble);
            return startArray.startsWith(startPEM);
        }
        catch (Exception e) {
            throw new DSSException("Unable to read InputStream");
        }
    }

    public static byte[] convertToDER(String pemCert) {
        String base64 = pemCert.replace(CERT_BEGIN, "");
        base64 = base64.replace(CERT_END, "");
        base64 = base64.replaceAll("\\s", "");
        return Utils.fromBase64((String)base64);
    }

    public static byte[] convertCRLToDER(String pemCRL) {
        String base64 = pemCRL.replace(CRL_BEGIN, "");
        base64 = base64.replace(CRL_END, "");
        base64 = base64.replaceAll("\\s", "");
        return Utils.fromBase64((String)base64);
    }

    public static CertificateToken loadCertificate(String path) throws DSSException {
        InputStream inputStream = DSSUtils.class.getResourceAsStream(path);
        return DSSUtils.loadCertificate(inputStream);
    }

    public static CertificateToken loadCertificate(File file) throws DSSException {
        InputStream inputStream = DSSUtils.toByteArrayInputStream(file);
        CertificateToken x509Certificate = DSSUtils.loadCertificate(inputStream);
        return x509Certificate;
    }

    public static CertificateToken loadCertificate(InputStream inputStream) throws DSSException {
        List<CertificateToken> certificates = DSSUtils.loadCertificates(inputStream);
        if (certificates.size() == 1) {
            return certificates.get(0);
        }
        throw new DSSException("Could not parse certificate");
    }

    public static Collection<CertificateToken> loadCertificateFromP7c(InputStream is) {
        return DSSUtils.loadCertificates(is);
    }

    private static List<CertificateToken> loadCertificates(InputStream is) {
        ArrayList<CertificateToken> certificates = new ArrayList<CertificateToken>();
        try {
            Collection<? extends Certificate> certificatesCollection = certificateFactory.generateCertificates(is);
            if (certificatesCollection != null) {
                for (X509Certificate x509Certificate : certificatesCollection) {
                    certificates.add(new CertificateToken(x509Certificate));
                }
            }
            if (certificates.isEmpty()) {
                throw new DSSException("Could not parse certificate(s)");
            }
            return certificates;
        }
        catch (Exception e) {
            throw new DSSException((Throwable)e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static CertificateToken loadCertificate(byte[] input) throws DSSException {
        if (input == null) {
            throw new NullPointerException("X509 certificate");
        }
        try (ByteArrayInputStream inputStream = new ByteArrayInputStream(input);){
            CertificateToken certificateToken = DSSUtils.loadCertificate(inputStream);
            return certificateToken;
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static CertificateToken loadCertificateFromBase64EncodedString(String base64Encoded) {
        byte[] bytes = Utils.fromBase64((String)base64Encoded);
        return DSSUtils.loadCertificate(bytes);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static Collection<CertificateToken> loadIssuerCertificates(CertificateToken cert, DataLoader loader) {
        List<String> urls = DSSASN1Utils.getCAAccessLocations(cert);
        if (Utils.isCollectionEmpty(urls)) {
            LOG.info("There is no AIA extension for certificate download. CA " + cert.getIssuerX500Principal().getName());
            return null;
        }
        if (loader == null) {
            LOG.warn("There is no DataLoader defined to load Certificates from AIA extension (urls : " + urls + ")");
            return null;
        }
        Iterator<String> i$ = urls.iterator();
        while (i$.hasNext()) {
            String url = i$.next();
            LOG.debug("Loading certificate from {}", (Object)url);
            byte[] bytes = loader.get(url);
            if (Utils.isArrayNotEmpty((byte[])bytes)) {
                LOG.debug("Base64 content : " + Utils.toBase64((byte[])bytes));
                try {
                    ByteArrayInputStream is = new ByteArrayInputStream(bytes);
                    Throwable throwable = null;
                    try {
                        CertificateToken issuerCert;
                        Collection<CertificateToken> issuerCerts;
                        block22: {
                            issuerCerts = null;
                            issuerCert = null;
                            try {
                                issuerCert = DSSUtils.loadCertificate(bytes);
                                issuerCerts = Collections.singletonList(issuerCert);
                            }
                            catch (DSSException dssEx) {
                                if (issuerCert != null) break block22;
                                Collection<CertificateToken> certsCollection = DSSUtils.loadCertificateFromP7c(is);
                                for (CertificateToken token : certsCollection) {
                                    if (!cert.isSignedBy(token)) continue;
                                    issuerCert = token;
                                    issuerCerts = certsCollection;
                                }
                            }
                        }
                        if (issuerCert == null) continue;
                        if (!cert.getIssuerX500Principal().equals(issuerCert.getSubjectX500Principal())) {
                            LOG.info("There is AIA extension, but the issuer subject name and subject name does not match.");
                            LOG.info("CERT ISSUER    : " + cert.getIssuerX500Principal().toString());
                            LOG.info("ISSUER SUBJECT : " + issuerCert.getSubjectX500Principal().toString());
                        }
                        List<CertificateToken> list = issuerCerts;
                        return list;
                    }
                    catch (Throwable throwable2) {
                        throwable = throwable2;
                        throw throwable2;
                    }
                    finally {
                        if (is == null) continue;
                        if (throwable != null) {
                            try {
                                ((InputStream)is).close();
                            }
                            catch (Throwable x2) {
                                throwable.addSuppressed(x2);
                            }
                            continue;
                        }
                        ((InputStream)is).close();
                    }
                }
                catch (Exception e) {
                    LOG.warn("Unable to parse certficate from AIA (url:" + url + ") : " + e.getMessage());
                }
                continue;
            }
            LOG.error("Unable to read data from {}.", (Object)url);
        }
        return null;
    }

    public static X509CRL loadCRLBase64Encoded(String base64Encoded) {
        byte[] derEncoded = Utils.fromBase64((String)base64Encoded);
        return DSSUtils.loadCRL(derEncoded);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static X509CRL loadCRL(byte[] byteArray) {
        try (ByteArrayInputStream inputStream = new ByteArrayInputStream(byteArray);){
            X509CRL x509CRL = DSSUtils.loadCRL(inputStream);
            return x509CRL;
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static X509CRL loadCRL(InputStream inputStream) {
        try {
            return (X509CRL)certificateFactory.generateCRL(inputStream);
        }
        catch (CRLException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static String getSHA1Digest(String stringToDigest) {
        byte[] digest = DSSUtils.getMessageDigest(DigestAlgorithm.SHA1).digest(stringToDigest.getBytes());
        return Utils.toHex((byte[])digest);
    }

    public static String getSHA1Digest(InputStream inputStream) throws IOException {
        return Utils.toHex((byte[])DSSUtils.digest(DigestAlgorithm.SHA1, inputStream));
    }

    public static StringBuilder replaceStrStr(StringBuilder string, String oldPattern, String newPattern) {
        int idxOld;
        if (string == null || oldPattern == null || oldPattern.equals("") || newPattern == null) {
            return string;
        }
        StringBuilder replaced = new StringBuilder();
        int startIdx = 0;
        while ((idxOld = string.indexOf(oldPattern, startIdx)) >= 0) {
            replaced.append(string.substring(startIdx, idxOld));
            replaced.append(newPattern);
            startIdx = idxOld + oldPattern.length();
        }
        replaced.append(string.substring(startIdx));
        return replaced;
    }

    public static String replaceStrStr(String string, String oldPattern, String newPattern) {
        StringBuilder stringBuilder = DSSUtils.replaceStrStr(new StringBuilder(string), oldPattern, newPattern);
        return stringBuilder.toString();
    }

    public static byte[] digest(DigestAlgorithm digestAlgorithm, byte[] data) throws DSSException {
        MessageDigest messageDigest = DSSUtils.getMessageDigest(digestAlgorithm);
        byte[] digestValue = messageDigest.digest(data);
        return digestValue;
    }

    public static MessageDigest getMessageDigest(DigestAlgorithm digestAlgorithm) {
        try {
            String digestAlgorithmOid = digestAlgorithm.getOid();
            MessageDigest messageDigest = MessageDigest.getInstance(digestAlgorithmOid, "BC");
            return messageDigest;
        }
        catch (GeneralSecurityException e) {
            throw new DSSException("Digest algorithm '" + digestAlgorithm.getName() + "' error: " + e.getMessage(), (Throwable)e);
        }
    }

    public static byte[] digest(DigestAlgorithm digestAlgo, InputStream inputStream) throws DSSException {
        try {
            MessageDigest messageDigest = DSSUtils.getMessageDigest(digestAlgo);
            byte[] buffer = new byte[4096];
            int count = 0;
            while ((count = inputStream.read(buffer)) > 0) {
                messageDigest.update(buffer, 0, count);
            }
            byte[] digestValue = messageDigest.digest();
            return digestValue;
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static byte[] digest(DigestAlgorithm digestAlgorithm, DSSDocument document) {
        try (InputStream is = document.openStream();){
            byte[] byArray = DSSUtils.digest(digestAlgorithm, is);
            return byArray;
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static byte[] digest(DigestAlgorithm digestAlgorithm, byte[] ... data) {
        MessageDigest messageDigest = DSSUtils.getMessageDigest(digestAlgorithm);
        for (byte[] bytes : data) {
            messageDigest.update(bytes);
        }
        byte[] digestValue = messageDigest.digest();
        return digestValue;
    }

    public static InputStream toInputStream(String filePath) throws DSSException {
        File file = DSSUtils.getFile(filePath);
        InputStream inputStream = DSSUtils.toInputStream(file);
        return inputStream;
    }

    public static InputStream toInputStream(File file) throws DSSException {
        if (file == null) {
            throw new NullPointerException();
        }
        try {
            FileInputStream fileInputStream = DSSUtils.openInputStream(file);
            return fileInputStream;
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static InputStream toInputStream(String string, String charset) throws DSSException {
        try {
            ByteArrayInputStream inputStream = new ByteArrayInputStream(string.getBytes(charset));
            return inputStream;
        }
        catch (UnsupportedEncodingException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static FileOutputStream toFileOutputStream(String path) throws DSSException {
        try {
            return new FileOutputStream(path);
        }
        catch (FileNotFoundException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static InputStream toByteArrayInputStream(File file) {
        if (file == null) {
            throw new NullPointerException();
        }
        try {
            byte[] bytes = DSSUtils.readFileToByteArray(file);
            ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(bytes);
            return byteArrayInputStream;
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static byte[] toByteArray(File file) throws DSSException {
        if (file == null) {
            throw new NullPointerException();
        }
        try {
            byte[] bytes = DSSUtils.readFileToByteArray(file);
            return bytes;
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static byte[] readFileToByteArray(File file) throws IOException {
        FileInputStream in = null;
        try {
            in = DSSUtils.openInputStream(file);
            byte[] byArray = Utils.toByteArray((InputStream)in);
            return byArray;
        }
        finally {
            Utils.closeQuietly((Closeable)in);
        }
    }

    private static FileInputStream openInputStream(File file) throws IOException {
        if (file.exists()) {
            if (file.isDirectory()) {
                throw new IOException("File '" + file + "' exists but is a directory");
            }
            if (!file.canRead()) {
                throw new IOException("File '" + file + "' cannot be read");
            }
        } else {
            throw new FileNotFoundException("File '" + file + "' does not exist");
        }
        return new FileInputStream(file);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static byte[] toByteArray(DSSDocument document) {
        try (InputStream is = document.openStream();){
            byte[] byArray = DSSUtils.toByteArray(is);
            return byArray;
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static byte[] toByteArray(InputStream inputStream) {
        if (inputStream == null) {
            throw new NullPointerException();
        }
        try {
            byte[] bytes = Utils.toByteArray((InputStream)inputStream);
            return bytes;
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static String toString(byte[] bytes) {
        if (bytes == null) {
            throw new NullPointerException();
        }
        String string = new String(bytes);
        return string;
    }

    public static void saveToFile(byte[] bytes, File file) throws DSSException {
        file.getParentFile().mkdirs();
        ByteArrayInputStream is = null;
        FileOutputStream os = null;
        try {
            os = new FileOutputStream(file);
            is = new ByteArrayInputStream(bytes);
            Utils.copy((InputStream)is, (OutputStream)os);
        }
        catch (IOException e) {
            try {
                throw new DSSException((Throwable)e);
            }
            catch (Throwable throwable) {
                Utils.closeQuietly(is);
                Utils.closeQuietly((Closeable)os);
                throw throwable;
            }
        }
        Utils.closeQuietly((Closeable)is);
        Utils.closeQuietly((Closeable)os);
    }

    public static void saveToFile(InputStream inputStream, String path) throws IOException {
        FileOutputStream fileOutputStream = DSSUtils.toFileOutputStream(path);
        Utils.copy((InputStream)inputStream, (OutputStream)fileOutputStream);
        Utils.closeQuietly((Closeable)fileOutputStream);
    }

    public static X509CRL toX509CRL(X509CRLHolder x509CRLHolder) {
        try {
            JcaX509CRLConverter jcaX509CRLConverter = new JcaX509CRLConverter();
            X509CRL x509CRL = jcaX509CRLConverter.getCRL(x509CRLHolder);
            return x509CRL;
        }
        catch (CRLException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static byte[] getEncoded(X509CRL x509CRL) {
        try {
            byte[] encoded = x509CRL.getEncoded();
            return encoded;
        }
        catch (CRLException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static byte[] getEncoded(BasicOCSPResp basicOCSPResp) {
        try {
            byte[] encoded = basicOCSPResp.getEncoded();
            return encoded;
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static byte[] getEncoded(OCSPResp ocspResp) {
        try {
            byte[] encoded = ocspResp.getEncoded();
            return encoded;
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static String getDeterministicId(Date signingTime, TokenIdentifier id) {
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            if (signingTime != null) {
                baos.write(Long.toString(signingTime.getTime()).getBytes());
            }
            if (id != null) {
                baos.write(id.asXmlId().getBytes());
            }
            String deterministicId = "id-" + DSSUtils.getMD5Digest(baos.toByteArray());
            return deterministicId;
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static String getMD5Digest(byte[] bytes) {
        try {
            byte[] digestValue = DSSUtils.digest(DigestAlgorithm.MD5, bytes);
            return Utils.toHex((byte[])digestValue);
        }
        catch (Exception e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static Date getLocalDate(Date gtmDate, Date localDate) {
        Date newLocalDate = new Date(gtmDate.getTime() + (long)TimeZone.getDefault().getOffset(localDate.getTime()));
        return newLocalDate;
    }

    public static long toLong(byte[] bytes) {
        ByteBuffer buffer = ByteBuffer.allocate(8);
        buffer.put(bytes, 0, 8);
        buffer.flip();
        return buffer.getLong();
    }

    public static X500Principal getX500PrincipalOrNull(String x500PrincipalString) {
        try {
            X500Principal x500Principal = new X500Principal(x500PrincipalString);
            return x500Principal;
        }
        catch (Exception e) {
            LOG.warn(e.getMessage());
            return null;
        }
    }

    public static boolean x500PrincipalAreEquals(X500Principal firstX500Principal, X500Principal secondX500Principal) {
        if (firstX500Principal == null || secondX500Principal == null) {
            return false;
        }
        if (firstX500Principal.equals(secondX500Principal)) {
            return true;
        }
        Map<String, String> firstStringStringHashMap = DSSASN1Utils.get(firstX500Principal);
        Map<String, String> secondStringStringHashMap = DSSASN1Utils.get(secondX500Principal);
        boolean containsAll = firstStringStringHashMap.entrySet().containsAll(secondStringStringHashMap.entrySet());
        return containsAll;
    }

    public static X500Principal getX500Principal(String x509SubjectName) throws DSSException {
        try {
            X500Principal x500Principal = new X500Principal(x509SubjectName);
            return DSSUtils.getNormalizedX500Principal(x500Principal);
        }
        catch (IllegalArgumentException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static X500Principal getNormalizedX500Principal(X500Principal x500Principal) {
        String utf8Name = DSSASN1Utils.getUtf8String(x500Principal);
        X500Principal x500PrincipalNormalized = new X500Principal(utf8Name);
        return x500PrincipalNormalized;
    }

    public static InputStream getResource(String resourcePath) {
        InputStream resourceAsStream = DSSUtils.class.getClassLoader().getResourceAsStream(resourcePath);
        return resourceAsStream;
    }

    public static Date getUtcDate(int year, int month, int day) {
        Calendar calendar = Calendar.getInstance(TimeZone.getTimeZone("UTC"));
        calendar.set(year, month, day, 0, 0, 0);
        Date date = calendar.getTime();
        return date;
    }

    public static Date getDate(Date date, int days) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(5, days);
        Date newDate = calendar.getTime();
        return newDate;
    }

    public static byte[] getUtf8Bytes(String string) {
        if (string == null) {
            return null;
        }
        try {
            byte[] bytes = string.getBytes("UTF-8");
            return bytes;
        }
        catch (UnsupportedEncodingException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static String getMessageId(String message) {
        String message_ = message.toLowerCase().replaceAll("[^a-z_]", " ");
        StringBuilder nameId = new StringBuilder();
        StringTokenizer stringTokenizer = new StringTokenizer(message_);
        while (stringTokenizer.hasMoreElements()) {
            String word = (String)stringTokenizer.nextElement();
            nameId.append(word.charAt(0));
        }
        String nameIdString = nameId.toString();
        return nameIdString.toUpperCase();
    }

    public static int available(InputStream is) throws DSSException {
        try {
            return is.available();
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static void printSecurityProvides() {
        Provider[] providers;
        for (Provider provider : providers = Security.getProviders()) {
            System.out.println("PROVIDER: " + provider.getName());
            Set<Provider.Service> services = provider.getServices();
            for (Provider.Service service : services) {
                System.out.println("\tALGORITHM: " + service.getAlgorithm() + " / " + service.getType() + " / " + service.getClassName());
            }
        }
    }

    public static String getSummaryMessage(Exception exception, Class<?> javaClass) {
        String javaClassName = javaClass.getName();
        StackTraceElement[] stackTrace = exception.getStackTrace();
        String message = "See log file for full stack trace.\n";
        message = message + exception.toString() + '\n';
        for (StackTraceElement element : stackTrace) {
            String className = element.getClassName();
            if (className.equals(javaClassName)) {
                message = message + element.toString() + '\n';
                break;
            }
            message = message + element.toString() + '\n';
        }
        return message;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static int readToArray(DSSDocument dssDocument, int headerLength, byte[] destinationByteArray) {
        try (InputStream inputStream = dssDocument.openStream();){
            int read;
            int n = read = inputStream.read(destinationByteArray, 0, headerLength);
            return n;
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static long getDateDiff(Date date1, Date date2, TimeUnit timeUnit) {
        long diff = date2.getTime() - date1.getTime();
        return timeUnit.convert(diff, TimeUnit.MILLISECONDS);
    }

    public static byte[] concatenate(byte[] ... arrays) {
        if (arrays == null || arrays.length == 0 || arrays.length == 1 && arrays[0] == null) {
            return null;
        }
        if (arrays.length == 1) {
            return (byte[])arrays[0].clone();
        }
        int joinedLength = 0;
        for (byte[] array : arrays) {
            if (array == null) continue;
            joinedLength += array.length;
        }
        byte[] joinedArray = new byte[joinedLength];
        int destinationIndex = 0;
        for (byte[] array : arrays) {
            if (array == null) continue;
            System.arraycopy(array, 0, joinedArray, destinationIndex, array.length);
            destinationIndex += array.length;
        }
        return joinedArray;
    }

    public static String getFinalFileName(DSSDocument originalFile, SigningOperation operation, SignatureLevel level, ASiCContainerType containerType) {
        StringBuilder finalName = new StringBuilder();
        String originalName = null;
        originalName = containerType != null ? "container" : originalFile.getName();
        if (Utils.isStringNotEmpty((String)originalName)) {
            int dotPosition = originalName.lastIndexOf(46);
            if (dotPosition > 0) {
                finalName.append(originalName.substring(0, dotPosition));
            } else {
                finalName.append(originalName);
            }
        } else {
            finalName.append("document");
        }
        if (SigningOperation.SIGN.equals((Object)operation)) {
            finalName.append("-signed-");
        } else if (SigningOperation.EXTEND.equals((Object)operation)) {
            finalName.append("-extended-");
        }
        finalName.append(Utils.lowerCase((String)level.name().replaceAll("_", "-")));
        finalName.append('.');
        if (containerType != null) {
            switch (containerType) {
                case ASiC_S: {
                    finalName.append("asics");
                    break;
                }
                case ASiC_E: {
                    finalName.append("asice");
                    break;
                }
            }
        } else {
            SignatureForm signatureForm = level.getSignatureForm();
            switch (signatureForm) {
                case XAdES: {
                    finalName.append("xml");
                    break;
                }
                case CAdES: {
                    finalName.append("pkcs7");
                    break;
                }
                case PAdES: {
                    finalName.append("pdf");
                    break;
                }
            }
        }
        return finalName.toString();
    }

    public static String getFinalFileName(DSSDocument originalFile, SigningOperation operation, SignatureLevel level) {
        return DSSUtils.getFinalFileName(originalFile, operation, level, null);
    }

    public static String decodeUrl(String uri) {
        try {
            return URLDecoder.decode(uri, "UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            LOG.error("Unable to decode '" + uri + "' : " + e.getMessage(), (Throwable)e);
            return uri;
        }
    }

    static {
        EMPTY_BYTE_ARRAY = new byte[0];
        try {
            Security.addProvider((Provider)securityProvider);
            certificateFactory = CertificateFactory.getInstance("X.509", "BC");
        }
        catch (CertificateException e) {
            LOG.error(e.getMessage(), (Throwable)e);
            throw new DSSException("Platform does not support X509 certificate", (Throwable)e);
        }
        catch (NoSuchProviderException e) {
            LOG.error(e.getMessage(), (Throwable)e);
            throw new DSSException("Platform does not support BouncyCastle", (Throwable)e);
        }
    }
}

