/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.tsl.service;

import eu.europa.esig.dss.DSSException;
import eu.europa.esig.dss.DSSUtils;
import eu.europa.esig.dss.DigestAlgorithm;
import eu.europa.esig.dss.tsl.Condition;
import eu.europa.esig.dss.tsl.ServiceInfo;
import eu.europa.esig.dss.tsl.ServiceInfoStatus;
import eu.europa.esig.dss.tsl.TLInfo;
import eu.europa.esig.dss.tsl.TSLConditionsForQualifiers;
import eu.europa.esig.dss.tsl.TSLLoaderResult;
import eu.europa.esig.dss.tsl.TSLParserResult;
import eu.europa.esig.dss.tsl.TSLService;
import eu.europa.esig.dss.tsl.TSLServiceProvider;
import eu.europa.esig.dss.tsl.TSLServiceStatusAndInformationExtensions;
import eu.europa.esig.dss.tsl.TSLValidationModel;
import eu.europa.esig.dss.tsl.TSLValidationResult;
import eu.europa.esig.dss.tsl.TrustedListsCertificateSource;
import eu.europa.esig.dss.tsl.service.IgnorePivotFilenameFilter;
import eu.europa.esig.dss.util.MutableTimeDependentValues;
import eu.europa.esig.dss.util.TimeDependent;
import eu.europa.esig.dss.util.TimeDependentValues;
import eu.europa.esig.dss.utils.Utils;
import eu.europa.esig.dss.x509.CertificateToken;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import javax.xml.bind.DatatypeConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TSLRepository {
    private static final Logger LOG = LoggerFactory.getLogger(TSLRepository.class);
    private String cacheDirectoryPath = System.getProperty("java.io.tmpdir") + File.separator + "dss-cache-tsl" + File.separator;
    private Map<String, TSLValidationModel> tsls = new HashMap<String, TSLValidationModel>();
    private Map<String, TSLValidationModel> pivots = new HashMap<String, TSLValidationModel>();
    private TrustedListsCertificateSource trustedListsCertificateSource;

    public void setCacheDirectoryPath(String cacheDirectoryPath) {
        this.cacheDirectoryPath = cacheDirectoryPath;
    }

    public void setTrustedListsCertificateSource(TrustedListsCertificateSource trustedListsCertificateSource) {
        this.trustedListsCertificateSource = trustedListsCertificateSource;
    }

    public TSLValidationModel getByCountry(String countryIsoCode) {
        return this.tsls.get(countryIsoCode);
    }

    public TSLValidationModel getPivotByUrl(String pivotUrl) {
        return this.pivots.get(pivotUrl);
    }

    public Map<String, TSLValidationModel> getAllMapTSLValidationModels() {
        return Collections.unmodifiableMap(new TreeMap<String, TSLValidationModel>(this.tsls));
    }

    public void clearRepository() {
        try {
            Utils.cleanDirectory((File)new File(this.cacheDirectoryPath));
            this.tsls.clear();
        }
        catch (IOException e) {
            LOG.error("Unable to clean cache directory : " + e.getMessage(), (Throwable)e);
        }
    }

    boolean isLastCountryVersion(TSLLoaderResult resultLoader) {
        TSLValidationModel validationModel = this.getByCountry(resultLoader.getCountryCode());
        return this.isLastVersion(validationModel, resultLoader);
    }

    boolean isLastPivotVersion(TSLLoaderResult resultLoader) {
        TSLValidationModel validationModel = this.getPivotByUrl(resultLoader.getUrl());
        return this.isLastVersion(validationModel, resultLoader);
    }

    private boolean isLastVersion(TSLValidationModel validationModel, TSLLoaderResult resultLoader) {
        if (validationModel == null) {
            return false;
        }
        if (Utils.isArrayEmpty((byte[])resultLoader.getContent())) {
            return true;
        }
        validationModel.setUrl(resultLoader.getUrl());
        validationModel.setLoadedDate(new Date());
        String lastSha256 = this.getSHA256(resultLoader.getContent());
        return Utils.areStringsEqual((String)lastSha256, (String)validationModel.getSha256FileContent());
    }

    void updateParseResult(TSLParserResult tslParserResult) {
        TSLValidationModel validationModel = this.getByCountry(tslParserResult.getTerritory());
        if (validationModel != null) {
            validationModel.setParseResult(tslParserResult);
        }
    }

    void updateValidationResult(TSLValidationResult tslValidationResult) {
        TSLValidationModel validationModel = this.getByCountry(tslValidationResult.getCountryCode());
        if (validationModel != null) {
            validationModel.setValidationResult(tslValidationResult);
        }
    }

    TSLValidationModel storeInCache(TSLLoaderResult resultLoader) {
        TSLValidationModel validationModel = new TSLValidationModel();
        validationModel.setUrl(resultLoader.getUrl());
        validationModel.setSha256FileContent(this.getSHA256(resultLoader.getContent()));
        validationModel.setLoadedDate(new Date());
        validationModel.setFilepath(this.storeOnFileSystem(resultLoader.getCountryCode(), resultLoader));
        validationModel.setCertificateSourceSynchronized(false);
        this.tsls.put(resultLoader.getCountryCode(), validationModel);
        LOG.info("New version of " + resultLoader.getCountryCode() + " TSL is stored in cache");
        return validationModel;
    }

    TSLValidationModel storePivotInCache(TSLLoaderResult resultLoader) {
        TSLValidationModel validationModel = new TSLValidationModel();
        validationModel.setUrl(resultLoader.getUrl());
        validationModel.setSha256FileContent(this.getSHA256(resultLoader.getContent()));
        validationModel.setLoadedDate(new Date());
        String filename = resultLoader.getUrl();
        filename = filename.replaceAll("\\W", "_");
        validationModel.setFilepath(this.storeOnFileSystem(filename, resultLoader));
        this.pivots.put(resultLoader.getUrl(), validationModel);
        LOG.info("New version of the pivot LOTL '" + resultLoader.getUrl() + "' is stored in cache");
        return validationModel;
    }

    void addParsedResultFromCacheToMap(TSLParserResult tslParserResult) {
        TSLValidationModel validationModel = new TSLValidationModel();
        String countryCode = tslParserResult.getTerritory();
        String filePath = this.getFilePath(countryCode);
        validationModel.setFilepath(filePath);
        try (FileInputStream fis = new FileInputStream(filePath);){
            byte[] data = Utils.toByteArray((InputStream)fis);
            validationModel.setSha256FileContent(this.getSHA256(data));
        }
        catch (Exception e) {
            LOG.error("Unable to read '" + filePath + "' : " + e.getMessage());
        }
        validationModel.setParseResult(tslParserResult);
        validationModel.setCertificateSourceSynchronized(false);
        this.tsls.put(countryCode, validationModel);
    }

    private String storeOnFileSystem(String filename, TSLLoaderResult resultLoader) {
        this.ensureCacheDirectoryExists();
        String filePath = this.getFilePath(filename);
        File fileToCreate = new File(filePath);
        try (FileOutputStream os = new FileOutputStream(fileToCreate);){
            Utils.write((byte[])resultLoader.getContent(), (OutputStream)os);
        }
        catch (Exception e) {
            throw new DSSException("Cannot create file in cache : " + e.getMessage(), (Throwable)e);
        }
        return filePath;
    }

    private void ensureCacheDirectoryExists() {
        File cacheDir = new File(this.cacheDirectoryPath);
        if (!cacheDir.exists() || !cacheDir.isDirectory()) {
            cacheDir.mkdirs();
        }
    }

    private String getFilePath(String filename) {
        return this.cacheDirectoryPath + filename + ".xml";
    }

    private String getSHA256(byte[] data) {
        return DatatypeConverter.printBase64Binary((byte[])DSSUtils.digest((DigestAlgorithm)DigestAlgorithm.SHA256, (byte[])data));
    }

    List<File> getStoredFiles() {
        this.ensureCacheDirectoryExists();
        File cacheDir = new File(this.cacheDirectoryPath);
        File[] listFiles = cacheDir.listFiles(new IgnorePivotFilenameFilter());
        return Arrays.asList(listFiles);
    }

    void synchronize() {
        if (this.trustedListsCertificateSource != null) {
            Map<String, TSLValidationModel> allMapTSLValidationModels = this.getAllMapTSLValidationModels();
            for (Map.Entry<String, TSLValidationModel> entry : allMapTSLValidationModels.entrySet()) {
                String countryCode = entry.getKey();
                TSLValidationModel model = entry.getValue();
                if (!model.isCertificateSourceSynchronized()) {
                    TSLParserResult parseResult = model.getParseResult();
                    if (parseResult != null) {
                        List<TSLServiceProvider> serviceProviders = parseResult.getServiceProviders();
                        for (TSLServiceProvider serviceProvider : serviceProviders) {
                            for (TSLService service : serviceProvider.getServices()) {
                                for (CertificateToken certificate : service.getCertificates()) {
                                    this.trustedListsCertificateSource.removeCertificate(certificate);
                                    this.trustedListsCertificateSource.addCertificate(certificate, this.getServiceInfo(serviceProvider, service, countryCode));
                                }
                            }
                        }
                    }
                    model.setCertificateSourceSynchronized(true);
                }
                this.trustedListsCertificateSource.updateTlInfo(countryCode, this.getTlInfo(countryCode, model));
            }
            LOG.info("Nb of loaded trusted lists : " + allMapTSLValidationModels.size());
            LOG.info("Nb of trusted certificates : " + this.trustedListsCertificateSource.getNumberOfTrustedCertificates());
        }
    }

    private TLInfo getTlInfo(String countryCode, TSLValidationModel model) {
        TSLValidationResult validationResult;
        TLInfo info = new TLInfo();
        info.setCountryCode(countryCode);
        info.setUrl(model.getUrl());
        info.setLastLoading(model.getLoadedDate());
        info.setLotl(model.isLotl());
        TSLParserResult parseResult = model.getParseResult();
        if (parseResult != null) {
            info.setIssueDate(parseResult.getIssueDate());
            info.setNextUpdate(parseResult.getNextUpdateDate());
            info.setSequenceNumber(parseResult.getSequenceNumber());
            info.setVersion(parseResult.getVersion());
            int nbServiceProviders = 0;
            int nbServices = 0;
            int nbCertificates = 0;
            List<TSLServiceProvider> serviceProviders = parseResult.getServiceProviders();
            if (serviceProviders != null) {
                nbServiceProviders = serviceProviders.size();
                for (TSLServiceProvider tslServiceProvider : serviceProviders) {
                    List<TSLService> services = tslServiceProvider.getServices();
                    if (services == null) continue;
                    nbServices += services.size();
                    for (TSLService tslService : services) {
                        List<CertificateToken> certificates = tslService.getCertificates();
                        nbCertificates += Utils.collectionSize(certificates);
                    }
                }
            }
            info.setNbServiceProviders(nbServiceProviders);
            info.setNbServices(nbServices);
            info.setNbCertificates(nbCertificates);
        }
        if ((validationResult = model.getValidationResult()) != null) {
            info.setWellSigned(validationResult.isValid());
        }
        return info;
    }

    private ServiceInfo getServiceInfo(TSLServiceProvider serviceProvider, TSLService service, String countryCode) {
        ServiceInfo serviceInfo = new ServiceInfo();
        serviceInfo.setTspName(serviceProvider.getName());
        serviceInfo.setTspTradeName(serviceProvider.getTradeName());
        serviceInfo.setTspPostalAddress(serviceProvider.getPostalAddress());
        serviceInfo.setTspElectronicAddress(serviceProvider.getElectronicAddress());
        serviceInfo.setServiceName(service.getName());
        MutableTimeDependentValues status = new MutableTimeDependentValues();
        TimeDependentValues<TSLServiceStatusAndInformationExtensions> serviceStatus = service.getStatusAndInformationExtensions();
        if (serviceStatus != null) {
            for (TSLServiceStatusAndInformationExtensions tslServiceStatus : serviceStatus) {
                Map<String, List<Condition>> qualifiersAndConditions = this.getMapConditionsByQualifier(tslServiceStatus);
                ServiceInfoStatus s = new ServiceInfoStatus(tslServiceStatus.getType(), tslServiceStatus.getStatus(), qualifiersAndConditions, tslServiceStatus.getAdditionalServiceInfoUris(), tslServiceStatus.getServiceSupplyPoints(), tslServiceStatus.getExpiredCertsRevocationInfo(), tslServiceStatus.getStartDate(), tslServiceStatus.getEndDate());
                status.addOldest((TimeDependent)s);
            }
        }
        serviceInfo.setStatus((TimeDependentValues)status);
        serviceInfo.setTlCountryCode(countryCode);
        return serviceInfo;
    }

    private Map<String, List<Condition>> getMapConditionsByQualifier(TSLServiceStatusAndInformationExtensions tslServiceStatus) {
        List<TSLConditionsForQualifiers> conditionsForQualifiers = tslServiceStatus.getConditionsForQualifiers();
        HashMap<String, List<Condition>> qualifiersAndConditions = new HashMap<String, List<Condition>>();
        if (conditionsForQualifiers != null) {
            for (TSLConditionsForQualifiers tslConditionsForQualifiers : conditionsForQualifiers) {
                Condition condition = tslConditionsForQualifiers.getCondition();
                for (String qualifier : tslConditionsForQualifiers.getQualifiers()) {
                    ArrayList<Condition> conditionsForQualif = (ArrayList<Condition>)qualifiersAndConditions.get(qualifier);
                    if (conditionsForQualif == null) {
                        conditionsForQualif = new ArrayList<Condition>();
                        qualifiersAndConditions.put(qualifier, conditionsForQualif);
                    }
                    conditionsForQualif.add(condition);
                }
            }
        }
        return qualifiersAndConditions;
    }

    public Map<String, TLInfo> getSummary() {
        if (this.trustedListsCertificateSource != null) {
            return Collections.unmodifiableMap(new TreeMap(this.trustedListsCertificateSource.getSummary()));
        }
        return Collections.emptyMap();
    }
}

