/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.tsl.service;

import eu.europa.esig.dss.DSSException;
import eu.europa.esig.dss.client.http.DataLoader;
import eu.europa.esig.dss.tsl.TSLLoaderResult;
import eu.europa.esig.dss.tsl.TSLParserResult;
import eu.europa.esig.dss.tsl.TSLPointer;
import eu.europa.esig.dss.tsl.TSLValidationModel;
import eu.europa.esig.dss.tsl.TSLValidationResult;
import eu.europa.esig.dss.tsl.service.TSLLoader;
import eu.europa.esig.dss.tsl.service.TSLParser;
import eu.europa.esig.dss.tsl.service.TSLRepository;
import eu.europa.esig.dss.tsl.service.TSLValidator;
import eu.europa.esig.dss.utils.Utils;
import eu.europa.esig.dss.x509.CertificateToken;
import eu.europa.esig.dss.x509.KeyStoreCertificateSource;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TSLValidationJob {
    private static final Logger LOG = LoggerFactory.getLogger(TSLValidationJob.class);
    private ExecutorService executorService = Executors.newCachedThreadPool();
    private DataLoader dataLoader;
    private TSLRepository repository;
    private String lotlCode;
    private String lotlUrl;
    private String lotlRootSchemeInfoUri;
    private String ojUrl;
    private KeyStoreCertificateSource ojContentKeyStore;
    private boolean checkLOTLSignature = true;
    private boolean checkTSLSignatures = true;
    private List<String> filterTerritories;

    public void setExecutorService(ExecutorService executorService) {
        if (this.executorService != null && !this.executorService.isShutdown()) {
            this.executorService.shutdownNow();
        }
        this.executorService = executorService;
    }

    public void setDataLoader(DataLoader dataLoader) {
        this.dataLoader = dataLoader;
    }

    public void setRepository(TSLRepository repository) {
        this.repository = repository;
    }

    public void setLotlCode(String lotlCode) {
        this.lotlCode = lotlCode;
    }

    public void setLotlUrl(String lotlUrl) {
        this.lotlUrl = lotlUrl;
    }

    public void setLotlRootSchemeInfoUri(String lotlRootSchemeInfoUri) {
        this.lotlRootSchemeInfoUri = lotlRootSchemeInfoUri;
    }

    public void setOjUrl(String ojUrl) {
        this.ojUrl = ojUrl;
    }

    public void setOjContentKeyStore(KeyStoreCertificateSource ojContentKeyStore) {
        this.ojContentKeyStore = ojContentKeyStore;
    }

    public void setCheckLOTLSignature(boolean checkLOTLSignature) {
        this.checkLOTLSignature = checkLOTLSignature;
    }

    public void setCheckTSLSignatures(boolean checkTSLSignatures) {
        this.checkTSLSignatures = checkTSLSignatures;
    }

    public void setFilterTerritories(List<String> filterTerritories) {
        this.filterTerritories = filterTerritories;
    }

    public void initRepository() {
        LOG.info("Initialization of the TSL repository ...");
        int loadedTSL = 0;
        List<File> cachedFiles = this.repository.getStoredFiles();
        if (Utils.isCollectionNotEmpty(cachedFiles)) {
            ArrayList<Future<TSLParserResult>> futureParseResults = new ArrayList<Future<TSLParserResult>>();
            for (File file : cachedFiles) {
                try {
                    futureParseResults.add(this.executorService.submit(new TSLParser(file.getAbsolutePath())));
                }
                catch (Exception e) {
                    LOG.error("Unable to parse file '" + file.getAbsolutePath() + "' : " + e.getMessage(), (Throwable)e);
                }
            }
            for (Future future : futureParseResults) {
                try {
                    TSLParserResult tslParserResult = (TSLParserResult)future.get();
                    this.repository.addParsedResultFromCacheToMap(tslParserResult);
                    ++loadedTSL;
                }
                catch (Exception e) {
                    LOG.error("Unable to get parsing result : " + e.getMessage(), (Throwable)e);
                }
            }
            TSLValidationModel europeanModel = this.repository.getByCountry(this.lotlCode);
            if (this.checkLOTLSignature && europeanModel != null) {
                try {
                    TSLValidationResult tSLValidationResult = this.validateLOTL(europeanModel, this.ojContentKeyStore.getCertificates());
                    europeanModel.setValidationResult(tSLValidationResult);
                }
                catch (Exception exception) {
                    LOG.error("Unable to validate the LOTL : " + exception.getMessage(), (Throwable)exception);
                }
            }
            if (this.checkTSLSignatures && europeanModel != null && europeanModel.getParseResult() != null) {
                List<TSLPointer> list = europeanModel.getParseResult().getPointers();
                ArrayList<Future<TSLValidationResult>> futureValidationResults = new ArrayList<Future<TSLValidationResult>>();
                Map<String, TSLValidationModel> map = this.repository.getAllMapTSLValidationModels();
                for (Map.Entry<String, TSLValidationModel> entry : map.entrySet()) {
                    String countryCode = entry.getKey();
                    if (this.lotlCode.equals(countryCode)) continue;
                    TSLValidationModel countryModel = entry.getValue();
                    TSLValidator tslValidator = new TSLValidator(new File(countryModel.getFilepath()), countryCode, this.getPotentialSigners(list, countryCode));
                    futureValidationResults.add(this.executorService.submit(tslValidator));
                }
                this.storeValidationResults(futureValidationResults);
            }
            this.repository.synchronize();
        }
        LOG.info(loadedTSL + " loaded TSL from cached files in the repository");
    }

    public void refresh() {
        LOG.debug("TSL Validation Job is starting ...");
        TSLLoaderResult resultLoaderLOTL = null;
        Future<TSLLoaderResult> result = this.executorService.submit(new TSLLoader(this.dataLoader, this.lotlCode, this.lotlUrl));
        try {
            resultLoaderLOTL = result.get();
        }
        catch (Exception e) {
            LOG.error("Unable to load the LOTL : " + e.getMessage(), (Throwable)e);
            throw new DSSException("Unable to load the LOTL : " + e.getMessage(), (Throwable)e);
        }
        if (resultLoaderLOTL.getContent() == null) {
            LOG.error("Unable to load the LOTL: content is empty");
            throw new DSSException("Unable to load the LOTL: content is empty");
        }
        TSLValidationModel europeanModel = null;
        boolean newLotl = !this.repository.isLastCountryVersion(resultLoaderLOTL);
        europeanModel = newLotl ? this.repository.storeInCache(resultLoaderLOTL) : this.repository.getByCountry(resultLoaderLOTL.getCountryCode());
        TSLParserResult parseResult = europeanModel.getParseResult();
        if (parseResult == null) {
            try {
                parseResult = this.parseLOTL(europeanModel);
                europeanModel.setParseResult(parseResult);
            }
            catch (Exception e) {
                LOG.error("Unable to parse the LOTL : " + e.getMessage(), (Throwable)e);
                return;
            }
        }
        if (!this.isLatestOjKeystore(parseResult)) {
            LOG.warn("OJ keystore is out-dated !");
        }
        this.checkLOTLLocation(parseResult);
        ArrayList<CertificateToken> allowedLotlSigners = new ArrayList<CertificateToken>();
        allowedLotlSigners.addAll(this.ojContentKeyStore.getCertificates());
        if (this.isPivotLOTL(parseResult)) {
            this.extractAllowedLotlSignersFromPivots(parseResult, allowedLotlSigners);
        }
        if (this.checkLOTLSignature && (europeanModel.getValidationResult() == null || !europeanModel.getValidationResult().isValid())) {
            try {
                TSLValidationResult validationResult = this.validateLOTL(europeanModel, allowedLotlSigners);
                europeanModel.setValidationResult(validationResult);
            }
            catch (Exception e) {
                LOG.error("Unable to validate the LOTL : " + e.getMessage(), (Throwable)e);
            }
        }
        this.analyzeCountryPointers(parseResult.getPointers(), newLotl);
        this.repository.synchronize();
        LOG.debug("TSL Validation Job is finishing ...");
    }

    private void checkLOTLLocation(TSLParserResult parseResult) {
        List<TSLPointer> pointers = parseResult.getPointers();
        for (TSLPointer tslPointer : pointers) {
            if (!Utils.areStringsEqual((String)this.lotlCode, (String)tslPointer.getTerritory())) continue;
            if (Utils.areStringsEqual((String)this.lotlUrl, (String)tslPointer.getUrl())) break;
            LOG.warn("The LOTL URL has been changed ! Please update your properties (new value : {})", (Object)tslPointer.getUrl());
            break;
        }
    }

    private void extractAllowedLotlSignersFromPivots(TSLParserResult parseResult, List<CertificateToken> allowedLotlSigners) {
        LinkedList<Future<TSLLoaderResult>> pivotLoaderResults = new LinkedList<Future<TSLLoaderResult>>();
        List<String> pivotUris = this.getPivotUris(parseResult);
        for (String string : pivotUris) {
            pivotLoaderResults.add(this.executorService.submit(new TSLLoader(this.dataLoader, this.lotlCode, string)));
        }
        for (Future future : pivotLoaderResults) {
            try {
                TSLLoaderResult loaderResult = (TSLLoaderResult)future.get();
                if (loaderResult == null || loaderResult.getContent() == null) continue;
                TSLValidationModel pivotModel = null;
                pivotModel = !this.repository.isLastPivotVersion(loaderResult) ? this.repository.storePivotInCache(loaderResult) : this.repository.getPivotByUrl(loaderResult.getUrl());
                TSLParserResult pivotParseResult = pivotModel.getParseResult();
                if (pivotParseResult == null) {
                    Future<TSLParserResult> parseResultFuture = this.executorService.submit(new TSLParser(pivotModel.getFilepath()));
                    pivotParseResult = parseResultFuture.get();
                }
                TSLValidationResult pivotValidationResult = pivotModel.getValidationResult();
                if (this.checkLOTLSignature && pivotValidationResult == null) {
                    TSLValidator tslValidator = new TSLValidator(new File(pivotModel.getFilepath()), loaderResult.getCountryCode(), allowedLotlSigners);
                    Future<TSLValidationResult> pivotValidationFuture = this.executorService.submit(tslValidator);
                    pivotValidationResult = pivotValidationFuture.get();
                }
                if (pivotValidationResult.isValid()) {
                    List<CertificateToken> certs = this.getCertificatesForLOTLPointer(loaderResult, pivotParseResult);
                    allowedLotlSigners.clear();
                    allowedLotlSigners.addAll(certs);
                    continue;
                }
                LOG.warn("Pivot '" + loaderResult.getUrl() + "' is not valid");
            }
            catch (Exception e) {
                LOG.error("Unable to validate the pivot LOTL : " + e.getMessage(), (Throwable)e);
            }
        }
    }

    private List<CertificateToken> getCertificatesForLOTLPointer(TSLLoaderResult loaderResult, TSLParserResult pivotParseResult) {
        List<TSLPointer> pointers = pivotParseResult.getPointers();
        for (TSLPointer tslPointer : pointers) {
            if (!Utils.areStringsEqual((String)tslPointer.getTerritory(), (String)this.lotlCode)) continue;
            return tslPointer.getPotentialSigners();
        }
        LOG.warn("No LOTL pointer in pivot '" + loaderResult.getUrl() + "'");
        return new ArrayList<CertificateToken>();
    }

    private boolean isLatestOjKeystore(TSLParserResult parseResult) {
        List<String> englishSchemeInformationURIs = parseResult.getEnglishSchemeInformationURIs();
        return englishSchemeInformationURIs.contains(this.ojUrl);
    }

    private boolean isPivotLOTL(TSLParserResult parseResult) {
        return Utils.isCollectionNotEmpty(this.getPivotUris(parseResult));
    }

    private List<String> getPivotUris(TSLParserResult parseResult) {
        LinkedList<String> pivotUris = new LinkedList<String>();
        LinkedList englishSchemeInformationURIs = (LinkedList)parseResult.getEnglishSchemeInformationURIs();
        Iterator itr = englishSchemeInformationURIs.descendingIterator();
        while (itr.hasNext()) {
            String uri = (String)itr.next();
            if (Utils.areStringsEqual((String)this.ojUrl, (String)uri) || uri.startsWith(this.lotlRootSchemeInfoUri)) continue;
            pivotUris.add(uri);
        }
        return pivotUris;
    }

    private void analyzeCountryPointers(List<TSLPointer> pointers, boolean newLotl) {
        ArrayList<Future<TSLLoaderResult>> futureLoaderResults = new ArrayList<Future<TSLLoaderResult>>();
        for (TSLPointer tslPointer : pointers) {
            if (!Utils.isCollectionEmpty(this.filterTerritories) && !this.filterTerritories.contains(tslPointer.getTerritory())) continue;
            TSLLoader tslLoader = new TSLLoader(this.dataLoader, tslPointer.getTerritory(), tslPointer.getUrl());
            futureLoaderResults.add(this.executorService.submit(tslLoader));
        }
        ArrayList<Future<TSLParserResult>> futureParseResults = new ArrayList<Future<TSLParserResult>>();
        ArrayList<Future<TSLValidationResult>> futureValidationResults = new ArrayList<Future<TSLValidationResult>>();
        for (Future future : futureLoaderResults) {
            try {
                TSLLoaderResult loaderResult = (TSLLoaderResult)future.get();
                if (loaderResult.getContent() == null || loaderResult.getContent().length == 0) {
                    LOG.error("Failed to load TSl for country '" + loaderResult.getCountryCode() + "' from '" + loaderResult.getUrl() + "'");
                    continue;
                }
                if (loaderResult == null || loaderResult.getContent() == null) continue;
                TSLValidationModel countryModel = null;
                countryModel = !this.repository.isLastCountryVersion(loaderResult) ? this.repository.storeInCache(loaderResult) : this.repository.getByCountry(loaderResult.getCountryCode());
                TSLParserResult countryParseResult = countryModel.getParseResult();
                if (countryParseResult == null) {
                    futureParseResults.add(this.executorService.submit(new TSLParser(countryModel.getFilepath())));
                }
                if (!this.checkTSLSignatures || countryModel.getValidationResult() != null && !newLotl) continue;
                TSLValidator tslValidator = new TSLValidator(new File(countryModel.getFilepath()), loaderResult.getCountryCode(), this.getPotentialSigners(pointers, loaderResult.getCountryCode()));
                futureValidationResults.add(this.executorService.submit(tslValidator));
            }
            catch (Exception e) {
                LOG.error("Unable to load/parse TSL : " + e.getMessage(), (Throwable)e);
            }
        }
        for (Future future : futureParseResults) {
            try {
                TSLParserResult tslParserResult = (TSLParserResult)future.get();
                this.repository.updateParseResult(tslParserResult);
            }
            catch (Exception e) {
                LOG.error("Unable to get parsing result : " + e.getMessage(), (Throwable)e);
            }
        }
        this.storeValidationResults(futureValidationResults);
    }

    private void storeValidationResults(List<Future<TSLValidationResult>> futureValidationResults) {
        for (Future<TSLValidationResult> futureValidationResult : futureValidationResults) {
            try {
                TSLValidationResult tslValidationResult = futureValidationResult.get();
                this.repository.updateValidationResult(tslValidationResult);
            }
            catch (Exception e) {
                LOG.error("Unable to get validation result : " + e.getMessage(), (Throwable)e);
            }
        }
    }

    private List<CertificateToken> getPotentialSigners(List<TSLPointer> pointers, String countryCode) {
        if (Utils.isCollectionNotEmpty(pointers)) {
            for (TSLPointer tslPointer : pointers) {
                if (!Utils.areStringsEqual((String)countryCode, (String)tslPointer.getTerritory())) continue;
                return tslPointer.getPotentialSigners();
            }
        }
        return Collections.emptyList();
    }

    private TSLValidationResult validateLOTL(TSLValidationModel validationModel, List<CertificateToken> allowedSigners) throws Exception {
        validationModel.setLotl(true);
        TSLValidator tslValidator = new TSLValidator(new File(validationModel.getFilepath()), this.lotlCode, allowedSigners);
        Future<TSLValidationResult> future = this.executorService.submit(tslValidator);
        return future.get();
    }

    private TSLParserResult parseLOTL(TSLValidationModel validationModel) throws Exception {
        Future<TSLParserResult> future = this.executorService.submit(new TSLParser(validationModel.getFilepath()));
        return future.get();
    }
}

