/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.http;

import java.net.URI;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.function.Predicate;
import org.eclipse.jetty.http.HttpCookie;
import org.eclipse.jetty.http.HttpScheme;
import org.eclipse.jetty.util.NanoTime;
import org.eclipse.jetty.util.StringUtil;
import org.eclipse.jetty.util.thread.AutoLock;

public interface HttpCookieStore {
    public boolean add(URI var1, HttpCookie var2);

    public List<HttpCookie> all();

    public List<HttpCookie> match(URI var1);

    public boolean remove(URI var1, HttpCookie var2);

    public boolean clear();

    public static class Default
    implements HttpCookieStore {
        private final AutoLock lock = new AutoLock();
        private final Map<String, List<StoredHttpCookie>> cookies = new HashMap<String, List<StoredHttpCookie>>();

        @Override
        public boolean add(URI uri, HttpCookie cookie) {
            String resolvedDomain = this.resolveDomain(uri, cookie);
            if (resolvedDomain == null) {
                return false;
            }
            String resolvedPath = this.resolvePath(uri, cookie);
            boolean[] added = new boolean[1];
            StoredHttpCookie storedCookie = new StoredHttpCookie(cookie, uri, resolvedDomain, resolvedPath);
            try (AutoLock ignored = this.lock.lock();){
                String key = resolvedDomain.toLowerCase(Locale.ENGLISH);
                this.cookies.compute(key, (k, v) -> {
                    if (v != null) {
                        v.remove(storedCookie);
                    }
                    if (cookie.isExpired()) {
                        return v == null || v.isEmpty() ? null : v;
                    }
                    added[0] = true;
                    if (v == null) {
                        v = new ArrayList<StoredHttpCookie>();
                    }
                    v.add(storedCookie);
                    return v;
                });
            }
            return added[0];
        }

        private String resolveDomain(URI uri, HttpCookie cookie) {
            String uriDomain = uri.getHost();
            if (uriDomain == null) {
                return null;
            }
            String cookieDomain = cookie.getDomain();
            if (cookieDomain == null) {
                return uriDomain;
            }
            String resolvedDomain = cookieDomain;
            if (resolvedDomain.startsWith(".")) {
                resolvedDomain = cookieDomain.substring(1);
            }
            if (resolvedDomain.endsWith(".")) {
                resolvedDomain = uriDomain;
            }
            if (!this.allowDomain(resolvedDomain)) {
                return null;
            }
            if (!Default.isSameOrSubDomain(uriDomain, resolvedDomain)) {
                return null;
            }
            return resolvedDomain;
        }

        private String resolvePath(URI uri, HttpCookie cookie) {
            String resolvedPath = cookie.getPath();
            if (resolvedPath == null || !resolvedPath.startsWith("/")) {
                String uriPath = uri.getRawPath();
                if (StringUtil.isBlank(uriPath) || !uriPath.startsWith("/")) {
                    resolvedPath = "/";
                } else {
                    int lastSlash = uriPath.lastIndexOf(47);
                    resolvedPath = uriPath.substring(0, lastSlash);
                    if (resolvedPath.isEmpty()) {
                        resolvedPath = "/";
                    }
                }
            }
            return resolvedPath;
        }

        protected boolean allowDomain(String domain) {
            if (domain.endsWith(".")) {
                domain = domain.substring(0, domain.length() - 1);
            }
            if (domain.contains(".")) {
                return true;
            }
            if (domain.equals("localhost")) {
                return true;
            }
            return domain.startsWith("[") && domain.endsWith("]");
        }

        @Override
        public List<HttpCookie> all() {
            try (AutoLock ignored = this.lock.lock();){
                List<HttpCookie> list = this.cookies.values().stream().flatMap(Collection::stream).filter(Predicate.not(StoredHttpCookie::isExpired)).map(HttpCookie.class::cast).toList();
                return list;
            }
        }

        @Override
        public List<HttpCookie> match(URI uri) {
            String uriDomain = uri.getHost();
            if (uriDomain == null) {
                return List.of();
            }
            String path = uri.getRawPath();
            if (path == null || path.isBlank()) {
                path = "/";
            }
            boolean secure = HttpScheme.isSecure(uri.getScheme());
            ArrayList<HttpCookie> result = new ArrayList<HttpCookie>();
            try (AutoLock ignored = this.lock.lock();){
                String domain = uriDomain.toLowerCase(Locale.ENGLISH);
                while (domain != null) {
                    Iterator<Object> iterator;
                    List<StoredHttpCookie> stored = this.cookies.get(domain);
                    Iterator<Object> iterator2 = iterator = stored == null ? Collections.emptyIterator() : stored.iterator();
                    while (iterator.hasNext()) {
                        StoredHttpCookie cookie = (StoredHttpCookie)iterator.next();
                        if (cookie.isExpired()) {
                            iterator.remove();
                            continue;
                        }
                        if (cookie.isSecure() && !secure || !Default.domainMatches(uriDomain, cookie.domain, cookie.getWrapped().getDomain()) || !Default.pathMatches(path, cookie.path)) continue;
                        result.add(cookie);
                    }
                    domain = this.parentDomain(domain);
                }
            }
            return result;
        }

        private static boolean domainMatches(String uriDomain, String domain, String cookieDomain) {
            if (cookieDomain == null || cookieDomain.endsWith(".")) {
                return uriDomain.equalsIgnoreCase(domain);
            }
            return Default.isSameOrSubDomain(uriDomain, cookieDomain);
        }

        private static boolean isSameOrSubDomain(String subDomain, String domain) {
            int domainLength;
            int subDomainLength = subDomain.length();
            if (!subDomain.regionMatches(true, subDomainLength - (domainLength = domain.length()), domain, 0, domainLength)) {
                return false;
            }
            int beforeMatch = subDomainLength - domainLength - 1;
            if (beforeMatch < 0) {
                return true;
            }
            return subDomain.charAt(beforeMatch) == '.';
        }

        private static boolean pathMatches(String uriPath, String cookiePath) {
            if (cookiePath == null) {
                return true;
            }
            if (uriPath.equals(cookiePath)) {
                return true;
            }
            if (uriPath.startsWith(cookiePath)) {
                return cookiePath.endsWith("/") || uriPath.charAt(cookiePath.length()) == '/';
            }
            return false;
        }

        @Override
        public boolean remove(URI uri, HttpCookie cookie) {
            String uriDomain = uri.getHost();
            if (uriDomain == null) {
                return false;
            }
            String resolvedPath = this.resolvePath(uri, cookie);
            boolean[] removed = new boolean[1];
            try (AutoLock ignored = this.lock.lock();){
                String domain = uriDomain.toLowerCase(Locale.ENGLISH);
                while (domain != null) {
                    this.cookies.compute(domain, (k, v) -> {
                        if (v == null) {
                            return null;
                        }
                        Iterator iterator = v.iterator();
                        while (iterator.hasNext()) {
                            StoredHttpCookie storedCookie = (StoredHttpCookie)iterator.next();
                            if (!uriDomain.equalsIgnoreCase(storedCookie.uri.getHost()) || !storedCookie.path.equals(resolvedPath) || !storedCookie.getWrapped().getName().equals(cookie.getName())) continue;
                            iterator.remove();
                            removed[0] = true;
                        }
                        return v.isEmpty() ? null : v;
                    });
                    domain = this.parentDomain(domain);
                }
            }
            return removed[0];
        }

        private String parentDomain(String domain) {
            int dot = domain.indexOf(46);
            if (dot < 0) {
                return null;
            }
            if ((domain = domain.substring(dot + 1)).indexOf(46) < 0) {
                return null;
            }
            return domain;
        }

        @Override
        public boolean clear() {
            try (AutoLock ignored = this.lock.lock();){
                if (this.cookies.isEmpty()) {
                    boolean bl = false;
                    return bl;
                }
                this.cookies.clear();
                boolean bl = true;
                return bl;
            }
        }

        private static class StoredHttpCookie
        extends HttpCookie.Wrapper {
            private final long creationNanoTime = NanoTime.now();
            private final URI uri;
            private final String domain;
            private final String path;

            private StoredHttpCookie(HttpCookie wrapped, URI uri, String domain, String path) {
                super(wrapped);
                this.uri = Objects.requireNonNull(uri);
                this.domain = Objects.requireNonNull(domain);
                this.path = Objects.requireNonNull(path);
            }

            @Override
            public boolean isExpired() {
                long maxAge = this.getMaxAge();
                if (maxAge >= 0L && NanoTime.secondsSince(this.creationNanoTime) > maxAge) {
                    return true;
                }
                Instant expires = this.getExpires();
                return expires != null && Instant.now().isAfter(expires);
            }

            @Override
            public int hashCode() {
                return Objects.hash(this.getWrapped().getName(), this.domain.toLowerCase(Locale.ENGLISH), this.path);
            }

            @Override
            public boolean equals(Object obj) {
                if (this == obj) {
                    return true;
                }
                if (!(obj instanceof StoredHttpCookie)) {
                    return false;
                }
                StoredHttpCookie that = (StoredHttpCookie)obj;
                return this.getName().equals(that.getName()) && this.domain.equalsIgnoreCase(that.domain) && this.path.equals(that.path);
            }
        }
    }

    public static class Empty
    implements HttpCookieStore {
        @Override
        public boolean add(URI uri, HttpCookie cookie) {
            return false;
        }

        @Override
        public List<HttpCookie> all() {
            return List.of();
        }

        @Override
        public List<HttpCookie> match(URI uri) {
            return List.of();
        }

        @Override
        public boolean remove(URI uri, HttpCookie cookie) {
            return false;
        }

        @Override
        public boolean clear() {
            return false;
        }
    }
}

