/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.start;

import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.eclipse.jetty.start.BaseHome;
import org.eclipse.jetty.start.FS;
import org.eclipse.jetty.start.Module;
import org.eclipse.jetty.start.StartArgs;
import org.eclipse.jetty.start.StartLog;
import org.eclipse.jetty.start.UsageException;
import org.eclipse.jetty.util.TopologicalSort;

public class Modules
implements Iterable<Module> {
    private final List<Module> _modules = new ArrayList<Module>();
    private final Map<String, Module> _names = new HashMap<String, Module>();
    private final Map<String, Set<Module>> _provided = new HashMap<String, Set<Module>>();
    private final BaseHome _baseHome;
    private final StartArgs _args;
    private final Properties _deprecated = new Properties();

    public Modules(BaseHome basehome, StartArgs args) {
        String javaVersion;
        this._baseHome = basehome;
        this._args = args;
        if (!args.getProperties().containsKey("java.version") && (javaVersion = System.getProperty("java.version")) != null) {
            args.setProperty("java.version", javaVersion, "<internal>");
        }
        try {
            Path deprecatedPath = this._baseHome.getPath("modules/deprecated.properties");
            if (deprecatedPath != null && FS.exists(deprecatedPath)) {
                this._deprecated.load(new FileInputStream(deprecatedPath.toFile()));
            }
        }
        catch (IOException e) {
            StartLog.debug(e);
        }
    }

    public void dump(List<String> tags) {
        Set exclude = tags.stream().filter(t -> t.startsWith("-")).map(t -> t.substring(1)).collect(Collectors.toSet());
        Set include = tags.stream().filter(t -> !t.startsWith("-")).collect(Collectors.toSet());
        boolean all = include.contains("*") || include.isEmpty();
        AtomicReference tag = new AtomicReference();
        this._modules.stream().filter(m -> {
            boolean included = all || m.getTags().stream().anyMatch(t -> include.contains(t));
            boolean excluded = m.getTags().stream().anyMatch(t -> exclude.contains(t));
            return included && !excluded;
        }).sorted().forEach(module -> {
            if (!module.getPrimaryTag().equals(tag.get())) {
                tag.set(module.getPrimaryTag());
                System.out.printf("%nModules for tag '%s':%n", module.getPrimaryTag());
                System.out.print("-------------------");
                int i = module.getPrimaryTag().length();
                while (i-- > 0) {
                    System.out.print("-");
                }
                System.out.println();
            }
            Set<String> provides = module.getProvides();
            provides.remove(module.getName());
            System.out.printf("%n     Module: %s %s%n", module.getName(), provides.size() > 0 ? provides : "");
            for (String description : module.getDescription()) {
                System.out.printf("           : %s%n", description);
            }
            if (!module.getTags().isEmpty()) {
                String label = "       Tags: %s";
                for (String t : module.getTags()) {
                    System.out.printf(label, t);
                    label = ", %s";
                }
                System.out.println();
            }
            if (!module.getDepends().isEmpty()) {
                String label = "     Depend: %s";
                for (String parent : module.getDepends()) {
                    System.out.printf(label, parent);
                    label = ", %s";
                }
                System.out.println();
            }
            if (!module.getOptional().isEmpty()) {
                String label = "   Optional: %s";
                for (String parent : module.getOptional()) {
                    System.out.printf(label, parent);
                    label = ", %s";
                }
                System.out.println();
            }
            for (String lib : module.getLibs()) {
                System.out.printf("        LIB: %s%n", lib);
            }
            for (String xml : module.getXmls()) {
                System.out.printf("        XML: %s%n", xml);
            }
            for (String jpms : module.getJPMS()) {
                System.out.printf("        JPMS: %s%n", jpms);
            }
            for (String jvm : module.getJvmArgs()) {
                System.out.printf("        JVM: %s%n", jvm);
            }
            if (module.isEnabled()) {
                for (String selection : module.getEnableSources()) {
                    System.out.printf("    Enabled: %s%n", selection);
                }
            }
        });
    }

    public void dumpEnabled() {
        int i = 0;
        List<Module> enabled = this.getEnabled();
        for (Module module : enabled) {
            String name = module.getName();
            String index = i++ + ")";
            for (String s : module.getEnableSources()) {
                System.out.printf("  %4s %-15s %s%n", index, name, s);
                index = "";
                name = "";
            }
            if (!module.isTransitive() || !module.hasIniTemplate()) continue;
            System.out.printf("                       init template available with --add-to-start=%s%n", module.getName());
        }
    }

    public void registerAll() throws IOException {
        for (Path path : this._baseHome.getPaths("modules/*.mod")) {
            this.registerModule(path);
        }
    }

    private Module registerModule(Path file) {
        if (!FS.canReadFile(file)) {
            throw new IllegalStateException("Cannot read file: " + file);
        }
        String shortName = this._baseHome.toShortForm(file);
        try {
            StartLog.debug("Registering Module: %s", shortName);
            Module module = new Module(this._baseHome, file);
            this._modules.add(module);
            this._names.put(module.getName(), module);
            module.getProvides().forEach(n -> this._provided.computeIfAbsent((String)n, k -> new HashSet()).add(module));
            return module;
        }
        catch (Error | RuntimeException t) {
            throw t;
        }
        catch (Throwable t) {
            throw new IllegalStateException("Unable to register module: " + shortName, t);
        }
    }

    public String toString() {
        StringBuilder str = new StringBuilder();
        str.append("Modules[");
        str.append("count=").append(this._modules.size());
        str.append(",<");
        AtomicBoolean delim = new AtomicBoolean(false);
        this._modules.forEach(m -> {
            if (delim.get()) {
                str.append(',');
            }
            str.append(m.getName());
            delim.set(true);
        });
        str.append(">");
        str.append("]");
        return str.toString();
    }

    public List<Module> getEnabled() {
        List<Module> enabled = this._modules.stream().filter(m -> m.isEnabled()).collect(Collectors.toList());
        TopologicalSort sort = new TopologicalSort();
        for (Module module : enabled) {
            Consumer<String> add = name -> {
                Set<Module> provided;
                Module dependency = this._names.get(name);
                if (dependency != null && dependency.isEnabled()) {
                    sort.addDependency((Object)module, (Object)dependency);
                }
                if ((provided = this._provided.get(name)) != null) {
                    for (Module p : provided) {
                        if (!p.isEnabled()) continue;
                        sort.addDependency((Object)module, (Object)p);
                    }
                }
            };
            module.getDepends().forEach(add);
            module.getOptional().forEach(add);
        }
        sort.sort(enabled);
        return enabled;
    }

    public Set<String> enable(String name, String enabledFrom) {
        Module module = this.get(name);
        if (module == null) {
            throw new UsageException(-9, "Unknown module='%s'. List available with --list-modules", name);
        }
        HashSet<String> enabled = new HashSet<String>();
        this.enable(enabled, module, enabledFrom, false);
        return enabled;
    }

    private void enable(Set<String> newlyEnabled, Module module, String enabledFrom, boolean transitive) {
        Set<Module> providers;
        StartLog.debug("enable %s from %s transitive=%b", module, enabledFrom, transitive);
        if (newlyEnabled.contains(module.getName())) {
            StartLog.debug("Cycle at %s", module);
            return;
        }
        for (String name : module.getProvides()) {
            providers = this._provided.get(name);
            if (providers == null) continue;
            for (Module p : providers) {
                if (p.equals(module) || !p.isEnabled()) continue;
                if (p.isTransitive() && !transitive) {
                    p.clearTransitiveEnable();
                    continue;
                }
                throw new UsageException("Module %s provides %s, which is already provided by %s enabled in %s", module.getName(), name, p.getName(), p.getEnableSources());
            }
        }
        if (module.enable(enabledFrom, transitive)) {
            StartLog.debug("enabled %s", module.getName());
            newlyEnabled.add(module.getName());
            module.expandDependencies(this._args.getProperties());
            if (module.hasDefaultConfig()) {
                for (String line : module.getDefaultConfig()) {
                    this._args.parse(line, module.getName() + "[ini]");
                }
                for (Module m2 : this._modules) {
                    m2.expandDependencies(this._args.getProperties());
                }
            }
        }
        StartLog.debug("Enabled module %s depends on %s", module.getName(), module.getDepends());
        for (String dependsOn : module.getDepends()) {
            Optional dftProvider;
            providers = this.getAvailableProviders(dependsOn);
            StartLog.debug("Module %s depends on %s provided by %s", module, dependsOn, providers);
            if (providers.isEmpty()) {
                if (dependsOn.contains("/")) {
                    Path file = this._baseHome.getPath("modules/" + dependsOn + ".mod");
                    this.registerModule(file).expandDependencies(this._args.getProperties());
                    providers = this._provided.get(dependsOn);
                    if (providers == null || providers.isEmpty()) {
                        throw new UsageException("Module %s does not provide %s", this._baseHome.toShortForm(file), dependsOn);
                    }
                    this.enable(newlyEnabled, (Module)providers.stream().findFirst().get(), "dynamic dependency of " + module.getName(), true);
                    continue;
                }
                throw new UsageException("No module found to provide %s for %s", dependsOn, module);
            }
            if (providers.stream().filter(Module::isEnabled).count() > 0L) {
                providers.stream().filter(m -> m.isEnabled() && !m.equals(module)).forEach(m -> this.enable(newlyEnabled, (Module)m, "transitive provider of " + dependsOn + " for " + module.getName(), true));
                continue;
            }
            Optional<Object> optional = dftProvider = providers.size() == 1 ? providers.stream().findFirst() : providers.stream().filter(m -> m.getName().equals(dependsOn)).findFirst();
            if (dftProvider.isPresent()) {
                this.enable(newlyEnabled, (Module)dftProvider.get(), "transitive provider of " + dependsOn + " for " + module.getName(), true);
                continue;
            }
            if (!StartLog.isDebugEnabled()) continue;
            StartLog.debug("Module %s requires a %s implementation from one of %s", module, dependsOn, providers);
        }
    }

    private Set<Module> getAvailableProviders(String name) {
        Set<Module> providers = this._provided.get(name);
        StartLog.debug("Providers of %s are %s", name, providers);
        if (providers == null || providers.isEmpty()) {
            return Collections.emptySet();
        }
        providers = new HashSet<Module>(providers);
        HashSet<String> provided = new HashSet<String>();
        for (Module m : this._modules) {
            if (!m.isEnabled()) continue;
            provided.add(m.getName());
            provided.addAll(m.getProvides());
        }
        Iterator<Module> i = providers.iterator();
        block1: while (i.hasNext()) {
            Module provider = i.next();
            if (provider.isEnabled()) continue;
            for (String p : provider.getProvides()) {
                if (!provided.contains(p)) continue;
                StartLog.debug("Removing provider %s because %s already enabled", provider, p);
                i.remove();
                continue block1;
            }
        }
        StartLog.debug("Available providers of %s are %s", name, providers);
        return providers;
    }

    public Module get(String name) {
        String reason;
        Module module = this._names.get(name);
        if (module == null && (reason = this._deprecated.getProperty(name)) != null) {
            StartLog.warn("Module %s is no longer available: %s", name, reason);
        }
        return module;
    }

    @Override
    public Iterator<Module> iterator() {
        return this._modules.iterator();
    }

    public Stream<Module> stream() {
        return this._modules.stream();
    }

    public void checkEnabledModules() {
        StringBuilder unsatisfied = new StringBuilder();
        this._modules.stream().filter(Module::isEnabled).forEach(m -> m.getDepends().forEach(d -> {
            Set<Module> providers = this.getAvailableProviders((String)d);
            if (providers.stream().filter(Module::isEnabled).count() == 0L) {
                if (unsatisfied.length() > 0) {
                    unsatisfied.append(',');
                }
                unsatisfied.append(m.getName());
                StartLog.error("Module %s requires a module providing %s from one of %s%n", m.getName(), d, providers);
            }
        }));
        if (unsatisfied.length() > 0) {
            throw new UsageException(-1, "Unsatisfied module dependencies: " + unsatisfied);
        }
    }
}

