/*
 *    Geotoolkit.org - An Open Source Java GIS Toolkit
 *    http://www.geotoolkit.org
 *
 *    (C) 1999-2011, Open Source Geospatial Foundation (OSGeo)
 *    (C) 2009-2011, Geomatys
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation;
 *    version 2.1 of the License.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    This package contains formulas from the PROJ package of USGS.
 *    USGS's work is fully acknowledged here. This derived work has
 *    been relicensed under LGPL with Frank Warmerdam's permission.
 */
package org.geotoolkit.referencing.operation.projection;

import java.awt.geom.Point2D;
import net.jcip.annotations.Immutable;

import org.opengis.parameter.ParameterValueGroup;
import org.opengis.parameter.ParameterDescriptor;
import org.opengis.parameter.ParameterDescriptorGroup;
import org.opengis.referencing.operation.MathTransform2D;
import org.opengis.referencing.operation.Matrix;

import org.geotoolkit.resources.Errors;
import org.geotoolkit.referencing.operation.matrix.Matrix2;

import static java.lang.Math.*;
import static org.geotoolkit.referencing.operation.provider.PolarStereographic.*;


/**
 * The polar case of the {@linkplain Stereographic stereographic} projection.
 * This default implementation uses USGS equation (i.e. iteration) for computing
 * the {@linkplain #inverseTransform inverse transform}.
 *
 * @author Gerald Evenden (USGS)
 * @author André Gosselin (MPO)
 * @author Martin Desruisseaux (MPO, IRD, Geomatys)
 * @author Rueben Schulz (UBC)
 * @author Rémi Maréchal (Geomatys)
 * @version 3.19
 *
 * @see EquatorialStereographic
 * @see ObliqueStereographic
 *
 * @since 2.0
 * @module
 */
@Immutable
public class PolarStereographic extends Stereographic {
    /**
     * For compatibility with different versions during deserialization.
     */
    private static final long serialVersionUID = -6635298308431138524L;

    /**
     * Creates a Polar Stereographic projection from the given parameters. The descriptor argument
     * is usually {@link org.geotoolkit.referencing.operation.provider.PolarStereographic#PARAMETERS},
     * but is not restricted to. If a different descriptor is supplied, it is user's responsibility
     * to ensure that it is suitable to a Polar Stereographic projection.
     *
     * @param  descriptor Typically {@code Polar Stereographic.PARAMETERS}.
     * @param  values The parameter values of the projection to create.
     * @return The map projection.
     *
     * @since 3.00
     */
    public static MathTransform2D create(final ParameterDescriptorGroup descriptor,
                                         final ParameterValueGroup values)
    {
        final Parameters parameters = new Parameters(descriptor, values);
        final PolarStereographic projection = create(parameters);
        return projection.createConcatenatedTransform();
    }

    /**
     * Creates a Polar Stereographic projection from the given parameters.
     *
     * @param parameters The parameters of the projection to be created.
     * @return The map projection.
     */
    static PolarStereographic create(final Parameters parameters) {
        final boolean isSpherical = parameters.isSpherical();
        if (parameters.nameMatches(North.PARAMETERS)) {
            if (isSpherical) {
                return new PolarStereographic.Spherical(parameters, false, Boolean.FALSE);
            } else {
                return new PolarStereographic(parameters, false, Boolean.FALSE);
            }
        } else if (parameters.nameMatches(South.PARAMETERS)) {
            if (isSpherical) {
                return new PolarStereographic.Spherical(parameters, false, Boolean.TRUE);
            } else {
                return new PolarStereographic(parameters, false, Boolean.TRUE);
            }
        } else if (parameters.nameMatches(VariantB.PARAMETERS)) {
            if (isSpherical) {
                return new PolarStereographic.Spherical(parameters, false, null);
            } else {
                return new PolarStereographic.Series(parameters, false, null);
            }
        } if (isSpherical) {
            return new PolarStereographic.Spherical(parameters, true, null);
        } else {
            return new PolarStereographic.Series(parameters, true, null);
        }
    }

    /**
     * Constructs an oblique stereographic projection (USGS equations).
     *
     * @param parameters The parameters of the projection to be created.
     */
    protected PolarStereographic(final Parameters parameters) {
        this(parameters, parameters.nameMatches(PARAMETERS), null);
    }

    /**
     * Gets {@code "standard_parallel_1"} parameter value. This parameter should be mutually
     * exclusive with {@code "latitude_of_origin"}, but this is not a strict requirement for
     * the constructor.
     *
     * {@preformat text
     *   ┌───────────────────────────────────┬────────────────────┬─────────────┐
     *   │ Projection                        │ Parameter          │ Force pole  │
     *   ├───────────────────────────────────┼────────────────────┼─────────────┤
     *   │ Polar Stereographic (variant A)   │ Latitude of origin │ auto detect │
     *   │ Polar Stereographic (variant B)   │ Standard Parallel  │ auto detect │
     *   │ Stereographic North Pole          │ Standard Parallel  │ North pole  │
     *   │ Stereographic South Pole          │ Standard Parallel  │ South pole  │
     *   └───────────────────────────────────┴────────────────────┴─────────────┘
     * }
     *
     * "Standard Parallel" (a.k.a. "Latitude true scale") default to 90°N for
     * every cases (including Polar A, but it is meanless in this case), except
     * for "Stereographic South Pole" where it default to 90°S.
     */
    private static double latitudeTrueScale(final Parameters parameters,
                final boolean isVariantA, final Boolean forceSouthPole)
    {
        final ParameterDescriptor<Double> trueScaleDescriptor =
                Boolean.TRUE.equals(forceSouthPole) ? South.STANDARD_PARALLEL : North.STANDARD_PARALLEL;
        final double latitudeTrueScale;
        if (isVariantA) {
            // Polar A case
            latitudeTrueScale = copySign(90, parameters.latitudeOfOrigin);
        } else {
            // Any cases except Polar A
            latitudeTrueScale = (parameters.standardParallels.length != 0) ?
                    parameters.standardParallels[0] : trueScaleDescriptor.getDefaultValue();
        }
        Parameters.ensureLatitudeInRange(trueScaleDescriptor, latitudeTrueScale, true);
        return toRadians(latitudeTrueScale);
    }

    /**
     * Constructs a polar stereographic projection.
     *
     * @param parameters The parameters of the projection to be created.
     * @param isVariantA {@code true} for Polar Stereographic variant A,
     *        or {@code false} for all other cases.
     * @param forceSouthPole Forces projection to North pole if {@link Boolean#FALSE},
     *        to South pole if {@link Boolean#TRUE}, or do not force (i.e. detect
     *        from other parameter values) if {@code null}.
     */
    private PolarStereographic(final Parameters parameters,
            final boolean isVariantA, final Boolean forceSouthPole)
    {
        /*
         * Sets unconditionally the latitude of origin to 90°N, because the South case will
         * be handle by reverting the sign of y through the (de)normalize affine transforms.
         */
        super(parameters, 90);
        double latitudeTrueScale = latitudeTrueScale(parameters, isVariantA, forceSouthPole);
        final boolean southPole;
        if (forceSouthPole != null) {
            southPole = forceSouthPole.booleanValue();
        } else {
            southPole = (latitudeTrueScale < 0);
        }
        latitudeTrueScale = abs(latitudeTrueScale); // May be anything in [0 ... π/2] range.
        final double k0;
        if (abs(latitudeTrueScale - PI/2) >= ANGLE_TOLERANCE) {
            // Derives from (21-32 and 21-33)
            final double t = sin(latitudeTrueScale);
            if (excentricity != 0) {
                k0 = msfn(t, cos(latitudeTrueScale)) / tsfn(latitudeTrueScale, t);
            } else {
                // Simplification of the above equation in the spherical case,
                // derived from (21-7) and (21-11).
                k0 = 1 + t;
            }
        } else {
            if (excentricity != 0) {
                // True scale at pole (part of (21-33))
                k0 = 2 / sqrt(pow(1+excentricity, 1+excentricity) * pow(1-excentricity, 1-excentricity));
            } else {
                // Simplification of the above equation in the spherical case.
                k0 = 2;
            }
        }
        /*
         * At this point, all parameters have been processed. Now process to their
         * validation and the initialization of (de)normalize affine transforms.
         */
        double yk0 = k0;
        if (southPole) {
            parameters.normalize(true).scale(1, -1);
        } else {
            yk0 = -yk0;
        }
        parameters.validate();
        parameters.normalize(false).scale(k0, yk0);
        finish();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void transform(final double[] srcPts, final int srcOff,
                             final double[] dstPts, final int dstOff)
            throws ProjectionException
    {
        final double λ = rollLongitude(srcPts[srcOff]);
        final double φ = srcPts[srcOff + 1];
        final double ρ = tsfn(φ, sin(φ));
        dstPts[dstOff  ] = ρ * sin(λ);
        dstPts[dstOff+1] = ρ * cos(λ);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void inverseTransform(final double[] srcPts, final int srcOff,
                                    final double[] dstPts, final int dstOff)
            throws ProjectionException
    {
        final double x = srcPts[srcOff  ];
        final double y = srcPts[srcOff+1];
        final double ρ = hypot(x, y);
        /*
         * Compute latitude using iterative technique.
         */
        final double halfe = 0.5 * excentricity;
        double φ = 0;
        for (int i=MAXIMUM_ITERATIONS;;) {
            final double esinφ = excentricity * sin(φ);
            final double next = (PI/2) - 2*atan(ρ*pow((1-esinφ)/(1+esinφ), halfe));
            if (abs(φ - (φ=next)) < ITERATION_TOLERANCE) {
                break;
            }
            if (--i < 0) {
                throw new ProjectionException(Errors.Keys.NO_CONVERGENCE);
            }
        }
        dstPts[dstOff  ] = unrollLongitude(atan2(x, y));
        dstPts[dstOff+1] = φ;
    }




    /**
     * Provides the transform equations for the spherical case of the polar
     * stereographic projection.
     *
     * @author Gerald Evenden (USGS)
     * @author André Gosselin (MPO)
     * @author Martin Desruisseaux (MPO, IRD, Geomatys)
     * @author Rueben Schulz (UBC)
     * @version 3.00
     *
     * @since 2.4
     * @module
     */
    @Immutable
    static final class Spherical extends PolarStereographic {
        /**
         * For compatibility with different versions during deserialization.
         */
        private static final long serialVersionUID = 1655096575897215547L;

        /**
         * Constructs a spherical stereographic projection.
         *
         * @param parameters The parameters of the projection to be created.
         * @param isVariantA {@code true} for Polar Stereographic variant A,
         *        or {@code false} for all other cases.
         * @param forceSouthPole For projection to North pole if {@link Boolean#FALSE},
         *        to South pole if {@link Boolean#TRUE}, or do not force (i.e. detect
         *        from other parameter values) if {@code null}.
         */
        Spherical(final Parameters parameters, final boolean isVariantA, final Boolean forceSouthPole) {
            super(parameters, isVariantA, forceSouthPole);
            parameters.ensureSpherical();
        }

        /**
         * Returns {@code true} since this class uses spherical formulas.
         */
        @Override
        final boolean isSpherical() {
            return true;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        protected void transform(final double[] srcPts, final int srcOff,
                                 final double[] dstPts, final int dstOff)
                throws ProjectionException
        {
            double x = rollLongitude(srcPts[srcOff]);
            double y = srcPts[srcOff + 1];
            final double f = cos(y) / (1+sin(y)); // == tan (pi/4 - φ/2)
            y = f * cos(x); // (21-6)
            x = f * sin(x); // (21-5)
            assert checkTransform(srcPts, srcOff, dstPts, dstOff, x, y);
            dstPts[dstOff]   = x;
            dstPts[dstOff+1] = y;
        }

        /**
         * Computes using ellipsoidal formulas and compare with the
         * result from spherical formulas. Used in assertions only.
         */
        private boolean checkTransform(final double[] srcPts, final int srcOff,
                                       final double[] dstPts, final int dstOff,
                                       final double x, final double y)
                throws ProjectionException
        {
            super.transform(srcPts, srcOff, dstPts, dstOff);
            return Assertions.checkTransform(dstPts, dstOff, x, y);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        protected void inverseTransform(final double[] srcPts, final int srcOff,
                                        final double[] dstPts, final int dstOff)
                throws ProjectionException
        {
            double x = srcPts[srcOff  ];
            double y = srcPts[srcOff+1];
            final double ρ = hypot(x, y);
            x = unrollLongitude(atan2(x, y));
            y = PI/2 - abs(2*atan(ρ)); // (20-14) with phi1=90° and cos(y) = sin(π/2 + y).
            assert checkInverseTransform(srcPts, srcOff, dstPts, dstOff, x, y);
            dstPts[dstOff  ] = x;
            dstPts[dstOff+1] = y;
        }

        /**
         * Computes using ellipsoidal formulas and compare with the
         * result from spherical formulas. Used in assertions only.
         */
        private boolean checkInverseTransform(final double[] srcPts, final int srcOff,
                                              final double[] dstPts, final int dstOff,
                                              final double λ, final double φ)
                throws ProjectionException
        {
            super.inverseTransform(srcPts, srcOff, dstPts, dstOff);
            return Assertions.checkInverseTransform(dstPts, dstOff, λ, φ);
        }

        /**
         * Gets the derivative of this transform at a point.
         *
         * @param  point The coordinate point where to evaluate the derivative.
         * @return The derivative at the specified point as a 2&times;2 matrix.
         * @throws ProjectionException if the derivative can't be evaluated at the specified point.
         *
         * @since 3.19
         */
        @Override
        public Matrix derivative(final Point2D point) throws ProjectionException {
            final double λ = rollLongitude(point.getX());
            final double φ = point.getY();
            final double sinφp = sin(φ) + 1;
            final double sinλ  = sin(λ);
            final double cosλ  = cos(λ);
            final double F     = cos(φ) / sinφp; // == tan (pi/4 - phi/2)
            final Matrix derivative = new Matrix2(
                     cosλ * F,        // ∂x/∂λ
                    -sinλ / sinφp,    // ∂x/∂φ
                    -sinλ * F,        // ∂y/∂λ
                    -cosλ / sinφp);   // ∂y/∂φ
            assert Assertions.checkDerivative(derivative, super.derivative(point));
            return derivative;
        }
    }

    /**
     * Gets the derivative of this transform at a point.
     *
     * @param  point The coordinate point where to evaluate the derivative.
     * @return The derivative at the specified point as a 2&times;2 matrix.
     * @throws ProjectionException if the derivative can't be evaluated at the specified point.
     *
     * @since 3.19
     */
    @Override
    public Matrix derivative(final Point2D point) throws ProjectionException {
        final double λ = rollLongitude(point.getX());
        final double φ = point.getY();
        final double sinφ  = sin(φ);
        final double sinλ  = sin(λ);
        final double cosλ  = cos(λ);
        final double ρ     = tsfn(φ, sinφ);
        final double dρ_dφ = dtsfn_dφ(φ, sinφ,cos(φ))*ρ;
        return new Matrix2(
                 cosλ*ρ,       // ∂x/∂λ
                 sinλ*dρ_dφ,   // ∂x/∂φ
                -sinλ*ρ,       // ∂y/∂λ
                 cosλ*dρ_dφ);  // ∂y/∂φ
    }



    /**
     * Overrides {@link PolarStereographic} to use the a series for the {@link #inverseTransform
     * inverse transform} method. This is the equation specified by the EPSG. Allows for a
     * {@code "latitude_true_scale"} parameter to be used, but this parameter is not listed
     * by the EPSG and is not given as a parameter by the provider.
     * <p>
     * Compared to the default {@link PolarStereographic} implementation, the series implementation
     * is a little bit faster at the expense of a little bit less accuracy. The default
     * {@link PolarStereographic} implementation is a derivated work of Proj4, and is therefore
     * better tested.
     *
     * @author Rueben Schulz (UBC)
     * @author Martin Desruisseaux (MPO, IRD, Geomatys)
     * @version 3.00
     *
     * @since 2.4
     * @module
     */
    @Immutable
    static final class Series extends PolarStereographic {
        /**
         * For compatibility with different versions during deserialization.
         */
        private static final long serialVersionUID = 2795404156883313290L;

        /**
         * Constants used for the inverse polar series
         */
        private final double a, b;

        /**
         * Constants used for the inverse polar series
         */
        private final double c, d;

        /**
         * Constructs a polar stereographic projection (series inverse equations).
         *
         * @param parameters The parameters of the projection to be created.
         * @param isVariantA {@code true} for Polar Stereographic variant A,
         *        or {@code false} for all other cases.
         * @param forceSouthPole For projection to North pole if {@link Boolean#FALSE},
         *        to South pole if {@link Boolean#TRUE}, or do not force (i.e. detect
         *        from other parameter values) if {@code null}.
         */
        Series(final Parameters parameters, final boolean isVariantA, final Boolean forceSouthPole) {
            super(parameters, isVariantA, forceSouthPole);
            // See Snyde P. 19, "Computation of Series"
            final double e4 = excentricitySquared * excentricitySquared;
            final double e6 = e4 * excentricitySquared;
            final double e8 = e4 * e4;
            final double ci = 7/120.0 * e6 + 81/1120.0 * e8;
            final double di = 4279/161280.0 * e8;
            a = excentricitySquared*0.5 + 5/24.0*e4 + e6/12.0 + 13/360.0*e8 - ci;
            b = 2 * (7/48.0*e4 + 29/240.0*e6 + 811/11520.0*e8) - 4*di;
            c = ci * 4;
            d = di * 8;
            /*
             * Proj4 was calculating a k0 constant here. This constant divised by the one calculated
             * by the super-class (this division was required because the k0 calculated by the super-
             * class scales the denormalize affine transform) simplifies to:
             *
             *     k0 = sqrt(pow(1+excentricity, 1+excentricity)*
             *               pow(1-excentricity, 1-excentricity)) / 2
             *
             * This constant was used only as a divisor of χ in inverseTransform(...), but was
             * identical to the expression that multiplies χ just a few instructions further.
             * Consequently, it vanishes completely.
             */
        }

        /**
         * {@inheritDoc}
         */
        @Override
        protected void inverseTransform(final double[] srcPts, final int srcOff,
                                        final double[] dstPts, final int dstOff)
                throws ProjectionException
        {
            double x = srcPts[srcOff];
            double y = srcPts[srcOff+1];
            final double t = hypot(x, y);
            final double χ = PI/2 - 2*atan(t);
            x = unrollLongitude(atan2(x, y));

            // See Snyde P. 19, "Computation of Series"
            final double sin2χ = sin(2 * χ);
            final double cos2χ = cos(2 * χ);
            y = χ + sin2χ*(a + cos2χ*(b + cos2χ*(c + d*cos2χ)));
            assert checkInverseTransform(srcPts, srcOff, dstPts, dstOff, x, y);
            dstPts[dstOff]   = x;
            dstPts[dstOff+1] = y;
        }

        /**
         * Computes using ellipsoidal formulas and compare with the
         * result from spherical formulas. Used in assertions only.
         */
        private boolean checkInverseTransform(final double[] srcPts, final int srcOff,
                                              final double[] dstPts, final int dstOff,
                                              final double λ, final double φ)
                throws ProjectionException
        {
            super.inverseTransform(srcPts, srcOff, dstPts, dstOff);
            return Assertions.checkInverseTransform(dstPts, dstOff, λ, φ);
        }
    }
}
