/*
 * Copyright 2020 GridGain Systems, Inc. and Contributors.
 *
 * Licensed under the GridGain Community Edition License (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.gridgain.com/products/software/community-edition/gridgain-community-edition-license
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.internal.commandline.property.tasks;

import java.util.List;
import java.util.stream.Collectors;
import org.apache.ignite.IgniteException;
import org.apache.ignite.compute.ComputeJobResult;
import org.apache.ignite.internal.commandline.meta.subcommands.VoidDto;
import org.apache.ignite.internal.processors.task.GridInternal;
import org.apache.ignite.internal.visor.VisorJob;
import org.apache.ignite.internal.visor.VisorMultiNodeTask;
import org.jetbrains.annotations.Nullable;

import static org.apache.ignite.plugin.security.SecurityPermission.ADMIN_READ_DISTRIBUTED_PROPERTY;

/**
 * Task for property operations.
 */
@GridInternal
public class PropertiesListTask extends VisorMultiNodeTask<VoidDto, PropertiesListResult, PropertiesListResult> {
    /** */
    private static final long serialVersionUID = 0L;

    /** {@inheritDoc} */
    @Override protected VisorJob<VoidDto, PropertiesListResult> job(VoidDto arg) {
        return new PropertiesListJob(debug);
    }

    /** {@inheritDoc} */
    @Nullable @Override protected PropertiesListResult reduce0(List<ComputeJobResult> results) {
        if (results.isEmpty())
            throw new IgniteException("Empty job results");

        if (results.size() > 1)
            throw new IgniteException("Invalid job results: " + results);

        if (results.get(0).getException() != null)
            throw results.get(0).getException();
        else
            return results.get(0).getData();
    }

    /**
     * Job for property operations (get/set).
     */
    private static class PropertiesListJob extends VisorJob<VoidDto, PropertiesListResult> {
        /** */
        private static final long serialVersionUID = 0L;

        /**
         * @param debug Debug.
         */
        protected PropertiesListJob(boolean debug) {
            super(null, debug);
        }

        /** {@inheritDoc} */
        @Override protected PropertiesListResult run(@Nullable VoidDto arg) {
            ignite.context().security().authorize(ADMIN_READ_DISTRIBUTED_PROPERTY);

            return new PropertiesListResult(
                ignite.context().distributedConfiguration().properties().stream()
                    .map(pd -> pd.getName())
                    .collect(Collectors.toList())
            );
        }
    }
}
