/*
 * Copyright 2020 GridGain Systems, Inc. and Contributors.
 *
 * Licensed under the GridGain Community Edition License (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.gridgain.com/products/software/community-edition/gridgain-community-edition-license
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.internal.processors.cache.compress;

import org.apache.ignite.configuration.CacheConfiguration;
import org.apache.ignite.internal.GridKernalContext;

/**
 * Interface for compressing cache entries.
 */
public interface EntryCompressionStrategy extends AutoCloseable {
    /**
     * Tries to compress cache key or value data, as array of bytes.
     * If successful, returns compressed array, else {@code null}.
     *
     * @param bytes Array of uncompressed data.
     * @return Array of compressed data, or {@code null}.
     */
    byte[] tryCompress(byte[] bytes);

    /**
     * Decompresses an array of previously compressed cache data.
     *
     * @param bytes Array of compressed data.
     * @return Array of original data,.
     */
    byte[] decompress(byte[] bytes);

    /**
     * Gets flag indicating whether cache keys will be considered for compression.
     *
     * @return {@code True} if cache keys may be compressed.
     */
    boolean compressKeys();

    /**
     * Gets flag indicating whether this strategy is stateless. Only stateless strategies may be used with snapshots.
     * Strategies which require storing local dictionaries are not stateless.
     *
     * @return {@code True} if strategy is stateless.
     */
    boolean stateless();

    /**
     * Initializes entry compression strategy.
     *
     * @param ctx Kernal context.
     * @param ccfg Cache configuration.
     */
    void start(GridKernalContext ctx, CacheConfiguration ccfg);
}

