/*
 * Copyright 2021 GridGain Systems, Inc. and Contributors.
 *
 * Licensed under the GridGain Community Edition License (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.gridgain.com/products/software/community-edition/gridgain-community-edition-license
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.internal.processors.cache.persistence.pagemem;

/**
 * Logic used to protect memory (mainly, Checkpoint Buffer) from exhaustion using exponential backoff.
 */
class ExponentialBackoffThrottlingStrategy {
    /**
     * Starting throttle time. Limits write speed to 1000 MB/s.
     */
    private static final long STARTING_THROTTLE_NANOS = 4000;

    /**
     * Backoff ratio. Each next park will be this times longer.
     */
    private static final double BACKOFF_RATIO = 1.05;

    /**
     * Exponential backoff used to throttle threads.
     */
    private final ExponentialBackoff backoff = new ExponentialBackoff(STARTING_THROTTLE_NANOS, BACKOFF_RATIO);

    /**
     * Computes next duration (in nanos) to throttle a thread to protect Checkpoint Buffer.
     *
     * @return park time in nanos
     */
    long protectionParkTime() {
        return backoff.nextDuration();
    }

    /**
     * Resets the backoff counter. Invoked when no throttling is needed anymore.
     *
     * @return {@code true} iff the backoff was not already in a reset state
     */
    boolean resetBackoff() {
        return backoff.reset();
    }
}
