/*
 * Copyright 2019 GridGain Systems, Inc. and Contributors.
 *
 * Licensed under the GridGain Community Edition License (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.gridgain.com/products/software/community-edition/gridgain-community-edition-license
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.ignite.internal.processors.cache.persistence.wal.serializer;

import org.apache.ignite.IgniteCheckedException;
import org.apache.ignite.internal.managers.encryption.EncryptionCacheKeyProvider;
import org.apache.ignite.internal.pagemem.wal.WALPointer;
import org.apache.ignite.internal.pagemem.wal.record.MarshalledRecord;
import org.apache.ignite.internal.pagemem.wal.record.WALRecord;
import org.apache.ignite.internal.processors.cache.GridCacheSharedContext;
import org.apache.ignite.lang.IgniteBiPredicate;
import org.jetbrains.annotations.Nullable;

/**
 *
 */
public class RecordSerializerFactoryImpl implements RecordSerializerFactory {
    /** Context. */
    private GridCacheSharedContext cctx;

    /** Write pointer. */
    private boolean needWritePointer;

    /** Read record filter. */
    private @Nullable IgniteBiPredicate<WALRecord.RecordType, WALPointer> recordDeserializeFilter;

    /**
     * Marshalled mode flag.
     * Records are not deserialized in this mode, {@link MarshalledRecord} with binary representation are read instead.
     */
    private boolean marshalledMode;

    /** Skip position check flag. Should be set for reading compacted wal file with skipped physical records. */
    private boolean skipPositionCheck;

    /** Key provider for WAL records related to encrypted caches. */
    private final EncryptionCacheKeyProvider cacheKeyProvider;

    /**
     * @param cctx Cache shared context.
     */
    public RecordSerializerFactoryImpl(GridCacheSharedContext cctx) {
        this(cctx, null);
    }

    /**
     * @param cctx Cache shared context.
     * @param readTypeFilter Type filter.
     */
    public RecordSerializerFactoryImpl(
        GridCacheSharedContext cctx,
        @Nullable IgniteBiPredicate<WALRecord.RecordType, WALPointer> readTypeFilter
    ) {
        this(cctx, readTypeFilter, null);
    }

    /**
     * @param cctx Cache shared context.
     * @param readTypeFilter Type filter.
     * @param cacheKeyProvider Provider of encryption keys.
     */
    public RecordSerializerFactoryImpl(
        GridCacheSharedContext cctx,
        @Nullable IgniteBiPredicate<WALRecord.RecordType, WALPointer> readTypeFilter,
        EncryptionCacheKeyProvider cacheKeyProvider
    ) {
        this.cctx = cctx;
        this.recordDeserializeFilter = readTypeFilter;
        this.cacheKeyProvider = cacheKeyProvider == null ? cctx.kernalContext().encryption() : cacheKeyProvider;
    }

    /** {@inheritDoc} */
    @Override public RecordSerializer createSerializer(int ver) throws IgniteCheckedException {
        if (ver <= 0)
            throw new IgniteCheckedException("Failed to create a serializer (corrupted WAL file).");

        switch (ver) {
            case 1:
                return new RecordV1Serializer(
                    new RecordDataV1Serializer(cctx, cacheKeyProvider),
                    needWritePointer,
                    marshalledMode,
                    skipPositionCheck,
                    recordDeserializeFilter);

            case 2:
                return new RecordV2Serializer(
                    new RecordDataV2Serializer(cctx, cacheKeyProvider),
                    needWritePointer,
                    marshalledMode,
                    skipPositionCheck,
                    recordDeserializeFilter
                );

            default:
                throw new IgniteCheckedException("Failed to create a serializer with the given version " +
                    "(forward compatibility is not supported): " + ver);
        }
    }

    /**
     * @return Write pointer.
     */
    public boolean writePointer() {
        return needWritePointer;
    }

    /** {@inheritDoc} */
    @Override public RecordSerializerFactoryImpl writePointer(boolean writePointer) {
        this.needWritePointer = writePointer;

        return this;
    }

    /**
     * @return Read type filter.
     */
    public IgniteBiPredicate<WALRecord.RecordType, WALPointer> recordDeserializeFilter() {
        return recordDeserializeFilter;
    }

    /** {@inheritDoc} */
    @Override public RecordSerializerFactoryImpl recordDeserializeFilter(
        @Nullable IgniteBiPredicate<WALRecord.RecordType, WALPointer> readTypeFilter
    ) {
        this.recordDeserializeFilter = readTypeFilter;

        return this;
    }

    /**
     * @return Marshalled mode. Records are not deserialized in this mode,  with binary representation are read instead.
     */
    public boolean marshalledMode() {
        return marshalledMode;
    }

    /** {@inheritDoc} */
    @Override public RecordSerializerFactoryImpl marshalledMode(boolean marshalledMode) {
        this.marshalledMode = marshalledMode;

        return this;
    }

    /**
     * @return Skip position check flag. Should be set for reading compacted wal file with skipped physical records.
     */
    public boolean skipPositionCheck() {
        return skipPositionCheck;
    }

    /** {@inheritDoc} */
    @Override public RecordSerializerFactoryImpl skipPositionCheck(boolean skipPositionCheck) {
        this.skipPositionCheck = skipPositionCheck;

        return this;
    }

    /** {@inheritDoc} */
    @Override public RecordSerializerFactory clearSegmentLocalState() {
        skipPositionCheck = false;

        return this;
    }
}
