/*
 * Copyright 2019 GridGain Systems, Inc. and Contributors.
 *
 * Licensed under the GridGain Community Edition License (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.gridgain.com/products/software/community-edition/gridgain-community-edition-license
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.ignite.internal.processors.subscription;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import org.apache.ignite.internal.GridKernalContext;
import org.apache.ignite.internal.processors.GridProcessorAdapter;
import org.apache.ignite.internal.processors.cache.persistence.DatabaseLifecycleListener;
import org.apache.ignite.internal.processors.cache.persistence.metastorage.MetastorageLifecycleListener;
import org.apache.ignite.internal.processors.configuration.distributed.DistributedConfigurationLifecycleListener;
import org.apache.ignite.internal.processors.metastorage.DistributedMetastorageLifecycleListener;
import org.jetbrains.annotations.NotNull;

import static java.util.Objects.requireNonNull;

/**
 * Processor enables grid components to register listeners for events generated by other components on local node.
 *
 * It starts very first during node startup procedure so any components could use it.
 *
 * As only local events are supported, no network communication occurs during notification process.
 */
public class GridInternalSubscriptionProcessor extends GridProcessorAdapter {
    /** */
    private final List<MetastorageLifecycleListener> metastorageListeners = new ArrayList<>();

    /** */
    private final List<DistributedMetastorageLifecycleListener> distributedMetastorageListeners =
        new CopyOnWriteArrayList<>();

    /** */
    private final List<DatabaseLifecycleListener> dbListeners = new ArrayList<>();

    /**
     * Listeners of distributed configuration controlled by {@link org.apache.ignite.internal.processors.configuration.distributed.DistributedConfigurationProcessor}.
     */
    private List<DistributedConfigurationLifecycleListener> distributedConfigurationListeners = new ArrayList<>();

    /**
     * @param ctx Kernal context.
     */
    public GridInternalSubscriptionProcessor(GridKernalContext ctx) {
        super(ctx);
    }

    /** */
    public void registerMetastorageListener(@NotNull MetastorageLifecycleListener metastorageListener) {
        requireNonNull(metastorageListener, "Metastorage subscriber should be not-null.");

        metastorageListeners.add(metastorageListener);
    }

    /** */
    public List<MetastorageLifecycleListener> getMetastorageSubscribers() {
        return metastorageListeners;
    }

    /** */
    public void registerDistributedMetastorageListener(@NotNull DistributedMetastorageLifecycleListener lsnr) {
        requireNonNull(lsnr, "Global metastorage subscriber should be not-null.");

        distributedMetastorageListeners.add(lsnr);
    }

    /** */
    public List<DistributedMetastorageLifecycleListener> getDistributedMetastorageSubscribers() {
        return distributedMetastorageListeners;
    }

    /** */
    public void registerDatabaseListener(@NotNull DatabaseLifecycleListener databaseListener) {
        requireNonNull(databaseListener, "Database subscriber should be not-null.");

        dbListeners.add(databaseListener);
    }

    /** */
    public List<DatabaseLifecycleListener> getDatabaseListeners() {
        return dbListeners;
    }

    /**
     * Register distributed configuration lifecycle listener.
     *
     * @param lifecycleListener DistributedConfigurationLifecycleListener to register.
     */
    public void registerDistributedConfigurationListener(
        @NotNull DistributedConfigurationLifecycleListener lifecycleListener
    ) {
        requireNonNull(distributedConfigurationListeners, "Distributed configuration subscriber should be not-null.");

        distributedConfigurationListeners.add(lifecycleListener);
    }

    /** */
    public List<DistributedConfigurationLifecycleListener> getDistributedConfigurationListeners() {
        return distributedConfigurationListeners;
    }
}
