/*
 * Copyright 2019 GridGain Systems, Inc. and Contributors.
 *
 * Licensed under the GridGain Community Edition License (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.gridgain.com/products/software/community-edition/gridgain-community-edition-license
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.ignite.internal.util.ipc.loopback;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketException;
import java.sql.SQLException;
import org.apache.ignite.IgniteCheckedException;
import org.apache.ignite.internal.processors.odbc.SqlStateCode;
import org.apache.ignite.internal.util.ipc.IpcEndpoint;
import org.apache.ignite.internal.util.typedef.internal.U;

/**
 * Loopback IPC endpoint based on socket.
 */
public class IpcClientTcpEndpoint implements IpcEndpoint {
    /** Client socket. */
    private Socket clientSock;

    /**
     * Creates connected client IPC endpoint.
     *
     * @param clientSock Connected client socket.
     */
    public IpcClientTcpEndpoint(Socket clientSock) {
        assert clientSock != null;

        this.clientSock = clientSock;
    }

    /**
     * Creates and connects client IPC endpoint.
     *
     * @param port Port.
     * @param host Host.
     * @throws IgniteCheckedException If connection fails.
     */
    public IpcClientTcpEndpoint(String host, int port) throws IgniteCheckedException {
        clientSock = new Socket();

        try {
            clientSock.connect(new InetSocketAddress(host, port));
        }
        catch (IOException e) {
            throw new IgniteCheckedException("Failed to connect to endpoint [host=" + host + ", port=" + port + ']', e);
        }
    }

    /** {@inheritDoc} */
    @Override public InputStream inputStream() throws IgniteCheckedException {
        try {
            return clientSock.getInputStream();
        }
        catch (IOException e) {
            throw new IgniteCheckedException(e);
        }
    }

    /** {@inheritDoc} */
    @Override public OutputStream outputStream() throws IgniteCheckedException {
        try {
            return clientSock.getOutputStream();
        }
        catch (IOException e) {
            throw new IgniteCheckedException(e);
        }
    }

    /** {@inheritDoc} */
    @Override public void close() {
        U.closeQuiet(clientSock);
    }

    /**
     * Enable/disable socket timeout with specified timeout.
     *
     * @param ms the specified timeout, in milliseconds.
     * @throws SQLException if there is an error in the underlying protocol.
     */
    public void timeout(int ms) throws SQLException {
        try {
            clientSock.setSoTimeout(ms);
        }
        catch (SocketException e) {
            throw new SQLException("Failed to set connection timeout.", SqlStateCode.INTERNAL_ERROR, e);
        }
    }

    /**
     * Returns socket timeout.
     *
     * @throws SQLException if there is an error in the underlying protocol.
     */
    public int timeout() throws SQLException {
        try {
            return clientSock.getSoTimeout();
        }
        catch (SocketException e) {
            throw new SQLException("Failed to set connection timeout.", SqlStateCode.INTERNAL_ERROR, e);
        }
    }
}