/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.infinispan.atomic;

import org.infinispan.marshall.AbstractExternalizer;
import org.infinispan.marshall.Ids;
import org.infinispan.util.Util;
import org.infinispan.util.logging.Log;
import org.infinispan.util.logging.LogFactory;

import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

/**
 * Changes that have occurred on an AtomicHashMap
 *
 * @author Manik Surtani (<a href="mailto:manik AT jboss DOT org">manik AT jboss DOT org</a>)
 * @since 4.0
 */
public class AtomicHashMapDelta implements Delta {
   private static final Log log = LogFactory.getLog(AtomicHashMapDelta.class);
   private static final boolean trace = log.isTraceEnabled();

   private List<Operation> changeLog;

   public DeltaAware merge(DeltaAware d) {
      AtomicHashMap other;
      if (d != null && (d instanceof AtomicHashMap))
         other = (AtomicHashMap) d;
      else
         other = new AtomicHashMap();
      if (changeLog != null) {
         for (Operation o : changeLog) o.replay(other.delegate);
      }
      other.commit();
      return other;
   }

   public void addOperation(Operation o) {
      if (changeLog == null) {
         // lazy init
         changeLog = new LinkedList<Operation>();
      }
      changeLog.add(o);
   }

   @Override
   public String toString() {
      return "AtomicHashMapDelta{" +
            "changeLog=" + changeLog +
            '}';
   }

   public int getChangeLogSize() {
      return changeLog == null ? 0 : changeLog.size();
   }

   public static class Externalizer extends AbstractExternalizer<AtomicHashMapDelta> {
      @Override
      public void writeObject(ObjectOutput output, AtomicHashMapDelta delta) throws IOException {
         if (trace) log.trace("Serializing changeLog " + delta.changeLog);
         output.writeObject(delta.changeLog);
      }

      @Override
      public AtomicHashMapDelta readObject(ObjectInput input) throws IOException, ClassNotFoundException {
         AtomicHashMapDelta delta = new AtomicHashMapDelta();
         delta.changeLog = (List<Operation>) input.readObject();
         if (trace) log.trace("Deserialized changeLog " + delta.changeLog);
         return delta;
      }

      @Override
      public Integer getId() {
         return Ids.ATOMIC_HASH_MAP_DELTA;
      }

      @Override
      public Set<Class<? extends AtomicHashMapDelta>> getTypeClasses() {
         return Util.<Class<? extends AtomicHashMapDelta>>asSet(AtomicHashMapDelta.class);
      }
   }
}