package org.infinispan.commands;

import org.infinispan.CacheException;
import org.infinispan.commands.control.LockControlCommand;
import org.infinispan.commands.control.RehashControlCommand;
import org.infinispan.commands.control.StateTransferControlCommand;
import org.infinispan.commands.module.ModuleCommandFactory;
import org.infinispan.commands.read.GetKeyValueCommand;
import org.infinispan.commands.remote.ClusteredGetCommand;
import org.infinispan.commands.remote.RemoveRecoveryInfoCommand;
import org.infinispan.commands.remote.GetInDoubtTransactionsCommand;
import org.infinispan.commands.remote.MultipleRpcCommand;
import org.infinispan.commands.remote.SingleRpcCommand;
import org.infinispan.commands.tx.CommitCommand;
import org.infinispan.commands.tx.PrepareCommand;
import org.infinispan.commands.tx.RollbackCommand;
import org.infinispan.commands.write.ClearCommand;
import org.infinispan.commands.write.InvalidateCommand;
import org.infinispan.commands.write.InvalidateL1Command;
import org.infinispan.commands.write.PutKeyValueCommand;
import org.infinispan.commands.write.PutMapCommand;
import org.infinispan.commands.write.RemoveCommand;
import org.infinispan.commands.write.ReplaceCommand;
import org.infinispan.factories.GlobalComponentRegistry;
import org.infinispan.factories.KnownComponentNames;
import org.infinispan.factories.annotations.ComponentName;
import org.infinispan.factories.annotations.Inject;
import org.infinispan.factories.scopes.Scope;
import org.infinispan.factories.scopes.Scopes;
import org.infinispan.manager.EmbeddedCacheManager;
import org.infinispan.remoting.transport.Transport;

import java.util.Map;

/**
 * Specifically used to create un-initialized {@link org.infinispan.commands.ReplicableCommand}s from a byte stream.
 * This is a {@link Scopes#GLOBAL} component and doesn't have knowledge of initializing a command by injecting
 * cache-specific components into it.
 * <p />
 * Usually a second step to unmarshalling a command from a byte stream (after
 * creating an un-initialized version using this factory) is to pass the command though {@link CommandsFactory#initializeReplicableCommand(ReplicableCommand,boolean)}.
 *
 * @see CommandsFactory#initializeReplicableCommand(ReplicableCommand,boolean)
 * @author Manik Surtani
 * @author Mircea.Markus@jboss.com
 * @since 4.0
 */
@Scope(Scopes.GLOBAL)
public class RemoteCommandsFactory {
   Transport transport;
   EmbeddedCacheManager cacheManager;
   GlobalComponentRegistry registry;
   Map<Byte,ModuleCommandFactory> commandFactories;

   @Inject
   public void inject(Transport transport, EmbeddedCacheManager cacheManager, GlobalComponentRegistry registry,
                      @ComponentName(KnownComponentNames.MODULE_COMMAND_FACTORIES) Map<Byte, ModuleCommandFactory> commandFactories) {
      this.transport = transport;
      this.cacheManager = cacheManager;
      this.registry = registry;
      this.commandFactories = commandFactories;
   }

   /**
    * Creates an un-initialized command.  Un-initialized in the sense that parameters will be set, but any components
    * specific to the cache in question will not be set.
    * <p/>
    * You would typically set these parameters using {@link CommandsFactory#initializeReplicableCommand(ReplicableCommand,boolean)}
    * <p/>
    *
    * @param id         id of the command
    * @param parameters parameters to set
    * @return a replicable command
    */
   public ReplicableCommand fromStream(byte id, Object[] parameters) {
      ReplicableCommand command;
      switch (id) {
         case PutKeyValueCommand.COMMAND_ID:
            command = new PutKeyValueCommand();
            break;
         case LockControlCommand.COMMAND_ID:
            command = new LockControlCommand();
            break;
         case PutMapCommand.COMMAND_ID:
            command = new PutMapCommand();
            break;
         case RemoveCommand.COMMAND_ID:
            command = new RemoveCommand();
            break;
         case ReplaceCommand.COMMAND_ID:
            command = new ReplaceCommand();
            break;
         case GetKeyValueCommand.COMMAND_ID:
            command = new GetKeyValueCommand();
            break;
         case ClearCommand.COMMAND_ID:
            command = new ClearCommand();
            break;
         case PrepareCommand.COMMAND_ID:
            command = new PrepareCommand();
            break;
         case CommitCommand.COMMAND_ID:
            command = new CommitCommand();
            break;
         case RollbackCommand.COMMAND_ID:
            command = new RollbackCommand();
            break;
         case MultipleRpcCommand.COMMAND_ID:
            command = new MultipleRpcCommand();
            break;
         case SingleRpcCommand.COMMAND_ID:
            command = new SingleRpcCommand();
            break;
         case InvalidateCommand.COMMAND_ID:
            command = new InvalidateCommand();
            break;
         case InvalidateL1Command.COMMAND_ID:
            command = new InvalidateL1Command();
            break;
         case StateTransferControlCommand.COMMAND_ID:
            command = new StateTransferControlCommand();
            ((StateTransferControlCommand) command).init(transport);
            break;
         case ClusteredGetCommand.COMMAND_ID:
            command = new ClusteredGetCommand();
            break;
         case RehashControlCommand.COMMAND_ID:
            command = new RehashControlCommand(transport);
            break;
         case RemoveCacheCommand.COMMAND_ID:
            command = new RemoveCacheCommand(cacheManager, registry);
            break;
         case RemoveRecoveryInfoCommand.COMMAND_ID:
            command = new RemoveRecoveryInfoCommand();
            break;
         case GetInDoubtTransactionsCommand.COMMAND_ID:
            command = new GetInDoubtTransactionsCommand();
            break;
         default:
            ModuleCommandFactory mcf = commandFactories.get(id);
            if (mcf != null)
               return mcf.fromStream(id, parameters);
            else
               throw new CacheException("Unknown command id " + id + "!");
      }
      command.setParameters(id, parameters);
      return command;
   }
}
