/*
 * ==========================================================================================
 * =                            JAHIA'S ENTERPRISE DISTRIBUTION                             =
 * ==========================================================================================
 *
 *                                  http://www.jahia.com
 *
 * JAHIA'S ENTERPRISE DISTRIBUTIONS LICENSING - IMPORTANT INFORMATION
 * ==========================================================================================
 *
 *     Copyright (C) 2002-2019 Jahia Solutions Group. All rights reserved.
 *
 *     This file is part of a Jahia's Enterprise Distribution.
 *
 *     Jahia's Enterprise Distributions must be used in accordance with the terms
 *     contained in the Jahia Solutions Group Terms &amp; Conditions as well as
 *     the Jahia Sustainable Enterprise License (JSEL).
 *
 *     For questions regarding licensing, support, production usage...
 *     please contact our team at sales@jahia.com or go to http://www.jahia.com/license.
 *
 * ==========================================================================================
 */
package org.jahia.jdbc;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.sql.*;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * A {@link Connection} implementation that prevents execution of queries
 * altering the database when read-only mode is enabled.
 *
 * @author cmoitrier
 */
final class ReadOnlyModeAwareConnection extends DelegatingConnection {

    @FunctionalInterface
    interface ReadOnlyModeStatus {
        /**
         * Checks whether or not read-only mode is enabled
         * @return {@code true} if read-only mode is enabled, {@code false otherwise}
         */
        boolean isReadOnlyEnabled();
    }

    final ReadOnlyModeStatus status;

    ReadOnlyModeAwareConnection(Connection connection, ReadOnlyModeStatus status) {
        super(connection);
        this.status = status;
    }

    @Override
    public Statement createStatement() throws SQLException {
        Statement statement = super.createStatement();
        return createProxy(statement);
    }

    @Override
    public PreparedStatement prepareStatement(String sql) throws SQLException {
        PreparedStatement statement = super.prepareStatement(sql);
        return createProxy(statement, sql);
    }

    @Override
    public CallableStatement prepareCall(String sql) throws SQLException {
        CallableStatement call = super.prepareCall(sql);
        return createProxy(call, sql);
    }

    @Override
    public Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
        Statement statement = super.createStatement(resultSetType, resultSetConcurrency);
        return createProxy(statement);
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        PreparedStatement statement = super.prepareStatement(sql, resultSetType, resultSetConcurrency);
        return createProxy(statement, sql);
    }

    @Override
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        CallableStatement call = super.prepareCall(sql, resultSetType, resultSetConcurrency);
        return createProxy(call, sql);
    }

    @Override
    public Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        Statement statement = super.createStatement(resultSetType, resultSetConcurrency, resultSetHoldability);
        return createProxy(statement);
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        PreparedStatement statement = super.prepareStatement(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
        return createProxy(statement, sql);
    }

    @Override
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        CallableStatement call = super.prepareCall(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
        return createProxy(call, sql);
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
        PreparedStatement statement = super.prepareStatement(sql, autoGeneratedKeys);
        return createProxy(statement, sql);
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
        PreparedStatement statement = super.prepareStatement(sql, columnIndexes);
        return createProxy(statement, sql);
    }

    @Override
    public PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
        PreparedStatement statement = super.prepareStatement(sql, columnNames);
        return createProxy(statement, sql);
    }

    private Statement createProxy(Statement statement) {
        InvocationHandler handler = new ReadOnlyAwareStatementHandler<>(statement, status);
        return createProxy(Statement.class, handler);
    }

    private PreparedStatement createProxy(PreparedStatement statement, String sql) {
        InvocationHandler handler = new ReadOnlyAwareStatementHandler<>(statement, status, sql);
        return createProxy(PreparedStatement.class, handler);
    }

    private CallableStatement createProxy(CallableStatement statement, String sql) {
        InvocationHandler handler = new ReadOnlyAwareStatementHandler<>(statement, status, sql);
        return createProxy(CallableStatement.class, handler);
    }

    private <T extends Statement> T createProxy(Class<T> type, InvocationHandler handler) {
        return (T) Proxy.newProxyInstance(getClass().getClassLoader(), new Class[] {type}, handler);
    }

    static final class ReadOnlyAwareStatementHandler<T extends Statement> implements InvocationHandler {
        private static final Logger logger = LoggerFactory.getLogger(ReadOnlyModeAwareDataSource.class);
        private static final String GUARDED_OPERATION_PREFIX = "execute";
        private static final String ADD_BATCH_OPERATION = "addBatch";
        private static final String CLEAR_BATCH_OPERATION = "clearBatch";

        private final List<String> batches = new ArrayList<>();
        private final String sql;
        private final T target;
        private final ReadOnlyModeStatus status;

        private ReadOnlyAwareStatementHandler(T target, ReadOnlyModeStatus status) {
            this(target, status, null);
        }

        private ReadOnlyAwareStatementHandler(T target, ReadOnlyModeStatus status, String sql) {
            this.target = target;
            this.status = status;
            this.sql = sql;
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            if (status.isReadOnlyEnabled() && isGuardedOperation(method, args)) {
                logDismissedOperation(args);
                throw new ReadOnlySQLException();
            }

            Object result = method.invoke(target, args);

            if (ADD_BATCH_OPERATION.equals(method.getName())) {
                // Those queries need to be tracked in order to prevent their execution if one is forbidden
                if ((args != null) && (args.length > 0) && (args[0] instanceof String)) {
                    batches.add((String) args[0]);
                }

            } else if (CLEAR_BATCH_OPERATION.equals(method.getName())) {
                batches.clear();
            }

            return result;
        }

        /**
         * Checks whether or not an operation is forbidden
         *
         * @param method the method to check
         * @param args the arguments of {@code method}
         * @return {@code true} if execution is forbidden, {@code false} otherwise
         */
        private boolean isGuardedOperation(Method method, Object[] args) {
            if (method.getName().startsWith(GUARDED_OPERATION_PREFIX)) {
                if ((args == null) || (args.length == 0)) {
                    if ((target instanceof CallableStatement) || (target instanceof PreparedStatement)) {
                        // query template is provided at creation time
                        return !isQueryAllowed(sql);
                    } else {
                        // batch execution - those methods have no arguments
                        // and queries are added through addBatch(String)
                        // As per javadoc addBatch "cannot be called on a PreparedStatement or CallableStatement"
                        return batches.stream().anyMatch(s -> !isQueryAllowed(s));
                    }
                } else if (args[0] instanceof String) {
                    // any other execution - the query is the first argument
                    return !isQueryAllowed((String) args[0]);
                }
            }
            return false;
        }

        /*
         * Checks whether or not the given query is permitted in read-only mode
         *
         * @param sql the SQL query to test
         * @return {@code true} if the query is allowed, {@code false} otherwise
         */
        private boolean isQueryAllowed(String sql) {
            return (sql == null) || sql.toLowerCase().startsWith("select ");
        }

        private void logDismissedOperation(Object[] args) {
            if (!logger.isDebugEnabled()) {
                return;
            }

            if ((args == null) || (args.length == 0)) {
                if ((target instanceof CallableStatement) || (target instanceof PreparedStatement)) {
                    // query template is provided at creation time
                    logger.debug("Dismissed query: {}", sql);
                } else {
                    // batches
                    String queries = batches.stream().collect(Collectors.joining(",", "{", "}"));
                    logger.debug("Dismissed {} {}: {}", new Object[] {batches.size(), (batches.size() > 1 ? "queries" : "query"), queries});
                }
            } else {
                // any other execution
                logger.debug("Dismissed query: {}", args[0]);
            }
        }

    }

}
