/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.security.auth.message.config;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.security.auth.Subject;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.message.AuthException;
import javax.security.auth.message.AuthStatus;
import javax.security.auth.message.MessageInfo;
import javax.security.auth.message.ServerAuth;
import javax.security.auth.message.config.ServerAuthContext;
import javax.security.auth.message.module.ServerAuthModule;

//$Id: JBossServerAuthContext.java 73529 2008-05-20 14:46:23Z anil.saldhana@jboss.com $

/**
 *  Default Server Authentication Context
 *  @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 *  @since  May 17, 2006 
 *  @version $Revision: 73529 $
 */
public class JBossServerAuthContext implements ServerAuthContext
{  
   private List<ServerAuthModule> modules = new ArrayList<ServerAuthModule>(); 
   
   @SuppressWarnings("unchecked")
   private Map<String,Map> moduleOptionsByName = new HashMap<String,Map>();
   
   @SuppressWarnings("unchecked")
   public JBossServerAuthContext(List<ServerAuthModule> modules,
         Map<String,Map> moduleNameToOptions, CallbackHandler cbh) throws AuthException
   {
      this.modules = modules;
      this.moduleOptionsByName = moduleNameToOptions;
      for(ServerAuthModule sam:modules)
      {
         sam.initialize(null, null, cbh, 
               moduleOptionsByName.get(sam.getClass().getName())); 
      }
   }
   
   
   /**
    * @see ServerAuth#cleanSubject(Subject, Map)
    */
   public void cleanSubject(MessageInfo messageInfo, Subject subject) throws AuthException
   { 
      for(ServerAuthModule sam:modules)
      {
         sam.cleanSubject(messageInfo, subject);
      }
   }
   
   /**
    * @see ServerAuth#secureResponse(AuthParam, Subject, Map)
    */
   public AuthStatus secureResponse(MessageInfo messageInfo, Subject serviceSubject) throws AuthException
   { 
      AuthStatus status = null; 
      for(ServerAuthModule sam:modules)
      {
         status = sam.secureResponse(messageInfo, serviceSubject);
      }
      return status;
   }
   
   /**
    * @see ServerAuth#validateRequest(AuthParam, Subject, Subject, Map)
    */
   @SuppressWarnings("unchecked")
   public AuthStatus validateRequest(MessageInfo messageInfo, Subject clientSubject, 
         Subject serviceSubject) throws AuthException
   { 
      List<ServerAuthModule> supportingModules = new ArrayList<ServerAuthModule>();
      
      Class requestType = messageInfo.getRequestMessage().getClass();
      Class[] requestInterfaces = requestType.getInterfaces(); 
      
      List<Class> intfaee = Arrays.asList(requestInterfaces);
      AuthStatus status = null; 
      for(ServerAuthModule sam:modules)
      { 
         List<Class> supportedTypes = Arrays.asList(sam.getSupportedMessageTypes());
         
         //Check the interfaces
         for(Class clazz:intfaee)
         {
            if(supportedTypes.contains(clazz) && !supportingModules.contains(sam)) 
               supportingModules.add(sam);
         } 
         
         //Check the class type also
         if((supportedTypes.contains(Object.class) || supportedTypes.contains(requestType))
               && !supportingModules.contains(sam)) 
            supportingModules.add(sam); 
      }
      if(supportingModules.size() == 0)
         throw new RuntimeException("No ServerAuthModule configured to support type:"+requestType);
      
      for(ServerAuthModule sam:supportingModules)
      {
         status = sam.validateRequest(messageInfo, clientSubject, serviceSubject);
         if(status == AuthStatus.FAILURE)
            break;
      }
      return status;
   } 
}