/*
 * The MIT License
 *
 * Copyright 2022 CloudBees, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package org.jenkinsci.plugins.pipeline.modeldefinition.agent;

import org.kohsuke.stapler.DataBoundSetter;

/**
 * A type of {@code agent} option that supports automatic retries.
 * Usage from your {@link DeclarativeAgentScript#run} would look something like:
 * <pre>{@code
 * Closure run = {
 *     script.node {
 *         CheckoutScript.doCheckout(script, describable, null, body).call()
 *     }
 * }
 * if (describable.retries > 1) {
 *     return {
 *         script.retry(count: describable.retries, conditions: [script.agent(), script.nonresumable()]) {
 *             run.call()
 *         }
 *     }
 * } else {
 *     run
 * }}</pre>
 */
public abstract class RetryableDeclarativeAgent<A extends RetryableDeclarativeAgent<A>> extends DeclarativeAgent<A> {

    private int retries = 1;

    public int getRetries() {
        return retries;
    }

    @DataBoundSetter
    public void setRetries(int retries) {
        this.retries = retries;
    }

}
