/*
 * Decompiled with CFR 0.152.
 */
package org.junit.platform.commons.util;

import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.junit.platform.commons.PreconditionViolationException;
import org.junit.platform.commons.function.Try;
import org.junit.platform.commons.io.Resource;
import org.junit.platform.commons.io.ResourceFilter;
import org.junit.platform.commons.logging.Logger;
import org.junit.platform.commons.logging.LoggerFactory;
import org.junit.platform.commons.support.scanning.ClassFilter;
import org.junit.platform.commons.support.scanning.ClasspathScanner;
import org.junit.platform.commons.util.ClasspathFileVisitor;
import org.junit.platform.commons.util.CloseablePath;
import org.junit.platform.commons.util.Preconditions;
import org.junit.platform.commons.util.SearchPathUtils;
import org.junit.platform.commons.util.StringUtils;
import org.junit.platform.commons.util.UnrecoverableExceptions;

class DefaultClasspathScanner
implements ClasspathScanner {
    private static final Logger logger = LoggerFactory.getLogger(DefaultClasspathScanner.class);
    private static final char CLASSPATH_RESOURCE_PATH_SEPARATOR = '/';
    private static final String CLASSPATH_RESOURCE_PATH_SEPARATOR_STRING = String.valueOf('/');
    private static final String MALFORMED_CLASS_NAME_ERROR_MESSAGE = "Malformed class name";
    private final Supplier<ClassLoader> classLoaderSupplier;
    private final BiFunction<String, ClassLoader, Try<Class<?>>> loadClass;

    DefaultClasspathScanner(Supplier<ClassLoader> classLoaderSupplier, BiFunction<String, ClassLoader, Try<Class<?>>> loadClass) {
        this.classLoaderSupplier = classLoaderSupplier;
        this.loadClass = loadClass;
    }

    @Override
    public List<Class<?>> scanForClassesInPackage(String basePackageName, ClassFilter classFilter) {
        Preconditions.condition("".equals(basePackageName) || StringUtils.isNotBlank(basePackageName), "basePackageName must not be null or blank");
        Preconditions.notNull(classFilter, "classFilter must not be null");
        basePackageName = basePackageName.strip();
        List<URI> roots = this.getRootUrisForPackageNameOnClassPathAndModulePath(basePackageName);
        return this.findClassesForUris(roots, basePackageName, classFilter);
    }

    @Override
    public List<Class<?>> scanForClassesInClasspathRoot(URI root, ClassFilter classFilter) {
        Preconditions.notNull(root, "root must not be null");
        Preconditions.notNull(classFilter, "classFilter must not be null");
        return this.findClassesForUri(root, "", classFilter);
    }

    public List<Resource> scanForResourcesInPackage(String basePackageName, ResourceFilter resourceFilter) {
        Preconditions.condition("".equals(basePackageName) || StringUtils.isNotBlank(basePackageName), "basePackageName must not be null or blank");
        Preconditions.notNull(resourceFilter, "resourceFilter must not be null");
        basePackageName = basePackageName.strip();
        List<URI> roots = this.getRootUrisForPackageNameOnClassPathAndModulePath(basePackageName);
        return this.findResourcesForUris(roots, basePackageName, resourceFilter);
    }

    public List<Resource> scanForResourcesInClasspathRoot(URI root, ResourceFilter resourceFilter) {
        Preconditions.notNull(root, "root must not be null");
        Preconditions.notNull(resourceFilter, "resourceFilter must not be null");
        return this.findResourcesForUri(root, "", resourceFilter);
    }

    private List<Class<?>> findClassesForUris(List<URI> baseUris, String basePackageName, ClassFilter classFilter) {
        return baseUris.stream().map(baseUri -> this.findClassesForUri((URI)baseUri, basePackageName, classFilter)).flatMap(Collection::stream).distinct().toList();
    }

    private List<Class<?>> findClassesForUri(URI baseUri, String basePackageName, ClassFilter classFilter) {
        ArrayList classes = new ArrayList();
        DefaultClasspathScanner.walkFilesForUri(baseUri, SearchPathUtils::isClassOrSourceFile, (baseDir, file) -> this.processClassFileSafely((Path)baseDir, basePackageName, classFilter, (Path)file, classes::add));
        return classes;
    }

    private List<Resource> findResourcesForUris(List<URI> baseUris, String basePackageName, ResourceFilter resourceFilter) {
        return baseUris.stream().map(baseUri -> this.findResourcesForUri((URI)baseUri, basePackageName, resourceFilter)).flatMap(Collection::stream).distinct().toList();
    }

    private List<Resource> findResourcesForUri(URI baseUri, String basePackageName, ResourceFilter resourceFilter) {
        ArrayList<Resource> resources = new ArrayList<Resource>();
        DefaultClasspathScanner.walkFilesForUri(baseUri, SearchPathUtils::isResourceFile, (baseDir, file) -> this.processResourceFileSafely((Path)baseDir, basePackageName, resourceFilter, (Path)file, resources::add));
        return resources;
    }

    private static void walkFilesForUri(URI baseUri, Predicate<Path> filter, BiConsumer<Path, Path> consumer) {
        try (CloseablePath closeablePath = CloseablePath.create(baseUri);){
            Path baseDir = closeablePath.getPath();
            Preconditions.condition(Files.exists(baseDir, new LinkOption[0]), () -> "baseDir must exist: " + String.valueOf(baseDir));
            try {
                Files.walkFileTree(baseDir, new ClasspathFileVisitor(baseDir, filter, consumer));
            }
            catch (IOException ex) {
                logger.warn(ex, () -> "I/O error scanning files in " + String.valueOf(baseDir));
            }
        }
        catch (PreconditionViolationException ex) {
            throw ex;
        }
        catch (Exception ex) {
            logger.warn(ex, () -> "Error scanning files for URI " + String.valueOf(baseUri));
        }
    }

    private void processClassFileSafely(Path baseDir, String basePackageName, ClassFilter classFilter, Path file, Consumer<Class<?>> classConsumer) {
        try {
            String fullyQualifiedClassName = this.determineFullyQualifiedClassName(baseDir, basePackageName, file);
            if (classFilter.match(fullyQualifiedClassName)) {
                try {
                    this.loadClass.apply(fullyQualifiedClassName, this.getClassLoader()).toOptional().filter(classFilter::match).ifPresent(classConsumer);
                }
                catch (InternalError internalError) {
                    this.handleInternalError(file, fullyQualifiedClassName, internalError);
                }
            }
        }
        catch (Throwable throwable) {
            this.handleThrowable(file, throwable);
        }
    }

    private void processResourceFileSafely(Path baseDir, String basePackageName, ResourceFilter resourceFilter, Path resourceFile, Consumer<Resource> resourceConsumer) {
        try {
            String fullyQualifiedResourceName = this.determineFullyQualifiedResourceName(baseDir, basePackageName, resourceFile);
            Resource resource = Resource.of(fullyQualifiedResourceName, resourceFile.toUri());
            if (resourceFilter.match(resource)) {
                resourceConsumer.accept(resource);
            }
        }
        catch (Throwable throwable) {
            this.handleThrowable(resourceFile, throwable);
        }
    }

    private String determineFullyQualifiedClassName(Path baseDir, String basePackageName, Path file) {
        return Stream.of(basePackageName, this.determineSubpackageName(baseDir, file), SearchPathUtils.determineSimpleClassName(file)).filter(value -> !value.isEmpty()).collect(Collectors.joining(SearchPathUtils.PACKAGE_SEPARATOR_STRING));
    }

    private String determineFullyQualifiedResourceName(Path baseDir, String basePackageName, Path resourceFile) {
        return Stream.of(DefaultClasspathScanner.packagePath(basePackageName), DefaultClasspathScanner.packagePath(this.determineSubpackageName(baseDir, resourceFile)), this.determineSimpleResourceName(resourceFile)).filter(value -> !value.isEmpty()).collect(Collectors.joining(CLASSPATH_RESOURCE_PATH_SEPARATOR_STRING));
    }

    private String determineSimpleResourceName(Path resourceFile) {
        return resourceFile.getFileName().toString();
    }

    private String determineSubpackageName(Path baseDir, Path file) {
        Path relativePath = baseDir.relativize(file.getParent());
        String pathSeparator = baseDir.getFileSystem().getSeparator();
        return relativePath.toString().replace(pathSeparator, SearchPathUtils.PACKAGE_SEPARATOR_STRING);
    }

    private void handleInternalError(Path classFile, String fullyQualifiedClassName, InternalError ex) {
        if (MALFORMED_CLASS_NAME_ERROR_MESSAGE.equals(ex.getMessage())) {
            this.logMalformedClassName(classFile, fullyQualifiedClassName, ex);
        } else {
            this.logGenericFileProcessingException(classFile, ex);
        }
    }

    private void handleThrowable(Path classFile, Throwable throwable) {
        UnrecoverableExceptions.rethrowIfUnrecoverable(throwable);
        this.logGenericFileProcessingException(classFile, throwable);
    }

    private void logMalformedClassName(Path classFile, String fullyQualifiedClassName, InternalError ex) {
        try {
            logger.debug(ex, () -> "The java.lang.Class loaded from path [%s] has a malformed class name [%s].".formatted(classFile.toAbsolutePath(), fullyQualifiedClassName));
        }
        catch (Throwable t) {
            UnrecoverableExceptions.rethrowIfUnrecoverable(t);
            ex.addSuppressed(t);
            this.logGenericFileProcessingException(classFile, ex);
        }
    }

    private void logGenericFileProcessingException(Path classpathFile, Throwable throwable) {
        logger.debug(throwable, () -> "Failed to load [%s] during classpath scanning.".formatted(classpathFile.toAbsolutePath()));
    }

    private ClassLoader getClassLoader() {
        return this.classLoaderSupplier.get();
    }

    private List<URI> getRootUrisForPackageNameOnClassPathAndModulePath(String basePackageName) {
        LinkedHashSet<URI> uriSet = new LinkedHashSet<URI>(this.getRootUrisForPackage(basePackageName));
        if (!basePackageName.isEmpty() && !basePackageName.endsWith(SearchPathUtils.PACKAGE_SEPARATOR_STRING)) {
            this.getRootUrisForPackage(basePackageName + SearchPathUtils.PACKAGE_SEPARATOR_STRING).stream().map(DefaultClasspathScanner::removeTrailingClasspathResourcePathSeparator).forEach(uriSet::add);
        }
        return new ArrayList<URI>(uriSet);
    }

    private static URI removeTrailingClasspathResourcePathSeparator(URI uri) {
        String string = uri.toString();
        if (string.endsWith(CLASSPATH_RESOURCE_PATH_SEPARATOR_STRING)) {
            return URI.create(string.substring(0, string.length() - 1));
        }
        return uri;
    }

    private static String packagePath(String packageName) {
        if (packageName.isEmpty()) {
            return "";
        }
        return packageName.replace('.', '/');
    }

    private List<URI> getRootUrisForPackage(String basePackageName) {
        ArrayList<URI> uris = new ArrayList<URI>();
        try {
            Enumeration<URL> resources = this.getClassLoader().getResources(DefaultClasspathScanner.packagePath(basePackageName));
            while (resources.hasMoreElements()) {
                URL resource = resources.nextElement();
                uris.add(resource.toURI());
            }
        }
        catch (Exception ex) {
            logger.warn(ex, () -> "Error reading URIs from class loader for base package " + basePackageName);
        }
        return uris;
    }
}

