/*
 * Decompiled with CFR 0.152.
 */
package apoc.periodic;

import apoc.Pools;
import apoc.util.Util;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.LockSupport;
import java.util.function.Consumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Result;
import org.neo4j.helpers.collection.Iterators;
import org.neo4j.helpers.collection.Pair;
import org.neo4j.logging.Log;
import org.neo4j.procedure.Context;
import org.neo4j.procedure.Description;
import org.neo4j.procedure.Mode;
import org.neo4j.procedure.Name;
import org.neo4j.procedure.Procedure;
import org.neo4j.procedure.TerminationGuard;

public class Periodic {
    public static final Pattern RUNTIME_PATTERN = Pattern.compile("\\bruntime\\s*=", 2);
    public static final Pattern CYPHER_PREFIX_PATTERN = Pattern.compile("\\bcypher\\b", 2);
    public static final String CYPHER_RUNTIME_SLOTTED = "cypher runtime=slotted ";
    @Context
    public GraphDatabaseService db;
    @Context
    public TerminationGuard terminationGuard;
    @Context
    public Log log;
    static final Map<JobInfo, Future> list = new ConcurrentHashMap<JobInfo, Future>();
    static final Pattern LIMIT_PATTERN = Pattern.compile("\\slimit\\s", 2);

    @Procedure
    @Description(value="apoc.periodic.list - list all jobs")
    public Stream<JobInfo> list() {
        return list.entrySet().stream().map(e -> ((JobInfo)e.getKey()).update((Future)e.getValue()));
    }

    @Procedure(mode=Mode.WRITE)
    @Description(value="apoc.periodic.commit(statement,params) - runs the given statement in separate transactions until it returns 0")
    public Stream<RundownResult> commit(@Name(value="statement") String statement, @Name(value="params", defaultValue="") Map<String, Object> parameters) throws ExecutionException, InterruptedException {
        this.validateQuery(statement);
        Map<Object, Object> params = parameters == null ? Collections.emptyMap() : parameters;
        long total = 0L;
        long executions = 0L;
        long updates = 0L;
        long start = System.nanoTime();
        if (!LIMIT_PATTERN.matcher(statement).find()) {
            throw new IllegalArgumentException("the statement sent to apoc.periodic.commit must contain a `limit`");
        }
        AtomicInteger batches = new AtomicInteger();
        AtomicInteger failedCommits = new AtomicInteger();
        ConcurrentHashMap<String, Long> commitErrors = new ConcurrentHashMap<String, Long>();
        AtomicInteger failedBatches = new AtomicInteger();
        ConcurrentHashMap<String, Long> batchErrors = new ConcurrentHashMap<String, Long>();
        do {
            Map<String, Object> window = Util.map("_count", updates, "_total", total);
            updates = Util.getFuture(Pools.SCHEDULED.submit(() -> {
                batches.incrementAndGet();
                try {
                    return this.executeNumericResultStatement(statement, Util.merge(window, params));
                }
                catch (Exception e) {
                    failedBatches.incrementAndGet();
                    this.recordError(batchErrors, e);
                    return 0L;
                }
            }), commitErrors, failedCommits, 0L);
            total += updates;
            if (updates <= 0L) continue;
            ++executions;
        } while (updates > 0L && !Util.transactionIsTerminated(this.terminationGuard));
        long timeTaken = TimeUnit.NANOSECONDS.toSeconds(System.nanoTime() - start);
        boolean wasTerminated = Util.transactionIsTerminated(this.terminationGuard);
        return Stream.of(new RundownResult(total, executions, timeTaken, batches.get(), failedBatches.get(), batchErrors, failedCommits.get(), commitErrors, wasTerminated));
    }

    private void recordError(Map<String, Long> executionErrors, Exception e) {
        executionErrors.compute(this.getMessages(e), (s, i) -> i == null ? 1L : i + 1L);
    }

    private String getMessages(Throwable e) {
        LinkedHashSet<String> errors = new LinkedHashSet<String>();
        do {
            errors.add(e.getMessage());
        } while ((e = e.getCause()).getCause() != null && !e.getCause().equals(e));
        return String.join((CharSequence)"\n", errors);
    }

    private long executeNumericResultStatement(@Name(value="statement") String statement, @Name(value="params") Map<String, Object> parameters) {
        long sum = 0L;
        try (Result result = this.db.execute(statement, parameters);){
            while (result.hasNext()) {
                Collection row = result.next().values();
                for (Object value : row) {
                    if (!(value instanceof Number)) continue;
                    sum += ((Number)value).longValue();
                }
            }
        }
        return sum;
    }

    @Procedure
    @Description(value="apoc.periodic.cancel(name) - cancel job with the given name")
    public Stream<JobInfo> cancel(@Name(value="name") String name) {
        JobInfo info = new JobInfo(name);
        Future future = list.remove(info);
        if (future != null) {
            future.cancel(false);
            return Stream.of(info.update(future));
        }
        return Stream.empty();
    }

    @Procedure(mode=Mode.WRITE)
    @Description(value="apoc.periodic.submit('name',statement) - submit a one-off background statement")
    public Stream<JobInfo> submit(@Name(value="name") String name, @Name(value="statement") String statement) {
        this.validateQuery(statement);
        JobInfo info = Periodic.submit(name, () -> {
            try {
                Iterators.count((Iterator)this.db.execute(statement));
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
        return Stream.of(info);
    }

    @Procedure(mode=Mode.WRITE)
    @Description(value="apoc.periodic.repeat('name',statement,repeat-rate-in-seconds, config) submit a repeatedly-called background statement. Fourth parameter 'config' is optional and can contain 'params' entry for nested statement.")
    public Stream<JobInfo> repeat(@Name(value="name") String name, @Name(value="statement") String statement, @Name(value="rate") long rate, @Name(value="config", defaultValue="{}") Map<String, Object> config) {
        this.validateQuery(statement);
        Map params = config.getOrDefault("params", Collections.emptyMap());
        JobInfo info = Periodic.schedule(name, () -> Iterators.count((Iterator)this.db.execute(statement, params)), 0L, rate);
        return Stream.of(info);
    }

    private void validateQuery(String statement) {
        this.db.execute("EXPLAIN " + statement).close();
    }

    @Procedure(mode=Mode.WRITE)
    @Description(value="apoc.periodic.countdown('name',statement,repeat-rate-in-seconds) submit a repeatedly-called background statement until it returns 0")
    public Stream<JobInfo> countdown(@Name(value="name") String name, @Name(value="statement") String statement, @Name(value="rate") long rate) {
        this.validateQuery(statement);
        JobInfo info = Periodic.submit(name, new Countdown(name, statement, rate));
        info.rate = rate;
        return Stream.of(info);
    }

    public static <T> JobInfo submit(String name, Runnable task) {
        JobInfo info = new JobInfo(name);
        Future future = list.remove(info);
        if (future != null && !future.isDone()) {
            future.cancel(false);
        }
        Future<?> newFuture = Pools.SCHEDULED.submit(task);
        list.put(info, newFuture);
        return info;
    }

    public static JobInfo schedule(String name, Runnable task, long delay, long repeat) {
        JobInfo info = new JobInfo(name, delay, repeat);
        Future future = list.remove(info);
        if (future != null && !future.isDone()) {
            future.cancel(false);
        }
        ScheduledFuture<?> newFuture = Pools.SCHEDULED.scheduleWithFixedDelay(task, delay, repeat, TimeUnit.SECONDS);
        list.put(info, newFuture);
        return info;
    }

    @Procedure(mode=Mode.WRITE)
    @Description(value="apoc.periodic.rock_n_roll_while('some cypher for knowing when to stop', 'some cypher for iteration', 'some cypher as action on each iteration', 10000) YIELD batches, total - run the action statement in batches over the iterator statement's results in a separate thread. Returns number of batches and total processed rows")
    public Stream<LoopingBatchAndTotalResult> rock_n_roll_while(@Name(value="cypherLoop") String cypherLoop, @Name(value="cypherIterate") String cypherIterate, @Name(value="cypherAction") String cypherAction, @Name(value="batchSize") long batchSize) {
        Map<String, String> fieldStatement = Util.map("cypherLoop", cypherLoop, "cypherIterate", cypherIterate);
        this.validateQueries(fieldStatement);
        Stream<LoopingBatchAndTotalResult> allResults = Stream.empty();
        HashMap<String, Object> loopParams = new HashMap<String, Object>(1);
        Object value = null;
        while (true) {
            loopParams.put("previous", value);
            try (Result result = this.db.execute(cypherLoop, loopParams);){
                value = result.next().get("loop");
                if (!Util.toBoolean(value)) {
                    Stream<LoopingBatchAndTotalResult> stream = allResults;
                    return stream;
                }
            }
            this.log.info("starting batched operation using iteration `%s` in separate thread", new Object[]{cypherIterate});
            result = this.db.execute(cypherIterate);
            var11_10 = null;
            try {
                Stream<BatchAndTotalResult> oneResult = this.iterateAndExecuteBatchedInSeparateThread((int)batchSize, false, false, 0L, (Iterator<Map<String, Object>>)result, params -> this.db.execute(cypherAction, params), 50, -1);
                Object loopParam = value;
                allResults = Stream.concat(allResults, oneResult.map(r -> r.inLoop(loopParam)));
                continue;
            }
            catch (Throwable throwable) {
                var11_10 = throwable;
                throw throwable;
            }
            finally {
                if (result == null) continue;
                if (var11_10 != null) {
                    try {
                        result.close();
                    }
                    catch (Throwable throwable) {
                        var11_10.addSuppressed(throwable);
                    }
                    continue;
                }
                result.close();
                continue;
            }
            break;
        }
    }

    private void validateQueries(Map<String, String> fieldStatement) {
        String error = fieldStatement.entrySet().stream().map(e -> {
            try {
                this.validateQuery((String)e.getValue());
                return null;
            }
            catch (Exception exception) {
                return String.format("Exception for field `%s`, message: %s", e.getKey(), exception.getMessage());
            }
        }).filter(e -> e != null).collect(Collectors.joining("\n"));
        if (!error.isEmpty()) {
            throw new RuntimeException(error);
        }
    }

    @Procedure(mode=Mode.WRITE)
    @Description(value="apoc.periodic.iterate('statement returning items', 'statement per item', {batchSize:1000,iterateList:true,parallel:false,params:{},concurrency:50,retries:0}) YIELD batches, total - run the second statement for each item returned by the first statement. Returns number of batches and total processed rows")
    public Stream<BatchAndTotalResult> iterate(@Name(value="cypherIterate") String cypherIterate, @Name(value="cypherAction") String cypherAction, @Name(value="config") Map<String, Object> config) {
        this.validateQuery(cypherIterate);
        long batchSize = Util.toLong(config.getOrDefault("batchSize", 10000));
        int concurrency = Util.toInteger(config.getOrDefault("concurrency", 50));
        boolean parallel = Util.toBoolean(config.getOrDefault("parallel", false));
        boolean iterateList = Util.toBoolean(config.getOrDefault("iterateList", true));
        long retries = Util.toLong(config.getOrDefault("retries", 0));
        Map params = config.getOrDefault("params", Collections.emptyMap());
        int failedParams = Util.toInteger(config.getOrDefault("failedParams", -1));
        try (Result result = this.db.execute(Periodic.slottedRuntime(cypherIterate), params);){
            Pair<String, Boolean> prepared = this.prepareInnerStatement(cypherAction, iterateList, result.columns(), "_batch");
            String innerStatement = (String)prepared.first();
            iterateList = (Boolean)prepared.other();
            this.log.info("starting batching from `%s` operation using iteration `%s` in separate thread", new Object[]{cypherIterate, cypherAction});
            Stream<BatchAndTotalResult> stream = this.iterateAndExecuteBatchedInSeparateThread((int)batchSize, parallel, iterateList, retries, (Iterator<Map<String, Object>>)result, p -> this.db.execute(innerStatement, Util.merge(params, p)).close(), concurrency, failedParams);
            return stream;
        }
    }

    static String slottedRuntime(String cypherIterate) {
        if (RUNTIME_PATTERN.matcher(cypherIterate).find()) {
            return cypherIterate;
        }
        Matcher matcher = CYPHER_PREFIX_PATTERN.matcher(cypherIterate.substring(0, Math.min(15, cypherIterate.length())));
        return matcher.find() ? CYPHER_PREFIX_PATTERN.matcher(cypherIterate).replaceFirst(CYPHER_RUNTIME_SLOTTED) : CYPHER_RUNTIME_SLOTTED + cypherIterate;
    }

    public long retry(Consumer<Map<String, Object>> executor, Map<String, Object> params, long retry, long maxRetries) {
        try {
            executor.accept(Util.merge(params, Collections.singletonMap("_retry", retry)));
            return retry;
        }
        catch (Exception e) {
            if (retry >= maxRetries) {
                throw e;
            }
            this.log.warn("Retrying operation " + retry + " of " + maxRetries);
            Util.sleep(100);
            return this.retry(executor, params, retry + 1L, maxRetries);
        }
    }

    public Pair<String, Boolean> prepareInnerStatement(String cypherAction, boolean iterateList, List<String> columns, String iterator) {
        String names = columns.stream().map(Util::quote).collect(Collectors.joining("|"));
        boolean withCheck = this.regNoCaseMultiLine("[{$](" + names + ")\\}?\\s+AS\\s+").matcher(cypherAction).find();
        if (withCheck) {
            return Pair.of((Object)cypherAction, (Object)false);
        }
        if (iterateList) {
            if (this.regNoCaseMultiLine("UNWIND\\s+[{$]" + iterator + "\\}?\\s+AS\\s+").matcher(cypherAction).find()) {
                return Pair.of((Object)cypherAction, (Object)true);
            }
            String with = Util.withMapping(columns.stream(), c -> Util.quote(iterator) + "." + Util.quote(c) + " AS " + Util.quote(c));
            return Pair.of((Object)("UNWIND " + Util.param(iterator) + " AS " + Util.quote(iterator) + with + " " + cypherAction), (Object)true);
        }
        return Pair.of((Object)(Util.withMapping(columns.stream(), c -> Util.param(c) + " AS " + Util.quote(c)) + cypherAction), (Object)false);
    }

    public Pattern regNoCaseMultiLine(String pattern) {
        return Pattern.compile(pattern, 42);
    }

    @Procedure(mode=Mode.WRITE)
    @Description(value="apoc.periodic.rock_n_roll('some cypher for iteration', 'some cypher as action on each iteration', 10000) YIELD batches, total - run the action statement in batches over the iterator statement's results in a separate thread. Returns number of batches and total processed rows")
    public Stream<BatchAndTotalResult> rock_n_roll(@Name(value="cypherIterate") String cypherIterate, @Name(value="cypherAction") String cypherAction, @Name(value="batchSize") long batchSize) {
        Map<String, String> fieldStatement = Util.map("cypherIterate", cypherIterate, "cypherAction", cypherAction);
        this.validateQueries(fieldStatement);
        this.log.info("starting batched operation using iteration `%s` in separate thread", new Object[]{cypherIterate});
        try (Result result = this.db.execute(cypherIterate);){
            Stream<BatchAndTotalResult> stream = this.iterateAndExecuteBatchedInSeparateThread((int)batchSize, false, false, 0L, (Iterator<Map<String, Object>>)result, p -> this.db.execute(cypherAction, p).close(), 50, -1);
            return stream;
        }
    }

    private Stream<BatchAndTotalResult> iterateAndExecuteBatchedInSeparateThread(int batchsize, boolean parallel, boolean iterateList, long retries, Iterator<Map<String, Object>> iterator, Consumer<Map<String, Object>> consumer, int concurrency, int failedParams) {
        boolean wasTerminated;
        ExecutorService pool = parallel ? Pools.DEFAULT : Pools.SINGLE;
        ArrayList<Future<Long>> futures = new ArrayList<Future<Long>>(concurrency);
        long batches = 0L;
        long start = System.nanoTime();
        AtomicLong count = new AtomicLong();
        AtomicInteger failedOps = new AtomicInteger();
        AtomicLong retried = new AtomicLong();
        ConcurrentHashMap<String, Long> operationErrors = new ConcurrentHashMap<String, Long>();
        AtomicInteger failedBatches = new AtomicInteger();
        HashMap<String, Long> batchErrors = new HashMap<String, Long>();
        ConcurrentHashMap<String, List<Map<String, Object>>> failedParamsMap = new ConcurrentHashMap<String, List<Map<String, Object>>>();
        long successes = 0L;
        while (!Util.transactionIsTerminated(this.terminationGuard)) {
            Callable<Long> task;
            long finalBatches;
            if (this.log.isDebugEnabled()) {
                this.log.debug("execute in batch no " + batches + " batch size " + batchsize);
            }
            List<Map<String, Object>> batch = Util.take(iterator, batchsize);
            long currentBatchSize = batch.size();
            if (iterateList) {
                finalBatches = batches;
                task = () -> {
                    long c = count.addAndGet(currentBatchSize);
                    if (Util.transactionIsTerminated(this.terminationGuard)) {
                        return 0L;
                    }
                    try {
                        Map<String, Object> params = Util.map("_count", c, "_batch", batch);
                        retried.addAndGet(this.retry(consumer, params, 0L, retries));
                    }
                    catch (Exception e) {
                        failedOps.addAndGet(batchsize);
                        if (failedParams >= 0) {
                            failedParamsMap.put(Long.toString(finalBatches), new ArrayList(batch.subList(0, Math.min(failedParams + 1, batch.size()))));
                        }
                        this.recordError(operationErrors, e);
                    }
                    return currentBatchSize;
                };
            } else {
                finalBatches = batches;
                task = () -> {
                    if (Util.transactionIsTerminated(this.terminationGuard)) {
                        return 0L;
                    }
                    return batch.stream().map(p -> {
                        long c = count.incrementAndGet();
                        if (c % 1000L == 0L && Util.transactionIsTerminated(this.terminationGuard)) {
                            return 0;
                        }
                        try {
                            Map<String, Object> params = Util.merge(p, Util.map("_count", c, "_batch", batch));
                            retried.addAndGet(this.retry(consumer, params, 0L, retries));
                        }
                        catch (Exception e) {
                            failedOps.incrementAndGet();
                            if (failedParams >= 0) {
                                failedParamsMap.put(Long.toString(finalBatches), new ArrayList(batch.subList(0, Math.min(failedParams + 1, batch.size()))));
                            }
                            this.recordError(operationErrors, e);
                        }
                        return 1;
                    }).mapToLong(l -> l.intValue()).sum();
                };
            }
            futures.add(Util.inTxFuture(pool, this.db, task));
            ++batches;
            if (futures.size() > concurrency) {
                while (futures.stream().noneMatch(Future::isDone)) {
                    LockSupport.parkNanos(1000L);
                }
                Iterator it = futures.iterator();
                while (it.hasNext()) {
                    Future future = (Future)it.next();
                    if (!future.isDone()) continue;
                    successes += Util.getFuture(future, batchErrors, failedBatches, 0L).longValue();
                    it.remove();
                }
            }
            if (iterator.hasNext()) continue;
        }
        successes = (wasTerminated = Util.transactionIsTerminated(this.terminationGuard)) ? (successes += futures.stream().mapToLong(f -> Util.getFutureOrCancel(f, batchErrors, failedBatches, 0L)).sum()) : (successes += futures.stream().mapToLong(f -> Util.getFuture(f, batchErrors, failedBatches, 0L)).sum());
        Util.logErrors("Error during iterate.commit:", batchErrors, this.log);
        Util.logErrors("Error during iterate.execute:", operationErrors, this.log);
        long timeTaken = TimeUnit.NANOSECONDS.toSeconds(System.nanoTime() - start);
        BatchAndTotalResult result = new BatchAndTotalResult(batches, count.get(), timeTaken, successes, failedOps.get(), failedBatches.get(), retried.get(), operationErrors, batchErrors, wasTerminated, failedParamsMap);
        return Stream.of(result);
    }

    public static JobInfo schedule(String name, Runnable task, long delay) {
        JobInfo info = new JobInfo(name, delay, 0L);
        Future future = list.remove(info);
        if (future != null) {
            future.cancel(false);
        }
        ScheduledFuture<?> newFuture = Pools.SCHEDULED.schedule(task, delay, TimeUnit.SECONDS);
        list.put(info, newFuture);
        return info;
    }

    static {
        Runnable runnable = () -> {
            Iterator<Map.Entry<JobInfo, Future>> it = list.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry<JobInfo, Future> entry = it.next();
                if (!entry.getValue().isDone() && !entry.getValue().isCancelled()) continue;
                it.remove();
            }
        };
        Pools.SCHEDULED.scheduleAtFixedRate(runnable, 10L, 10L, TimeUnit.SECONDS);
    }

    private class Countdown
    implements Runnable {
        private final String name;
        private final String statement;
        private final long rate;

        public Countdown(String name, String statement, long rate) {
            this.name = name;
            this.statement = statement;
            this.rate = rate;
        }

        @Override
        public void run() {
            if (Periodic.this.executeNumericResultStatement(this.statement, Collections.emptyMap()) > 0L) {
                Pools.SCHEDULED.schedule(() -> Periodic.submit(this.name, this), this.rate, TimeUnit.SECONDS);
            }
        }
    }

    public static class JobInfo {
        public final String name;
        public long delay;
        public long rate;
        public boolean done;
        public boolean cancelled;

        public JobInfo(String name) {
            this.name = name;
        }

        public JobInfo(String name, long delay, long rate) {
            this.name = name;
            this.delay = delay;
            this.rate = rate;
        }

        public JobInfo update(Future future) {
            this.done = future.isDone();
            this.cancelled = future.isCancelled();
            return this;
        }

        public boolean equals(Object o) {
            return this == o || o instanceof JobInfo && this.name.equals(((JobInfo)o).name);
        }

        public int hashCode() {
            return this.name.hashCode();
        }
    }

    public static class LoopingBatchAndTotalResult {
        public Object loop;
        public long batches;
        public long total;

        public LoopingBatchAndTotalResult(Object loop, long batches, long total) {
            this.loop = loop;
            this.batches = batches;
            this.total = total;
        }
    }

    public static class BatchAndTotalResult {
        public final long batches;
        public final long total;
        public final long timeTaken;
        public final long committedOperations;
        public final long failedOperations;
        public final long failedBatches;
        public final long retries;
        public final Map<String, Long> errorMessages;
        public final Map<String, Object> batch;
        public final Map<String, Object> operations;
        public final boolean wasTerminated;
        public final Map<String, List<Map<String, Object>>> failedParams;

        public BatchAndTotalResult(long batches, long total, long timeTaken, long committedOperations, long failedOperations, long failedBatches, long retries, Map<String, Long> operationErrors, Map<String, Long> batchErrors, boolean wasTerminated, Map<String, List<Map<String, Object>>> failedParams) {
            this.batches = batches;
            this.total = total;
            this.timeTaken = timeTaken;
            this.committedOperations = committedOperations;
            this.failedOperations = failedOperations;
            this.failedBatches = failedBatches;
            this.retries = retries;
            this.errorMessages = operationErrors;
            this.wasTerminated = wasTerminated;
            this.failedParams = failedParams;
            this.batch = Util.map("total", batches, "failed", failedBatches, "committed", batches - failedBatches, "errors", batchErrors);
            this.operations = Util.map("total", total, "failed", failedOperations, "committed", committedOperations, "errors", operationErrors);
        }

        public LoopingBatchAndTotalResult inLoop(Object loop) {
            return new LoopingBatchAndTotalResult(loop, this.batches, this.total);
        }
    }

    public static class RundownResult {
        public final long updates;
        public final long executions;
        public final long runtime;
        public final long batches;
        public final long failedBatches;
        public final Map<String, Long> batchErrors;
        public final long failedCommits;
        public final Map<String, Long> commitErrors;
        public final boolean wasTerminated;

        public RundownResult(long total, long executions, long timeTaken, long batches, long failedBatches, Map<String, Long> batchErrors, long failedCommits, Map<String, Long> commitErrors, boolean wasTerminated) {
            this.updates = total;
            this.executions = executions;
            this.runtime = timeTaken;
            this.batches = batches;
            this.failedBatches = failedBatches;
            this.batchErrors = batchErrors;
            this.failedCommits = failedCommits;
            this.commitErrors = commitErrors;
            this.wasTerminated = wasTerminated;
        }
    }
}

