/*
 * (C) Copyright 2006-2015 Nuxeo SA (http://nuxeo.com/) and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-2.1.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     bstefanescu
 */
package org.nuxeo.connect.update.task;

import java.util.Map;

import org.nuxeo.connect.update.PackageException;
import org.nuxeo.connect.update.PackageUpdateService;
import org.nuxeo.connect.update.ValidationStatus;
import org.nuxeo.connect.update.xml.XmlWriter;
import org.w3c.dom.Element;

/**
 * An <b>atomic</b> command that can be executed as part of a task. Commands are logged so that they can be rollbacked
 * if needed. A command is assumed to be atomic - so that if anything goes wrong inside the execute method the command
 * will not be rollbacked - only previous commands are rollbacked. If you need to implement a non atomic command then
 * you should take care yourself of the partial rollback if an error occurs during the execution. The generic T
 * represent the task type where this command can be executed
 *
 * @author <a href="mailto:bs@nuxeo.com">Bogdan Stefanescu</a>
 */
public interface Command {

    /**
     * Sets the package update service.
     *
     * @since 8.4
     */
    default void setPackageUpdateService(PackageUpdateService packageUpdateService) {
    }

    /**
     * Test if the command must be run at the end in an installation process. This is useful for flush like commands
     * that reset application cache. These commands musty be executed at the install/uninstall end. By default the
     * uninstall log is generated by using the inverse of the installation commands executed in the inverse order. This
     * is not good for flush like commands that should always be executed at the end. For example if you have a copy and
     * a flush command in the install.xml the generated uninstall.xml will have a flush and a delete command - but he
     * order is not good since flush should be done at the end. By marking your command as post-install it will be put
     * To implement a post-install command simply override <a href=
     * "http://community.nuxeo.com/api/nuxeo/latest/javadoc/org/nuxeo/connect/update/task/standalone/commands/PostInstallCommand.html"
     * >org.nuxeo.connect.update.task.standalone.commands.PostInstallCommand</a>.
     */
    boolean isPostInstall();

    /**
     * The command id
     */
    String getId();

    /**
     * Execute the command and return the inverse command that can be used to rollback. Can return null if no rollback
     * command is needed.
     *
     * @param task
     * @param prefs user preferences - can be null if no preferences are given
     * @throws PackageException
     */
    Command run(Task task, Map<String, String> prefs) throws PackageException;

    /**
     * Validate if the command can be safely executed on the running platform. The command should check if the artifacts
     * it needs to copy or replace are consistent with the command configuration. If inconsistency is detected errors or
     * warnings must be added to the status.
     *
     * @param status
     */
    void validate(Task task, ValidationStatus status) throws PackageException;

    /**
     * Initialize a command parameters from a DOM element. This method will be called only once by the task when the
     * commands are loaded from an XML file.
     *
     * @param element
     * @throws PackageException
     */
    void initialize(Element element) throws PackageException;

    /**
     * Write the command as XML using the given {@link XmlWriter}. This method will be called by the task to create an
     * uninstall.xml file.
     *
     * @param writer
     */
    void writeTo(XmlWriter writer);

}
