// Copyright (C) 2008 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.


/**
 * This file combines the JSON.parse method defined by the original
 * json_sans_eval.js with the stringify method from the original
 * json2.js. Like json2.js, it defines a JSON object if one does not
 * already exist, and it initializes its parse and stringify methods
 * only if JSON does not currently have such methods (functions at
 * those property names). Additionally, if there is no
 * <tt>Date.prototype.toJSON</tt>, this file defines an ES5 compliant
 * one as well as the <tt>toJSON</tt> methods for <tt>String</tt>,
 * <tt>Number</tt>, and <tt>Boolean</tt>. The latter three are no
 * longer part of ES5, but are expected by the parts of this file
 * derived from json2.js.
 *
 * Of course, the reason this is included in the Caja distribution is
 * so that Caja can expose an ES5 compliant but Caja-safe JSON object
 * to cajoled code. Caja's wrapping of the provided JSON therefore
 * must work with either a real ES5 JSON or the one defined in this
 * file. To do so, Caja uses the replacer and reviver
 * hooks. Fortunately, ES5 and json2.js both specify that only own
 * properties of an object are stringified, and the the replacer is
 * called on the result of a <tt>toJSON</tt> call, making it possible
 * for the replacer to do its job.
 *
 * Comment from original json2.js:
 *
    http://www.JSON.org/json2.js
    2009-08-17

    Public Domain.

    NO WARRANTY EXPRESSED OR IMPLIED. USE AT YOUR OWN RISK.

    Based on json2.js from http://www.JSON.org/js.html
    but with the parse method to be provided by json_sans_eval.js

    This file creates a global JSON object containing two methods: stringify
    and parse.

        JSON.stringify(value, replacer, space)
            value       any JavaScript value, usually an object or array.

            replacer    an optional parameter that determines how object
                        values are stringified for objects. It can be a
                        function or an array of strings.

            space       an optional parameter that specifies the indentation
                        of nested structures. If it is omitted, the text will
                        be packed without extra whitespace. If it is a number,
                        it will specify the number of spaces to indent at each
                        level. If it is a string (such as '\t' or '&nbsp;'),
                        it contains the characters used to indent at each level.

            This method produces a JSON text from a JavaScript value.

            When an object value is found, if the object contains a toJSON
            method, its toJSON method will be called and the result will be
            stringified. A toJSON method does not serialize: it returns the
            value represented by the name/value pair that should be serialized,
            or undefined if nothing should be serialized. The toJSON method
            will be passed the key associated with the value, and this will be
            bound to the value

            For example, this would serialize Dates as ISO strings.

                Date.prototype.toJSON = function (key) {
                    function f(n) {
                        // Format integers to have at least two digits.
                        return n < 10 ? '0' + n : n;
                    }

                    return this.getUTCFullYear()   + '-' +
                         f(this.getUTCMonth() + 1) + '-' +
                         f(this.getUTCDate())      + 'T' +
                         f(this.getUTCHours())     + ':' +
                         f(this.getUTCMinutes())   + ':' +
                         f(this.getUTCSeconds())   + 'Z';
                };

            You can provide an optional replacer method. It will be passed the
            key and value of each member, with this bound to the containing
            object. The value that is returned from your method will be
            serialized. If your method returns undefined, then the member will
            be excluded from the serialization.

            If the replacer parameter is an array of strings, then it will be
            used to select the members to be serialized. It filters the results
            such that only members with keys listed in the replacer array are
            stringified.

            Values that do not have JSON representations, such as undefined or
            functions, will not be serialized. Such values in objects will be
            dropped; in arrays they will be replaced with null. You can use
            a replacer function to replace those with JSON values.
            JSON.stringify(undefined) returns undefined.

            The optional space parameter produces a stringification of the
            value that is filled with line breaks and indentation to make it
            easier to read.

            If the space parameter is a non-empty string, then that string will
            be used for indentation. If the space parameter is a number, then
            the indentation will be that many spaces.

            Example:

            text = JSON.stringify(['e', {pluribus: 'unum'}]);
            // text is '["e",{"pluribus":"unum"}]'


            text = JSON.stringify(['e', {pluribus: 'unum'}], null, '\t');
            // text is '[\n\t"e",\n\t{\n\t\t"pluribus": "unum"\n\t}\n]'

            text = JSON.stringify([new Date()], function (key, value) {
                return this[key] instanceof Date ?
                    'Date(' + this[key] + ')' : value;
            });
            // text is '["Date(---current time---)"]'


        JSON.parse(text, reviver)
            This method parses a JSON text to produce an object or array.
            It can throw a SyntaxError exception.

            The optional reviver parameter is a function that can filter and
            transform the results. It receives each of the keys and values,
            and its return value is used instead of the original value.
            If it returns what it received, then the structure is not modified.
            If it returns undefined then the member is deleted.

            Example:

            // Parse the text. Values that look like ISO date strings will
            // be converted to Date objects.

            myData = JSON.parse(text, function (key, value) {
                var a;
                if (typeof value === 'string') {
                    a =
/^(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2}(?:\.\d*)?)Z$/.exec(value);
                    if (a) {
                        return new Date(Date.UTC(+a[1], +a[2] - 1, +a[3], +a[4],
                            +a[5], +a[6]));
                    }
                }
                return value;
            });

            myData = JSON.parse('["Date(09/09/2001)"]', function (key, value) {
                var d;
                if (typeof value === 'string' &&
                        value.slice(0, 5) === 'Date(' &&
                        value.slice(-1) === ')') {
                    d = new Date(value.slice(5, -1));
                    if (d) {
                        return d;
                    }
                }
                return value;
            });


    This is a reference implementation. You are free to copy, modify, or
    redistribute.

    This code should be minified before deployment.
    See http://javascript.crockford.com/jsmin.html

    USE YOUR OWN COPY. IT IS EXTREMELY UNWISE TO LOAD CODE FROM SERVERS YOU DO
    NOT CONTROL.
 *
 * Comment from original json_sans_eval.js:
 *
 * Parses a string of well-formed JSON text.
 *
 * If the input is not well-formed, then behavior is undefined, but it is
 * deterministic and is guaranteed not to modify any object other than its
 * return value.
 *
 * This does not use `eval` so is less likely to have obscure security bugs than
 * json2.js.
 * It is optimized for speed, so is much faster than json_parse.js.
 *
 * This library should be used whenever security is a concern (when JSON may
 * come from an untrusted source), speed is a concern, and erroring on malformed
 * JSON is *not* a concern.
 *
 *                      Pros                   Cons
 *                    +-----------------------+-----------------------+
 * json_sans_eval.js  | Fast, secure          | Not validating        |
 *                    +-----------------------+-----------------------+
 * json_parse.js      | Validating, secure    | Slow                  |
 *                    +-----------------------+-----------------------+
 * json2.js           | Fast, some validation | Potentially insecure  |
 *                    +-----------------------+-----------------------+
 *
 * json2.js is very fast, but potentially insecure since it calls `eval` to
 * parse JSON data, so an attacker might be able to supply strange JS that
 * looks like JSON, but that executes arbitrary javascript.
 * If you do have to use json2.js with untrusted data, make sure you keep
 * your version of json2.js up to date so that you get patches as they're
 * released.
 *
 * @param {string} json per RFC 4627
 * @param {function} opt_reviver optional function that reworks JSON objects
 *     post-parse per Chapter 15.12 of EcmaScript3.1.
 *     If supplied, the function is called with a string key, and a value.
 *     The value is the property of 'this'.  The reviver should return
 *     the value to use in its place.  So if dates were serialized as
 *     {@code { "type": "Date", "time": 1234 }}, then a reviver might look like
 *     {@code
 *     function (key, value) {
 *       if (value && typeof value === 'object' && 'Date' === value.type) {
 *         return new Date(value.time);
 *       } else {
 *         return value;
 *       }
 *     }}.
 *     If the reviver returns {@code undefined} then the property named by key
 *     will be deleted from its container.
 *     {@code this} is bound to the object containing the specified property.
 * @return {Object|Array}
 * @author Mike Samuel <mikesamuel@gmail.com>
 */

if (typeof Date.prototype.toJSON !== 'function') {
  Date.prototype.toJSON = function (key) {
    return isFinite(this.valueOf()) ?
    this.getUTCFullYear()   + '-' +
    f(this.getUTCMonth() + 1) + '-' +
    f(this.getUTCDate())      + 'T' +
    f(this.getUTCHours())     + ':' +
    f(this.getUTCMinutes())   + ':' +
    f(this.getUTCSeconds())   + 'Z' : null;
  };

  String.prototype.toJSON =
  Number.prototype.toJSON =
  Boolean.prototype.toJSON = function (key) {
    return this.valueOf();
  };
}

var json_sans_eval = (function() {

   var hop = Object.hasOwnProperty;

   ///////////////////// from json2.js //////////////////////////

   function f(n) {
     // Format integers to have at least two digits.
     return n < 10 ? '0' + n : n;
   }

   var cx = /[\u0000\u00ad\u0600-\u0604\u070f\u17b4\u17b5\u200c-\u200f\u2028-\u202f\u2060-\u206f\ufeff\ufff0-\uffff]/g,
   escapable = /[\\\"\x00-\x1f\x7f-\x9f\u00ad\u0600-\u0604\u070f\u17b4\u17b5\u200c-\u200f\u2028-\u202f\u2060-\u206f\ufeff\ufff0-\uffff]/g,
   gap,
   indent,
   meta = {    // table of character substitutions
     '\b': '\\b',
     '\t': '\\t',
     '\n': '\\n',
     '\f': '\\f',
     '\r': '\\r',
     '"' : '\\"',
     '\\': '\\\\'
   },
   rep;


   function quote(string) {

     // If the string contains no control characters, no quote
     // characters, and no
     // backslash characters, then we can safely slap some quotes around it.
     // Otherwise we must also replace the offending characters with safe escape
     // sequences.

     escapable.lastIndex = 0;
     return escapable.test(string) ?
       '"' + string.replace(escapable, function (a) {
                              var c = meta[a];
                              return typeof c === 'string' ? c :
                                '\\u' + ('0000' + a.charCodeAt(0).toString(16)).slice(-4);
                            }) + '"' :
     '"' + string + '"';
   }


   function str(key, holder) {

     // Produce a string from holder[key].

     var i,          // The loop counter.
     k,          // The member key.
     v,          // The member value.
     length,
     mind = gap,
     partial,
     value = holder[key];

     // If the value has a toJSON method, call it to obtain a replacement value.

     if (value && typeof value === 'object' &&
         typeof value.toJSON === 'function') {
       value = value.toJSON(key);
     }

     // If we were called with a replacer function, then call the replacer to
     // obtain a replacement value.

     if (typeof rep === 'function') {
       value = rep.call(holder, key, value);
     }

     // What happens next depends on the value's type.

     switch (typeof value) {
     case 'string':
       return quote(value);

     case 'number':

       // JSON numbers must be finite. Encode non-finite numbers as null.

       return isFinite(value) ? String(value) : 'null';

     case 'boolean':
     case 'null':

       // If the value is a boolean or null, convert it to a string. Note:
       // typeof null does not produce 'null'. The case is included here in
       // the remote chance that this gets fixed someday.

       return String(value);

       // If the type is 'object', we might be dealing with an object
       // or an array or
       // null.

     case 'object':

       // Due to a specification blunder in ECMAScript, typeof null is 'object',
       // so watch out for that case.

       if (!value) {
         return 'null';
       }

       // Make an array to hold the partial results of stringifying
       // this object value.

       gap += indent;
       partial = [];

       // Is the value an array?

       if (Object.prototype.toString.apply(value) === '[object Array]') {

         // The value is an array. Stringify every element. Use null
         // as a placeholder
         // for non-JSON values.

         length = value.length;
         for (i = 0; i < length; i += 1) {
           partial[i] = str(i, value) || 'null';
         }

         // Join all of the elements together, separated with commas,
         // and wrap them in
         // brackets.

         v = partial.length === 0 ? '[]' :
           gap ? '[\n' + gap +
           partial.join(',\n' + gap) + '\n' +
           mind + ']' :
           '[' + partial.join(',') + ']';
         gap = mind;
         return v;
       }

       // If the replacer is an array, use it to select the members to
       // be stringified.

       if (rep && typeof rep === 'object') {
         length = rep.length;
         for (i = 0; i < length; i += 1) {
           k = rep[i];
           if (typeof k === 'string') {
             v = str(k, value);
             if (v) {
               partial.push(quote(k) + (gap ? ': ' : ':') + v);
             }
           }
         }
       } else {

         // Otherwise, iterate through all of the keys in the object.

         for (k in value) {
           if (hop.call(value, k)) {
             v = str(k, value);
             if (v) {
               partial.push(quote(k) + (gap ? ': ' : ':') + v);
             }
           }
         }
       }

       // Join all of the member texts together, separated with commas,
       // and wrap them in braces.

       v = partial.length === 0 ? '{}' :
         gap ? '{\n' + gap + partial.join(',\n' + gap) + '\n' +
         mind + '}' : '{' + partial.join(',') + '}';
       gap = mind;
       return v;
     }
   }

   function stringify (value, replacer, space) {

     // The stringify method takes a value and an optional replacer,
     // and an optional space parameter, and returns a JSON
     // text. The replacer can be a function that can replace
     // values, or an array of strings that will select the keys. A
     // default replacer method can be provided. Use of the space
     // parameter can produce text that is more easily readable.

     var i;
     gap = '';
     indent = '';

     // If the space parameter is a number, make an indent string
     // containing that
     // many spaces.

     if (typeof space === 'number') {
       for (i = 0; i < space; i += 1) {
         indent += ' ';
       }

       // If the space parameter is a string, it will be used as the
       // indent string.

     } else if (typeof space === 'string') {
       indent = space;
     }

     // If there is a replacer, it must be a function or an array.
     // Otherwise, throw an error.

     rep = replacer;
     if (replacer && typeof replacer !== 'function' &&
         (typeof replacer !== 'object' ||
          typeof replacer.length !== 'number')) {
       throw new Error('json_sans_eval.stringify');
     }

     // Make a fake root object containing our value under the key of ''.
     // Return the result of stringifying the value.

     return str('', {'': value});
   }

   var number
       = '(?:-?\\b(?:0|[1-9][0-9]*)(?:\\.[0-9]+)?(?:[eE][+-]?[0-9]+)?\\b)';
   var oneChar = '(?:[^\\0-\\x08\\x0a-\\x1f\"\\\\]'
       + '|\\\\(?:[\"/\\\\bfnrt]|u[0-9A-Fa-f]{4}))';
   var string = '(?:\"' + oneChar + '*\")';

   // Will match a value in a well-formed JSON file.
   // If the input is not well-formed, may match strangely, but not in an unsafe
   // way.
   // Since this only matches value tokens, it does not match
   // whitespace, colons,
   // or commas.
   var significantToken = new RegExp(
       '(?:false|true|null|[\\{\\}\\[\\]]'
       + '|' + number
       + '|' + string
       + ')', 'g');

   // Matches escape sequences in a string literal
   var escapeSequence = new RegExp('\\\\(?:([^u])|u(.{4}))', 'g');

   // Decodes escape sequences in object literals
   var escapes = {
     '"': '"',
     '/': '/',
     '\\': '\\',
     'b': '\b',
     'f': '\f',
     'n': '\n',
     'r': '\r',
     't': '\t'
   };
   function unescapeOne(_, ch, hex) {
     return ch ? escapes[ch] : String.fromCharCode(parseInt(hex, 16));
   }

   // A non-falsy value that coerces to the empty string when used as a key.
   var EMPTY_STRING = new String('');
   var SLASH = '\\';

   var completeToken = new RegExp(
       '(?:false|true|null|[ \t\r\n]+|[\\{\\}\\[\\],:]'
       + '|' + number
       + '|' + string
       + '|.)', 'g');

   function blank(arr, s, e) { while (--e >= s) { arr[e] = ''; } }

   function checkSyntax(text, keyFilter) {
     var toks = ('' + text).match(completeToken);
     var i = 0, n = toks.length;
     checkArray();
     if (i < n) {
       throw new Error('Trailing tokens ' + toks.slice(i - 1).join(''));
     }
     return toks.join('');

     function checkArray() {
       while (i < n) {
         var t = toks[i++];
         switch (t) {
           case ']': return;
           case '[': checkArray(); break;
           case '{': checkObject(); break;
         }
       }
     }
     function checkObject() {
       // For the tokens    {  "a"  :  null  ,  "b" ...
       // the state is         0    1  2     3  0
       var state = 0;
       // If we need to sanitize instead of validating, uncomment:
       // var skip = 0;  // The index of the first token to skip or 0.
       while (i < n) {
         var t = toks[i++];
         switch (t.charCodeAt(0)) {
           case 0x09: case 0x0a: case 0x0d: case 0x20: continue; // space chars
           case 0x22: // "
             var len = t.length;
             if (len === 1) { throw new Error(t); }
             if (state === 0) {
               if (keyFilter && !keyFilter(
                       t.substring(1, len - 1)
                       .replace(escapeSequence, unescapeOne))) {
                 throw new Error(t);
                 // If we need to sanitize instead of validating, uncomment:
                 // skip = i - 1;
               }
             } else if (state !== 2) { throw new Error(t); }
             break;
           case 0x27: throw new Error(t);  // '
           case 0x2c: // ,
             if (state !== 3) { throw new Error(t); }
             state = 0;
             // If we need to sanitize instead of validating, uncomment:
             // if (skip) { blank(toks, skip, i); skip = 0; }
             continue;
           case 0x3a: // :
             if (state !== 1) { throw new Error(t); }
             break;
           case 0x5b:  // [
             if (state !== 2) { throw new Error(t); }
             checkArray();
             break;
           case 0x7b:  // {
             if (state !== 2) { throw new Error(t); }
             checkObject();
             break;
           case 0x7d:  // }
             // If we need to sanitize instead of validating, uncomment:
             // if (skip) { blank(toks, skip, i - 1); skip = 0; }
             return;
           default:
             if (state !== 2) { throw new Error(t); }
             break;
         }
         ++state;
       }
     }
   };

   function parse (json, opt_reviver) {
     // Split into tokens
     var toks = json.match(significantToken);
     // Construct the object to return
     var result;
     var tok = toks[0];
     if ('{' === tok) {
       result = {};
     } else if ('[' === tok) {
       result = [];
     } else {
       throw new Error(tok);
     }

     // If undefined, the key in an object key/value record to use
     // for the next
     // value parsed.
     var key;
     // Loop over remaining tokens maintaining a stack of
     // uncompleted objects and
     // arrays.
     var stack = [result];
     for (var i = 1, n = toks.length; i < n; ++i) {
       tok = toks[i];

       var cont;
       switch (tok.charCodeAt(0)) {
       default:  // sign or digit
         cont = stack[0];
         cont[key || cont.length] = +(tok);
         key = void 0;
         break;
       case 0x22:  // '"'
         tok = tok.substring(1, tok.length - 1);
         if (tok.indexOf(SLASH) !== -1) {
           tok = tok.replace(escapeSequence, unescapeOne);
         }
         cont = stack[0];
         if (!key) {
           if (cont instanceof Array) {
             key = cont.length;
           } else {
             key = tok || EMPTY_STRING;  // Use as key for next value seen.
             break;
           }
         }
         cont[key] = tok;
         key = void 0;
         break;
       case 0x5b:  // '['
         cont = stack[0];
         stack.unshift(cont[key || cont.length] = []);
         key = void 0;
         break;
       case 0x5d:  // ']'
         stack.shift();
         break;
       case 0x66:  // 'f'
         cont = stack[0];
         cont[key || cont.length] = false;
         key = void 0;
         break;
       case 0x6e:  // 'n'
         cont = stack[0];
         cont[key || cont.length] = null;
         key = void 0;
         break;
       case 0x74:  // 't'
         cont = stack[0];
         cont[key || cont.length] = true;
         key = void 0;
         break;
       case 0x7b:  // '{'
         cont = stack[0];
         stack.unshift(cont[key || cont.length] = {});
         key = void 0;
         break;
       case 0x7d:  // '}'
         stack.shift();
         break;
       }
     }
     // Fail if we've got an uncompleted object.
     if (stack.length) { throw new Error(); }

     if (opt_reviver) {
       // Based on walk as implemented in http://www.json.org/json2.js
       var walk = function (holder, key) {
         var value = holder[key];
         if (value && typeof value === 'object') {
           var toDelete = null;
           for (var k in value) {
             if (hop.call(value, k) && value !== holder) {
               // Recurse to properties first.  This has the effect of causing
               // the reviver to be called on the object graph depth-first.

               // Since 'this' is bound to the holder of the property, the
               // reviver can access sibling properties of k including ones
               // that have not yet been revived.

               // The value returned by the reviver is used in place of the
               // current value of property k.
               // If it returns undefined then the property is deleted.
               var v = walk(value, k);
               if (v !== void 0) {
                 value[k] = v;
               } else {
                 // Deleting properties inside the loop has vaguely defined
                 // semantics in ES3.
                 if (!toDelete) { toDelete = []; }
                 toDelete.push(k);
               }
             }
           }
           if (toDelete) {
             for (var i = toDelete.length; --i >= 0;) {
               delete value[toDelete[i]];
             }
           }
         }
         return opt_reviver.call(holder, key, value);
       };
       result = walk({ '': result }, '');
     }

     return result;
   }

   return {
     checkSyntax: checkSyntax,
     parse: parse,
     stringify: stringify
   };
})();

if (typeof JSON === 'undefined') { var JSON = {}; }
if (typeof JSON.stringify !== 'function') {
  JSON.stringify = json_sans_eval.stringify;
}
if (typeof JSON.parse !== 'function') {
  JSON.parse = json_sans_eval.parse;
}
;
// Copyright (C) 2007 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @fileoverview the Cajita runtime library.
 * It is written in Javascript, not Cajita, and would be rejected by the Cajita
 * translator. This module exports two globals:<ol>
 * <li>"___" for use by the output of the Cajita translator and by some
 *     other untranslated Javascript code.
 * <li>"cajita" providing some common services to the Cajita programmer.
 * </ol>
 *
 * @author erights@gmail.com
 * @requires this, json_sans_eval
 * @provides ___, cajita, safeJSON 
 * @overrides Array, Boolean, Date, Function, Number, Object, RegExp, String
 * @overrides Error, EvalError, RangeError, ReferenceError, SyntaxError,
 *   TypeError, URIError
 * @overrides escape, JSON
 */

// TODO(erights): All code text in comments should be enclosed in {@code ...}.

// TODO(ihab.awad): Missing tests in CajitaTest.java for the functionality
// in this file.

// FIXME(erights): After the Caja runtime was started, some of Caja's
// ideas for controlling ES3 attribute helped inspire the new ES5
// attribute control APIs. But in the process, different terms were
// adopted for these attributes. The current correspondences:
// ES3          Caja         ES5
// ReadOnly     canSet       writable
// DontEnum     canEnum      enumerable
// DontDelete   canDelete    configurable
// We should consider migrating our terminology towards ES5's,
// especially changing "set" to "write".


// Add a tag to whitelisted builtin constructors so we can check the class
// cross-frame. Note that Function is not on the list.

Array.typeTag___ = 'Array';
Object.typeTag___ = 'Object';
String.typeTag___ = 'String';
Boolean.typeTag___ = 'Boolean';
Number.typeTag___ = 'Number';
Date.typeTag___ = 'Date';
RegExp.typeTag___ = 'RegExp';
Error.typeTag___ = 'Error';
EvalError.typeTag___ = 'EvalError';
RangeError.typeTag___ = 'RangeError';
ReferenceError.typeTag___ = 'ReferenceError';
SyntaxError.typeTag___ = 'SyntaxError';
TypeError.typeTag___ = 'TypeError';
URIError.typeTag___ = 'URIError';

Object.prototype.proto___ = null;

////////////////////////////////////////////////////////////////////////
// Cajita adds the following common Javascript extensions to ES3
// TODO(erights): Move such extensions to a separate extensions.js.
////////////////////////////////////////////////////////////////////////

/** In anticipation of ES5 */
if (Date.prototype.toISOString === void 0 && 
    typeof Date.prototype.toJSON === 'function') {
  // These are separate functions in ES5. As separate functions, they
  // also tame separately.
  // TODO(erights): Really, to follow ES5, Date.prototype.toISOString
  // should be non-generic and Date.prototype.toJSON should be generic
  // and defined by calling this.toISOString().
  Date.prototype.toISOString = function() {
    return Date.prototype.toJSON.call(this);
  };
}


/**
 * Make Function.prototype.apply tolerant of emulated ES5 arguments
 * objects.
 * 
 * <p>If the existing Function.prototype.apply is not tolerant of
 * array-like args that are neither arrays nor genuine arguments
 * objects, then replace it with a wrapper that is at least tolerant
 * of emulated ES5 arguments objects. Better would be to be tolerant
 * of array-likes in general, as ES5 is, but we'd need a cheap enough
 * test to know when we'd need to copy.
 */
try {
  (function(){}).apply({},{length:0});
} catch (ex) {
  if (ex instanceof TypeError) {
    (function() {
       Function.prototype.apply___ = Function.prototype.apply;

       Function.prototype.apply = function applyGuard(self, args) {
         if (args && args.CLASS___ === 'Arguments') {
           args = Array.slice(args, 0);
         }
         return this.apply___(self, args);
       };
     })();
  }
}


/**
 * Provide Array.slice similar to Firefox.  You can slice anything, and the
 * result is always an array.  Hazards:
 *
 *  - In IE, Array.prototype.slice.call(void 0) throws an error, and we
 *    need it to return [].
 *
 *  - In IE[678] and Firefox 3, x.slice(0, undefined) returns [] rather
 *    than x.  We don't care about that incompatibility, but we do need
 *    Array.slice(x) to return x.
 *
 *  - In Firefox 3.x, Array.slice works on any array-like x.  We only
 *    handle typeof x === 'object'.
 */
if (Array.slice === void 0) {
  Array.slice = function(self, opt_start, opt_end) {
    if (self && typeof self === 'object') {
      if (opt_end === void 0) { opt_end = self.length; }
      return Array.prototype.slice.call(self, opt_start, opt_end);
    } else {
      return [];
    }
  };
}


/**
 * In anticipation of ES5.
 * <p>
 * Bind this function to <tt>self</tt>, which will serve
 * as the value of <tt>this</tt> during invocation. Curry on a
 * partial set of arguments in <tt>var_args</tt>. Return the curried
 * result as a new function object.
 * <p>
 * Note: Like the built-in Function.prototype.call and
 * Function.prototype.apply, this one is not whitelisted. Instead,
 * it is provided as it would be in future JavaScripts without
 * special knowledge of Caja. This allows Caja code using bind() to
 * work today uncajoled as well. It also suppresses the overriding
 * of 'bind' by the 'in' test in setStatic().
 * <p>
 * Note that this is distinct from the tamed form of bind() made
 * available to Cajita code.
 */
if (Function.prototype.bind === void 0) {
  Function.prototype.bind = function(self, var_args) {
    var thisFunc = this;
    var leftArgs = Array.slice(arguments, 1);
    function funcBound(var_args) {
      var args = leftArgs.concat(Array.slice(arguments, 0));
      return thisFunc.apply(self, args);
    }
    return funcBound;
  };
}

// The following may or may not exist in the browser-supplied
// global scope; declare as a 'var' to avoid errors when we
// check for its existence later.
var escape;

// cajita.js exports the following names to the Javascript global
// namespace. Cajita code can only use the "cajita" object. The "___"
// object is for use by code generated by the Cajita translator, and by
// Javascript code (such as a powerbox) in the embedding application.

var cajita;
var ___;
var safeJSON;

// Explicitly passing in the actual global object to avoid
// ReferenceErrors when referring to potentially nonexistent objects
// like HTMLDivElement.

(function(global) {

  function ToInt32(alleged_int) {
    return alleged_int >> 0;
  }

  function ToUInt32(alleged_int) {
    return alleged_int >>> 0;
  }

  /**
   * Returns the first index at which the specimen is found (by
   * "identical()") or -1 if none, starting at offset i, if present.
   * If i < 0, the offset is relative to the end of the array.
   */
  function arrayIndexOf(specimen, i) {
    var len = ToUInt32(this.length);
    i = ToInt32(i);
    if (i < 0) {
      if ((i += len) < 0) {
        i = 0;
      }
    }
    for (; i < len; ++i) {
      if (i in this && identical(this[i], specimen)) {
        return i;
      }
    }
    return -1;
  }
  Array.prototype.indexOf = arrayIndexOf;

  /**
   * Returns the last index at which the specimen is found (by
   * "identical()") or -1 if none, starting at offset i, if present.
   * If i < 0, the offset is relative to the end of the array.
   */
  function arrayLastIndexOf(specimen, i) {
    var len = ToUInt32(this.length);

    if (isNaN(i)) {
      i = len - 1;
    } else {
      i = ToInt32(i);
      if (i < 0) {
        i += len;
        if (i < 0) {
          return -1;
        }
      } else if (i >= len) {
        i = len - 1;
      }
    }

    for (; i >= 0 ; --i) {
      if (i in this && identical(this[i], specimen)) {
        return i;
      }
    }
    return -1;
  }
  Array.prototype.lastIndexOf = arrayLastIndexOf;

  ////////////////////////////////////////////////////////////////////////
  // Some regular expressions checking for specific suffixes.
  ////////////////////////////////////////////////////////////////////////

  var endsWith_canDelete___ = /_canDelete___$/;
  var endsWith_canRead___ = /_canRead___$/;
  var endsWith_canSet___ = /_canSet___$/;
  var endsWith___ = /___$/;
  var endsWith__ = /__$/;

  ////////////////////////////////////////////////////////////////////////
  // Some very basic primordial methods
  ////////////////////////////////////////////////////////////////////////

  /**
   * A reliable typeof for use by Cajita code, and by uncajoled code
   * (like parts of cajita.js) that require a reliable typeof.
   * <p>
   * Cajita specifies that <tt>typeof new RegExp("x")</tt>
   * evaluate to <tt>'object'</tt>. Unfortunately, on some of Cajita's
   * current target platforms (including at least Safari 3 and Rhino),
   * it returns <tt>'function'</tt> instead. Since the distinction
   * between functions and non-functions is crucial to Cajita, we
   * translate the Cajita <tt>typeof</tt> operator into calls to this
   * <tt>typeOf</tt> function.
   */
  function typeOf(obj) {
    var result = typeof obj;
    if (result !== 'function') { return result; }
    var ctor = obj.constructor;
    if (typeof ctor === 'function' &&
        ctor.typeTag___ === 'RegExp' &&
        obj instanceof ctor) {
      return 'object';
    }
    return 'function';
  }

  if (typeof new RegExp('x') === 'object') {
    // Since typeof does what we want on FF3, IE6, Chrome, and Opera,
    // on these platforms we shouldn't pay for the more expensive
    // typeOf. Note that we declare the safe one but conditionally
    // reset it to the fast one. This is safe even if hoisting causes
    // the declared function to be called before we can replace it.
    typeOf = function fastTypeof(obj) {
      return typeof obj;
    };
  }

  var myOriginalHOP = Object.prototype.hasOwnProperty;
  var myOriginalToString = Object.prototype.toString;

  /**
   * <tt>hasOwnProp(obj, name)</tt> means what
   * <tt>obj.hasOwnProperty(name)</tt> would normally mean in an
   * unmodified Javascript system.
   */
  function hasOwnProp(obj, name) {
    if (!obj) { return false; }
    var t = typeof obj;
    if (t !== 'object' && t !== 'function') {
      // If obj is a primitive, Object(obj) still has no own properties.
      return false;
    }
    return myOriginalHOP.call(obj, name);
  }

  /**
   * Are x and y not observably distinguishable?
   */
  function identical(x, y) {
    if (x === y) {
      // 0 === -0, but they are not identical
      return x !== 0 || 1/x === 1/y;
    } else {
      // NaN !== NaN, but they are identical.
      // NaNs are the only non-reflexive value, i.e., if x !== x,
      // then x is a NaN.
      return x !== x && y !== y;
    }
  }

  /**
   * Inherited by non-frozen simple functions, which freezes them and
   * installs an overriding fastpath CALL___ to themselves.
   */
  function callFault(var_args) {
    return asFunc(this).apply(USELESS, arguments);
  }
  Object.prototype.CALL___ = callFault;

  ////////////////////////////////////////////////////////////////////////
  // Diagnostics and condition enforcement
  ////////////////////////////////////////////////////////////////////////

  /**
   * The initial default logging function does nothing.
   * <p>
   * Note: JavaScript has no macros, so even in the "does nothing"
   * case, remember that the arguments are still evaluated.
   */
  function defaultLogger(str, opt_stop) {}
  var myLogFunc = markFuncFreeze(defaultLogger);

  /**
   * Gets the currently registered logging function.
   */
  function getLogFunc() { return myLogFunc; }

  /**
   * Register newLogFunc as the current logging function, to be called
   * by <tt>___.log(str)</tt> and <tt>___.fail(...)</tt>.
   * <p>
   * A logging function is assumed to have the signature
   * <tt>(str, opt_stop)</tt>, where<ul>
   * <li><tt>str</tt> is the diagnostic string to be logged, and
   * <li><tt>opt_stop</tt>, if present and <tt>true</tt>, indicates
   *     that normal flow control is about to be terminated by a
   *     throw. This provides the logging function the opportunity to
   *     terminate normal control flow in its own way, such as by
   *     invoking an undefined method, in order to trigger a Firebug
   *     stacktrace.
   * </ul>
   */
  function setLogFunc(newLogFunc) { myLogFunc = newLogFunc; }

  /**
   * Calls the currently registered logging function.
   */
  function log(str) { myLogFunc(String(str)); }


  /**
   * Throw, and optionally log, an error whose message is the
   * concatenation of the arguments.
   * <p>
   * The arguments are converted to strings (presumably by an
   * implicit call to ".toString()") and appended together to make
   * the message of the Error that's thrown.
   */
  function fail(var_args) {
    var message = Array.slice(arguments, 0).join('');
    myLogFunc(message, true);
    throw new Error(message);
  }

  /**
   * Like an assert that can't be turned off.
   * <p>
   * Either returns true (on success) or throws (on failure). The
   * arguments starting with <tt>var_args</tt> are converted to
   * strings and appended together to make the message of the Error
   * that's thrown.
   * <p>
   * TODO(erights) We may deprecate this in favor of <pre>
   *     test || fail(var_args...)
   * </pre> or <pre>
   *     if (!test) { fail(var_args...); }
   * </pre>
   */
  function enforce(test, var_args) {
    return test || fail.apply(USELESS, Array.slice(arguments, 1));
  }

  /**
   * Enforces <tt>typeOf(specimen) === typename</tt>, in which case
   * specimen is returned.
   * <p>
   * If not, throws an informative TypeError
   * <p>
   * opt_name, if provided, should be a name or description of the
   * specimen used only to generate friendlier error messages.
   */
  function enforceType(specimen, typename, opt_name) {
    if (typeOf(specimen) !== typename) {
      fail('expected ', typename, ' instead of ', typeOf(specimen),
           ': ', (opt_name || specimen));
    }
    return specimen;
  }

  /**
   * Enforces that specimen is a non-negative integer within the range
   * of exactly representable consecutive integers, in which case
   * specimen is returned.
   * <p>
   * "Nat" is short for "Natural number".
   */
  function enforceNat(specimen) {
    enforceType(specimen, 'number');
    if (Math.floor(specimen) !== specimen) {
      fail('Must be integral: ', specimen);
    }
    if (specimen < 0) {
      fail('Must not be negative: ', specimen);
    }
    // Could pre-compute precision limit, but probably not faster
    // enough to be worth it.
    if (Math.floor(specimen - 1) !== specimen - 1) {
      fail('Beyond precision limit: ', specimen);
    }
    if (Math.floor(specimen - 1) >= specimen) {
      fail('Must not be infinite: ', specimen);
    }
    return specimen;
  }

  /**
   * Returns a function that can typically be used in lieu of
   * <tt>func</tt>, but that logs a deprecation warning on first use.
   * <p>
   * Currently for internal use only, though may make this available
   * on <tt>___</tt> or even <tt>cajita</tt> at a later time, after
   * making it safe for such use. Forwards only arguments to
   * <tt>func</tt> and returns results back, without forwarding
   * <tt>this</tt>. If you want to deprecate an exophoric function,
   * deprecate a bind()ing of that function instead.
   */
  function deprecate(func, badName, advice) {
    var warningNeeded = true;
    return function() {
      if (warningNeeded) {
        log('"' + badName + '" is deprecated.\n' + advice);
        warningNeeded = false;
      }
      return func.apply(USELESS, arguments);
    };
  }

  ////////////////////////////////////////////////////////////////////////
  // Privileged fault handlers
  ////////////////////////////////////////////////////////////////////////

  function debugReference(obj) {
    switch (typeOf(obj)) {
      case 'object': {
        if (obj === null) { return '<null>'; }
        var constr = directConstructor(obj);
        return '[' + ((constr && constr.name) || 'Object') + ']';
      }
      default: {
        return '(' + obj + ':' + typeOf(obj) + ')';
      }
    }
  }

  var myKeeper = {

    toString: function toString() { return '<Logging Keeper>'; },

    handleRead: function handleRead(obj, name) {
      //log('Not readable: (' + debugReference(obj) + ').' + name);
      return void 0;
    },

    handleCall: function handleCall(obj, name, args) {
      fail('Not callable: (', debugReference(obj), ').', name);
    },

    handleSet: function handleSet(obj, name, val) {
      fail('Not writable: (', debugReference(obj), ').', name);
    },

    handleDelete: function handleDelete(obj, name) {
      fail('Not deletable: (', debugReference(obj), ').', name);
    }
  };

  Object.prototype.handleRead___ = function handleRead___(name) {
    var handlerName = name + '_getter___';
    if (this[handlerName]) {
      return this[handlerName]();
    }
    return myKeeper.handleRead(this, name);
  };

  Object.prototype.handleCall___ = function handleCall___(name, args) {
    var handlerName = name + '_handler___';
    if (this[handlerName]) {
      return this[handlerName].call(this, args);
    }
    return myKeeper.handleCall(this, name, args);
  };

  Object.prototype.handleSet___ = function handleSet___(name, val) {
    var handlerName = name + '_setter___';
    if (this[handlerName]) {
      return this[handlerName](val);
    }
    return myKeeper.handleSet(this, name, val);
  };

  Object.prototype.handleDelete___ = function handleDelete___(name) {
    var handlerName = name + '_deleter___';
    if (this[handlerName]) {
      return this[handlerName]();
    }
    return myKeeper.handleDelete(this, name);
  };

  ////////////////////////////////////////////////////////////////////////
  // walking prototype chain, checking JSON containers
  ////////////////////////////////////////////////////////////////////////

  /**
   * Returns the 'constructor' property of obj's prototype.
   * <p>
   * SECURITY TODO(erights): Analyze the security implications
   * of exposing this as a property of the cajita object.
   * <p>
   * By "obj's prototype", we mean the prototypical object that obj
   * most directly inherits from, not the value of its 'prototype'
   * property. We memoize the apparent prototype into 'proto___' to
   * speed up future queries.
   * <p>
   * If obj is a function or not an object, return undefined.
   */
  function directConstructor(obj) {
    if (obj === null) { return void 0; }
    if (obj === void 0) { return void 0; }
    if (typeOf(obj) === 'function') {
      // Since functions return undefined,
      // directConstructor() doesn't provide access to the
      // forbidden Function constructor.
      return void 0;
    }
    obj = Object(obj);
    var result;
    if (myOriginalHOP.call(obj, 'proto___')) {
      var proto = obj.proto___;
      // At this point we know that (typeOf(proto) === 'object')
      if (proto === null) { return void 0; }
      result = proto.constructor;
      // rest of: if (!isPrototypical(result))
      if (result.prototype !== proto || typeOf(result) !== 'function') {
        result = directConstructor(proto);
      }

    } else {
      if (!myOriginalHOP.call(obj, 'constructor')) {
        // TODO(erights): Detect whether this is a valid constructor
        // property in the sense that result is a proper answer. If
        // not, at least give a sensible error, which will be hard to
        // phrase.
        result = obj.constructor;
      } else {
        var oldConstr = obj.constructor;
        if (delete obj.constructor) {
          result = obj.constructor;
          obj.constructor = oldConstr;
        } else if (isPrototypical(obj)) {
          // A difficult case. In Safari, and perhaps according to
          // ES3, the prototypical object created for the default
          // value of a function's 'prototype' property has a
          // non-deletable 'constructor' property. If this is what we
          // have, then we assume it inherits directly from
          // Object.prototype, so the result should be Object.
          log('Guessing the directConstructor of : ' + obj);
          result = Object;
        } else {
          return fail('Discovery of direct constructors unsupported when the ',
                      'constructor property is not deletable: ',
                      obj, '.constructor === ', oldConstr, 
                      '(', obj === global, ')');
        }
      }

      if (typeOf(result) !== 'function' || !(obj instanceof result)) {
        fail('Discovery of direct constructors for foreign begotten ',
             'objects not implemented on this platform.\n');
      }
      if (result.prototype.constructor === result) {
        // Memoize, so it'll be faster next time.
        obj.proto___ = result.prototype;
      }
    }
    return result;
  }

  /**
   * The function category of the whitelisted global constructors
   * defined in ES is the string name of the constructor, allowing
   * isInstanceOf() to work cross-frame. Otherwise, the function
   * category of a function is just the function itself.
   */
  function getFuncCategory(fun) {
    enforceType(fun, 'function');
    if (fun.typeTag___) {
      return fun.typeTag___;
    } else {
      return fun;
    }
  }

  /**
   * Is <tt>obj</tt> a direct instance of a function whose category is
   * the same as the category of <tt>ctor</tt>?
   */
  function isDirectInstanceOf(obj, ctor) {
    var constr = directConstructor(obj);
    if (constr === void 0) { return false; }
    return getFuncCategory(constr) === getFuncCategory(ctor);
  }

  /**
   * Is <tt>obj</tt> an instance of a function whose category is
   * the same as the category of <tt>ctor</tt>?
   */
  function isInstanceOf(obj, ctor) {
    if (obj instanceof ctor) { return true; }
    if (isDirectInstanceOf(obj, ctor)) { return true; }
    // BUG TODO(erights): walk prototype chain.
    // In the meantime, this will fail should it encounter a
    // cross-frame instance of a "subclass" of ctor.
    return false;
  }

  /**
   * A Record is an object whose direct constructor is Object. 
   * <p>
   * These are the kinds of objects that can be expressed as
   * an object literal ("<tt>{...}</tt>") in the JSON language.
   */
  function isRecord(obj) {
    if (!obj) { return false; }
    if (obj.RECORD___ === obj) { return true; }
    if (isDirectInstanceOf(obj, Object)) {
      obj.RECORD___ = obj;
      return true;
    }
    return false;
  }

  /**
   * An Array is an object whose direct constructor is Array.
   * <p>
   * These are the kinds of objects that can be expressed as
   * an array literal ("<tt>[...]</tt>") in the JSON language.
   */
  function isArray(obj) {
    return isDirectInstanceOf(obj, Array);
  }

  /**
   * A JSON container is a non-prototypical object whose direct
   * constructor is Object or Array.
   * <p>
   * These are the kinds of non-primitive objects that can be
   * expressed in the JSON language.
   */
  function isJSONContainer(obj) {
    if (!obj) { return false; }
    if (obj.RECORD___ === obj) { return true; }
    var constr = directConstructor(obj);
    if (constr === void 0) { return false; }
    var typeTag = constr.typeTag___;
    if (typeTag !== 'Object' && typeTag !== 'Array') { return false; }
    return !isPrototypical(obj);
  }

  /**
   * If obj is frozen, Cajita code cannot directly assign to
   * own properties of obj, nor directly add or delete own properties to
   * obj.
   * <p>
   * The status of being frozen is not inherited. If A inherits from
   * B (i.e., if A's prototype is B), A and B each may or may not be
   * frozen independently. (Though if B is prototypical, then it must
   * be frozen.)
   * <p>
   * If <tt>typeof obj</tt> is neither 'object' nor 'function', then
   * it's currently considered frozen.
   */
  function isFrozen(obj) {
    if (!obj) { return true; }
    // TODO(erights): Object(<primitive>) wrappers should also be
    // considered frozen.
    if (obj.FROZEN___ === obj) { return true; }
    var t = typeof obj;
    return t !== 'object' && t !== 'function';
  }

  /**
   * Mark obj as frozen so that Cajita code cannot directly assign to its
   * own properties.
   * <p>
   * If obj is a function, also freeze obj.prototype.
   * <p>
   * This appears as <tt>___.primFreeze(obj)</tt> and is wrapped by
   * <tt>cajita.freeze(obj)</tt>, which applies only to JSON containers.
   * It does a shallow freeze, i.e., if record y inherits from record x,
   * ___.primFreeze(y) will not freeze x.
   */
  function primFreeze(obj) {
    // Fail silently on undefined, since
    //   (function(){
    //     var f = Foo;
    //     if (true) { function Foo() {} }
    //   })();
    // gets translated to (roughly)
    //   (function(){
    //     var Foo;
    //     var f = ___.primFreeze(Foo);
    //     if (true) { Foo = function Foo() {}; }
    //   })();
    if (isFrozen(obj)) { return obj; }

    if (obj.SLOWFREEZE___) {
      // Still true even if SLOWFREEZE___ is inherited.

      // badFlags are names of properties we need to turn off.
      // We accumulate these first, so that we're not in the midst of a
      // for/in loop on obj while we're deleting properties from obj.
      var badFlags = [];
      for (var k in obj) {
        if (endsWith_canSet___.test(k) || endsWith_canDelete___.test(k)) {
          if (obj[k]) {
            badFlags.push(k);
          }
        }
      }
      for (var i = 0; i < badFlags.length; i++) {
        var flag = badFlags[i];
        if (myOriginalHOP.call(obj, flag)) {
          if (!(delete obj[flag])) {
            fail('internal: failed delete: ', debugReference(obj), '.', flag);
          }
        }
        if (obj[flag]) {
          obj[flag] = false;
        }
      }
      // Will only delete it as an own property, so may legitimately
      // fail if obj only inherits SLOWFREEZE___.
      delete obj.SLOWFREEZE___;
    }
    obj.FROZEN___ = obj;
    if (typeOf(obj) === 'function') {
      if (isFunc(obj)) {
        grantCall(obj, 'call');
        grantCall(obj, 'apply');
        obj.CALL___ = obj;
      }
      // Do last to avoid possible infinite recursion.
      if (obj.prototype) { primFreeze(obj.prototype); }
    }
    return obj;
  }

  /**
   * Like primFreeze(obj), but applicable only to JSON containers,
   * (pointlessly but harmlessly) to functions, and to Errors.
   * <p>
   * Errors are constructed objects whose only whitelisted
   * properties are <tt>name</tt> and <tt>message</tt>, both of which
   * are strings on at least all A-grade browsers and on all browsers
   * that conform to either the ES3 or ES5 specs. Therefore, we
   * could consider Errors to simply be frozen. We don't only to avoid
   * slowing down <tt>isFrozen()</tt>, which needs to be really
   * fast. Instead, we allow cajoled code to freeze it. We also
   * freeze it in <tt>tameException()</tt>, so all Errors are frozen
   * by the time cajoled code can catch them.
   */
  function freeze(obj) {
    if (isJSONContainer(obj)) {
      return primFreeze(obj);
    }
    if (typeOf(obj) === 'function') {
      enforce(isFrozen(obj), 'Internal: non-frozen function: ' + obj);
      return obj;
    }
    if (isInstanceOf(obj, Error)) {
      return primFreeze(obj);
    }
    fail('cajita.freeze(obj) applies only to JSON Containers, ',
         'functions, and Errors: ',
         debugReference(obj));
  }
  

  /**
   * Makes a mutable copy of a JSON container.
   * <p>
   * Even if the original is frozen, the copy will still be mutable.
   * It does a shallow copy, i.e., if record y inherits from record x,
   * ___.copy(y) will also inherit from x.
   */
  function copy(obj) {
    if (!isJSONContainer(obj)) {
      fail('cajita.copy(obj) applies only to JSON Containers: ',
           debugReference(obj));
    }
    var result = isArray(obj) ? [] : {};
    forOwnKeys(obj, markFuncFreeze(function(k, v) {
      result[k] = v;
    }));
    return result;
  }

  /**
   * A snapshot of a JSON container is a frozen shallow copy of that
   * container.
   */
  function snapshot(obj) {
    return primFreeze(copy(obj));
  }


  ////////////////////////////////////////////////////////////////////////
  // Accessing property attributes
  ////////////////////////////////////////////////////////////////////////

  /**
   * Tests whether the fast-path canRead flag is set.
   */
  function canRead(obj, name)   {
    if (obj === void 0 || obj === null) { return false; }
    return !!obj[name + '_canRead___'];
  }

  /**
   * Tests whether the fast-path canEnum flag is set.
   */
  function canEnum(obj, name)   {
    if (obj === void 0 || obj === null) { return false; }
    return !!obj[name + '_canEnum___'];
  }

  /**
   * Tests whether the fast-path canCall flag is set, or grantCall() has been
   * called.
   */
  function canCall(obj, name)   {
    if (obj === void 0 || obj === null) { return false; }
    if (obj[name + '_canCall___']) { return true; }
    if (obj[name + '_grantCall___']) {
      fastpathCall(obj, name);
      return true;
    }
    return false;
  }
  /**
   * Tests whether the fast-path canSet flag is set, or grantSet() has been
   * called, on this object itself as an own (non-inherited) attribute.
   */
  function canSet(obj, name) {
    if (obj === void 0 || obj === null) { return false; }
    if (obj[name + '_canSet___'] === obj) { return true; }
    if (obj[name + '_grantSet___'] === obj) {
      fastpathSet(obj, name);
      return true;
    }
    return false;
  }

  /**
   * Tests whether the fast-path canDelete flag is set, on this
   * object itself as an own (non-inherited) attribute.
   */
  function canDelete(obj, name) {
    if (obj === void 0 || obj === null) { return false; }
    return obj[name + '_canDelete___'] === obj;
  }

  /**
   * Sets the fast-path canRead flag.
   * <p>
   * These are called internally to memoize decisions arrived at by
   * other means.
   */
  function fastpathRead(obj, name) {
    if (name === 'toString') { fail("internal: Can't fastpath .toString"); }
    obj[name + '_canRead___'] = obj;
  }

  function fastpathEnum(obj, name) {
    obj[name + '_canEnum___'] = obj;
  }

  /**
   * Simple functions should callable and readable, but methods
   * should only be callable.
   */
  function fastpathCall(obj, name) {
    if (name === 'toString') { fail("internal: Can't fastpath .toString"); }
    if (obj[name + '_canSet___']) {
      obj[name + '_canSet___'] = false;
    }
    if (obj[name + '_grantSet___']) {
      obj[name + '_grantSet___'] = false;
    }
    obj[name + '_canCall___'] = obj;
  }

  /**
   * fastpathSet implies fastpathEnum and fastpathRead.
   * <p>
   * fastpathSet also disables the ability to call and records that
   * more work must be done to freeze this object. 
   */
  function fastpathSet(obj, name) {
    if (name === 'toString') { fail("internal: Can't fastpath .toString"); }
    if (isFrozen(obj)) {
      fail("Can't set .", name, ' on frozen (', debugReference(obj), ')');
    }
    if (typeOf(obj) === 'function') {
      fail("Can't make .", name, 
           ' writable on a function (', debugReference(obj), ')');
    }
    fastpathEnum(obj, name);
    fastpathRead(obj, name);
    if (obj[name + '_canCall___']) {
      obj[name + '_canCall___'] = false;
    }
    if (obj[name + '_grantCall___']) {
      obj[name + '_grantCall___'] = false;
    }
    obj.SLOWFREEZE___ = obj;
    obj[name + '_canSet___'] = obj;
  }

  /**
   * fastpathDelete allows delete of a member on a constructed object via
   * the public API.
   * <p>
   * fastpathDelete also records that more work must be done to freeze
   * this object.  
   * <p>
   * TODO(erights): Having a fastpath flag for this probably doesn't
   * make sense.
   */
  function fastpathDelete(obj, name) {
    if (name === 'toString') { fail("internal: Can't fastpath .toString"); }
    if (isFrozen(obj)) {
      fail("Can't delete .", name, ' on frozen (', debugReference(obj), ')');
    }
    if (typeOf(obj) === 'function') {
      fail("Can't make .", name, 
           ' deletable on a function (', debugReference(obj), ')');
    }
    obj.SLOWFREEZE___ = obj;
    obj[name + '_canDelete___'] = obj;
  }

  /**
   * The various <tt>grant*</tt> functions are called externally by
   * Javascript code to express whitelisting taming decisions.
   */
  function grantRead(obj, name) {
    fastpathRead(obj, name);
  }

  function grantEnum(obj, name) {
    fastpathEnum(obj, name);
  }

  function grantCall(obj, name) {
    fastpathCall(obj, name);
    obj[name + '_grantCall___'] = obj;
  }

  function grantSet(obj, name) {
    fastpathSet(obj, name);
    obj[name + '_grantSet___'] = obj;
  }

  function grantDelete(obj, name) {
    fastpathDelete(obj, name);
  }

  ////////////////////////////////////////////////////////////////////////
  // Primitive objective membrane
  ////////////////////////////////////////////////////////////////////////

  /**
   * Records that f is t's feral twin and t is f's tame twin.
   * <p>
   * A <i>feral</i> object is one safe to make accessible to trusted
   * but possibly innocent uncajoled code. A <i>tame</i> object is one
   * safe to make accessible to untrusted cajoled
   * code. ___tamesTo(f, t) records that f is feral, that t is tamed,
   * and that they are in one-to-one correspondence so that 
   * ___.tame(f) === t and ___.untame(t) === f.
   * <p>
   * All primitives already tame and untame to themselves, so tamesTo
   * only accepts non-primitive arguments. The characteristic of being
   * marked tame or feral only applies to the object itself, not to
   * objects which inherit from it. TODO(erights): We should probably
   * check that a derived object does not get markings that conflict
   * with the markings on its base object.
   * <p>
   * Initialization code can express some taming decisions by calling
   * tamesTo to preregister some feral/tame pairs. 
   * <p>
   * Unlike the subjective membranes created by Domita, in this one,
   * the objects in a tame/feral pair point directly at each other,
   * and thereby retain each other. So long as one is non-garbage the
   * other will be as well. 
   */
  function tamesTo(f, t) {
    var ftype = typeof f;
    if (!f || (ftype !== 'function' && ftype !== 'object')) { 
      fail('Unexpected feral primitive: ', f); 
    }
    var ttype = typeof t;
    if (!t || (ttype !== 'function' && ttype !== 'object')) {
      fail('Unexpected tame primitive: ', t); 
    }

    if (f.TAMED_TWIN___ === t && t.FERAL_TWIN___ === f) { 
      // Just a transient diagnostic until we understand how often
      // this happens.
      log('multiply tamed: ' + f + ', ' + t);
      return; 
    }

    // TODO(erights): Given that we maintain the invariant that 
    // (f.TAMED_TWIN___ === t && hasOwnProp(f, 'TAMED_TWIN___')) iff
    // (t.FERAL_TWIN___ === f && hasOwnProp(t, 'FERAL_TWIN___')), then we
    // could decide to more delicately rely on this invariant and test
    // the backpointing rather than hasOwnProp below.

    if (f.TAMED_TWIN___ && hasOwnProp(f, 'TAMED_TWIN___')) { 
      fail('Already tames to something: ', f); 
    }
    if (t.FERAL_TWIN___ && hasOwnProp(t, 'FERAL_TWIN___')) { 
      fail('Already untames to something: ', t); 
    }
    if (f.FERAL_TWIN___ && hasOwnProp(f, 'FERAL_TWIN___')) { 
      fail('Already tame: ', f); 
    }
    if (t.TAMED_TWIN___ && hasOwnProp(t, 'TAMED_TWIN___')) { 
      fail('Already feral: ', t); 
    }

    f.TAMED_TWIN___ = t;
    t.FERAL_TWIN___ = f;
  }

  /**
   * ___.tamesToSelf(obj) marks obj as both tame and feral.
   * <p>
   * Most tamed objects should be both feral and tame, i.e.,
   * safe to be accessed from both the feral and tame worlds.
   * <p>
   * This is equivalent to tamesTo(obj, obj) but a bit faster by
   * exploiting the knowledge that f and t are the same object.
   */
  function tamesToSelf(obj) {
    var otype = typeof obj;
    if (!obj || (otype !== 'function' && otype !== 'object')) { 
      fail('Unexpected primitive: ', obj); 
    }
    if (obj.TAMED_TWIN___ === obj && obj.FERAL_TWIN___ === obj) { 
      // Just a transient diagnostic until we understand how often
      // this happens.
      log('multiply tamed: ' + obj);
      return; 
    }

    // TODO(erights): Given that we maintain the invariant that 
    // (f.TAMED_TWIN___ === t && hasOwnProp(f, 'TAMED_TWIN___')) iff
    // (t.FERAL_TWIN___ === f && hasOwnProp(t, 'FERAL_TWIN___')), then we
    // could decide to more delicately rely on this invariant and test
    // the backpointing rather than hasOwnProp below.
    if (obj.TAMED_TWIN___ && hasOwnProp(obj, 'TAMED_TWIN___')) { 
      fail('Already tames to something: ', obj); 
    }
    if (obj.FERAL_TWIN___ && hasOwnProp(obj, 'FERAL_TWIN___')) { 
      fail('Already untames to something: ', obj); 
    }

    obj.TAMED_TWIN___ = obj.FERAL_TWIN___ = obj;
  }

  /**
   * 
   * Returns a tame object representing f, or undefined on failure.
   * <ol>
   * <li>All primitives tame and untame to themselves. Therefore,
   *     undefined is only a valid failure indication after checking
   *     that the argument is not undefined. 
   * <li>If f has a registered tame twin, return that.
   * <li>If f is marked tame, then f already is a tame object 
   *     representing f, so return f.
   * <li>If f has an AS_TAMED___() method, call it and then register 
   *     the result as f's tame twin. Unlike the tame/feral
   *     registrations, this method applies even if it is inherited.
   * <li>If f is a Record, call tameRecord(f). We break Records out as
   *     a special case since the only thing all Records inherit from
   *     is Object.prototype, which everything else inherits from as
   *     well. 
   * <li>Indicate failure by returning undefined.
   * </ol>
   * Record taming does not (yet?) deal with record inheritance. 
   * <p>
   * The AS_TAMED___() methods may assume that they are called only by
   * tame() and only on unmarked non-primitive objects. They must
   * therefore either return another unmarked non-primitive object
   * (possibly the same one) or undefined for failure. On returning
   * successfully, tame() will register the pair so AS_TAMED___() does
   * not need to. 
   */
  function tame(f) {
    var ftype = typeof f;
    if (!f || (ftype !== 'function' && ftype !== 'object')) { 
      return f; 
    }
    var t = f.TAMED_TWIN___;
    // Here we do use the backpointing test as a cheap hasOwnProp test.
    if (t && t.FERAL_TWIN___ === f) { return t; }

    var realFeral = f.FERAL_TWIN___;
    if (realFeral && realFeral.TAMED_TWIN___ === f) {
      // If f has a feral twin, then f itself is tame.
      log('Tame-only object from feral side: ' + f);
      return f;
    }
    if (f.AS_TAMED___) {
      t = f.AS_TAMED___();
      if (t) { tamesTo(f, t); }
      return t;
    }
    if (isRecord(f)) {
      t = tameRecord(f);
      // tameRecord does not actually have any possibility of failure,
      // but we can't assume that here.
      if (t) { tamesTo(f, t); }
      return t;
    }
    return undefined;
  }

  /**
   * Returns a feral object representing t, or undefined on failure.
   * <ol>
   * <li>All primitives tame and untame to themselves. Therefore,
   *     undefined is only a valid failure indication after checking
   *     that the argument is not undefined. 
   * <li>If t has a registered feral twin, return that.
   * <li>If t is marked feral, then t already is a feral object 
   *     representing t, so return t.
   * <li>If t has an AS_FERAL___() method, call it and then register 
   *     the result as t's feral twin. Unlike the tame/feral
   *     registrations, this method applies even if it is inherited.
   * <li>If t is a Record, call untameRecord(t).
   * <li>Indicate failure by returning undefined.
   * </ol>
   * Record untaming does not (yet?) deal with record inheritance. 
   * <p>
   * The AS_FERAL___() methods may assume that they are called only by
   * untame() and only on unmarked non-primitive objects. They must
   * therefore either return another unmarked non-primitive object
   * (possibly the same one) or undefined for failure. On returning
   * successfully, untame() will register the pair so AS_FERAL___() does
   * not need to. 
   */
  function untame(t) {
    var ttype = typeof t;
    if (!t || (ttype !== 'function' && ttype !== 'object')) { 
      return t; 
    }
    var f = t.FERAL_TWIN___;
    // Here we do use the backpointing test as a cheap hasOwnProp test.
    if (f && f.TAMED_TWIN___ === t) { return f; }

    var realTame = t.TAMED_TWIN___;
    if (realTame && realTame.FERAL_TWIN___ === t) {
      // If t has a tamed twin, then t itself is feral.
      log('Feral-only object from tame side: ' + t);
      return t;
    }
    if (t.AS_FERAL___) {
      f = t.AS_FERAL___();
      if (f) { tamesTo(f, t); }
      return f;
    }
    if (isRecord(t)) {
      f = untameRecord(t);
      // untameRecord does not actually have any possibility of
      // failure, but we can't assume that here.
      if (f) { tamesTo(f, t); }
      return f;
    }
    return undefined;
  }

  ////////////////////////////////////////////////////////////////////////
  // Taming helpers to be called only by tame() and untame().
  ////////////////////////////////////////////////////////////////////////

  global.AS_TAMED___ = function() {
    fail('global object almost leaked');
  };

  global.AS_FERAL___ = function() {
    fail('global object leaked');
  };

  /**
   * Used in lieu of an AS_TAMED___() method for Records.
   * <p>
   * Assume f is an unmarked Record. Recursively tame all its
   * mentionable enumerable own properties, being careful to
   * handle cycles. Failure to tame a property value only causes
   * that property to be omitted. Freeze the resulting record. If
   * the original record were frozen and all properties tame to
   * themselves, then the Record should tame to itself. 
   */
  function tameRecord(f) {
    var t = {};
    var changed = !isFrozen(f);
    // To handle cycles, provisionally mark f as taming to a fresh
    // t going in and see how the rest tames. Set up a try/finally
    // block to remove these provisional markings even on
    // exceptional exit.
    tamesTo(f, t);      
    try {
      var keys = ownKeys(f);
      var len = keys.length;
      for (var i = 0; i < len; i++) {
        var k = keys[i];
        var fv = f[k];
        var tv = tame(fv);
        if (tv === void 0 && fv !== void 0) {
          changed = true;
        } else {
          if (fv !== tv && fv === fv) { // I hate NaNs
            changed = true;
          }
          t[k] = tv;
        }
      }
    } finally {
      delete f.TAMED_TWIN___;
      delete t.FERAL_TWIN___;
    }
    if (changed) {
      // Although the provisional marks have been removed, our caller
      // will restore them. We do it this way to make tameRecord()
      // more similar to AS_TAMED___() methods.
      return primFreeze(t);
    } else {
      return f;
    }
  }
 
  /**
   * Used in lieu of an AS_FERAL___() method for Records.
   * <p>
   * Assume t is an unmarked Record. Recursively untame all its
   * mentionable enumerable own properties, being careful to
   * handle cycles. Failure to untame a property value only causes
   * that property to be omitted. Freeze the resulting record. If
   * the original record were frozen and all properties untame to
   * themselves, then the Record should untame to itself. 
   */
  function untameRecord(t) {
    var f = {};
    var changed = !isFrozen(t);
    // To handle cycles, provisionally mark t as untaming to a fresh
    // f going in and see how the rest untames. Set up a try/finally
    // block to remove these provisional markings even on
    // exceptional exit.
    tamesTo(f, t);      
    try {
      var keys = ownKeys(t);
      var len = keys.length;
      for (var i = 0; i < len; i++) {
        var k = keys[i];
        var tv = t[k];
        var fv = untame(tv);
        if (fv === void 0 && tv !== void 0) {
          changed = true;
        } else {
          if (tv !== fv && tv === tv) { // I hate NaNs
            changed = true;
          }
          f[k] = fv;
        }
      }
    } finally {
      delete t.FERAL_TWIN___;
      delete f.TAMED_TWIN___;
    }
    if (changed) {
      // Although the provisional marks have been removed, our caller
      // will restore them. We do it this way to make untameRecord()
      // more similar to AS_FERAL___() methods.
      return primFreeze(f);
    } else {
      return t;
    }
  }

  /**
   * 
   * Tame an array into a frozen dense array of tamed elements.
   * <p>
   * Assume f is an unmarked array. Recursively tame all its
   * elements (the values of its uint-named properties between 0 and
   * length-1), being careful to handle cycles. Absence of an index or
   * failure to tame a value only causes the taming at that index to
   * be undefined. Freeze the resulting array. If the original array
   * were frozen and all elements are present and tame to themselves,
   * then the array should tame to itself.
   * <p>
   * SECURITY HAZARD: Having the array tame to itself under the above
   * rule isn't safe if the array contains non-index-named properties
   * with non-tame values, as these will become accessible to
   * untrusted cajoled code. However, it is too expensive to check for
   * these here, so it is the responsibility of the trusted uncajoled
   * code never to cause an array containing such properties to be
   * tamed. TODO(erights): Should add a debugging flag to enable
   * expensive safety checks.
   */
  Array.prototype.AS_TAMED___ = function tameArray() {
    var f = this;
    var t = [];
    var changed = !isFrozen(f);
    // To handle cycles, provisionally mark f as taming to a fresh
    // t going in and see how the rest tames. Set up a try/finally
    // block to remove these provisional markings even on
    // exceptional exit.
    tamesTo(f, t);      
    try {
      var len = f.length;
      for (var i = 0; i < len; i++) {
        if (i in f) {
          var fv = f[i];
          var tv = tame(fv);
          if (fv !== tv && fv === fv) { // I hate NaNs
            changed = true;
          }
          t[i] = tv;
        } else {
          changed = true;
          t[i] = void 0;          
        }
      }
    } finally {
      delete f.TAMED_TWIN___;
      delete t.FERAL_TWIN___;
    }
    if (changed) {
      // Although the provisional marks have been removed, our caller
      // will restore them.
      return primFreeze(t);
    } else {
      // See SECURITY HAZARD note in doc-comment.
      return f;
    }
  };

  /**
   * Untame an array into a frozen dense array of feral elements.
   * <p>
   * Assume f is an unmarked array. Recursively untame all its
   * elements (the values of its uint-named properties between 0 and
   * length-1), being careful to handle cycles. Absence of an index or
   * failure to untame a value only causes the untaming at that index to
   * be undefined. Freeze the resulting array. If the original array
   * were frozen and all elements are present and untame to themselves,
   * then the array should untame to itself.
   * <p>
   * SECURITY HAZARD: Having the array untame to itself under the above
   * rule isn't safe if the array contains non-index-named properties
   * with non-feral values, as these might be accessed by
   * trusted uncajoled code. However, it is too expensive to check for
   * these here, so it is the responsibility of the trusted uncajoled
   * code not to <i>innocently</i> access such properties on a feral
   * array resulting from untaming a tame array. TODO(erights): Should
   * add a debugging flag to enable expensive safety checks.
   */
  Array.prototype.AS_FERAL___ = function untameArray() {
    var t = this;
    var f = [];
    var changed = !isFrozen(t);
    // To handle cycles, provisionally mark t as untaming to a fresh
    // f going in and see how the rest untames. Set up a try/finally
    // block to remove these provisional markings even on
    // exceptional exit.
    tamesTo(f, t);      
    try {
      var len = t.length;
      for (var i = 0; i < len; i++) {
        if (i in t) {
          var tv = t[i];
          var fv = untame(tv);
          if (tv !== fv && tv === tv) { // I hate NaNs
            changed = true;
          }
          f[i] = fv;
        } else {
          changed = true;
          f[i] = void 0;
        }
      }
    } finally {
      delete t.FERAL_TWIN___;
      delete f.TAMED_TWIN___;
    }
    if (changed) {
      // Although the provisional marks have been removed, our caller
      // will restore them.
      return primFreeze(f);
    } else {
      // See SECURITY HAZARD note in doc-comment.
      return t;
    }
  };
  
  /**
   * Constructors and simple-functions tame to themselves by default.
   */
  Function.prototype.AS_TAMED___ = function defaultTameFunc() {
    var f = this;
    if (isFunc(f) || isCtor(f)) { return f; }
    return void 0;
  };
  
  /**
   * Constructors and simple-function untame to themselves by default.
   */
  Function.prototype.AS_FERAL___ = function defaultUntameFunc() {
    var t = this;
    if (isFunc(t) || isCtor(t)) { return t; }
    return void 0;    
  };

  /**
   * Prevent privilege escalation by passing USELESS rather than null,
   * undefined or the global object as the <tt>this</tt> of a call to
   * a real <tt>call</tt>, <tt>apply</tt>, or <tt>bind</tt> method.
   */
  function stopEscalation(val) {
    if (val === null || val === void 0 || val === global) {
      return USELESS;
    }
    return val;
  }

  /**
   * To be installed as the AS_TAMED___() method on feral functions
   * marked by markXo4a().
   * <p>
   * A feral function is marked as xo4a iff it may be given a tame
   * <tt>this</tt> and arguments and must return a tame result. We
   * therefore tame it to a frozen pseudo-function whose call and
   * apply methods calls the original function's apply method
   * directly. 
   */
  function tameXo4a() {
    var xo4aFunc = this;
    function tameApplyFuncWrapper(self, opt_args) {
      return xo4aFunc.apply(stopEscalation(self), opt_args || []);
    }
    markFuncFreeze(tameApplyFuncWrapper);

    function tameCallFuncWrapper(self, var_args) {
      return tameApplyFuncWrapper(self, Array.slice(arguments, 1));
    }
    markFuncFreeze(tameCallFuncWrapper);

    var result = PseudoFunction(tameCallFuncWrapper, tameApplyFuncWrapper);
    result.length = xo4aFunc.length;
    result.toString = markFuncFreeze(xo4aFunc.toString.bind(xo4aFunc));
    return primFreeze(result);
  }

  /**
   * To be installed as the AS_TAMED___() method on feral functions
   * marked by markInnocent().
   * <p>
   * An innocent feral function must be assumed to be exophoric, to
   * expect only a feral this and arguments and to return a feral
   * result. We therefore tame it to a frozen pseudo-function whose
   * call and apply methods calls the original feral function with an
   * untaming of its self and remaining arguments. It will then return 
   * a taming of what the feral function returns.
   */
  function tameInnocent() {
    var feralFunc = this;
    function tameApplyFuncWrapper(self, opt_args) {
      var feralThis = stopEscalation(untame(self));
      var feralArgs = untame(opt_args);
      var feralResult = feralFunc.apply(feralThis, feralArgs || []);
      return tame(feralResult);
    }
    markFuncFreeze(tameApplyFuncWrapper);

    function tameCallFuncWrapper(self, var_args) {
      return tameApplyFuncWrapper(self, Array.slice(arguments, 1));
    }
    markFuncFreeze(tameCallFuncWrapper);

    var result = PseudoFunction(tameCallFuncWrapper, tameApplyFuncWrapper);
    result.length = feralFunc.length;
    result.toString = markFuncFreeze(feralFunc.toString.bind(feralFunc));
    return primFreeze(result);
  }

  /**
   * Makes a [[ThrowTypeError]] function, as defined in section 13.2.3
   * of the ES5 spec.
   * 
   * <p>The informal name for the [[ThrowTypeError]] function, defined
   * in section 13.2.3 of the ES5 spec, is the "poison pill". The poison
   * pill is simply a no-argument function that, when called, always
   * throws a TypeError. Since we wish this TypeError to carry useful
   * diagnostic info, we violate the ES5 spec by defining 4 poison
   * pills with 4 distinct identities.
   * 
   * <p>A poison pill is installed as the getter & setter of the
   * de-jure (arguments.callee) and de-facto non-strict magic stack
   * inspection properties, which no longer work in ES5/strict, since
   * they violate encapsulation. Rather than simply remove them,
   * access to these properties is poisoned in order to catch errors
   * earlier when porting old non-strict code.
   */
  function makePoisonPill(badThing) {
    function poisonPill() {
      throw new TypeError('' + badThing + ' forbidden by ES5/strict');
    }
    return poisonPill;
  }
  var poisonArgsCallee = makePoisonPill('arguments.callee');
  var poisonArgsCaller = makePoisonPill('arguments.caller');
  var poisonFuncCaller = makePoisonPill("A function's .caller");
  var poisonFuncArgs = makePoisonPill("A function's .arguments");

  /**
   * Given either an array or an actual arguments object, return
   * Cajita's emulation of an ES5/strict arguments object.
   */
  function args(original) {
    var result = {length: 0};
    pushMethod.apply(result, original);
    result.CLASS___ = 'Arguments';
    useGetHandler(result, 'callee', poisonArgsCallee);
    useSetHandler(result, 'callee', poisonArgsCallee);
    useGetHandler(result, 'caller', poisonArgsCaller);
    useSetHandler(result, 'caller', poisonArgsCaller);
    return result;
  }
  var pushMethod = [].push;

  /**
   * A Record is a pseudo-function if it inherits from
   * some PseudoFunctionProto.
   * <p>
   * At the moment a pseudo-function is returned from
   * PseudoFunction(), it has working call, apply, and bind
   * methods. However, since pseudo-functions may be mutated, these
   * can be altered or deleted. Neverthess, having determined that an
   * object is a pseudo-function, its clients may (and generally will)
   * assume that all these methods are present and working.
   */
  var PseudoFunctionProto = {

    /**
     * A simple default that should generally be overridden.
     */
    toString: markFuncFreeze(function() {
      return 'pseudofunction(var_args) {\n    [some code]\n}';
    }),

    /**
     * Since there's no constructor, to have a general cross-frame
     * pseudo-function test, we test the truthiness of PFUNC___.
     */
    PFUNC___: true,

    /**
     * For Valija code, this has to be 'Function' or the standard cross-frame
     * test fails.  We can't override ObjectPrototype.toString in Valija
     * because of issue 953.
     */
    CLASS___: 'Function',

    /**
     * A pseudo-function untames to an exophoric feral function.
     * <p>
     * The resulting feral function will call the pseudo-function's
     * apply method with a taming of the feral this as the tame self
     * and a taming of its arguments. It will then return an untaming
     * of the result. 
     */
    AS_FERAL___: function untamePseudoFunction() {
      var tamePseudoFunc = this;
      function feralWrapper(var_args) {
        var feralArgs = Array.slice(arguments, 0);
        var tamedSelf = tame(stopEscalation(this));
        var tamedArgs = tame(feralArgs);
        var tameResult = callPub(tamePseudoFunc, 
                                 'apply', 
                                 [tamedSelf, tamedArgs]);
        return untame(tameResult);
      }
      return feralWrapper;
    }
  };
  useGetHandler(PseudoFunctionProto, 'caller', poisonFuncCaller);
  useSetHandler(PseudoFunctionProto, 'caller', poisonFuncCaller);
  useGetHandler(PseudoFunctionProto, 'arguments', poisonFuncArgs);
  useSetHandler(PseudoFunctionProto, 'arguments', poisonFuncArgs);
  primFreeze(PseudoFunctionProto);

  /**
   * Makes an unfrozen pseudo-function that inherits from
   * PseudoFunctionProto. 
   * <p>
   * Both the callFunc and the opt_applyFunc, if provided, must be
   * simple-functions. If the opt_applyFunc is omitted, it is
   * synthesized from the callFunc.
   * <p>
   * PseudoFunction is not a genuine constructor, because
   * pseudo-functions are Records, not constructed objects.
   * <p>
   * PseudoFunction's caller should set the pseudo-function's toString
   * method to something useful, overriding the default inherited from
   * PseudoFunctionProto. 
   */
  function PseudoFunction(callFunc, opt_applyFunc) {
    callFunc = asFunc(callFunc);
    var applyFunc;
    if (opt_applyFunc) {
      applyFunc = asFunc(opt_applyFunc);
    } else {
      applyFunc = markFuncFreeze(function applyFun(self, opt_args) {
        var args = [self];
        if (opt_args !== void 0 && opt_args !== null) {
          args.push.apply(args, opt_args);
        }
        return callFunc.apply(USELESS, args);
      });
    }

    var result = primBeget(PseudoFunctionProto);
    result.call = callFunc;
    result.apply = applyFunc;
    result.bind = markFuncFreeze(function bindFun(self, var_args) {
      self = stopEscalation(self);
      var args = [USELESS, self].concat(Array.slice(arguments, 1));
      return markFuncFreeze(callFunc.bind.apply(callFunc, args));
    });
    result.length = callFunc.length -1;
    return result;
  }

  ////////////////////////////////////////////////////////////////////////
  // Classifying functions and pseudo-functions
  ////////////////////////////////////////////////////////////////////////

  function isCtor(constr)    {
    return constr && !!constr.CONSTRUCTOR___;
  }
  function isFunc(fun) {
    return fun && !!fun.FUNC___;
  }
  function isXo4aFunc(func) {
    return func && !!func.XO4A___;
  }
  function isPseudoFunc(fun) {
    return fun && fun.PFUNC___;
  }

  /**
   * Mark <tt>constr</tt> as a constructor which tames to itself by 
   * default, and whose instances are constructed objects which tame 
   * and untame to themselves by default. 
   * <p>
   * A function is tamed and classified by calling one of
   * <tt>markCtor()</tt>, <tt>markXo4a()</tt>, or
   * <tt>markFuncFreeze()</tt>. Each of these checks that the function
   * hasn't already been classified by any of the others. A function
   * which has not been so classified is an <i>unclassifed function</i>.
   * <p>
   * If <tt>opt_Sup</tt> is provided, record that constr.prototype
   * inherits from opt_Sup.prototype. This bookkeeping helps
   * directConstructor().
   * <p>
   * <tt>opt_name</tt>, if provided, should be the name of the constructor
   * function. Currently, this is used only to generate friendlier
   * error messages.
   */
  function markCtor(constr, opt_Sup, opt_name) {
    enforceType(constr, 'function', opt_name);
    if (isFunc(constr)) {
      fail("Simple functions can't be constructors: ", constr);
    }
    if (isXo4aFunc(constr)) {
      fail("Exophoric functions can't be constructors: ", constr);
    }
    constr.CONSTRUCTOR___ = true;
    if (opt_Sup) {
      derive(constr, opt_Sup);
    } else if (constr !== Object) {
      fail('Only "Object" has no super: ', constr);
    }
    if (opt_name) {
      constr.NAME___ = String(opt_name);
    }
    if (constr !== Object && constr !== Array) {
      // Iff constr is not Object nor Array, then any object inheriting from
      // constr.prototype is a constructed object which therefore tames to
      // itself by default. We do this with AS_TAMED___ and AS_FERAL___
      // methods on the prototype so it can be overridden either by overriding
      // these methods or by pre-taming with ___.tamesTo or ___.tamesToSelf.
      constr.prototype.AS_TAMED___ =
        constr.prototype.AS_FERAL___ = function() {
          return this;
        };
    }
    return constr;  // translator freezes constructor later
  }

  function derive(constr, sup) {
    var proto = constr.prototype;
    sup = asCtor(sup);
    if (isFrozen(constr)) {
      fail('Derived constructor already frozen: ', constr);
    }
    if (!(proto instanceof sup)) {
      fail('"' + constr + '" does not derive from "', sup);
    }
    if ('__proto__' in proto && proto.__proto__ !== sup.prototype) {
      fail('"' + constr + '" does not derive directly from "', sup);
    }
    if (!isFrozen(proto)) {
      // Some platforms, like Safari, actually conform to the part
      // of the ES3 spec which states that the constructor property
      // of implicitly created prototypical objects are not
      // deletable. But this prevents the inheritance-walking
      // algorithm (kludge) in directConstructor from working. Thus,
      // we set proto___ here so that directConstructor can skip
      // that impossible case.
      proto.proto___ = sup.prototype;
    }
  }

  /**
   * Initialize argument constructor <i>feralCtor</i> so that it
   * represents a "subclass" of argument constructor <i>someSuper</i>,
   * and return a non-invokable taming of <i>feralCtor</i>. 
   *
   * Given: 
   *
   *   function FeralFoo() { ... some uncajoled constructor ... }
   *   var Foo = extend(FeralFoo, FeralSuper, 'Foo');
   *
   * it will be the case that:
   *
   *   new FeralFoo() instanceof Foo
   *
   * however -- and this is the crucial property -- cajoled code will get an
   * error if it invokes either of:
   *
   *   new Foo()
   *   Foo()
   *
   * This allows us to expose the tame Foo to cajoled code, allowing
   * it to sense that all the FeralFoo instances we give it are
   * instanceof Foo, without granting to cajoled code the means to
   * create any new such instances.
   * 
   * extend() also sets <i>feralCtor</i>.prototype to set up the
   * prototype chain so that 
   *
   *   new FeralFoo() instanceof FeralSuper
   * and
   *   new FeralFoo() instanceof Super
   *
   * @param feralCtor An feral-only uncajoled constructor. This must
   *        NOT be exposed to cajoled code by any other mechanism.
   * @param someSuper Some constructor representing the
   *        superclass. This can be <ul>
   *        <li>a feralCtor that had been provided as a first argument
   *            in a previous call to extend(), 
   *        <li>an inertCtor as returned by a previous call to
   *            extend(), or 
   *        <li>a constructor that has been marked as such by ___.markCtor().
   *        </ul>
   *        In all cases, someSuper.prototype.constructor must be
   *        a constructor that has been marked as such by
   *        ___.markCtor(). 
   * @param opt_name If the returned inert constructor is made
   *        available this should be the property name used.
   *
   * @return a tame inert class constructor as described above.
   */
  function extend(feralCtor, someSuper, opt_name) {
    if (!('function' === typeof feralCtor)) {
      fail('Internal: Feral constructor is not a function');
    }
    someSuper = asCtor(someSuper.prototype.constructor);
    var noop = function () {};
    noop.prototype = someSuper.prototype;
    feralCtor.prototype = new noop();
    feralCtor.prototype.proto___ = someSuper.prototype;

    var inert = function() {
      fail('This constructor cannot be called directly');
    };

    inert.prototype = feralCtor.prototype;
    feralCtor.prototype.constructor = inert;
    markCtor(inert, someSuper, opt_name);
    tamesTo(feralCtor, inert);
    return primFreeze(inert);
  }

  /**
   * Marks a function as a feral exophoric function whose
   * <tt>this</tt> and arguments may be tame and whose results will be
   * tame .
   */
  function markXo4a(func, opt_name) {
    enforceType(func, 'function', opt_name);
    if (isCtor(func)) {
      fail("Internal: Constructors can't be exophora: ", func);
    }
    if (isFunc(func)) {
      fail("Internal: Simple functions can't be exophora: ", func);
    }
    func.XO4A___ = true;
    if (opt_name) {
      func.NAME___ = opt_name;
    }
    func.AS_TAMED___ = tameXo4a;
    return primFreeze(func);
  }

  /**
   * Mark a function innocent if it might expect a feral <tt>this</tt>
   * and arguments and might return a feral result, but should still
   * tame to something that allows untrusted cajoled code to invoke it.
   */
  function markInnocent(func, opt_name) {
    enforceType(func, 'function', opt_name);
    if (isCtor(func)) {
      fail("Internal: Constructors aren't innocent: ", func);
    }
    if (isFunc(func)) {
      fail("Internal: Simple functions aren't innocent: ", func);
    }
    if (isXo4aFunc(func)) {
      fail("Internal: Exophoric functions aren't innocent: ", func);
    }
    if (opt_name) {
      func.NAME___ = opt_name;
    }
    func.AS_TAMED___ = tameInnocent;
    return primFreeze(func);
  }

  /**
   * Mark fun as a simple function and freeze it.
   * <p>
   * simple functions tame to themselves by default.
   * <p>
   * opt_name, if provided, should be the name of the
   * function. Currently, this is used only to generate friendlier
   * error messages. 
   */
  function markFuncFreeze(fun, opt_name) {
    // inline: enforceType(fun, 'function', opt_name);
    if (typeOf(fun) !== 'function') {
      fail('expected function instead of ', typeOf(fun),
           ': ', (opt_name || fun));
    }

    // inline: if (isCtor(fun)) {
    if (fun.CONSTRUCTOR___) {
      fail("Constructors can't be simple functions: ", fun);
    }
    // inline: if (isXo4aFunc(fun)) {
    if (fun.XO4A___) {
      fail("Exophoric functions can't be simple functions: ", fun);
    }
    fun.FUNC___ = opt_name ? String(opt_name) : true;
    return primFreeze(fun);
  }

  /** This "Only" form doesn't freeze */
  function asCtorOnly(constr) {
    if (isCtor(constr) || isFunc(constr)) {
      return constr;
    }
    enforceType(constr, 'function');
    fail("Untamed functions can't be called as constructors: ", constr);
  }

  /** Only constructors and simple functions can be called as constructors */
  function asCtor(constr) {
    return primFreeze(asCtorOnly(constr));
  }

  /**
   * Only simple functions (or Number, String, or Boolean) can be
   * called as simple functions. 
   * <p>
   * It is now <tt>asFunc</tt>'s responsibility to
   * <tt>primFreeze(fun)</tt>.
   */
  function asFunc(fun) {
    if (fun && fun.FUNC___) {
      // fastpath shortcut
      if (fun.FROZEN___ === fun) {
        return fun;
      } else {
        return primFreeze(fun);
      }
    }
    enforceType(fun, 'function');
    if (isCtor(fun)) {
      if (fun === Number || fun === String || fun === Boolean) {
        // TODO(erights): To avoid accidents, <tt>markXo4a</tt>,
        // <tt>markFuncFreeze</tt>, and <tt>markCtor</tt> each ensure
        // that these classifications are exclusive. A function can be
        // classified as in at most one of these categories. However,
        // some primordial type conversion functions like
        // <tt>String</tt> need to be invocable both ways, so we
        // should probably relax this constraint.
        // <p>
        // But before we do, we should reexamine other
        // implications. For example, simple-functions, when called
        // reflectively by <tt>call</tt> or <tt>apply</tt> (and
        // therefore <tt>bind</tt>), ignore their first argument,
        // whereas constructors can be called reflectively by
        // <tt>call</tt> to do super-initialization on behalf of a
        // derived constructor.
        // <p>
        // Curiously, ES3 also defines function behavior different
        // from constructor behavior for <tt>Object</tt>,
        // <tt>Date</tt>, <tt>RegExp</tt>, and <tt>Error</tt>. (Not
        // sure about <tt>Array</tt>.) We should understand these as
        // well before introducing a proper solution.
        return primFreeze(fun);
      }
      fail("Constructors can't be called as simple functions: ", fun);
    }
    if (isXo4aFunc(fun)) {
      fail("Exophoric functions can't be called as simple functions: ", fun);
    }
    fail("Untamed functions can't be called as simple functions: ", fun);
  }

  /**
   * Coerces fun to a genuine simple-function.
   * <p>
   * If fun is an pseudo-function, then return a simple-function that
   * invokes fun's apply method. Otherwise, asFunc().
   */
  function toFunc(fun) {
    if (isPseudoFunc(fun)) {
      return markFuncFreeze(function applier(var_args) {
        return callPub(fun, 'apply', [USELESS, Array.slice(arguments, 0)]);
      });
    }
    return asFunc(fun);
  }

  /**
   * An object is prototypical iff its 'constructor' property points
   * at a genuine function whose 'prototype' property points back at
   * it.
   * <p>
   * Cajita code cannot access or create prototypical objects since
   * the 'prototype' property of genuine functions is inaccessible,
   * and since the transient function used by <tt>beget</tt> to create
   * inheritance chains does not escape.
   */
  function isPrototypical(obj) {
    if (typeOf(obj) !== 'object') { return false; }
    if (obj === null) { return false; }
    var constr = obj.constructor;
    if (typeOf(constr) !== 'function') { return false; }
    return constr.prototype === obj;
  }

  /**
   * Throws an exception if the value is an unmarked function or a
   * prototypical object.
   */
  function asFirstClass(value) {
    switch (typeOf(value)) {
      case 'function': {
        if (isFunc(value) || isCtor(value)) {
          if (isFrozen(value)) {
            return value;
          }
          // TODO(metaweta): make this a cajita-uncatchable exception
          fail('Internal: non-frozen function encountered: ', value);
        } else if (isXo4aFunc(value)) {
          // TODO(metaweta): make this a cajita-uncatchable exception
          // TODO(erights): non-user-hostile error message
          fail('Internal: toxic exophora encountered: ', value);
        } else {
          // TODO(metaweta): make this a cajita-uncatchable exception
          fail('Internal: toxic function encountered: ', value);
        }
        break;
      }
      case 'object': {
        if (value !== null && isPrototypical(value)) {
          // TODO(metaweta): make this a cajita-uncatchable exception
          fail('Internal: prototypical object encountered: ', value);
        }
        return value;
      }
      default: {
        return value;
      }
    }
  }

  ////////////////////////////////////////////////////////////////////////
  // Accessing properties
  ////////////////////////////////////////////////////////////////////////

  /**
   * Can a Cajita client of <tt>obj</tt> read its {@code name} property?
   * <p>
   * If the property is unmentionable (i.e. ends in an '__'), then no.
   * If the property was defined by Cajita code, then yes. If it was
   * whitelisted, then yes. Or if the property is an own property of
   * <i>some</i> JSON container, then yes.
   * <p>
   * Why "some"? If record y inherits from record x, and 'foo' is an own
   * property of x, then canReadPub(y, 'foo') must be true.
   */
  function canReadPub(obj, name) {
    if (typeof name === 'number' && name >= 0) { return name in obj; }
    name = String(name);
    if (obj === null) { return false; }
    if (obj === void 0) { return false; }
    if (obj[name + '_canRead___']) { return (name in Object(obj)); }
    if (endsWith__.test(name)) { return false; }
    if (name === 'toString') { return false; }
    if (!isJSONContainer(obj)) { return false; }
    if (!myOriginalHOP.call(obj, name)) { return false; }
    fastpathRead(obj, name);
    return true;
  }

  function hasOwnPropertyOf(obj, name) {
    if (typeof name === 'number' && name >= 0) { return hasOwnProp(obj, name); }
    name = String(name);
    if (obj && obj[name + '_canRead___'] === obj) { return true; }
    return canReadPub(obj, name) && myOriginalHOP.call(obj, name);
  }

  /**
   * Implements Cajita's <tt><i>name</i> in <i>obj</i></tt>
   */
  function inPub(name, obj) {
    var t = typeof obj;
    if (!obj || (t !== 'object' && t !== 'function')) {
      throw new TypeError('invalid "in" operand: ' + obj);
    }
    obj = Object(obj);
    if (canReadPub(obj, name)) { return true; }
    if (canCallPub(obj, name)) { return true; }
    if ((name + '_getter___') in obj) { return true; }
    if ((name + '_handler___') in obj) { return true; }
    return false;
  }

  /**
   * Called by Caja code attempting to read a property.
   * <p>
   * If it can't then <tt>readPub</tt> returns <tt>undefined</tt> instead.
   */
  function readPub(obj, name) {
    if (typeof name === 'number' && name >= 0) {
      if (typeof obj === 'string') {
        // In partial anticipation of ES5.
        // TODO(erights): Once ES5 settles, revisit this and
        // correctly implement the agreed semantics.
        // Mike Samuel suggested also making it conditional on
        //  (+name) === (name & 0x7fffffff)
        // but then realized that it violates the requirement
        // that the string form be the canonical form of the
        // number. So 'foo'['00'] would be treated the same
        // as 'foo'['0'] which is incorrect. 
        return obj.charAt(name);
      } else {
        return obj[name];
      }
    }
    name = String(name);
    if (canReadPub(obj, name)) { return obj[name]; }
    if (obj === null || obj === void 0) {
      throw new TypeError("Can't read " + name + ' on ' + obj);
    }
    return obj.handleRead___(name);
  }

  /**
   * If <tt>obj</tt> is an object with a property <tt>name</tt> that
   * should be objectively readable from Valija, return
   * <tt>obj[name]</tt>, else <tt>pumpkin</tt>.
   * <p>
   * Provides a fastpath for Valija's <tt>read()</tt> function
   * <tt>$v.r()</tt>. The reason for returning the passed in pumpkin
   * rather than, for example, <tt>undefined</tt>, is so that the
   * caller can pass in a known unique value and distinguish it, on
   * return, from any possible valid value.
   * <p>
   * A property should be objectively readable iff<ul>
   * <li>It is readable from Cajita, and
   * <li><tt>obj</tt> is not a function, and
   * <li>either<ul>
   *     <li><tt>name</tt> is an own property of <tt>obj</tt>, or
   *     <li><tt>obj</tt> inherits <tt>name</tt> from an ancestor that
   *         Cajita considers first-class. The one such possibility is
   *         when <tt>obj</tt> is a record inheriting <tt>name</tt>
   *         from another record. (A record is a non-prototypical
   *         object whose directConstructor is Object.)
   *     </ul>
   * </ul>
   */
  function readOwn(obj, name, pumpkin) {
    if (typeof obj !== 'object' || !obj) {
      if (typeOf(obj) !== 'object') {
        return pumpkin;
      }
    }
    if (typeof name === 'number' && name >= 0) {
      if (myOriginalHOP.call(obj, name)) { return obj[name]; }
      return pumpkin;
    }
    name = String(name);
    if (obj[name + '_canRead___'] === obj) { return obj[name]; }
    if (!myOriginalHOP.call(obj, name)) { return pumpkin; }
    // inline remaining relevant cases from canReadPub
    if (endsWith__.test(name)) { return pumpkin; }
    if (name === 'toString') { return pumpkin; }
    if (!isJSONContainer(obj)) { return pumpkin; }
    fastpathRead(obj, name);
    return obj[name];
  }

  /**
   * Ensure that all the permitsUsed starting at result are forever
   * safe to allow without runtime checks.
   */
  function enforceStaticPath(result, permitsUsed) {
    forOwnKeys(permitsUsed, markFuncFreeze(function(name, subPermits) {
      // Don't factor out since we don't enforce frozen if permitsUsed
      // are empty.
      // TODO(erights): Once we have ES5ish attribute control, it
      // will suffice to enforce that each used property is frozen
      // independent of the object as a whole.
      enforce(isFrozen(result), 'Assumed frozen: ', result);
      if (name === '()') {
        // TODO(erights): Revisit this case
      } else {
        enforce(canReadPub(result, name),
                'Assumed readable: ', result, '.', name);
        if (inPub('()', subPermits)) {
          enforce(canCallPub(result, name),
                  'Assumed callable: ', result, '.', name, '()');
        }
        enforceStaticPath(readPub(result, name), subPermits);
      }
    }));
  }

  /**
   * Privileged code attempting to read an imported value from a module's
   * <tt>IMPORTS___</tt>. This function is NOT available to Cajita code.
   * <p>
   * This delegates to <tt>readOwn()</tt>, and so will only read
   * those properties from module_imports that are objectively visible
   * from both Cajita and Valija.
   */
  function readImport(module_imports, name, opt_permitsUsed) {
    var pumpkin = {};
    var result = readOwn(module_imports, name, pumpkin);
    if (result === pumpkin) {
      log('Linkage warning: ' + name + ' not importable');
      return void 0;
    }
    if (opt_permitsUsed) {
      enforceStaticPath(result, opt_permitsUsed);
    }
    return result;
  }

  /**
   * Can "innocent" code enumerate the named property on this object?
   * <p>
   * "Innocent" code is code which we assume to be ignorant of Caja,
   * not to be actively hostile, but which may be buggy (and
   * therefore accidentally harmful or exploitable). This
   * corresponds to legacy code, such as libraries, that we decide
   * to run untranslated, perhaps hidden or tamed, but which needs
   * to co-exist smoothly with the Caja runtime.
   * <p>
   * An earlier version of canInnocentEnum() filtered out exactly those
   * names ending with a double underbar. It now filters out exactly
   * those names ending in a triple underbar. Cajita code can't see names
   * ending in a double underbar, since existing platforms (like
   * Firefox) use such names for purposes that should be hidden from
   * Caja code. However, it is not up to Caja to shield innocent code
   * from seeing such platform properties. All the magic names Cajita
   * adds for its own internal bookkeeping end in triple underbar, so
   * that is all we need to hide from innocent code.
   */
  function canInnocentEnum(obj, name) {
    name = String(name);
    if (endsWith___.test(name)) { return false; }
    return true;
  }

  /**
   * Would a Cajita for/in loop by a client of obj see this name?
   * <p>
   * For properties defined in Cajita, this is generally the same as
   * canReadPub.  Otherwise according to whitelisting.
   */
  function canEnumPub(obj, name) {
    if (obj === null) { return false; }
    if (obj === void 0) { return false; }
    name = String(name);
    if (obj[name + '_canEnum___']) { return true; }
    if (endsWith__.test(name)) { return false; }
    if (!isJSONContainer(obj)) { return false; }
    if (!myOriginalHOP.call(obj, name)) { return false; }
    fastpathEnum(obj, name);
    if (name === 'toString') { return true; }
    fastpathRead(obj, name);
    return true;
  }

  /**
   * Like canEnumPub, but allows only non-inherited properties.
   */
  function canEnumOwn(obj, name) {
    name = String(name);
    if (obj && obj[name + '_canEnum___'] === obj) { return true; }
    return canEnumPub(obj, name) && myOriginalHOP.call(obj, name);
  }

  /**
   * Returns a new object whose only utility is its identity and (for
   * diagnostic purposes only) its name.
   */
  function Token(name) {
    name = String(name);
    return primFreeze({
      toString: markFuncFreeze(function tokenToString() { return name; }),
      throwable___: true
    });
  }
  markFuncFreeze(Token);

  /**
   * Inside a <tt>cajita.forOwnKeys()</tt>, or <tt>cajita.forAllKeys()</tt>, the
   * body function can terminate early, as if with a conventional
   * <tt>break;</tt>, by doing a <pre>return cajita.BREAK;</pre>
   */
  var BREAK = Token('BREAK');

  /**
   * A unique value that should never be made accessible to untrusted
   * code, for distinguishing the absence of a result from any
   * returnable result.
   * <p>
   * See makeNewModuleHandler's getLastOutcome().
   */
  var NO_RESULT = Token('NO_RESULT');

  /**
   * For each sensible key/value pair in obj, call fn with that
   * pair.
   * <p>
   * If obj is an array, then enumerate indexes. Otherwise, enumerate
   * the canEnumOwn() property names.
   */
  function forOwnKeys(obj, fn) {
    fn = toFunc(fn);
    var keys = ownKeys(obj);
    for (var i = 0; i < keys.length; i++) {
      if (fn(keys[i], readPub(obj, keys[i])) === BREAK) {
        return;
      }
    }
  }

  /**
   * For each sensible key/value pair in obj, call fn with that
   * pair.
   * <p>
   * If obj is an array, then enumerate indexes. Otherwise, enumerate
   * the canEnumPub() property names.
   */
  function forAllKeys(obj, fn) {
    fn = toFunc(fn);
    var keys = allKeys(obj);
    for (var i = 0; i < keys.length; i++) {
      if (fn(keys[i], readPub(obj, keys[i])) === BREAK) {
        return;
      }
    }
  }

  /**
   * Return an array of the publicly readable own keys of obj.
   * <p>
   * If obj is an array, then enumerate indexes. Otherwise, enumerate
   * the canEnumOwn() property names.
   */
  function ownKeys(obj) {
    var result = [];
    if (isArray(obj)) {
      var len = obj.length;
      for (var i = 0; i < len; i++) {
        result.push(i);
      }
    } else {
      for (var k in obj) {
        if (canEnumOwn(obj, k)) {
          result.push(k);
        }
      }
      if (obj !== void 0 && obj !== null && obj.handleEnum___) {
        result = result.concat(obj.handleEnum___(true));
      }
    }
    return result;
  }

  /**
   * Return an array of the publicly readable own and inherited keys of obj.
   * <p>
   * If obj is an array, then enumerate indexes. Otherwise, enumerate
   * the canEnumPub() property names.
   */
  function allKeys(obj) {
    if (isArray(obj)) {
      return ownKeys(obj);
    } else {
      var result = [];
      for (var k in obj) {
        if (canEnumPub(obj, k)) {
          result.push(k);
        }
      }
      if (obj !== void 0 && obj !== null && obj.handleEnum___) {
        result = result.concat(obj.handleEnum___(false));
      }
      return result;
    }
  }

  /**
   * Can this be called as a public method?
   * <p>
   * For genuine methods, they are only callable if the canCall
   * attribute is set. Otherwise, if this property is readable and
   * holds a simple function, then it's also callable as a function,
   * which we can memoize.
   */
  function canCallPub(obj, name) {
    if (obj === null) { return false; }
    if (obj === void 0) { return false; }
    name = String(name);
    if (obj[name + '_canCall___']) { return true; }
    if (obj[name + '_grantCall___']) {
      fastpathCall(obj, name);
      return true;
    }
    if (!canReadPub(obj, name)) { return false; }
    if (endsWith__.test(name)) { return false; }
    if (name === 'toString') { return false; }
    var func = obj[name];
    if (!isFunc(func) && !isXo4aFunc(func)) {
      return false;
    }
    fastpathCall(obj, name);
    return true;
  }

  /**
   * A client of obj tries to call one of its methods.
   */
  function callPub(obj, name, args) {
    name = String(name);
    if (obj === null || obj === void 0) {
      throw new TypeError("Can't call " + name + ' on ' + obj);
    }
    if (obj[name + '_canCall___'] || canCallPub(obj, name)) {
      return obj[name].apply(obj, args);
    }
    if (obj.handleCall___) { return obj.handleCall___(name, args); }
    fail('not callable:', debugReference(obj), '.', name);
  }

  /**
   * Can a client of obj directly assign to its name property?
   * <p>
   * If this property is unmentionable (i.e., ends with a '__') or if this
   * object is frozen, then no.
   * Else if this is an own property defined by Cajita code,
   * then yes. If the object is a JSON container, then
   * yes. Otherwise according to whitelisting decisions.
   */
  function canSetPub(obj, name) {
    name = String(name);
    if (canSet(obj, name)) { return true; }
    if (endsWith__.test(name)) { return false; }
    if (name === 'valueOf') { return false; }
    if (name === 'toString') { return false; }
    return !isFrozen(obj) && isJSONContainer(obj);
  }

  /** A client of obj attempts to assign to one of its properties. */
  function setPub(obj, name, val) {
    // asFirstClass() here would be a useful safety check, to prevent
    // the further propogation of, for example, a leaked toxic
    // function. However, its security benefit is questionable, and
    // the check is expensive in this position.
//  val = asFirstClass(val);
    if (typeof name === 'number' &&
        name >= 0 &&
        // See issue 875
        obj instanceof Array &&
        obj.FROZEN___ !== obj) {
      return obj[name] = val;
    }
    name = String(name);
    if (obj === null || obj === void 0) {
      throw new TypeError("Can't set " + name + ' on ' + obj);
    }
    if (obj[name + '_canSet___'] === obj) {
      return obj[name] = val;
    } else if (canSetPub(obj, name)) {
      fastpathSet(obj, name);
      return obj[name] = val;
    } else {
      return obj.handleSet___(name, val);
    }
  }

  /**
   * Can the given function have the given static method added to it?
   * @param {Function} fun
   * @param {string} staticMemberName an identifier in the public namespace.
   */
  function canSetStatic(fun, staticMemberName) {
    staticMemberName = '' + staticMemberName;
    if (typeOf(fun) !== 'function') {
      log('Cannot set static member of non function: ' + fun);
      return false;
    }
    if (isFrozen(fun)) {
      log('Cannot set static member of frozen function: ' + fun);
      return false;
    }
    if (!isFunc(fun)) {
      log('Can only set static members on simple-functions: ' + fun);
      return false;
    }
    if (staticMemberName === 'toString') {
      // no diagnostic as this is a normal fault-handling case.
      return false;
    }
    // statics are public
    if (endsWith__.test(staticMemberName) || staticMemberName === 'valueOf') {
      log('Illegal static member name: ' + staticMemberName);
      return false;
    }
    // disallows prototype, call, apply, bind
    if (staticMemberName in fun) {
      log('Cannot override static member: ' + staticMemberName);
      return false;
    }
    return true;
  }

  /**
   * Sets a static members of a fun, making sure that it can't be used to
   * override call/apply/bind and other builtin members of function.
   * @param {Function} fun
   * @param {string} staticMemberName an identifier in the public namespace.
   * @param staticMemberValue the value of the static member.
   */
  function setStatic(fun, staticMemberName, staticMemberValue) {
    staticMemberName = '' + staticMemberName;
    if (canSetStatic(fun, staticMemberName)) {
      fun[staticMemberName] = staticMemberValue;
      fastpathEnum(fun, staticMemberName);
      fastpathRead(fun, staticMemberName);
    } else {
      fun.handleSet___(staticMemberName, staticMemberValue);
    }
  }

  /**
   * Can a client of obj delete the named property?
   */
  function canDeletePub(obj, name) {
    name = String(name);
    if (isFrozen(obj)) { return false; }
    if (endsWith__.test(name)) { return false; }
    if (name === 'valueOf') { return false; }
    if (name === 'toString') { return false; }
    if (isJSONContainer(obj)) { return true; }
    return false;
  }

  /**
   * A client of obj can only delete a property of obj if obj is a
   * non-frozen JSON container.
   */
  function deletePub(obj, name) {
    name = String(name);
    if (obj === null || obj === void 0) {
      throw new TypeError("Can't delete " + name + ' on ' + obj);
    }
    if (canDeletePub(obj, name)) {
      // See deleteFieldEntirely for reasons why we don't cache deletability.
      return deleteFieldEntirely(obj, name);
    } else {
      return obj.handleDelete___(name);
    }
  }

  /**
   * Deletes a field removing any cached permissions.
   * @param {object} obj
   * @param {string} name of field in obj to delete.
   * @return {boolean}
   * @throws {Error} if field not deletable or name not in field.
   * @private
   */
  function deleteFieldEntirely(obj, name) {
    // Can't cache fastpath delete since deleting the field should remove
    // all privileges for that field.
    delete obj[name + '_canRead___'];
    delete obj[name + '_canEnum___'];
    delete obj[name + '_canCall___'];
    delete obj[name + '_grantCall___'];
    delete obj[name + '_grantSet___'];
    delete obj[name + '_canSet___'];
    delete obj[name + '_canDelete___'];
    return (delete obj[name]) || (fail('not deleted: ', name), false);
  }

  ////////////////////////////////////////////////////////////////////////
  // Other
  ////////////////////////////////////////////////////////////////////////

  /**
   * When a <tt>this</tt> value must be provided but nothing is
   * suitable, provide this useless object instead.
   */
  var USELESS = Token('USELESS');

  /**
   * A call to cajita.manifest(data) is dynamically ignored, but if the
   * data expression is valid static JSON text, its value is made
   * statically available to the module loader.
   * <p>
   * TODO(erights): Find out if this is still the plan.
   */
  function manifest(ignored) {}

  /**
   * All the extra fields observed in Error objects on any supported
   * browser which seem to carry possibly-useful diagnostic info.
   * <p>
   * By "extra", we means any fields other that those already
   * accessible to cajoled code, namely <tt>name</tt> and
   * <tt>message</tt>. 
   */
  var stackInfoFields = [
    'stack', 'fileName', 'lineNumer', // Seen in FF 3.0.3
    // fileName, lineNumber also seen in Rhino 1.7r1
    'description', // Seen in IE 6.0.2900, but seems identical to "message"
    'stackTrace', // Seen on Opera 9.51 after enabling
                  // "opera:config#UserPrefs|Exceptions Have Stacktrace"
    'sourceURL', 'line' // Seen on Safari 3.1.2
  ];

  /**
   * If given an Error in which hidden diagnostic info may be found,
   * return a record in which that diagnostic info is available to
   * cajoled code. 
   * <p>
   * This is so named because it used to be implemented as the
   * unsealer of a sealer/unsealer pair. TODO(erights) consider
   * renaming and deprecating the current name.
   */
  function callStackUnsealer(ex) {
    if (ex && isInstanceOf(ex, Error)) {
      var stackInfo = {};
      var numStackInfoFields = stackInfoFields.length;
      for (var i = 0; i < numStackInfoFields; i++) {
        var k = stackInfoFields[i];
        if (k in ex) { stackInfo[k] = ex[k]; }
      }
      if ('cajitaStack___' in ex) {
        // Set by cajita-debugmode.js
        stackInfo.cajitaStack = ex.cajitaStack___;
      }
      return primFreeze(stackInfo);
    }
    return void 0;
  }

  /**
   * Receives whatever was caught by a user defined try/catch block.
   *
   * @param ex A value caught in a try block.
   * @return The value to make available to the cajoled catch block.
   */
  function tameException(ex) {
    if (ex && ex.UNCATCHABLE___) { throw ex; }
    try {
      switch (typeOf(ex)) {
        case 'string':
        case 'number':
        case 'boolean': 
        case 'undefined': {
          // Immutable.
          return ex;
        }
        case 'object': {
          if (ex === null) { return null; }
          if (ex.throwable___) { return ex; }
          if (isInstanceOf(ex, Error)) { return primFreeze(ex); }
          return '' + ex;
        }
        case 'function': {
          // According to Pratap Lakhsman's "JScript Deviations" S2.11
          // If the caught object is a function, calling it within the catch
          // supplies the head of the scope chain as the "this value".  The
          // called function can add properties to this object.  This implies
          // that for code of this shape:
          //     var x;
          //     try {
          //       // ...
          //     } catch (E) {
          //       E();
          //       return s;
          //     }
          // The reference to 'x' within the catch is not necessarily to the
          // local declaration of 'x'; this gives Catch the same performance
          // problems as with.

          // We return a different, powerless function instead.
          var name = '' + (ex.name || ex);
          function inLieuOfThrownFunction() {
            return 'In lieu of thrown function: ' + name;
          };
          return markFuncFreeze(inLieuOfThrownFunction, name);
        }
        default: {
          log('Unrecognized exception type: ' + (typeOf(ex)));
          return 'Unrecognized exception type: ' + (typeOf(ex));
        }
      }
    } catch (_) {
      // Can occur if coercion to string fails, or if ex has getters
      // that fail. This function must never throw an exception
      // because doing so would cause control to leave a catch block
      // before the handler fires.
      log('Exception during exception handling.');
      return 'Exception during exception handling.';
    }
  }

  /**
   * Makes a new empty object that directly inherits from <tt>proto</tt>.
   */
  function primBeget(proto) {
    if (proto === null) { fail('Cannot beget from null.'); }
    if (proto === (void 0)) { fail('Cannot beget from undefined.'); }
    function F() {}
    F.prototype = proto;
    var result = new F();
    result.proto___ = proto;
    return result;
  }

  /**
   * Creates a well formed Cajita record from a list of alternating
   * keys and values. 
   * <p>
   * The translator translates Cajita object literals into calls to
   * <tt>initializeMap</tt> so that a potentially toxic function
   * cannot be made the <tt>toString</tt> property of even a temporary
   * object. 
   */
  function initializeMap(list) {
    var result = {};
    for (var i = 0; i < list.length; i += 2) {
      // Call asFirstClass() here to prevent, for example, a toxic
      // function being used as the toString property of an object
      // literal.
      setPub(result, list[i], asFirstClass(list[i + 1]));
    }
    return result;
  }

  ////////////////////////////////////////////////////////////////////////
  // Taming mechanism
  ////////////////////////////////////////////////////////////////////////

  /**
   * Arrange to handle read-faults on <tt>obj[name]</tt>
   * by calling <tt>getHandler()</tt> as a method on
   * the faulted object.
   * <p>
   * In order for this fault-handler to get control, it's important
   * that no one does a conflicting <tt>grantRead()</tt>.
   * FIXME(ben): and fastpathRead()?
   */
  function useGetHandler(obj, name, getHandler) {
    obj[name + '_getter___'] = getHandler;
  }

  /**
   * Arrange to handle call-faults on <tt>obj[name](args...)</tt> by
   * calling <tt>applyHandler(args)</tt> as a method on the faulted
   * object.
   * <p>
   * Note that <tt>applyHandler</tt> is called with a single argument,
   * which is the list of arguments in the original call.
   * <p>
   * In order for this fault-handler to get control, it's important
   * that no one does a conflicting grantCall() or other grants which
   * imply grantCall().
   * FIXME(ben): also fastpath?
   */
  function useApplyHandler(obj, name, applyHandler) {
    obj[name + '_handler___'] = applyHandler;
  }

  /**
   * Arrange to handle call-faults on <tt>obj[name](args...)</tt> by
   * calling <tt>callHandler(args...)</tt> as a method on the faulted
   * object.
   * <p>
   * Note that <tt>callHandler</tt> is called with the same arguments
   * as the original call.
   * <p>
   * In order for this fault-handler to get control, it's important
   * that no one does a conflicting grantCall() or other grants which
   * imply grantCall().
   * FIXME(ben): also fastpath?
   */
  function useCallHandler(obj, name, callHandler) {
    useApplyHandler(obj, name, function callApplier(args) {
      return callHandler.apply(this, args);
    });
  }

  /**
   * Arrange to handle set-faults on <tt>obj[name] = newValue</tt> by
   * calling <tt>setHandler(newValue)</tt> as a method on the faulted
   * object.
   * <p>
   * In order for this fault-handler to get control, it's important
   * that no one does a conflicting grantSet().
   * FIXME(ben): also fastpath?
   */
  function useSetHandler(obj, name, setHandler) {
    obj[name + '_setter___'] = setHandler;
  }

  /**
   * Arrange to handle delete-faults on <tt>delete obj[name]</tt> by
   * calling <tt>deleteHandler()</tt> as a method on the faulted object.
   * <p>
   * In order for this fault-handler to get control, it's important
   * that no one does a conflicting grantDelete().
   * FIXME(ben): also fastpath?
   */
  function useDeleteHandler(obj, name, deleteHandler) {
    obj[name + '_deleter___'] = deleteHandler;
  }

  /**
   * Whilelist obj[name] as a simple frozen function that can be either
   * called or read.
   */
  function grantFunc(obj, name) {
    markFuncFreeze(obj[name], name);
    grantCall(obj, name);
    grantRead(obj, name);
  }

  /**
   * Whitelist proto[name] as a generic exophoric function that can
   * safely be called with its <tt>this</tt> bound to other objects.
   * <p>
   * Since exophoric functions are not first-class, reading
   * proto[name] returns the corresponding pseudo-function -- a record
   * with simple-functions for its call, bind, and apply.
   */
  function grantGenericMethod(proto, name) {
    var func = markXo4a(proto[name], name);
    grantCall(proto, name);
    var pseudoFunc = tame(func);
    useGetHandler(proto, name, function xo4aGetter() {
      return pseudoFunc;
    });
  }

  /**
   * Use func as a virtual generic exophoric function.
   * <p>
   * Since exophoric functions are not first-class, reading
   * proto[name] returns the corresponding pseudo-function -- a record
   * with simple-functions for its call, bind, and apply.
   */
  function handleGenericMethod(obj, name, func) {
    var feral = obj[name];
    if (!hasOwnProp(obj, name)) {
      // TODO(erights): domita would currently generate this warning,
      // and generated warnings currently cause DomitaTest to fail. It
      // is not clear whether the right solution is to make domita not
      // generate these warnings or to not consider the triggering
      // condition to deserve a warning. In any case, DomitaTest
      // should probably become warning tolerant.
      //log('warning: possible taming mistake: (' + obj + ')[' + name + ']');
      feral = func;
    } else if (hasOwnProp(feral, 'TAMED_TWIN___')) {
      // TODO(erights): See above TODO note.
      //log('warning: already tamed: (' + obj + ')[' + name + ']');
      feral = func;      
    }
    useCallHandler(obj, name, func);
    var pseudoFunc = tameXo4a.call(func);
    tamesTo(feral, pseudoFunc);
    useGetHandler(obj, name, function genericGetter() {
      return pseudoFunc;
    });
  }

  /**
   * Virtually replace proto[name] with a fault-handler
   * wrapper that first verifies that <tt>this</tt> inherits from
   * proto.
   * <p>
   * When a pre-existing Javascript method may do something unsafe
   * when applied to a <tt>this</tt> of the wrong type, we need to
   * provide a fault-handler instead to prevent such mis-application.
   * <p>
   * In order for this fault handler to get control, it's important
   * that no one does an grantCall() or other grants which imply
   * grantCall().
   * FIXME(ben): also fastpath?
   */
  function grantTypedMethod(proto, name) {
    var original = proto[name];
    handleGenericMethod(proto, name, function guardedApplier(var_args) {
      if (!inheritsFrom(this, proto)) {
        fail("Can't call .", name, ' on a non ',
             directConstructor(proto), ': ', this);
      }
      return original.apply(this, arguments);
    });
  }

  /**
   * Virtually replace proto[name] with a fault-handler
   * wrapper that first verifies that <tt>this</tt> isn't frozen.
   * <p>
   * When a pre-existing Javascript method would mutate its object,
   * we need to provide a fault handler instead to prevent such
   * mutation from violating Cajita semantics.
   * <p>
   * In order for this fault handler to get control, it's important
   * that no one does an grantCall() or other grants which imply
   * grantCall().
   * FIXME(ben): also fastpath?
   */
  function grantMutatingMethod(proto, name) {
    var original = proto[name];
    handleGenericMethod(proto, name, function nonMutatingApplier(var_args) {
      if (isFrozen(this)) {
        fail("Can't .", name, ' a frozen object');
      }
      return original.apply(this, arguments);
    });
  }

  /**
   * Virtually replace proto[name] with a fault-handler wrapper under
   * the assumption that the original is a generic innocent method.
   * <p>
   * As an innocent method, we assume it is exophoric (uses its
   * <tt>this</tt> parameter), requires a feral <tt>this</tt> and
   * arguments, and returns a feral result. As a generic method, we
   * assume that its <tt>this</tt> may be bound to objects that do not
   * inherit from <tt>proto</tt>.
   * <p>
   * The wrapper will untame <tt>this</tt>. Note that typically
   * <tt>this</tt> will be a constructed object and so will untame to
   * itself. The wrapper will also untame the arguments and tame and
   * return the result.
   */
  function grantInnocentMethod(proto, name) {
    var original = proto[name];
    handleGenericMethod(proto, name, function guardedApplier(var_args) {
      // like tameApplyFuncWrapper() but restated to avoid triple wrapping.
      var feralThis = stopEscalation(untame(this));
      var feralArgs = untame(Array.slice(arguments, 0));
      var feralResult = original.apply(feralThis, feralArgs);
      return tame(feralResult);
    });
  }

  /**
   * Verifies that regexp is something that can appear as a
   * parameter to a Javascript method that would use it in a match.
   * <p>
   * If it is a RegExp, then this match might mutate it, which must
   * not be allowed if regexp is frozen. Otherwise it must be a string.
   */
  function enforceMatchable(regexp) {
    if (isInstanceOf(regexp, RegExp)) {
      if (isFrozen(regexp)) {
        fail("Can't match with frozen RegExp: ", regexp);
      }
    } else {
      enforceType(regexp, 'string');
    }
  }

  /**
   * A shorthand that happens to be useful here.
   * <p>
   * For all i in arg2s: func2(arg1,arg2s[i]).
   */
  function all2(func2, arg1, arg2s) {
    var len = arg2s.length;
    for (var i = 0; i < len; i += 1) {
      func2(arg1, arg2s[i]);
    }
  }

  ////////////////////////////////////////////////////////////////////////
  // Taming decisions
  ////////////////////////////////////////////////////////////////////////

  /// Math

  all2(grantRead, Math, [
    'E', 'LN10', 'LN2', 'LOG2E', 'LOG10E', 'PI', 'SQRT1_2', 'SQRT2'
  ]);
  all2(grantFunc, Math, [
    'abs', 'acos', 'asin', 'atan', 'atan2', 'ceil', 'cos', 'exp', 'floor',
    'log', 'max', 'min', 'pow', 'random', 'round', 'sin', 'sqrt', 'tan'
  ]);

  /// toString

  function grantToString(proto) {
    proto.TOSTRING___ = tame(markXo4a(proto.toString, 'toString'));
  }

  function makeToStringMethod(toStringValue) {
    function toStringMethod(var_args) {
      var args = Array.slice(arguments, 0);
      if (isFunc(toStringValue)) {
        return toStringValue.apply(this, args);
      }
      var toStringValueApply = readPub(toStringValue, 'apply');
      if (isFunc(toStringValueApply)) {
        return toStringValueApply.call(toStringValue, this, args);
      }
      var result = myOriginalToString.call(this);
      log('Not correctly printed: ' + result);
      return result;
    };
    return toStringMethod;
  }

  function toStringGetter() {
    if (hasOwnProp(this, 'toString') &&
        typeOf(this.toString) === 'function' &&
        !hasOwnProp(this, 'TOSTRING___')) {
      grantToString(this);
    }
    return this.TOSTRING___;
  }
  useGetHandler(Object.prototype, 'toString',
                toStringGetter);

  useApplyHandler(Object.prototype, 'toString',
                  function toStringApplier(args) {
    var toStringValue = toStringGetter.call(this);
    return makeToStringMethod(toStringValue).apply(this, args);
  });

  useSetHandler(Object.prototype, 'toString',
                function toStringSetter(toStringValue) {
    if (isFrozen(this) || !isJSONContainer(this)) {
      return myKeeper.handleSet(this, 'toString', toStringValue);
    }
    var firstClassToStringValue = asFirstClass(toStringValue);
    this.TOSTRING___ = firstClassToStringValue;
    this.toString = makeToStringMethod(firstClassToStringValue);
    return toStringValue;
  });

  useDeleteHandler(Object.prototype, 'toString',
                   function toStringDeleter() {
    if (isFrozen(this) || !isJSONContainer(this)) {
      return myKeeper.handleDelete(this, 'toString');
    }
    return (delete this.toString) && (delete this.TOSTRING___);
  });

  /// Object

  markCtor(Object, void 0, 'Object');
  Object.prototype.TOSTRING___ = tame(markXo4a(function() {
    if (this.CLASS___) {
      return '[object ' + this.CLASS___ + ']';
    } else {
      return myOriginalToString.call(this);
    }
  }, 'toString'));
  all2(grantGenericMethod, Object.prototype, [
    'toLocaleString', 'valueOf', 'isPrototypeOf'
  ]);
  grantRead(Object.prototype, 'length');
  handleGenericMethod(Object.prototype, 'hasOwnProperty',
                      function hasOwnPropertyHandler(name) {
    return hasOwnPropertyOf(this, name);
  });
  handleGenericMethod(Object.prototype, 'propertyIsEnumerable',
                      function propertyIsEnumerableHandler(name) {
    name = String(name);
    return canEnumPub(this, name);
  });
  useCallHandler(Object, 'freeze', markFuncFreeze(freeze));
  useGetHandler(Object, 'freeze', function(){return freeze;});

  /// Function

  grantToString(Function.prototype);
  handleGenericMethod(Function.prototype, 'apply',
                      function applyHandler(self, opt_args) {
    return toFunc(this).apply(USELESS, opt_args || []);
  });
  handleGenericMethod(Function.prototype, 'call',
                      function callHandler(self, var_args) {
    return toFunc(this).apply(USELESS, Array.slice(arguments, 1));
  });
  handleGenericMethod(Function.prototype, 'bind',
                      function bindHandler(self, var_args) {
    var thisFunc = toFunc(this);
    var leftArgs = Array.slice(arguments, 1);
    function boundHandler(var_args) {
      var args = leftArgs.concat(Array.slice(arguments, 0));
      return thisFunc.apply(USELESS, args);
    }
    return markFuncFreeze(boundHandler);
  });
  useGetHandler(Function.prototype, 'caller', poisonFuncCaller);
  useGetHandler(Function.prototype, 'arguments', poisonFuncArgs);

  /// Array

  markCtor(Array, Object, 'Array');
  grantFunc(Array, 'slice');
  grantToString(Array.prototype);
  all2(grantTypedMethod, Array.prototype, [ 'toLocaleString' ]);
  all2(grantGenericMethod, Array.prototype, [
    'concat', 'join', 'slice', 'indexOf', 'lastIndexOf'
  ]);
  all2(grantMutatingMethod, Array.prototype, [
    'pop', 'push', 'reverse', 'shift', 'splice', 'unshift'
  ]);
  handleGenericMethod(Array.prototype, 'sort',
                      function sortHandler(comparator) {
    if (isFrozen(this)) {
      fail("Can't sort a frozen array.");
    }
    if (comparator) {
      return Array.prototype.sort.call(this, toFunc(comparator));
    } else {
      return Array.prototype.sort.call(this);
    }
  });

  /// String

  markCtor(String, Object, 'String');
  grantFunc(String, 'fromCharCode');
  grantToString(String.prototype);
  all2(grantTypedMethod, String.prototype, [
    'indexOf', 'lastIndexOf'
  ]);
  all2(grantGenericMethod, String.prototype, [
    'charAt', 'charCodeAt', 'concat',
    'localeCompare', 'slice', 'substr', 'substring',
    'toLowerCase', 'toLocaleLowerCase', 'toUpperCase', 'toLocaleUpperCase'
  ]);

  handleGenericMethod(String.prototype, 'match',
                      function matchHandler(regexp) {
    enforceMatchable(regexp);
    return this.match(regexp);
  });
  handleGenericMethod(String.prototype, 'replace',
                      function replaceHandler(searcher, replacement) {
    enforceMatchable(searcher);
    if (isFunc(replacement)) {
      replacement = asFunc(replacement);
    } else if (isPseudoFunc(replacement)) {
      replacement = toFunc(replacement);
    } else {
      replacement = '' + replacement;
    }
    return this.replace(searcher, replacement);
  });
  handleGenericMethod(String.prototype, 'search',
                      function searchHandler(regexp) {
    enforceMatchable(regexp);
    return this.search(regexp);
  });
  handleGenericMethod(String.prototype, 'split',
                      function splitHandler(separator, limit) {
    enforceMatchable(separator);
    return this.split(separator, limit);
  });

  /// Boolean

  markCtor(Boolean, Object, 'Boolean');
  grantToString(Boolean.prototype);

  /// Number

  markCtor(Number, Object, 'Number');
  all2(grantRead, Number, [
    'MAX_VALUE', 'MIN_VALUE', 'NaN',
    'NEGATIVE_INFINITY', 'POSITIVE_INFINITY'
  ]);
  grantToString(Number.prototype);
  all2(grantTypedMethod, Number.prototype, [
    'toLocaleString', 'toFixed', 'toExponential', 'toPrecision'
  ]);

  /// Date

  markCtor(Date, Object, 'Date');
  grantFunc(Date, 'parse');
  grantFunc(Date, 'UTC');
  grantToString(Date.prototype);
  all2(grantTypedMethod, Date.prototype, [
    'toDateString','toTimeString', 'toUTCString',
    'toLocaleString', 'toLocaleDateString', 'toLocaleTimeString',
    'toISOString', 'toJSON',
    'getDay', 'getUTCDay', 'getTimezoneOffset',

    'getTime', 'getFullYear', 'getUTCFullYear', 'getMonth', 'getUTCMonth',
    'getDate', 'getUTCDate', 'getHours', 'getUTCHours',
    'getMinutes', 'getUTCMinutes', 'getSeconds', 'getUTCSeconds',
    'getMilliseconds', 'getUTCMilliseconds'
  ]);
  all2(grantMutatingMethod, Date.prototype, [
    'setTime', 'setFullYear', 'setUTCFullYear', 'setMonth', 'setUTCMonth',
    'setDate', 'setUTCDate', 'setHours', 'setUTCHours',
    'setMinutes', 'setUTCMinutes', 'setSeconds', 'setUTCSeconds',
    'setMilliseconds', 'setUTCMilliseconds'
  ]);

  /// RegExp

  markCtor(RegExp, Object, 'RegExp');
  grantToString(RegExp.prototype);
  handleGenericMethod(RegExp.prototype, 'exec',
                      function execHandler(specimen) {
    if (isFrozen(this)) {
      fail("Can't .exec a frozen RegExp");
    }
    specimen = String(specimen); // See bug 528
    return this.exec(specimen);
  });
  handleGenericMethod(RegExp.prototype, 'test',
                      function testHandler(specimen) {
    if (isFrozen(this)) {
      fail("Can't .test a frozen RegExp");
    }
    specimen = String(specimen); // See bug 528
    return this.test(specimen);
  });

  all2(grantRead, RegExp.prototype, [
    'source', 'global', 'ignoreCase', 'multiline', 'lastIndex'
  ]);

  /// errors

  markCtor(Error, Object, 'Error');
  grantToString(Error.prototype);
  grantRead(Error.prototype, 'name');
  grantRead(Error.prototype, 'message');
  markCtor(EvalError, Error, 'EvalError');
  markCtor(RangeError, Error, 'RangeError');
  markCtor(ReferenceError, Error, 'ReferenceError');
  markCtor(SyntaxError, Error, 'SyntaxError');
  markCtor(TypeError, Error, 'TypeError');
  markCtor(URIError, Error, 'URIError');


  var sharedImports;

  ////////////////////////////////////////////////////////////////////////
  // Module loading
  ////////////////////////////////////////////////////////////////////////

  var myNewModuleHandler;

  /**
   * Gets the current module handler.
   */
  function getNewModuleHandler() {
    return myNewModuleHandler;
  }

  /**
   * Registers a new-module-handler, to be called back when a new
   * module is loaded.
   * <p>
   * This callback mechanism is provided so that translated Cajita
   * modules can be loaded from a trusted site with the
   * &lt;script&gt; tag, which runs its script as a statement, not
   * an expression. The callback is of the form
   * <tt>newModuleHandler.handle(newModule)</tt>.
   */
  function setNewModuleHandler(newModuleHandler) {
    myNewModuleHandler = newModuleHandler;
  }

  /**
   * A new-module-handler which returns the new module without
   * instantiating it.
   */
  var obtainNewModule = freeze({
    handle: markFuncFreeze(function handleOnly(newModule){ return newModule; })
  });

  function registerClosureInspector(module) {
    if (this && this.CLOSURE_INSPECTOR___ 
        && this.CLOSURE_INSPECTOR___.supportsCajaDebugging) {
      this.CLOSURE_INSPECTOR___.registerCajaModule(module);
    }
  }

  /**
   * Makes and returns a fresh "normal" module handler whose imports
   * are initialized to a copy of the sharedImports.
   * <p>
   * This handles a new module by calling it, passing it the imports
   * object held in this handler. Successive modules handled by the
   * same "normal" handler thereby see a simulation of successive
   * updates to a shared global scope.
   */
  function makeNormalNewModuleHandler() {
    var imports = void 0;
    var lastOutcome = void 0;
    function getImports() {
      if (!imports) { imports = copy(sharedImports); }
      return imports;
    }
    return freeze({
      getImports: markFuncFreeze(getImports),
      setImports: markFuncFreeze(function setImports(newImports) {
        imports = newImports;
      }),

      /**
       * An outcome is a pair of a success flag and a value.
       * <p>
       * If the success flag is true, then the value is the normal
       * result of calling the module function. If the success flag is
       * false, then the value is the thrown error by which the module
       * abruptly terminated.
       * <p>
       * An html page is cajoled to a module that runs to completion,
       * but which reports as its outcome the outcome of its last
       * script block. In order to reify that outcome and report it
       * later, the html page initializes moduleResult___ to
       * NO_RESULT, the last script block is cajoled to set
       * moduleResult___ to something other than NO_RESULT on success
       * but to call handleUncaughtException() on
       * failure, and the html page returns moduleResult___ on
       * completion. handleUncaughtException() records a failed
       * outcome. This newModuleHandler's handle() method will not
       * overwrite an already reported outcome with NO_RESULT, so the
       * last script-block's outcome will be preserved.
       */
      getLastOutcome: markFuncFreeze(function getLastOutcome() {
        return lastOutcome;
      }),

      /**
       * If the last outcome is a success, returns its value;
       * otherwise <tt>undefined</tt>.
       */
      getLastValue: markFuncFreeze(function getLastValue() {
        if (lastOutcome && lastOutcome[0]) {
          return lastOutcome[1];
        } else {
          return void 0;
        }
      }),

      /**
       * Runs the newModule's module function.
       * <p>
       * Updates the last outcome to report the module function's
       * reported outcome. Propagate this outcome by terminating in
       * the same manner.
       */
      handle: markFuncFreeze(function handle(newModule) {
        registerClosureInspector(newModule);
        var outcome = void 0;
        try {
          var result = newModule.instantiate(___, getImports());
          if (result !== NO_RESULT) {
            outcome = [true, result];
          }
        } catch (ex) {
          outcome = [false, ex];
        }
        lastOutcome = outcome;
        if (outcome) {
          if (outcome[0]) {
            return outcome[1];
          } else {
            throw outcome[1];
          }
        } else {
          return void 0;
        }
      }),

      /**
       * This emulates HTML5 exception handling for scripts as discussed at
       * http://code.google.com/p/google-caja/wiki/UncaughtExceptionHandling
       * and see HtmlCompiler.java for the code that calls this.
       * @param exception a raw exception.  Since {@code throw} can raise any
       *   value, exception could be any value accessible to cajoled code, or
       *   any value thrown by an API imported by cajoled code.
       * @param onerror the value of the raw reference "onerror" in top level
       *   cajoled code.  This will likely be undefined much of the time, but
       *   could be anything.  If it is a func, it can be called with
       *   three strings (message, source, lineNum) as the
       *   {@code window.onerror} event handler.
       * @param {string} source a URI describing the source file from which the
       *   error originated.
       * @param {string} lineNum the approximate line number in source at which
       *   the error originated.
       */
      handleUncaughtException: function handleUncaughtException(exception,
                                                                onerror,
                                                                source,
                                                                lineNum) {
        lastOutcome = [false, exception];

        // Cause exception to be rethrown if it is uncatchable.
        var message = tameException(exception);
        if ('object' === typeOf(exception) && exception !== null) {
          message = String(exception.message || exception.desc || message);
        }

        // If we wanted to provide a hook for containers to get uncaught
        // exceptions, it would go here before onerror is invoked.

        // See the HTML5 discussion for the reasons behind this rule.
        if (isPseudoFunc(onerror)) { onerror = toFunc(onerror); }
        var shouldReport = (
            isFunc(onerror)
            ? onerror.CALL___(message, String(source), String(lineNum))
            : onerror !== null);
        if (shouldReport !== false) {
          log(source + ':' + lineNum + ': ' + message);
        }
      }
    });
  }

  /**
   * Produces a function module given an object literal module
   */
  function prepareModule(module, load) {
    registerClosureInspector(module);
    function theModule(imports) {
      // The supplied 'imports' contain arguments supplied by the caller of the
      // module. We need to add the primordials (Array, Object, ...) to these
      // before invoking the Cajita module.
      var completeImports = copy(sharedImports);
      completeImports.load = load;
      // Copy all properties, including Cajita-unreadable ones since these may
      // be used by privileged code.
      var k;
      for (k in imports) {
        if (hasOwnProp(imports, k)) { completeImports[k] = imports[k]; }
      }
      return module.instantiate(___, primFreeze(completeImports));
    }
    theModule.FUNC___ = 'theModule';

    // Whitelist certain module properties as visible to Cajita code. These
    // are all primitive values that do not allow two Cajita entities with
    // access to the same module object to communicate.
    setStatic(theModule, 'cajolerName', module.cajolerName);
    setStatic(theModule, 'cajolerVersion', module.cajolerVersion);
    setStatic(theModule, 'cajoledDate', module.cajoledDate);
    setStatic(theModule, 'moduleURL', module.moduleURL);
    // The below is a transitive freeze because includedModules is an array
    // of strings.
    if (!!module.includedModules) {
      setStatic(theModule, 'includedModules',
                ___.freeze(module.includedModules));
    }

    return primFreeze(theModule);
  }

  /**
   * A module is an object literal containing metadata and an
   * <code>instantiate</code> member, which is a plugin-maker function.
   * <p>
   * loadModule(module) marks module's <code>instantiate</code> member as a
   * func, freezes the module, asks the current new-module-handler to handle it
   * (thereby notifying the handler), and returns the new module.
   */
  function loadModule(module) {
    freeze(module);
    markFuncFreeze(module.instantiate);
    return callPub(myNewModuleHandler, 'handle', [module]);
  }

  var registeredImports = [];

  /**
   * Gets or assigns the id associated with this (assumed to be)
   * imports object, registering it so that
   * <tt>getImports(getId(imports)) === imports</tt>.
   * <p>
   * This system of registration and identification allows us to
   * cajole html such as
   * <pre>&lt;a onmouseover="alert(1)"&gt;Mouse here&lt;/a&gt;</pre>
   * into html-writing JavaScript such as<pre>
   * ___IMPORTS___.document.innerHTML = "
   *  &lt;a onmouseover=\"
   *    (function(___IMPORTS___) {
   *      ___IMPORTS___.alert(1);
   *    })(___.getImports(" + ___.getId(___IMPORTS___) + "))
   *  \"&gt;Mouse here&lt;/a&gt;
   * ";
   * </pre>
   * If this is executed by a plugin whose imports is assigned id 42,
   * it generates html with the same meaning as<pre>
   * &lt;a onmouseover="___.getImports(42).alert(1)"&gt;Mouse here&lt;/a&gt;
   * </pre>
   * <p>
   * An imports is not registered and no id is assigned to it until the
   * first call to <tt>getId</tt>. This way, an imports that is never
   * registered, or that has been <tt>unregister</tt>ed since the last
   * time it was registered, will still be garbage collectable.
   */
  function getId(imports) {
    enforceType(imports, 'object', 'imports');
    var id;
    if ('id___' in imports) {
      id = enforceType(imports.id___, 'number', 'id');
    } else {
      id = imports.id___ = registeredImports.length;
    }
    registeredImports[id] = imports;
    return id;
  }

  /**
   * Gets the imports object registered under this id.
   * <p>
   * If it has been <tt>unregistered</tt> since the last
   * <tt>getId</tt> on it, then <tt>getImports</tt> will fail.
   */
  function getImports(id) {
    var result = registeredImports[enforceType(id, 'number', 'id')];
    if (result === void 0) {
      fail('imports#', id, ' unregistered');
    }
    return result;
  }

  /**
   * If you know that this <tt>imports</tt> no longer needs to be
   * accessed by <tt>getImports</tt>, then you should
   * <tt>unregister</tt> it so it can be garbage collected.
   * <p>
   * After unregister()ing, the id is not reassigned, and the imports
   * remembers its id. If asked for another <tt>getId</tt>, it
   * reregisters itself at its old id.
   */
  function unregister(imports) {
    enforceType(imports, 'object', 'imports');
    if ('id___' in imports) {
      var id = enforceType(imports.id___, 'number', 'id');
      registeredImports[id] = void 0;
    }
  }


  ////////////////////////////////////////////////////////////////////////
  // Guards and Trademarks
  ////////////////////////////////////////////////////////////////////////

  /**
   * The identity function just returns its argument.
   */
  function identity(x) { return x; }

  /**
   * One-arg form is known in scheme as "call with escape
   * continuation" (call/ec), and is the semantics currently 
   * proposed for EcmaScript Harmony's "return to label".
   * 
   * <p>In this analogy, a call to <tt>callWithEjector</tt> emulates a
   * labeled statement. The ejector passed to the <tt>attemptFunc</tt>
   * emulates the label part. The <tt>attemptFunc</tt> itself emulates
   * the statement being labeled. And a call to <tt>eject</tt> with
   * this ejector emulates the return-to-label statement.
   * 
   * <p>We extend the normal notion of call/ec with an
   * <tt>opt_failFunc</tt> in order to give more the sense of a
   * <tt>try/catch</tt> (or similarly, the <tt>escape</tt> special
   * form in E). The <tt>attemptFunc</tt> is like the <tt>try</tt>
   * clause and the <tt>opt_failFunc</tt> is like the <tt>catch</tt>
   * clause. If omitted, <tt>opt_failFunc</tt> defaults to the
   * <tt>identity</tt> function. 
   * 
   * <p><tt>callWithEjector</tt> creates a fresh ejector -- a one
   * argument function -- for exiting from this attempt. It then calls
   * <tt>attemptFunc</tt> passing that ejector as argument. If
   * <tt>attemptFunc</tt> completes without calling the ejector, then
   * this call to <tt>callWithEjector</tt> completes
   * likewise. Otherwise, if the ejector is called with an argument,
   * then <tt>opt_failFunc</tt> is called with that argument. The
   * completion of <tt>opt_failFunc</tt> is then the completion of the
   * <tt>callWithEjector</tt> as a whole.
   * 
   * <p>The ejector stays live until <tt>attemptFunc</tt> is exited,
   * at which point the ejector is disabled. Calling a disabled
   * ejector throws.
   * 
   * <p>In order to emulate the semantics I expect of ES-Harmony's
   * return-to-label and to prevent the reification of the internal
   * token thrown in order to emulate call/ec, <tt>tameException</tt>
   * immediately rethrows this token, preventing Cajita and Valija
   * <tt>catch</tt> clauses from catching it. However,
   * <tt>finally</tt> clauses will still be run while unwinding an
   * ejection. If these do their own non-local exit, that takes
   * precedence over the ejection in progress but leave the ejector
   * live.
   * 
   * <p>Historic note: This was first invented by John C. Reynolds in 
   * <a href="http://doi.acm.org/10.1145/800194.805852"
   * >Definitional interpreters for higher-order programming 
   * languages</a>. Reynold's invention was a special form as in E, 
   * rather than a higher order function as here and in call/ec.
   */
  function callWithEjector(attemptFunc, opt_failFunc) {
    var failFunc = opt_failFunc || identity;
    var disabled = false;
    var token = new Token('ejection');
    token.UNCATCHABLE___ = true;
    var stash = void 0;
    function ejector(result) {
      if (disabled) {
        cajita.fail('ejector disabled');
      } else {
        // don't disable here.
        stash = result;
        throw token;
      }
    }
    markFuncFreeze(ejector);
    try {
      try {
        return callPub(attemptFunc, 'call', [USELESS, ejector]);
      } finally {
        disabled = true;
      }
    } catch (e) {
      if (e === token) {
        return callPub(failFunc, 'call', [USELESS, stash]);
      } else {
        throw e;
      }
    }
  }

  /**
   * Safely invokes <tt>opt_ejector</tt> with <tt>result</tt>.
   * <p>
   * If <tt>opt_ejector</tt> is falsy, disabled, or returns
   * normally, then <tt>eject</tt> throws. Under no conditions does
   * <tt>eject</tt> return normally.
   */
  function eject(opt_ejector, result) {
    if (opt_ejector) {
      callPub(opt_ejector, 'call', [USELESS, result]);
      fail('Ejector did not exit: ', opt_ejector);
    } else {
      fail(result);
    }
  }
  
  /**
   * Internal routine for making a trademark from a table.
   * <p>
   * To untangle a cycle, the guard made by <tt>makeTrademark</tt> is
   * not yet either stamped or frozen. The caller of
   * <tt>makeTrademark</tt> must do both before allowing it to
   * escape. 
   */
  function makeTrademark(typename, table) {
    typename = String(typename);
    return primFreeze({
      toString: markFuncFreeze(function() { return typename + 'Mark'; }),

      stamp: primFreeze({
        toString: markFuncFreeze(function() { return typename + 'Stamp'; }),
        mark___: markFuncFreeze(function(obj) {
          table.set(obj, true);
          return obj;
        })
      }),

      guard: {
        toString: markFuncFreeze(function() { return typename + 'T'; }),
        coerce: markFuncFreeze(function(specimen, opt_ejector) {
          if (table.get(specimen)) { return specimen; }
          eject(opt_ejector,
                'Specimen does not have the "' + typename + '" trademark');
        })
      }
    });
  }

  /**
   * Objects representing guards should be marked as such, so that
   * they will pass the <tt>GuardT</tt> guard.
   * <p>
   * <tt>GuardT</tt> is generally accessible as
   * <tt>cajita.GuardT</tt>. However, <tt>GuardStamp</tt> must not be
   * made generally accessible, but rather only given to code trusted
   * to use it to deem as guards things that act in a guard-like
   * manner: A guard MUST be immutable and SHOULD be idempotent. By
   * "idempotent", we mean that<pre>
   *     var x = g(specimen, ej); // may fail
   *     // if we're still here, then without further failure
   *     g(x) === x
   * </pre>
   */
  var GuardMark = makeTrademark('Guard', newTable(true));
  var GuardT = GuardMark.guard;
  var GuardStamp = GuardMark.stamp;
  primFreeze(GuardStamp.mark___(GuardT));  

  /**
   * The <tt>Trademark</tt> constructor makes a trademark, which is a
   * guard/stamp pair, where the stamp marks and freezes unfrozen
   * records as carrying that trademark and the corresponding guard
   * cerifies objects as carrying that trademark (and therefore as
   * having been marked by that stamp).
   * <p>
   * By convention, a guard representing the type-like concept 'Foo'
   * is named 'FooT'. The corresponding stamp is 'FooStamp'. And the
   * record holding both is 'FooMark'. Many guards also have
   * <tt>of</tt> methods for making guards like themselves but
   * parameterized by further constraints, which are usually other
   * guards. For example, <tt>T.ListT</tt> is the guard representing
   * frozen array, whereas <tt>T.ListT.of(cajita.GuardT)</tt>
   * represents frozen arrays of guards.
   */
  function Trademark(typename) {
    var result = makeTrademark(typename, newTable(true));
    primFreeze(GuardStamp.mark___(result.guard));
    return result;
  }
  markFuncFreeze(Trademark);

  /**
   * First ensures that g is a guard; then does 
   * <tt>g.coerce(specimen, opt_ejector)</tt>.
   */
  function guard(g, specimen, opt_ejector) {
    g = GuardT.coerce(g); // failure throws rather than ejects
    return g.coerce(specimen, opt_ejector);
  }

  /**
   * First ensures that g is a guard; then checks whether the specimen
   * passes that guard.
   * <p>
   * If g is a coercing guard, this only checks that g coerces the
   * specimen to something rather than failing. Note that trademark
   * guards are non-coercing, so if specimen passes a trademark guard,
   * then specimen itself has been marked with that trademark.
   */
  function passesGuard(g, specimen) {
    g = GuardT.coerce(g); // failure throws rather than ejects
    return callWithEjector(
      markFuncFreeze(function(opt_ejector) {
        g.coerce(specimen, opt_ejector);
        return true;
      }),
      markFuncFreeze(function(ignored) {
        return false;
      })
    );
  }

  /**
   * Given that <tt>stamps</tt> is a list of stamps and
   * <tt>record</tt> is a non-frozen record, this marks record with
   * the trademarks of all of these stamps, and then freezes and
   * returns the record.
   * <p>
   * If any of these conditions do not hold, this throws.
   */
  function stamp(stamps, record) {
    if (!isRecord(record)) {
      fail('Can only stamp records: ', record);
    }
    if (isFrozen(record)) {
      fail("Can't stamp frozen objects: ", record);
    }
    var numStamps = stamps.length >>> 0;
    // First ensure that we will succeed before applying any stamps to
    // the record. If we ever extend Cajita with mutating getters, we
    // will need to do more to ensure impossibility of failure after
    // partial stamping.
    for (var i = 0; i < numStamps; i++) {
      if (!('mark___' in stamps[i])) {
        fail("Can't stamp with a non-stamp: ", stamps[i]);
      }
    }
    // Looping again over the same untrusted stamps alleged-array is safe
    // assuming single-threaded execution and non-mutating accessors.
    // If we extend Cajita to allow getters/setters, we'll need to make a 
    // copy of the array above and loop over the copy below.
    for (var i = 0; i < numStamps; i++) {
      // Only works for real stamps, postponing the need for a
      // user-implementable auditing protocol.
      stamps[i].mark___(record);
    }
    return freeze(record);
  }

  ////////////////////////////////////////////////////////////////////////
  // Sealing and Unsealing
  ////////////////////////////////////////////////////////////////////////

  /**
   * Returns a pair of functions such that the seal(x) wraps x in an object
   * so that only unseal can get x back from the object.
   * <p>
   * TODO(erights): The only remaining use as of this writing is
   * in domita for css. Perhaps a refactoring is in order.
   *
   * @return {object} of the form
   *     { seal: function seal(x) { return Token('(box)'); },
   *       unseal: function unseal(box) { return x; } }.
   */
  function makeSealerUnsealerPair() {
    var table = newTable(true);
    var undefinedStandin = {};
    function seal(payload) {
      if (payload === void 0) {
        payload = undefinedStandin;
      }
      var box = Token('(box)');
      table.set(box, payload);
      return box;
    }
    function unseal(box) {
      var payload = table.get(box);
      if (payload === void 0) {
        fail('Sealer/Unsealer mismatch'); 
      } else if (payload === undefinedStandin) {
        return void 0;
      } else {
        return payload;
      }
    }
    return freeze({
      seal: markFuncFreeze(seal),
      unseal: markFuncFreeze(unseal)
    });
  }

  ////////////////////////////////////////////////////////////////////////
  // Needed for Valija
  ////////////////////////////////////////////////////////////////////////

  /**
   * <tt>cajita.construct(ctor, [args...])</tt> invokes a constructor
   * or simple function as a constructor using 'new'.
   */
  function construct(ctor, args) {
    ctor = asCtor(ctor);
    // This works around problems with (new Array()) and (new Date()) where
    // the returned object is not really a Date or Array on SpiderMonkey and
    // other interpreters.
    switch (args.length) {
      case 0:  return new ctor();
      case 1:  return new ctor(args[0]);
      case 2:  return new ctor(args[0], args[1]);
      case 3:  return new ctor(args[0], args[1], args[2]);
      case 4:  return new ctor(args[0], args[1], args[2], args[3]);
      case 5:  return new ctor(args[0], args[1], args[2], args[3], args[4]);
      case 6:  return new ctor(args[0], args[1], args[2], args[3], args[4],
                               args[5]);
      case 7:  return new ctor(args[0], args[1], args[2], args[3], args[4],
                               args[5], args[6]);
      case 8:  return new ctor(args[0], args[1], args[2], args[3], args[4],
                               args[5], args[6], args[7]);
      case 9:  return new ctor(args[0], args[1], args[2], args[3], args[4],
                               args[5], args[6], args[7], args[8]);
      case 10: return new ctor(args[0], args[1], args[2], args[3], args[4],
                               args[5], args[6], args[7], args[8], args[9]);
      case 11: return new ctor(args[0], args[1], args[2], args[3], args[4],
                               args[5], args[6], args[7], args[8], args[9],
                               args[10]);
      case 12: return new ctor(args[0], args[1], args[2], args[3], args[4],
                               args[5], args[6], args[7], args[8], args[9],
                               args[10], args[11]);
      default:
        if (ctor.typeTag___ === 'Array') {
          return ctor.apply(USELESS, args);
        }
        var tmp = function (args) {
          return ctor.apply(this, args);
        };
        tmp.prototype = ctor.prototype;
        return new tmp(args);
    }
  }

  /**
   * Create a unique identification of a given table identity that can
   * be used to invisibly (to Cajita code) annotate a key object to
   * index into a table.
   * <p>
   * magicCount and MAGIC_TOKEN together represent a
   * unique-across-frames value safe against collisions, under the
   * normal Caja threat model assumptions. magicCount and
   * MAGIC_NAME together represent a probably unique across frames
   * value, with which can generate strings in which collision is
   * unlikely but possible.
   * <p>
   * The MAGIC_TOKEN is a unique unforgeable per-Cajita runtime
   * value. magicCount is a per-Cajita counter, which increments each
   * time a new one is needed.
   */
  var magicCount = 0;
  var MAGIC_NUM = Math.random();
  var MAGIC_TOKEN = Token('MAGIC_TOKEN_FOR:' + MAGIC_NUM);
  // Using colons in the below causes it to fail on IE since getting a
  // property whose name contains a colon on a DOM table element causes
  // an exception.
  var MAGIC_NAME = '_index;'+ MAGIC_NUM + ';';

  /**
   * 
   * Creates a new mutable associative table mapping from the
   * identity of arbitrary keys (as defined by tt>identical()</tt>) to
   * arbitrary values.
   * <p>
   * Operates as specified by <a href=
   * "http://wiki.ecmascript.org/doku.php?id=strawman:weak_references#ephemeron_tables"
   * >ephemeron tables</a>, including the "Implementation
   * Considerations" section regarding emulation on ES3, except that,
   * when <tt>opt_useKeyLifetime</tt> is falsy or absent, the keys
   * here may be primitive types as well. 
   * <p>
   * To support Domita, the keys might be host objects.
   */
  function newTable(opt_useKeyLifetime, opt_expectedSize) {
    magicCount++;
    var myMagicIndexName = MAGIC_NAME + magicCount + '___';

    function setOnKey(key, value) {
      var ktype = typeof key;
      if (!key || (ktype !== 'function' && ktype !== 'object')) { 
        fail("Can't use key lifetime on primitive keys: ", key);
      }
      var list = key[myMagicIndexName];
      // To distinguish key from objects that derive from it,
      //    list[0] should be === key
      // For odd positive i,
      //    list[i] is the MAGIC_TOKEN for a Cajita runtime (i.e., a
      //            browser frame in which the Cajita runtime has been
      //            loaded). The myMagicName and the MAGIC_TOKEN
      //            together uniquely identify a table.
      //    list[i+1] is the value stored in that table under this key.
      if (!list || list[0] !== key) {
        key[myMagicIndexName] = [key, MAGIC_TOKEN, value];
      } else {
        var i;
        for (i = 1; i < list.length; i += 2) {
          if (list[i] === MAGIC_TOKEN) { break; }
        }
        list[i] = MAGIC_TOKEN;
        list[i + 1] = value;
      }
    }

    function getOnKey(key) {
      var ktype = typeof key;
      if (!key || (ktype !== 'function' && ktype !== 'object')) { 
        fail("Can't use key lifetime on primitive keys: ", key);
      }
      var list = key[myMagicIndexName];
      if (!list || list[0] !== key) {
        return void 0;
      } else {
        for (var i = 1; i < list.length; i += 2) {
          if (list[i] === MAGIC_TOKEN) { return list[i + 1]; }
        }
        return void 0;
      }
    }

    if (opt_useKeyLifetime) {
      return primFreeze({
        set: markFuncFreeze(setOnKey),
        get: markFuncFreeze(getOnKey)
      });
    }

    var myValues = [];

    function setOnTable(key, value) {
      var index;
      switch (typeof key) {
        case 'object':
        case 'function': {
          if (null === key) { myValues.prim_null = value; return; }
          index = getOnKey(key);
          if (value === void 0) {
            if (index === void 0) {
              return;
            } else {
              setOnKey(key, void 0);
            }
          } else {
            if (index === void 0) {
              index = myValues.length;
              setOnKey(key, index);
            }
          }
          break;
        }
        case 'string': {
          index = 'str_' + key;
          break;
        }
        default: { 
          index = 'prim_' + key;
          break; 
        }
      }
      if (value === void 0) {
        // TODO(erights): Not clear that this is the performant
        // thing to do when index is numeric and < length-1.
        delete myValues[index];
      } else {
        myValues[index] = value;
      }
    }

    /**
     * If the key is absent, returns <tt>undefined</tt>.
     * <p>
     * Users of this table cannot distinguish an <tt>undefined</tt>
     * value from an absent key.
     */
    function getOnTable(key) {
      switch (typeof key) {
        case 'object':
        case 'function': {
          if (null === key) { return myValues.prim_null; }
          var index = getOnKey(key);
          if (void 0 === index) { return void 0; }
          return myValues[index];
        }
        case 'string': { return myValues['str_' + key]; }
        default: { return myValues['prim_' + key]; }
      }
    }

    return primFreeze({
      set: markFuncFreeze(setOnTable),
      get: markFuncFreeze(getOnTable)
    });
  }


  /**
   * Is <tt>allegedParent</tt> on <obj>'s prototype chain?
   * <p>
   * Although in raw JavaScript <tt>'foo' instanceof String</tt> is
   * false, to reduce the differences between primitives and their
   * wrappers, <tt>inheritsFrom('foo', String.prototype)</tt> is true.
   */
  function inheritsFrom(obj, allegedParent) {
    if (null === obj) { return false; }
    if (void 0 === obj) { return false; }
    if (typeOf(obj) === 'function') { return false; }
    if (typeOf(allegedParent) !== 'object') { return false; }
    if (null === allegedParent) { return false; }
    function F() {}
    F.prototype = allegedParent;
    return Object(obj) instanceof F;
  }

  /**
   * Return func.prototype's directConstructor.
   * <p>
   * When following the "classical" inheritance pattern (simulating
   * class-style inheritance as a pattern of prototypical
   * inheritance), func may represent (the constructor of) a class; in
   * which case getSuperCtor() returns (the constructor of) its
   * immediate superclass.
   */
  function getSuperCtor(func) {
    enforceType(func, 'function');
    if (isCtor(func) || isFunc(func)) {
      var result = directConstructor(func.prototype);
      if (isCtor(result) || isFunc(result)) {
        return result;
      }
    }
    return void 0;
  }

  var attribute = new RegExp(
      '^([\\s\\S]*)_(?:canRead|canCall|getter|handler)___$');

  /**
   * Returns a list of all cajita-readable own properties, whether or
   * not they are cajita-enumerable.
   */
  function getOwnPropertyNames(obj) {
    var result = [];
    var seen = {};
    // TODO(erights): revisit once we do ES5ish attribute control.
    var implicit = isJSONContainer(obj);
    for (var k in obj) {
      if (hasOwnProp(obj, k)) {
        if (implicit && !endsWith__.test(k)) {
          if (!myOriginalHOP.call(seen, k)) {
            seen[k] = true;
            result.push(k);
          }
        } else {
          var match = attribute.exec(k);
          if (match !== null) {
            var base = match[1];
            if (!myOriginalHOP.call(seen, base)) {
              seen[base] = true;
              result.push(base);
            }
          }
        }
      }
    }
    return result;
  }

  /**
   * Return the names of the accessible own properties of
   * func.prototype.
   * <p>
   * Since prototypical objects are not themselves accessible in
   * Cajita, this means in effect: the properties contributed by
   * func.prototype that would be accessible on objects that inherit
   * from func.prototype.
   */
  function getProtoPropertyNames(func) {
    enforceType(func, 'function');
    return getOwnPropertyNames(func.prototype);
  }

  /**
   * Return the value associated with func.prototype[name].
   * <p>
   * Since prototypical objects are not themselves accessible in
   * Cajita, this means in effect: If x inherits name from
   * func.prototype, what would the value of x[name] be? If the value
   * associated with func.prototype[name] is an exophoric function
   * (resulting from taming a generic method), then return the
   * corresponding pseudo-function.
   */
  function getProtoPropertyValue(func, name) {
    return asFirstClass(readPub(func.prototype, name));
  }

  /**
   * Like primBeget(), but applicable only to records.
   */
  function beget(parent) {
    if (!isRecord(parent)) {
      fail('Can only beget() records: ', parent);
    }
    var result = primBeget(parent);
    result.RECORD___ = result;
    return result;
  }

  ////////////////////////////////////////////////////////////////////////
  // JSON
  ////////////////////////////////////////////////////////////////////////

  function jsonParseOk(json) {
    try {
      var x = json.parse('{"a":3}');
      return x.a === 3;
    } catch (e) {
      return false;
    }
  }

  function jsonStringifyOk(json) {
    try {
      var x = json.stringify({"a":3, "b__":4}, function replacer(k, v) {
        return (/__$/.test(k) ? void 0 : v);
      });
      if (x !== '{"a":3}') {
        return false;
      }
      // ie8 has a buggy JSON unless this update has been applied:
      //   http://support.microsoft.com/kb/976662
      // test for one of the known bugs.
      x = json.stringify(void 0, 'invalid');
      return x === void 0;
    } catch (e) {
      return false;
    }
  }

  var goodJSON = {};
  goodJSON.parse = jsonParseOk(global.JSON) ?
    global.JSON.parse : json_sans_eval.parse;
  goodJSON.stringify = jsonStringifyOk(global.JSON) ?
    global.JSON.stringify : json_sans_eval.stringify;

  safeJSON = primFreeze({
    CLASS___: 'JSON',
    parse: markFuncFreeze(function (text, opt_reviver) {
      var reviver = void 0;
      if (opt_reviver) {
        opt_reviver = toFunc(opt_reviver);
        reviver = function (key, value) {
          return opt_reviver.apply(this, arguments);
        };
      }
      return goodJSON.parse(
          json_sans_eval.checkSyntax(text, function (key) {
            return (key !== 'valueOf' && key !== 'toString'
                    && !endsWith__.test(key));
          }), reviver);
    }),
    stringify: markFuncFreeze(function (obj, opt_replacer, opt_space) {
      switch (typeof opt_space) {
        case 'number': case 'string': case 'undefined': break;
        default: throw new TypeError('space must be a number or string');
      }
      var replacer;
      if (opt_replacer) {
        opt_replacer = toFunc(opt_replacer);
        replacer = function (key, value) {
          if (!canReadPub(this, key)) { return void 0; }
          return opt_replacer.apply(this, arguments);
        };
      } else {
        replacer = function (key, value) {
          return (canReadPub(this, key)) ? value : void 0;
        };
      }
      return goodJSON.stringify(obj, replacer, opt_space);
    })
  });

  ////////////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////////////

  cajita = {
    // Diagnostics and condition enforcement
    log: log,
    fail: fail,
    enforce: enforce,
    enforceType: enforceType,

    // walking prototype chain, checking JSON containers
    directConstructor: directConstructor,
    getFuncCategory: getFuncCategory,
    isDirectInstanceOf: isDirectInstanceOf,
    isInstanceOf: isInstanceOf,
    isRecord: isRecord,           isArray: isArray,
    isJSONContainer: isJSONContainer,
    freeze: freeze,               isFrozen: isFrozen,
    copy: copy,                   snapshot: snapshot,

    // Accessing properties
    canReadPub: canReadPub,       readPub: readPub,
    hasOwnPropertyOf: hasOwnPropertyOf,
                                  readOwn: readOwn,
    canEnumPub: canEnumPub,
    canEnumOwn: canEnumOwn,
    canInnocentEnum: canInnocentEnum,
    BREAK: BREAK,
    allKeys: allKeys,             forAllKeys: forAllKeys,
    ownKeys: ownKeys,             forOwnKeys: forOwnKeys,
    canCallPub: canCallPub,       callPub: callPub,
    canSetPub: canSetPub,         setPub: setPub,
    canDeletePub: canDeletePub,   deletePub: deletePub,

    // Object indistinguishability and object-keyed tables
    Token: Token,
    identical: identical,
    newTable: newTable,

    // Guards and Trademarks
    identity: identity,
    callWithEjector: callWithEjector,
    eject: eject,
    GuardT: GuardT,
    Trademark: Trademark,
    guard: guard,
    passesGuard: passesGuard,
    stamp: stamp,

    // Sealing & Unsealing
    makeSealerUnsealerPair: makeSealerUnsealerPair,

    // Other
    USELESS: USELESS,
    manifest: manifest,

    // Needed for Valija
    args: args,
    construct: construct,
    inheritsFrom: inheritsFrom,
    getSuperCtor: getSuperCtor,
    getOwnPropertyNames: getOwnPropertyNames,
    getProtoPropertyNames: getProtoPropertyNames,
    getProtoPropertyValue: getProtoPropertyValue,
    beget: beget,

    PseudoFunctionProto: PseudoFunctionProto,
    PseudoFunction: PseudoFunction,
    isPseudoFunc: isPseudoFunc,

    // deprecated
    enforceNat: deprecate(enforceNat, '___.enforceNat',
                          'Use (x === x >>> 0) instead as a UInt32 test')
  };

  forOwnKeys(cajita, markFuncFreeze(function(k, v) {
    switch (typeOf(v)) {
      case 'object': {
        if (v !== null) { primFreeze(v); }
        break;
      }
      case 'function': {
        markFuncFreeze(v);
        break;
      }
    }
  }));

  sharedImports = {
    cajita: cajita,

    'null': null,
    'false': false,
    'true': true,
    'NaN': NaN,
    'Infinity': Infinity,
    'undefined': void 0,
    parseInt: markFuncFreeze(parseInt),
    parseFloat: markFuncFreeze(parseFloat),
    isNaN: markFuncFreeze(isNaN),
    isFinite: markFuncFreeze(isFinite),
    decodeURI: markFuncFreeze(decodeURI),
    decodeURIComponent: markFuncFreeze(decodeURIComponent),
    encodeURI: markFuncFreeze(encodeURI),
    encodeURIComponent: markFuncFreeze(encodeURIComponent),
    escape: escape ? markFuncFreeze(escape) : (void 0),
    Math: Math,
    JSON: safeJSON,

    Object: Object,
    Array: Array,
    String: String,
    Boolean: Boolean,
    Number: Number,
    Date: Date,
    RegExp: RegExp,

    Error: Error,
    EvalError: EvalError,
    RangeError: RangeError,
    ReferenceError: ReferenceError,
    SyntaxError: SyntaxError,
    TypeError: TypeError,
    URIError: URIError
  };

  forOwnKeys(sharedImports, markFuncFreeze(function(k, v) {
    switch (typeOf(v)) {
      case 'object': {
        if (v !== null) { primFreeze(v); }
        break;
      }
      case 'function': {
        primFreeze(v);
        break;
      }
    }
  }));
  primFreeze(sharedImports);

  ___ = {
    // Diagnostics and condition enforcement
    getLogFunc: getLogFunc,
    setLogFunc: setLogFunc,

    primFreeze: primFreeze,

    // Accessing property attributes.
    canRead: canRead,             grantRead: grantRead,
    canEnum: canEnum,             grantEnum: grantEnum,
    canCall: canCall,        
    canSet: canSet,               grantSet: grantSet,
    canDelete: canDelete,         grantDelete: grantDelete,

    // Module linkage
    readImport: readImport,

    // Classifying functions
    isCtor: isCtor,
    isFunc: isFunc,
    markCtor: markCtor,           extend: extend,
    markFuncFreeze: markFuncFreeze,
    markXo4a: markXo4a,           markInnocent: markInnocent,
    asFunc: asFunc,               toFunc: toFunc,

    // Accessing properties
    inPub: inPub,
    canSetStatic: canSetStatic,   setStatic: setStatic,

    // Other
    typeOf: typeOf,
    hasOwnProp: hasOwnProp,
    deleteFieldEntirely: deleteFieldEntirely,
    tameException: tameException,
    primBeget: primBeget,
    callStackUnsealer: callStackUnsealer,
    RegExp: RegExp,  // Available to rewrite rule w/o risk of masking
    GuardStamp: GuardStamp,
    asFirstClass: asFirstClass,
    initializeMap: initializeMap,
    iM: initializeMap,

    // Taming mechanism
    useGetHandler: useGetHandler,
    useSetHandler: useSetHandler,

    grantFunc: grantFunc,
    grantGenericMethod: grantGenericMethod,
    handleGenericMethod: handleGenericMethod,
    grantTypedMethod: grantTypedMethod,
    grantMutatingMethod: grantMutatingMethod,
    grantInnocentMethod: grantInnocentMethod,

    enforceMatchable: enforceMatchable,
    all2: all2,

    tamesTo: tamesTo,
    tamesToSelf: tamesToSelf,
    tame: tame,
    untame: untame,

    // Module loading
    getNewModuleHandler: getNewModuleHandler,
    setNewModuleHandler: setNewModuleHandler,
    obtainNewModule: obtainNewModule,
    makeNormalNewModuleHandler: makeNormalNewModuleHandler,
    loadModule: loadModule,
    prepareModule: prepareModule,
    NO_RESULT: NO_RESULT,

    getId: getId,
    getImports: getImports,
    unregister: unregister,

    // Deprecated
    grantEnumOnly: deprecate(grantEnum, '___.grantEnumOnly',
                             'Use ___.grantEnum instead.'),
    grantCall: deprecate(grantGenericMethod, '___.grantCall',
                         'Choose a method tamer (e.g., ___.grantFunc,' +
                         '___.grantGenericMethod,etc) according to the ' +
                         'safety properties of calling and reading the ' +
                         'method.'),
    grantGeneric: deprecate(grantGenericMethod, '___.grantGeneric',
                            'Use ___.grantGenericMethod instead.'),
    useApplyHandler: deprecate(useApplyHandler, '___.useApplyHandler',
                               'Use ___.handleGenericMethod instead.'),
    useCallHandler: deprecate(useCallHandler, '___.useCallHandler',
                              'Use ___.handleGenericMethod instead.'),
    handleGeneric: deprecate(useCallHandler, '___.handleGeneric',
                              'Use ___.handleGenericMethod instead.'),
    grantTypedGeneric: deprecate(useCallHandler, '___.grantTypedGeneric',
                              'Use ___.grantTypedMethod instead.'),
    grantMutator: deprecate(useCallHandler, '___.grantMutator',
                              'Use ___.grantMutatingMethod instead.'),
    useDeleteHandler: deprecate(useDeleteHandler, '___.useDeleteHandler',
                                'Refactor to avoid needing to handle ' +
                                'deletions.'),
    isXo4aFunc: deprecate(isXo4aFunc, '___.isXo4aFunc',
                          'Refactor to avoid needing to dynamically test ' +
                          'whether a function is marked exophoric.'),
    xo4a: deprecate(markXo4a, '___.xo4a',
                    'Consider refactoring to avoid needing to explicitly ' +
                    'mark a function as exophoric. Use one of the exophoric ' +
                    'method tamers (e.g., ___.grantGenericMethod) instead.' +
                    'Otherwise, use ___.markXo4a instead.'),
    ctor: deprecate(markCtor, '___.ctor',
                    'Use ___.markCtor instead.'),
    func: deprecate(markFuncFreeze, '___.func',
                    '___.func should not be called ' +
                    'from manually written code.'),
    frozenFunc: deprecate(markFuncFreeze, '___.frozenFunc',
                          'Use ___.markFuncFreeze instead.'),
    markFuncOnly: deprecate(markFuncFreeze, '___.markFuncOnly',
                            '___.markFuncOnly should not be called ' +
                            'from manually written code.'),

    // Taming decisions
    sharedImports: sharedImports
  };

  forOwnKeys(cajita, markFuncFreeze(function(k, v) {
    if (k in ___) {
      fail('internal: initialization conflict: ', k);
    }
    if (typeOf(v) === 'function') {
      grantFunc(cajita, k);
    }
    ___[k] = v;
  }));
  setNewModuleHandler(makeNormalNewModuleHandler());
})(this);
;
// Copyright (C) 2008 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @fileoverview decorates definitions from <tt>cajita.js</tt> to collect
 * debugging information at runtime.
 *
 * <h3>Usage</h3>
 * Currently a container loads cajita.js {@script <script src="cajita.js"/>}
 * to provide library support for cajoled code.
 * Containers will likely provide a sandbox to let developers to test their
 * gadgets.  This container can include both cajita.js and this file:<pre>
 *   <script src="cajita.js"/>
 *   <script src="cajita-debugmode.js"/>
 * </pre>.
 * TODO(mikesamuel): how to collect logging.
 *
 * <h3>Changes</h3>
 * This extends the {@code ___} object to maintain a stack of call sites.
 * It adds an operation, {@code ___.getCallerStack}, which returns the caller
 * stack, including special stack frames for delete/get/read/set operations.
 * <p>
 * In debug mode, the normal fasttracking is turned off, so all property
 * accessing/modifying operations go through {@code ___} methods.
 *
 * <h3>Interleaved operations</h3>
 * Interleaved operations, as from an XMLHTTPRequest's onReadyStateChange
 * handler or a cross-frame event, can corrupt the stack.  We try to detect
 * these on popping a stack frame, and mark the stack invalid.
 * <p>
 * {@code ___.startCallerStack} can be invoked by event handlers to set the
 * stack to a valid state.  If the stack is already valid, it's left alone.
 *
 * @author mikesamuel@gmail.com
 * @requires cajita, console
 * @overrides ___
 */

(function () {
  var orig = cajita.copy(___);

  function mixin(obj, members) {
    for (var i = 0, n = members.length; i < n; i += 2) {
      var k = members[i], v = members[i + 1];
      if (k in obj) { throw new Error('overriding ' + k + ' in ' + v); }
      obj[k] = v;
    }
  }

  function override_members(obj, members, extraOptionalParams) {
    for (var i = 0, n = members.length; i < n; i += 2) {
      var k = members[i], v = members[i + 1];
      if (!(k in obj)) { throw new Error('can\'t override ' + k + ' in ' + v); }
      if (obj[k] === v) continue;
      if (extraOptionalParams !== null
          && obj[k].length + extraOptionalParams !== v.length) {
        throw new Error('overriding ' + k + ' with a different signature');
      }
      obj[k] = v;
    }
  }

  // Define the stack, and accessors
  var stack;
  var stackInvalid = true;

  /**
   * Start the caller stack if it hasn't been started already.  Called by
   * event handler, {@code setTimeout}, and {@code setInterval} wrappers
   */
  function startCallerStack() {
    if (!stack || stackInvalid) {
      stack = [];
      stackInvalid = false;
    }
  }

  /**
   * Returns a call stack that will not be mutated by subsequent
   * changes.
   */
  function getCallerStack() {
    return stackInvalid ? void 0 : cajita.freeze(stack.slice(0));
  }

  function pushFrame(stackFrame) {
    stack.push(stackFrame);
    return stackFrame;
  }

  function popFrame(stackFrame) {
    // If we're not popping what we expect, something weird has screwed up
    // our push/pop symmetry, so we mark the stack invalid.
    var top = stack.length - 1;
    if (top >= 0 && stackFrame === stack[top]) {
      stack.length = top;
    } else {
      stackInvalid = true;
    }
  }

  function rethrowWith(ex, stackFrame) {
    pushFrame(stackFrame);
    attachCajaStack(ex);
    popFrame(stackFrame);
    throw ex;
  }

  function requireObject(obj, callerIdx) {
    switch (typeof obj) {
      case 'object':
        if (obj !== null) { return obj; }
        break;
      case 'function':
        return obj;
    }
    rethrowWith(new Error('Expected object not ' + obj),
                this.debugSymbols_[callerIdx]);
  }


  // Make sure that object accessors and mutators update the stack, and others
  // that can fail if obj is undefined or name is denied.

  /**
   * Associate the cajita call stack with an Error object if there is none there
   * already.
   */
  function attachCajaStack(ex) {
    // Associate the current stack with ex if it is an Error.
    if (ex && ex instanceof Error && !ex.cajitaStack___) {
      ex.cajitaStack___ = getCallerStack();
    }
  }

  function errorDecorator(fn) {
    var arity = fn.length;
    return function (var_args) {
      try {
        return fn.apply(this, arguments);
      } catch (ex) {
        rethrowWith(ex, this.debugSymbols_[arguments[arity]]);
      }
    };
  }

  function callPub(obj, name, args, callerIdx) {
    var stackFrame = pushFrame(this.debugSymbols_[callerIdx]);
    try {
      try {
        return orig.callPub.apply(this, arguments);
      } catch (ex) {
        attachCajaStack(ex);
        throw ex;
      }
    } finally {
      popFrame(stackFrame);
    }
  }

  function asFunc(fun, callerIdx) {
    return makeWrapper(
        fun, 'asFunc', this.debugSymbols_[callerIdx]);
  }

  function construct(fun, args, callerIdx) {
    var stackFrame = pushFrame(this.debugSymbols_[callerIdx]);
    try {
      try {
        return orig.construct(fun.callFn || fun, args);
      } catch (ex) {
        attachCajaStack(ex);
        throw ex;
      }
    } finally {
      popFrame(stackFrame);
    }
  }

  /**
   * Return a function of the same kind (simple/method/ctor) as fun, but
   * making sure that any Error thrown because fun is not of the required kind
   * has a stack attached.
   *
   * @param {Function} fun
   * @param {string} conditionName name of the condition that checks
   *     that fun is of the right kind.  E.g. 'asFunc'
   * @param stackFrame of the call of fun in original source code.
   * @return {Function} applies fun, but attaches a cajita stack trace to any
   *     Error raised by fun.
   */
  function makeWrapper(fun, conditionName, stackFrame) {
    try {
      fun = orig[conditionName](fun);
      if (!fun) { return fun; }
    } catch (ex) {
      rethrowWith(ex, stackFrame);
    }
    function wrapper() {
      pushFrame(stackFrame);
      try {
        try {
          return fun.apply(this, arguments);
        } catch (ex) {
          attachCajaStack(ex);
          throw ex;
        }
      } finally {
        popFrame(stackFrame);
      }
    }

    // fun might pass asCtor because it is simple.  Copy only the bits onto
    // wrapper that allow it to survive similar checks.
    if (fun.FUNC___) {
      wrapper.FUNC___ = fun.FUNC___;
    } else if (fun.XO4A___) {
      wrapper.XO4A___ = true;
    }

    return orig.primFreeze(wrapper);
  }

  function tameException(ex) {
    var tamedEx = orig.tameException(ex);
    // Make sure that tamed Errors propagate the cajitaStack___,
    // so that an exception can be rethrown.
    
    // We need to make sure tameException has the property that
    //     try { f(); } catch (ex) { throw ex; }
    // preserves stack information if it was captured by an earlier throw,
    // so it will be available to code with access to the unsealer.
    if (tamedEx && tamedEx instanceof Error && !tamedEx.cajitaStack___) {
      tamedEx.cajitaStack___ = getCallerStack();
    }
    return tamedEx;
  }

  // Extend to output the source file position with the message.
  var origLog = cajita.log;
  function log(msg) {
    if (!stackInvalid && stack.length > 0) {
      msg = stack[stack.length - 1] + ': ' + msg;
    }
    return origLog(msg);
  }

  // Dump stack traces during loading to the console.
  function loadModule(module) {
    cajita.log('starting loadModule');
    try {
      orig.loadModule(module);
      cajita.log('done loadModule');
    } catch (ex) {
      if ('undefined' !== typeof console) {
        if (ex && ex.cajitaStack___) {
          var stack = ex.cajitaStack___;
          if (stack) {
            console.group(
                ex.message + ' @ ' + ex.fileName + ':' + ex.lineNumber);
            console.error(stack.join('\n'));
            console.groupEnd();
          }
        } else if ('string' === typeof ex.stack) {
          console.log(ex.stack.match(/@\S*:\d+(?:\n|$)/g).join('\n\n'));
        } else {
          console.log('' + ex);
        }
      }
      throw ex;
    }
  }

  /**
   * Attach the stack to an exception before it is thrown from cajita code.
   * @param ex a value that cajita code is allowed to throw.
   */
  function userException(ex, callerIdx) {
    var stackFrame = pushFrame(this.debugSymbols_[callerIdx]);
    try {
      // TODO(mikesamuel): should userException be defined as identity in
      // cajita.js?  If so we should do ex = orig.userException(ex) inside this
      // try.
      // This would let us use userException to prevent user code from raising
      // InternalErrors.
      attachCajaStack(ex);
    } finally {
      popFrame(stackFrame);
    }
    return ex;
  }

  /**
   * Receive debugSymbols during module initialization, and set up the debugging
   * hooks for this module's version of ___.
   */
  function useDebugSymbols(var_args) {
    var newDebugSymbols = arguments;
    cajita.log('using debug symbols');
    if (!cajita.isJSONContainer(this)) { cajita.fail('called on bad ___'); }
    if (this.debugSymbols_ !== void 0) {
      cajita.log('___ reused with different debug symbols');
    }
    // Unpack the debugging symbols.

    // Per DebuggingSymbolsStage:
    //   The debugSymbols are a list of the form
    //       '[' <FilePosition> (',' <prefixLength> ',' <dFilePosition}>)* ']'
    //   where the dFilePositions are turned into FilePositions by
    //   prepending them with the first prefixLength characters of the
    //   preceding FilePosition.
    var debugSymbols = [];
    if (newDebugSymbols.length) {
      var last = newDebugSymbols[0];
      debugSymbols.push(last);
      for (var i = 1, n = newDebugSymbols.length; i < n; i += 2) {
        last = last.substring(0, newDebugSymbols[i]) + newDebugSymbols[i + 1];
        debugSymbols.push(last);
      }
    }
    this.debugSymbols_ = debugSymbols;

    // Maintain stack through calls, and attach a stack when an operation fails.
    override_members(
        this,
        [
         'asFunc', asFunc,
         'callPub', callPub,
         'construct', construct
        ], 1);
    override_members(
        this,
        [
         'canEnum', errorDecorator(orig.canEnum),
         'deletePub', errorDecorator(orig.deletePub),
         'inPub', errorDecorator(orig.inPub),
         'readPub', errorDecorator(orig.readPub),
         'setPub', errorDecorator(orig.setPub)
        ], null);
    // Make sure that tamed exceptions propagate stacktraces
    override_members(this, ['tameException', tameException], 0);
  }

  // Export useDebugSymbols and the rest of the debug API so that modules
  // compiled with debugging information can setup their ___, and so that
  // privileged exception handlers
  mixin(
      ___,
      [
       'getCallerStack', getCallerStack,
       'requireObject', requireObject,
       'startCallerStack', startCallerStack,
       'useDebugSymbols', useDebugSymbols,
       'userException', userException
      ]);

  // Include the top stack frame in log messages.
  override_members(cajita, ['log', ___.markFuncFreeze(log)], 0);
  // Dump stack traces during loading to the console.
  override_members(___, ['loadModule', ___.markFuncFreeze(loadModule)], 0);

  startCallerStack();
})();
;
// Copyright (C) 2007 Google Inc.
//      
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// .............................................................................

// If this module is loaded after cajita.js is loaded, and in an
// environment (such as produced by turning on Firebug) where
// <tt>console.log</tt> is a function, then it will register 
// a wrapper around <tt>console.log</tt> (or <tt>console.info</tt> 
// and <tt>console.error</tt> if available) using 
// <tt>___.setLogFunc()</tt>, so cajita.js will log its diagnostics
// to the Firebug console.

// If you load triv-logger.js and log-to-console.js into the same
// system, the last one loaded wins.

// This module is written in Javascript, not Caja, and would be
// rejected by the Caja translator. 


(function(global) {
  
  var console;

  if (global.___ && 
      (console = global.console) && 
      typeof console.log === 'function') {

    function logToConsole(str, opt_stop) {
      if (opt_stop && typeof console.error === 'function') {
        console.error(str);
      } else if (typeof console.info === 'function') {
        console.info(str);
      } else {
        console.log(str);
      }
      if (opt_stop) {
        // breakpoint here by uncommenting out the following line:
        debugger;
        // or by setting a breakpoint on this useless line:
        return;
      }
    };
    ___.setLogFunc(logToConsole);
  }

})(this);
;
// Copyright (C) 2008 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.


/**
 * @fileoverview
 * Unicode character classes.
 *
 * @see http://www.w3.org/TR/2000/REC-xml-20001006#CharClasses
 * @author mikesamuel@gmail.com
 * @provides unicode
 */


/** @namespace */
var unicode = {};

unicode.BASE_CHAR = (
    '\u0041-\u005A\u0061-\u007A\u00C0-\u00D6\u00D8-\u00F6\u00F8-\u00FF'
    + '\u0100-\u0131\u0134-\u013E\u0141-\u0148\u014A-\u017E\u0180-\u01C3'
    + '\u01CD-\u01F0\u01F4-\u01F5\u01FA-\u0217\u0250-\u02A8\u02BB-\u02C1'
    + '\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03CE\u03D0-\u03D6'
    + '\u03DA\u03DC\u03DE\u03E0\u03E2-\u03F3\u0401-\u040C\u040E-\u044F'
    + '\u0451-\u045C\u045E-\u0481\u0490-\u04C4\u04C7-\u04C8\u04CB-\u04CC'
    + '\u04D0-\u04EB\u04EE-\u04F5\u04F8-\u04F9\u0531-\u0556\u0559'
    + '\u0561-\u0586\u05D0-\u05EA\u05F0-\u05F2\u0621-\u063A\u0641-\u064A'
    + '\u0671-\u06B7\u06BA-\u06BE\u06C0-\u06CE\u06D0-\u06D3\u06D5'
    + '\u06E5-\u06E6\u0905-\u0939\u093D\u0958-\u0961\u0985-\u098C'
    + '\u098F-\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9'
    + '\u09DC-\u09DD\u09DF-\u09E1\u09F0-\u09F1\u0A05-\u0A0A\u0A0F-\u0A10'
    + '\u0A13-\u0A28\u0A2A-\u0A30\u0A32-\u0A33\u0A35-\u0A36\u0A38-\u0A39'
    + '\u0A59-\u0A5C\u0A5E\u0A72-\u0A74\u0A85-\u0A8B\u0A8D\u0A8F-\u0A91'
    + '\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2-\u0AB3\u0AB5-\u0AB9\u0ABD\u0AE0'
    + '\u0B05-\u0B0C\u0B0F-\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32-\u0B33'
    + '\u0B36-\u0B39\u0B3D\u0B5C-\u0B5D\u0B5F-\u0B61\u0B85-\u0B8A'
    + '\u0B8E-\u0B90\u0B92-\u0B95\u0B99-\u0B9A\u0B9C\u0B9E-\u0B9F'
    + '\u0BA3-\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB5\u0BB7-\u0BB9\u0C05-\u0C0C'
    + '\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C33\u0C35-\u0C39\u0C60-\u0C61'
    + '\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9'
    + '\u0CDE\u0CE0-\u0CE1\u0D05-\u0D0C\u0D0E-\u0D10\u0D12-\u0D28'
    + '\u0D2A-\u0D39\u0D60-\u0D61\u0E01-\u0E2E\u0E30\u0E32-\u0E33'
    + '\u0E40-\u0E45\u0E81-\u0E82\u0E84\u0E87-\u0E88\u0E8A\u0E8D'
    + '\u0E94-\u0E97\u0E99-\u0E9F\u0EA1-\u0EA3\u0EA5\u0EA7\u0EAA-\u0EAB'
    + '\u0EAD-\u0EAE\u0EB0\u0EB2-\u0EB3\u0EBD\u0EC0-\u0EC4\u0F40-\u0F47'
    + '\u0F49-\u0F69\u10A0-\u10C5\u10D0-\u10F6\u1100\u1102-\u1103'
    + '\u1105-\u1107\u1109\u110B-\u110C\u110E-\u1112\u113C\u113E\u1140'
    + '\u114C\u114E\u1150\u1154-\u1155\u1159\u115F-\u1161\u1163\u1165'
    + '\u1167\u1169\u116D-\u116E\u1172-\u1173\u1175\u119E\u11A8\u11AB'
    + '\u11AE-\u11AF\u11B7-\u11B8\u11BA\u11BC-\u11C2\u11EB\u11F0\u11F9'
    + '\u1E00-\u1E9B\u1EA0-\u1EF9\u1F00-\u1F15\u1F18-\u1F1D\u1F20-\u1F45'
    + '\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D'
    + '\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC'
    + '\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC'
    + '\u2126\u212A-\u212B\u212E\u2180-\u2182\u3041-\u3094\u30A1-\u30FA'
    + '\u3105-\u312C\uAC00-\uD7A3');
unicode.IDEOGRAPHIC = '\u4E00-\u9FA5\u3007\u3021-\u3029';
unicode.LETTER = unicode.BASE_CHAR + unicode.IDEOGRAPHIC;
unicode.COMBINING_CHAR = (
    '\u0300-\u0345\u0360-\u0361\u0483-\u0486\u0591-\u05A1\u05A3-\u05B9'
    + '\u05BB-\u05BD\u05BF\u05C1-\u05C2\u05C4\u064B-\u0652\u0670'
    + '\u06D6-\u06DC\u06DD-\u06DF\u06E0-\u06E4\u06E7-\u06E8\u06EA-\u06ED'
    + '\u0901-\u0903\u093C\u093E-\u094C\u094D\u0951-\u0954\u0962-\u0963'
    + '\u0981-\u0983\u09BC\u09BE\u09BF\u09C0-\u09C4\u09C7-\u09C8'
    + '\u09CB-\u09CD\u09D7\u09E2-\u09E3\u0A02\u0A3C\u0A3E\u0A3F'
    + '\u0A40-\u0A42\u0A47-\u0A48\u0A4B-\u0A4D\u0A70-\u0A71\u0A81-\u0A83'
    + '\u0ABC\u0ABE-\u0AC5\u0AC7-\u0AC9\u0ACB-\u0ACD\u0B01-\u0B03\u0B3C'
    + '\u0B3E-\u0B43\u0B47-\u0B48\u0B4B-\u0B4D\u0B56-\u0B57\u0B82-\u0B83'
    + '\u0BBE-\u0BC2\u0BC6-\u0BC8\u0BCA-\u0BCD\u0BD7\u0C01-\u0C03'
    + '\u0C3E-\u0C44\u0C46-\u0C48\u0C4A-\u0C4D\u0C55-\u0C56\u0C82-\u0C83'
    + '\u0CBE-\u0CC4\u0CC6-\u0CC8\u0CCA-\u0CCD\u0CD5-\u0CD6\u0D02-\u0D03'
    + '\u0D3E-\u0D43\u0D46-\u0D48\u0D4A-\u0D4D\u0D57\u0E31\u0E34-\u0E3A'
    + '\u0E47-\u0E4E\u0EB1\u0EB4-\u0EB9\u0EBB-\u0EBC\u0EC8-\u0ECD'
    + '\u0F18-\u0F19\u0F35\u0F37\u0F39\u0F3E\u0F3F\u0F71-\u0F84'
    + '\u0F86-\u0F8B\u0F90-\u0F95\u0F97\u0F99-\u0FAD\u0FB1-\u0FB7\u0FB9'
    + '\u20D0-\u20DC\u20E1\u302A-\u302F\u3099\u309A');
unicode.DIGIT = (
    '\u0030-\u0039\u0660-\u0669\u06F0-\u06F9\u0966-\u096F\u09E6-\u09EF'
    + '\u0A66-\u0A6F\u0AE6-\u0AEF\u0B66-\u0B6F\u0BE7-\u0BEF\u0C66-\u0C6F'
    + '\u0CE6-\u0CEF\u0D66-\u0D6F\u0E50-\u0E59\u0ED0-\u0ED9\u0F20-\u0F29');
unicode.EXTENDER = (
    '\u00B7\u02D0\u02D1\u0387\u0640\u0E46\u0EC6\u3005\u3031-\u3035'
    + '\u309D-\u309E\u30FC-\u30FE');
;
/* Copyright Google Inc.
 * Licensed under the Apache Licence Version 2.0
 * Autogenerated at Fri May 27 14:58:39 PDT 2011
 * @provides css
 */
var css = {
  'properties': (function () {
      var s = [ '|left|center|right', '|top|center|bottom',
        '#(?:[\\da-f]{3}){1,2}|aqua|black|blue|fuchsia|gray|green|lime|maroon|navy|olive|orange|purple|red|silver|teal|white|yellow|rgb\\(\\s*(?:-?\\d+|0|[+\\-]?\\d+(?:\\.\\d+)?%)\\s*,\\s*(?:-?\\d+|0|[+\\-]?\\d+(?:\\.\\d+)?%)\\s*,\\s*(?:-?\\d+|0|[+\\-]?\\d+(?:\\.\\d+)?%)\\)',
        '[+\\-]?\\d+(?:\\.\\d+)?(?:[cem]m|ex|in|p[ctx])',
        '\\d+(?:\\.\\d+)?(?:[cem]m|ex|in|p[ctx])',
        'none|hidden|dotted|dashed|solid|double|groove|ridge|inset|outset',
        '[+\\-]?\\d+(?:\\.\\d+)?%', '\\d+(?:\\.\\d+)?%',
        'url\\(\"[^()\\\\\"\\r\\n]+\"\\)',
        'repeat-x|repeat-y|(?:repeat|space|round|no-repeat)(?:\\s+(?:repeat|space|round|no-repeat)){0,2}'
      ], c = [ RegExp('^\\s*(?:\\s*(?:0|' + s[ 3 ] + '|' + s[ 6 ] +
          ')){1,2}\\s*$', 'i'), RegExp('^\\s*(?:\\s*(?:0|' + s[ 3 ] + '|' + s[
            6 ] + ')){1,4}(?:\\s*\\/(?:\\s*(?:0|' + s[ 3 ] + '|' + s[ 6 ] +
          ')){1,4})?\\s*$', 'i'), RegExp('^\\s*(?:\\s*none|(?:(?:\\s*(?:' + s[
            2 ] + ')\\s+(?:0|' + s[ 3 ] + ')(?:\\s*(?:0|' + s[ 3 ] +
          ')){1,4}(?:\\s*inset)?|(?:\\s*inset)?\\s+(?:0|' + s[ 3 ] +
          ')(?:\\s*(?:0|' + s[ 3 ] + ')){1,4}(?:\\s*(?:' + s[ 2 ] +
          '))?)\\s*,)*(?:\\s*(?:' + s[ 2 ] + ')\\s+(?:0|' + s[ 3 ] +
          ')(?:\\s*(?:0|' + s[ 3 ] +
          ')){1,4}(?:\\s*inset)?|(?:\\s*inset)?\\s+(?:0|' + s[ 3 ] +
          ')(?:\\s*(?:0|' + s[ 3 ] + ')){1,4}(?:\\s*(?:' + s[ 2 ] +
          '))?))\\s*$', 'i'), RegExp('^\\s*(?:' + s[ 2 ] +
          '|transparent|inherit)\\s*$', 'i'), RegExp('^\\s*(?:' + s[ 5 ] +
          '|inherit)\\s*$', 'i'), RegExp('^\\s*(?:thin|medium|thick|0|' + s[ 3
          ] + '|inherit)\\s*$', 'i'), RegExp('^\\s*(?:(?:thin|medium|thick|0|'
          + s[ 3 ] + '|' + s[ 5 ] + '|' + s[ 2 ] +
          '|transparent|inherit)(?:\\s+(?:thin|medium|thick|0|' + s[ 3 ] +
          ')|\\s+(?:' + s[ 5 ] +
          ')|\\s*#(?:[\\da-f]{3}){1,2}|\\s+aqua|\\s+black|\\s+blue|\\s+fuchsia|\\s+gray|\\s+green|\\s+lime|\\s+maroon|\\s+navy|\\s+olive|\\s+orange|\\s+purple|\\s+red|\\s+silver|\\s+teal|\\s+white|\\s+yellow|\\s+rgb\\(\\s*(?:-?\\d+|0|'
          + s[ 6 ] + ')\\s*,\\s*(?:-?\\d+|0|' + s[ 6 ] +
          ')\\s*,\\s*(?:-?\\d+|0|' + s[ 6 ] +
          ')\\)|\\s+transparent|\\s+inherit){0,2}|inherit)\\s*$', 'i'),
        /^\s*(?:none|inherit)\s*$/i, RegExp('^\\s*(?:' + s[ 8 ] +
          '|none|inherit)\\s*$', 'i'), RegExp('^\\s*(?:0|' + s[ 3 ] + '|' + s[
            6 ] + '|auto|inherit)\\s*$', 'i'), RegExp('^\\s*(?:0|' + s[ 4 ] +
          '|' + s[ 7 ] + '|none|inherit|auto)\\s*$', 'i'), RegExp('^\\s*(?:0|'
          + s[ 4 ] + '|' + s[ 7 ] + '|inherit|auto)\\s*$', 'i'),
        /^\s*(?:0(?:\.\d+)?|\.\d+|1(?:\.0+)?|inherit)\s*$/i,
        RegExp('^\\s*(?:(?:' + s[ 2 ] + '|invert|inherit|' + s[ 5 ] +
          '|thin|medium|thick|0|' + s[ 3 ] +
          ')(?:\\s*#(?:[\\da-f]{3}){1,2}|\\s+aqua|\\s+black|\\s+blue|\\s+fuchsia|\\s+gray|\\s+green|\\s+lime|\\s+maroon|\\s+navy|\\s+olive|\\s+orange|\\s+purple|\\s+red|\\s+silver|\\s+teal|\\s+white|\\s+yellow|\\s+rgb\\(\\s*(?:-?\\d+|0|'
          + s[ 6 ] + ')\\s*,\\s*(?:-?\\d+|0|' + s[ 6 ] +
          ')\\s*,\\s*(?:-?\\d+|0|' + s[ 6 ] +
          ')\\)|\\s+invert|\\s+inherit|\\s+(?:' + s[ 5 ] +
          '|inherit)|\\s+(?:thin|medium|thick|0|' + s[ 3 ] +
          '|inherit)){0,2}|inherit)\\s*$', 'i'), RegExp('^\\s*(?:' + s[ 2 ] +
          '|invert|inherit)\\s*$', 'i'),
        /^\s*(?:visible|hidden|scroll|auto|no-display|no-content)\s*$/i,
        RegExp('^\\s*(?:0|' + s[ 4 ] + '|' + s[ 7 ] + '|inherit)\\s*$', 'i'),
        /^\s*(?:auto|always|avoid|left|right|inherit)\s*$/i,
        RegExp('^\\s*(?:0|[+\\-]?\\d+(?:\\.\\d+)?m?s|' + s[ 6 ] +
          '|inherit)\\s*$', 'i'), /^\s*(?:0|[+\-]?\d+(?:\.\d+)?|inherit)\s*$/i,
        /^\s*(?:clip|ellipsis)\s*$/i, RegExp('^\\s*(?:normal|0|' + s[ 3 ] +
          '|inherit)\\s*$', 'i') ];
      return {
        '-moz-border-radius': c[ 1 ],
        '-moz-border-radius-bottomleft': c[ 0 ],
        '-moz-border-radius-bottomright': c[ 0 ],
        '-moz-border-radius-topleft': c[ 0 ],
        '-moz-border-radius-topright': c[ 0 ],
        '-moz-box-shadow': c[ 2 ],
        '-moz-opacity': c[ 12 ],
        '-moz-outline': c[ 13 ],
        '-moz-outline-color': c[ 14 ],
        '-moz-outline-style': c[ 4 ],
        '-moz-outline-width': c[ 5 ],
        '-o-text-overflow': c[ 20 ],
        '-webkit-border-bottom-left-radius': c[ 0 ],
        '-webkit-border-bottom-right-radius': c[ 0 ],
        '-webkit-border-radius': c[ 1 ],
        '-webkit-border-radius-bottom-left': c[ 0 ],
        '-webkit-border-radius-bottom-right': c[ 0 ],
        '-webkit-border-radius-top-left': c[ 0 ],
        '-webkit-border-radius-top-right': c[ 0 ],
        '-webkit-border-top-left-radius': c[ 0 ],
        '-webkit-border-top-right-radius': c[ 0 ],
        '-webkit-box-shadow': c[ 2 ],
        'azimuth':
        /^\s*(?:0|[+\-]?\d+(?:\.\d+)?(?:g?rad|deg)|(?:left-side|far-left|left|center-left|center|center-right|right|far-right|right-side|behind)(?:\s+(?:left-side|far-left|left|center-left|center|center-right|right|far-right|right-side|behind))?|leftwards|rightwards|inherit)\s*$/i,
        'background': RegExp('^\\s*(?:\\s*(?:' + s[ 8 ] + '|none|(?:(?:0|' + s[
            6 ] + '|' + s[ 3 ] + s[ 0 ] + ')(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3
          ] + s[ 1 ] + '))?|(?:center|(?:lef|righ)t(?:\\s+(?:0|' + s[ 6 ] + '|'
          + s[ 3 ] + '))?|(?:top|bottom)(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] +
          '))?)(?:\\s+(?:center|(?:lef|righ)t(?:\\s+(?:0|' + s[ 6 ] + '|' + s[
            3 ] + '))?|(?:top|bottom)(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] +
          '))?))?)(?:\\s*\\/\\s*(?:(?:0|' + s[ 4 ] + '|' + s[ 6 ] +
          '|auto)(?:\\s+(?:0|' + s[ 4 ] + '|' + s[ 6 ] +
          '|auto)){0,2}|cover|contain))?|\\/\\s*(?:(?:0|' + s[ 4 ] + '|' + s[ 6
          ] + '|auto)(?:\\s+(?:0|' + s[ 4 ] + '|' + s[ 6 ] +
          '|auto)){0,2}|cover|contain)|' + s[ 9 ] +
          '|scroll|fixed|local|(?:border|padding|content)-box)(?:\\s*' + s[ 8 ]
          + '|\\s+none|(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] + s[ 0 ] +
          ')(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] + s[ 1 ] +
          '))?|(?:\\s+(?:center|(?:lef|righ)t(?:\\s+(?:0|' + s[ 6 ] + '|' + s[
            3 ] + '))?|(?:top|bottom)(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] +
          '))?)){1,2})(?:\\s*\\/\\s*(?:(?:0|' + s[ 4 ] + '|' + s[ 6 ] +
          '|auto)(?:\\s+(?:0|' + s[ 4 ] + '|' + s[ 6 ] +
          '|auto)){0,2}|cover|contain))?|\\s*\\/\\s*(?:(?:0|' + s[ 4 ] + '|' +
          s[ 6 ] + '|auto)(?:\\s+(?:0|' + s[ 4 ] + '|' + s[ 6 ] +
          '|auto)){0,2}|cover|contain)|\\s+repeat-x|\\s+repeat-y|(?:\\s+(?:repeat|space|round|no-repeat)){1,2}|\\s+(?:scroll|fixed|local)|\\s+(?:border|padding|content)-box){0,4}\\s*,)*\\s*(?:'
          + s[ 2 ] + '|transparent|inherit|' + s[ 8 ] + '|none|(?:(?:0|' + s[ 6
          ] + '|' + s[ 3 ] + s[ 0 ] + ')(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] +
          s[ 1 ] + '))?|(?:center|(?:lef|righ)t(?:\\s+(?:0|' + s[ 6 ] + '|' +
          s[ 3 ] + '))?|(?:top|bottom)(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] +
          '))?)(?:\\s+(?:center|(?:lef|righ)t(?:\\s+(?:0|' + s[ 6 ] + '|' + s[
            3 ] + '))?|(?:top|bottom)(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] +
          '))?))?)(?:\\s*\\/\\s*(?:(?:0|' + s[ 4 ] + '|' + s[ 6 ] +
          '|auto)(?:\\s+(?:0|' + s[ 4 ] + '|' + s[ 6 ] +
          '|auto)){0,2}|cover|contain))?|\\/\\s*(?:(?:0|' + s[ 4 ] + '|' + s[ 6
          ] + '|auto)(?:\\s+(?:0|' + s[ 4 ] + '|' + s[ 6 ] +
          '|auto)){0,2}|cover|contain)|' + s[ 9 ] +
          '|scroll|fixed|local|(?:border|padding|content)-box)(?:\\s*#(?:[\\da-f]{3}){1,2}|\\s+aqua|\\s+black|\\s+blue|\\s+fuchsia|\\s+gray|\\s+green|\\s+lime|\\s+maroon|\\s+navy|\\s+olive|\\s+orange|\\s+purple|\\s+red|\\s+silver|\\s+teal|\\s+white|\\s+yellow|\\s+rgb\\(\\s*(?:-?\\d+|0|'
          + s[ 6 ] + ')\\s*,\\s*(?:-?\\d+|0|' + s[ 6 ] +
          ')\\s*,\\s*(?:-?\\d+|0|' + s[ 6 ] +
          ')\\)|\\s+transparent|\\s+inherit|\\s*' + s[ 8 ] +
          '|\\s+none|(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] + s[ 0 ] +
          ')(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] + s[ 1 ] +
          '))?|(?:\\s+(?:center|(?:lef|righ)t(?:\\s+(?:0|' + s[ 6 ] + '|' + s[
            3 ] + '))?|(?:top|bottom)(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] +
          '))?)){1,2})(?:\\s*\\/\\s*(?:(?:0|' + s[ 4 ] + '|' + s[ 6 ] +
          '|auto)(?:\\s+(?:0|' + s[ 4 ] + '|' + s[ 6 ] +
          '|auto)){0,2}|cover|contain))?|\\s*\\/\\s*(?:(?:0|' + s[ 4 ] + '|' +
          s[ 6 ] + '|auto)(?:\\s+(?:0|' + s[ 4 ] + '|' + s[ 6 ] +
          '|auto)){0,2}|cover|contain)|\\s+repeat-x|\\s+repeat-y|(?:\\s+(?:repeat|space|round|no-repeat)){1,2}|\\s+(?:scroll|fixed|local)|\\s+(?:border|padding|content)-box){0,5}\\s*$',
          'i'),
        'background-attachment':
        /^\s*(?:scroll|fixed|local)(?:\s*,\s*(?:scroll|fixed|local))*\s*$/i,
        'background-color': c[ 3 ],
        'background-image': RegExp('^\\s*(?:' + s[ 8 ] +
          '|none)(?:\\s*,\\s*(?:' + s[ 8 ] + '|none))*\\s*$', 'i'),
        'background-position': RegExp('^\\s*(?:(?:0|' + s[ 6 ] + '|' + s[ 3 ] +
          s[ 0 ] + ')(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] + s[ 1 ] +
          '))?|(?:center|(?:lef|righ)t(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] +
          '))?|(?:top|bottom)(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] +
          '))?)(?:\\s+(?:center|(?:lef|righ)t(?:\\s+(?:0|' + s[ 6 ] + '|' + s[
            3 ] + '))?|(?:top|bottom)(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] +
          '))?))?)(?:\\s*,\\s*(?:(?:0|' + s[ 6 ] + '|' + s[ 3 ] + s[ 0 ] +
          ')(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] + s[ 1 ] +
          '))?|(?:center|(?:lef|righ)t(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] +
          '))?|(?:top|bottom)(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] +
          '))?)(?:\\s+(?:center|(?:lef|righ)t(?:\\s+(?:0|' + s[ 6 ] + '|' + s[
            3 ] + '))?|(?:top|bottom)(?:\\s+(?:0|' + s[ 6 ] + '|' + s[ 3 ] +
          '))?))?))*\\s*$', 'i'),
        'background-repeat': RegExp('^\\s*(?:' + s[ 9 ] + ')(?:\\s*,\\s*(?:' +
          s[ 9 ] + '))*\\s*$', 'i'),
        'border': RegExp('^\\s*(?:(?:thin|medium|thick|0|' + s[ 3 ] + '|' + s[
            5 ] + '|' + s[ 2 ] + '|transparent)(?:\\s+(?:thin|medium|thick|0|'
          + s[ 3 ] + ')|\\s+(?:' + s[ 5 ] +
          ')|\\s*#(?:[\\da-f]{3}){1,2}|\\s+aqua|\\s+black|\\s+blue|\\s+fuchsia|\\s+gray|\\s+green|\\s+lime|\\s+maroon|\\s+navy|\\s+olive|\\s+orange|\\s+purple|\\s+red|\\s+silver|\\s+teal|\\s+white|\\s+yellow|\\s+rgb\\(\\s*(?:-?\\d+|0|'
          + s[ 6 ] + ')\\s*,\\s*(?:-?\\d+|0|' + s[ 6 ] +
          ')\\s*,\\s*(?:-?\\d+|0|' + s[ 6 ] +
          ')\\)|\\s+transparent){0,2}|inherit)\\s*$', 'i'),
        'border-bottom': c[ 6 ],
        'border-bottom-color': c[ 3 ],
        'border-bottom-left-radius': c[ 0 ],
        'border-bottom-right-radius': c[ 0 ],
        'border-bottom-style': c[ 4 ],
        'border-bottom-width': c[ 5 ],
        'border-collapse': /^\s*(?:collapse|separate|inherit)\s*$/i,
        'border-color': RegExp('^\\s*(?:(?:' + s[ 2 ] +
          '|transparent)(?:\\s*#(?:[\\da-f]{3}){1,2}|\\s+aqua|\\s+black|\\s+blue|\\s+fuchsia|\\s+gray|\\s+green|\\s+lime|\\s+maroon|\\s+navy|\\s+olive|\\s+orange|\\s+purple|\\s+red|\\s+silver|\\s+teal|\\s+white|\\s+yellow|\\s+rgb\\(\\s*(?:-?\\d+|0|'
          + s[ 6 ] + ')\\s*,\\s*(?:-?\\d+|0|' + s[ 6 ] +
          ')\\s*,\\s*(?:-?\\d+|0|' + s[ 6 ] +
          ')\\)|\\s+transparent){0,4}|inherit)\\s*$', 'i'),
        'border-left': c[ 6 ],
        'border-left-color': c[ 3 ],
        'border-left-style': c[ 4 ],
        'border-left-width': c[ 5 ],
        'border-radius': c[ 1 ],
        'border-right': c[ 6 ],
        'border-right-color': c[ 3 ],
        'border-right-style': c[ 4 ],
        'border-right-width': c[ 5 ],
        'border-spacing': RegExp('^\\s*(?:(?:\\s*(?:0|' + s[ 3 ] +
          ')){1,2}|\\s*inherit)\\s*$', 'i'),
        'border-style': RegExp('^\\s*(?:(?:' + s[ 5 ] + ')(?:\\s+(?:' + s[ 5 ]
          + ')){0,4}|inherit)\\s*$', 'i'),
        'border-top': c[ 6 ],
        'border-top-color': c[ 3 ],
        'border-top-left-radius': c[ 0 ],
        'border-top-right-radius': c[ 0 ],
        'border-top-style': c[ 4 ],
        'border-top-width': c[ 5 ],
        'border-width': RegExp('^\\s*(?:(?:thin|medium|thick|0|' + s[ 3 ] +
          ')(?:\\s+(?:thin|medium|thick|0|' + s[ 3 ] + ')){0,4}|inherit)\\s*$',
          'i'),
        'bottom': c[ 9 ],
        'box-shadow': c[ 2 ],
        'caption-side': /^\s*(?:top|bottom|inherit)\s*$/i,
        'clear': /^\s*(?:none|left|right|both|inherit)\s*$/i,
        'clip': RegExp('^\\s*(?:rect\\(\\s*(?:0|' + s[ 3 ] +
          '|auto)\\s*,\\s*(?:0|' + s[ 3 ] + '|auto)\\s*,\\s*(?:0|' + s[ 3 ] +
          '|auto)\\s*,\\s*(?:0|' + s[ 3 ] + '|auto)\\)|auto|inherit)\\s*$',
          'i'),
        'color': RegExp('^\\s*(?:' + s[ 2 ] + '|inherit)\\s*$', 'i'),
        'counter-increment': c[ 7 ],
        'counter-reset': c[ 7 ],
        'cue': RegExp('^\\s*(?:(?:' + s[ 8 ] + '|none|inherit)(?:\\s*' + s[ 8 ]
          + '|\\s+none|\\s+inherit)?|inherit)\\s*$', 'i'),
        'cue-after': c[ 8 ],
        'cue-before': c[ 8 ],
        'cursor': RegExp('^\\s*(?:(?:\\s*' + s[ 8 ] +
          '\\s*,)*\\s*(?:auto|crosshair|default|pointer|move|e-resize|ne-resize|nw-resize|n-resize|se-resize|sw-resize|s-resize|w-resize|text|wait|help|progress|all-scroll|col-resize|hand|no-drop|not-allowed|row-resize|vertical-text)|\\s*inherit)\\s*$',
          'i'),
        'direction': /^\s*(?:ltr|rtl|inherit)\s*$/i,
        'display':
        /^\s*(?:inline|block|list-item|run-in|inline-block|table|inline-table|table-row-group|table-header-group|table-footer-group|table-row|table-column-group|table-column|table-cell|table-caption|none|inherit|-moz-inline-box|-moz-inline-stack)\s*$/i,
        'elevation':
        /^\s*(?:0|[+\-]?\d+(?:\.\d+)?(?:g?rad|deg)|below|level|above|higher|lower|inherit)\s*$/i,
        'empty-cells': /^\s*(?:show|hide|inherit)\s*$/i,
        'filter': RegExp('^\\s*(?:\\s*alpha\\(\\s*opacity\\s*=\\s*(?:0|' + s[ 6
          ] + '|[+\\-]?\\d+(?:\\.\\d+)?)\\))+\\s*$', 'i'),
        'float': /^\s*(?:left|right|none|inherit)\s*$/i,
        'font':
        RegExp('^\\s*(?:(?:normal|italic|oblique|inherit|small-caps|bold|bolder|lighter|100|200|300|400|500|600|700|800|900)(?:\\s+(?:normal|italic|oblique|inherit|small-caps|bold|bolder|lighter|100|200|300|400|500|600|700|800|900)){0,2}\\s+(?:xx-small|x-small|small|medium|large|x-large|xx-large|(?:small|larg)er|0|'
          + s[ 4 ] + '|' + s[ 7 ] +
          '|inherit)(?:\\s*\\/\\s*(?:normal|0|\\d+(?:\\.\\d+)?|' + s[ 4 ] + '|'
          + s[ 7 ] +
          '|inherit))?(?:(?:\\s*\"\\w(?:[\\w-]*\\w)(?:\\s+\\w([\\w-]*\\w))*\"|\\s+(?:serif|sans-serif|cursive|fantasy|monospace))(?:\\s*,\\s*(?:\"\\w(?:[\\w-]*\\w)(?:\\s+\\w([\\w-]*\\w))*\"|serif|sans-serif|cursive|fantasy|monospace))*|\\s+inherit)|caption|icon|menu|message-box|small-caption|status-bar|inherit)\\s*$',
          'i'),
        'font-family':
        /^\s*(?:(?:"\w(?:[\w-]*\w)(?:\s+\w([\w-]*\w))*"|serif|sans-serif|cursive|fantasy|monospace)(?:\s*,\s*(?:"\w(?:[\w-]*\w)(?:\s+\w([\w-]*\w))*"|serif|sans-serif|cursive|fantasy|monospace))*|inherit)\s*$/i,
        'font-size':
        RegExp('^\\s*(?:xx-small|x-small|small|medium|large|x-large|xx-large|(?:small|larg)er|0|'
          + s[ 4 ] + '|' + s[ 7 ] + '|inherit)\\s*$', 'i'),
        'font-stretch':
        /^\s*(?:normal|wider|narrower|ultra-condensed|extra-condensed|condensed|semi-condensed|semi-expanded|expanded|extra-expanded|ultra-expanded)\s*$/i,
        'font-style': /^\s*(?:normal|italic|oblique|inherit)\s*$/i,
        'font-variant': /^\s*(?:normal|small-caps|inherit)\s*$/i,
        'font-weight':
        /^\s*(?:normal|bold|bolder|lighter|100|200|300|400|500|600|700|800|900|inherit)\s*$/i,
        'height': c[ 9 ],
        'left': c[ 9 ],
        'letter-spacing': c[ 21 ],
        'line-height': RegExp('^\\s*(?:normal|0|\\d+(?:\\.\\d+)?|' + s[ 4 ] +
          '|' + s[ 7 ] + '|inherit)\\s*$', 'i'),
        'list-style':
        RegExp('^\\s*(?:(?:disc|circle|square|decimal|decimal-leading-zero|lower-roman|upper-roman|lower-greek|lower-latin|upper-latin|armenian|georgian|lower-alpha|upper-alpha|none|inherit|inside|outside|'
          + s[ 8 ] +
          ')(?:\\s+(?:disc|circle|square|decimal|decimal-leading-zero|lower-roman|upper-roman|lower-greek|lower-latin|upper-latin|armenian|georgian|lower-alpha|upper-alpha|none|inherit)|\\s+(?:inside|outside|inherit)|\\s*'
          + s[ 8 ] + '|\\s+none|\\s+inherit){0,2}|inherit)\\s*$', 'i'),
        'list-style-image': c[ 8 ],
        'list-style-position': /^\s*(?:inside|outside|inherit)\s*$/i,
        'list-style-type':
        /^\s*(?:disc|circle|square|decimal|decimal-leading-zero|lower-roman|upper-roman|lower-greek|lower-latin|upper-latin|armenian|georgian|lower-alpha|upper-alpha|none|inherit)\s*$/i,
        'margin': RegExp('^\\s*(?:(?:0|' + s[ 3 ] + '|' + s[ 6 ] +
          '|auto)(?:\\s+(?:0|' + s[ 3 ] + '|' + s[ 6 ] +
          '|auto)){0,4}|inherit)\\s*$', 'i'),
        'margin-bottom': c[ 9 ],
        'margin-left': c[ 9 ],
        'margin-right': c[ 9 ],
        'margin-top': c[ 9 ],
        'max-height': c[ 10 ],
        'max-width': c[ 10 ],
        'min-height': c[ 11 ],
        'min-width': c[ 11 ],
        'opacity': c[ 12 ],
        'outline': c[ 13 ],
        'outline-color': c[ 14 ],
        'outline-style': c[ 4 ],
        'outline-width': c[ 5 ],
        'overflow': /^\s*(?:visible|hidden|scroll|auto|inherit)\s*$/i,
        'overflow-x': c[ 15 ],
        'overflow-y': c[ 15 ],
        'padding': RegExp('^\\s*(?:(?:\\s*(?:0|' + s[ 4 ] + '|' + s[ 7 ] +
          ')){1,4}|\\s*inherit)\\s*$', 'i'),
        'padding-bottom': c[ 16 ],
        'padding-left': c[ 16 ],
        'padding-right': c[ 16 ],
        'padding-top': c[ 16 ],
        'page-break-after': c[ 17 ],
        'page-break-before': c[ 17 ],
        'page-break-inside': /^\s*(?:avoid|auto|inherit)\s*$/i,
        'pause': RegExp('^\\s*(?:(?:\\s*(?:0|[+\\-]?\\d+(?:\\.\\d+)?m?s|' + s[
            6 ] + ')){1,2}|\\s*inherit)\\s*$', 'i'),
        'pause-after': c[ 18 ],
        'pause-before': c[ 18 ],
        'pitch':
        /^\s*(?:0|\d+(?:\.\d+)?k?Hz|x-low|low|medium|high|x-high|inherit)\s*$/i,
        'pitch-range': c[ 19 ],
        'play-during': RegExp('^\\s*(?:' + s[ 8 ] +
          '\\s*(?:mix|repeat)(?:\\s+(?:mix|repeat))?|auto|none|inherit)\\s*$',
          'i'),
        'position': /^\s*(?:static|relative|absolute|inherit)\s*$/i,
        'quotes': c[ 7 ],
        'richness': c[ 19 ],
        'right': c[ 9 ],
        'speak': /^\s*(?:normal|none|spell-out|inherit)\s*$/i,
        'speak-header': /^\s*(?:once|always|inherit)\s*$/i,
        'speak-numeral': /^\s*(?:digits|continuous|inherit)\s*$/i,
        'speak-punctuation': /^\s*(?:code|none|inherit)\s*$/i,
        'speech-rate':
        /^\s*(?:0|[+\-]?\d+(?:\.\d+)?|x-slow|slow|medium|fast|x-fast|faster|slower|inherit)\s*$/i,
        'stress': c[ 19 ],
        'table-layout': /^\s*(?:auto|fixed|inherit)\s*$/i,
        'text-align': /^\s*(?:left|right|center|justify|inherit)\s*$/i,
        'text-decoration':
        /^\s*(?:none|(?:underline|overline|line-through|blink)(?:\s+(?:underline|overline|line-through|blink)){0,3}|inherit)\s*$/i,
        'text-indent': RegExp('^\\s*(?:0|' + s[ 3 ] + '|' + s[ 6 ] +
          '|inherit)\\s*$', 'i'),
        'text-overflow': c[ 20 ],
        'text-shadow': c[ 2 ],
        'text-transform':
        /^\s*(?:capitalize|uppercase|lowercase|none|inherit)\s*$/i,
        'text-wrap': /^\s*(?:normal|unrestricted|none|suppress)\s*$/i,
        'top': c[ 9 ],
        'unicode-bidi': /^\s*(?:normal|embed|bidi-override|inherit)\s*$/i,
        'vertical-align':
        RegExp('^\\s*(?:baseline|sub|super|top|text-top|middle|bottom|text-bottom|0|'
          + s[ 6 ] + '|' + s[ 3 ] + '|inherit)\\s*$', 'i'),
        'visibility': /^\s*(?:visible|hidden|collapse|inherit)\s*$/i,
        'voice-family':
        /^\s*(?:(?:\s*(?:"\w(?:[\w-]*\w)(?:\s+\w([\w-]*\w))*"|male|female|child)\s*,)*\s*(?:"\w(?:[\w-]*\w)(?:\s+\w([\w-]*\w))*"|male|female|child)|\s*inherit)\s*$/i,
        'volume': RegExp('^\\s*(?:0|\\d+(?:\\.\\d+)?|' + s[ 7 ] +
          '|silent|x-soft|soft|medium|loud|x-loud|inherit)\\s*$', 'i'),
        'white-space':
        /^\s*(?:normal|pre|nowrap|pre-wrap|pre-line|inherit|-o-pre-wrap|-moz-pre-wrap|-pre-wrap)\s*$/i,
        'width': RegExp('^\\s*(?:0|' + s[ 4 ] + '|' + s[ 7 ] +
          '|auto|inherit)\\s*$', 'i'),
        'word-spacing': c[ 21 ],
        'word-wrap': /^\s*(?:normal|break-word)\s*$/i,
        'z-index': /^\s*(?:auto|-?\d+|inherit)\s*$/i,
        'zoom': RegExp('^\\s*(?:normal|0|\\d+(?:\\.\\d+)?|' + s[ 7 ] +
          ')\\s*$', 'i')
      };
    })(),
  'alternates': {
    'MozBoxShadow': [ 'boxShadow' ],
    'WebkitBoxShadow': [ 'boxShadow' ],
    'float': [ 'cssFloat', 'styleFloat' ]
  },
  'HISTORY_INSENSITIVE_STYLE_WHITELIST': {
    'display': true,
    'filter': true,
    'float': true,
    'height': true,
    'left': true,
    'opacity': true,
    'overflow': true,
    'position': true,
    'right': true,
    'top': true,
    'visibility': true,
    'width': true,
    'padding-left': true,
    'padding-right': true,
    'padding-top': true,
    'padding-bottom': true
  }
}
;
/* Copyright Google Inc.
 * Licensed under the Apache Licence Version 2.0
 * Autogenerated at Fri May 27 14:58:40 PDT 2011
 * @provides html4
 */
var html4 = {};
html4 .atype = {
  'NONE': 0,
  'URI': 1,
  'URI_FRAGMENT': 11,
  'SCRIPT': 2,
  'STYLE': 3,
  'ID': 4,
  'IDREF': 5,
  'IDREFS': 6,
  'GLOBAL_NAME': 7,
  'LOCAL_NAME': 8,
  'CLASSES': 9,
  'FRAME_TARGET': 10
};
html4 .ATTRIBS = {
  '*::class': 9,
  '*::dir': 0,
  '*::id': 4,
  '*::lang': 0,
  '*::onclick': 2,
  '*::ondblclick': 2,
  '*::onkeydown': 2,
  '*::onkeypress': 2,
  '*::onkeyup': 2,
  '*::onload': 2,
  '*::onmousedown': 2,
  '*::onmousemove': 2,
  '*::onmouseout': 2,
  '*::onmouseover': 2,
  '*::onmouseup': 2,
  '*::style': 3,
  '*::title': 0,
  'a::accesskey': 0,
  'a::coords': 0,
  'a::href': 1,
  'a::hreflang': 0,
  'a::name': 7,
  'a::onblur': 2,
  'a::onfocus': 2,
  'a::rel': 0,
  'a::rev': 0,
  'a::shape': 0,
  'a::tabindex': 0,
  'a::target': 10,
  'a::type': 0,
  'area::accesskey': 0,
  'area::alt': 0,
  'area::coords': 0,
  'area::href': 1,
  'area::nohref': 0,
  'area::onblur': 2,
  'area::onfocus': 2,
  'area::shape': 0,
  'area::tabindex': 0,
  'area::target': 10,
  'bdo::dir': 0,
  'blockquote::cite': 1,
  'br::clear': 0,
  'button::accesskey': 0,
  'button::disabled': 0,
  'button::name': 8,
  'button::onblur': 2,
  'button::onfocus': 2,
  'button::tabindex': 0,
  'button::type': 0,
  'button::value': 0,
  'canvas::height': 0,
  'canvas::width': 0,
  'caption::align': 0,
  'col::align': 0,
  'col::char': 0,
  'col::charoff': 0,
  'col::span': 0,
  'col::valign': 0,
  'col::width': 0,
  'colgroup::align': 0,
  'colgroup::char': 0,
  'colgroup::charoff': 0,
  'colgroup::span': 0,
  'colgroup::valign': 0,
  'colgroup::width': 0,
  'del::cite': 1,
  'del::datetime': 0,
  'dir::compact': 0,
  'div::align': 0,
  'dl::compact': 0,
  'font::color': 0,
  'font::face': 0,
  'font::size': 0,
  'form::accept': 0,
  'form::action': 1,
  'form::autocomplete': 0,
  'form::enctype': 0,
  'form::method': 0,
  'form::name': 7,
  'form::onreset': 2,
  'form::onsubmit': 2,
  'form::target': 10,
  'h1::align': 0,
  'h2::align': 0,
  'h3::align': 0,
  'h4::align': 0,
  'h5::align': 0,
  'h6::align': 0,
  'hr::align': 0,
  'hr::noshade': 0,
  'hr::size': 0,
  'hr::width': 0,
  'iframe::align': 0,
  'iframe::frameborder': 0,
  'iframe::height': 0,
  'iframe::marginheight': 0,
  'iframe::marginwidth': 0,
  'iframe::width': 0,
  'img::align': 0,
  'img::alt': 0,
  'img::border': 0,
  'img::height': 0,
  'img::hspace': 0,
  'img::ismap': 0,
  'img::name': 7,
  'img::src': 1,
  'img::usemap': 11,
  'img::vspace': 0,
  'img::width': 0,
  'input::accept': 0,
  'input::accesskey': 0,
  'input::align': 0,
  'input::alt': 0,
  'input::autocomplete': 0,
  'input::checked': 0,
  'input::disabled': 0,
  'input::ismap': 0,
  'input::maxlength': 0,
  'input::name': 8,
  'input::onblur': 2,
  'input::onchange': 2,
  'input::onfocus': 2,
  'input::onselect': 2,
  'input::readonly': 0,
  'input::size': 0,
  'input::src': 1,
  'input::tabindex': 0,
  'input::type': 0,
  'input::usemap': 11,
  'input::value': 0,
  'ins::cite': 1,
  'ins::datetime': 0,
  'label::accesskey': 0,
  'label::for': 5,
  'label::onblur': 2,
  'label::onfocus': 2,
  'legend::accesskey': 0,
  'legend::align': 0,
  'li::type': 0,
  'li::value': 0,
  'map::name': 7,
  'menu::compact': 0,
  'ol::compact': 0,
  'ol::start': 0,
  'ol::type': 0,
  'optgroup::disabled': 0,
  'optgroup::label': 0,
  'option::disabled': 0,
  'option::label': 0,
  'option::selected': 0,
  'option::value': 0,
  'p::align': 0,
  'pre::width': 0,
  'q::cite': 1,
  'select::disabled': 0,
  'select::multiple': 0,
  'select::name': 8,
  'select::onblur': 2,
  'select::onchange': 2,
  'select::onfocus': 2,
  'select::size': 0,
  'select::tabindex': 0,
  'table::align': 0,
  'table::bgcolor': 0,
  'table::border': 0,
  'table::cellpadding': 0,
  'table::cellspacing': 0,
  'table::frame': 0,
  'table::rules': 0,
  'table::summary': 0,
  'table::width': 0,
  'tbody::align': 0,
  'tbody::char': 0,
  'tbody::charoff': 0,
  'tbody::valign': 0,
  'td::abbr': 0,
  'td::align': 0,
  'td::axis': 0,
  'td::bgcolor': 0,
  'td::char': 0,
  'td::charoff': 0,
  'td::colspan': 0,
  'td::headers': 6,
  'td::height': 0,
  'td::nowrap': 0,
  'td::rowspan': 0,
  'td::scope': 0,
  'td::valign': 0,
  'td::width': 0,
  'textarea::accesskey': 0,
  'textarea::cols': 0,
  'textarea::disabled': 0,
  'textarea::name': 8,
  'textarea::onblur': 2,
  'textarea::onchange': 2,
  'textarea::onfocus': 2,
  'textarea::onselect': 2,
  'textarea::readonly': 0,
  'textarea::rows': 0,
  'textarea::tabindex': 0,
  'tfoot::align': 0,
  'tfoot::char': 0,
  'tfoot::charoff': 0,
  'tfoot::valign': 0,
  'th::abbr': 0,
  'th::align': 0,
  'th::axis': 0,
  'th::bgcolor': 0,
  'th::char': 0,
  'th::charoff': 0,
  'th::colspan': 0,
  'th::headers': 6,
  'th::height': 0,
  'th::nowrap': 0,
  'th::rowspan': 0,
  'th::scope': 0,
  'th::valign': 0,
  'th::width': 0,
  'thead::align': 0,
  'thead::char': 0,
  'thead::charoff': 0,
  'thead::valign': 0,
  'tr::align': 0,
  'tr::bgcolor': 0,
  'tr::char': 0,
  'tr::charoff': 0,
  'tr::valign': 0,
  'ul::compact': 0,
  'ul::type': 0
};
html4 .eflags = {
  'OPTIONAL_ENDTAG': 1,
  'EMPTY': 2,
  'CDATA': 4,
  'RCDATA': 8,
  'UNSAFE': 16,
  'FOLDABLE': 32,
  'SCRIPT': 64,
  'STYLE': 128
};
html4 .ELEMENTS = {
  'a': 0,
  'abbr': 0,
  'acronym': 0,
  'address': 0,
  'applet': 16,
  'area': 2,
  'b': 0,
  'base': 18,
  'basefont': 18,
  'bdo': 0,
  'big': 0,
  'blockquote': 0,
  'body': 49,
  'br': 2,
  'button': 0,
  'canvas': 0,
  'caption': 0,
  'center': 0,
  'cite': 0,
  'code': 0,
  'col': 2,
  'colgroup': 1,
  'dd': 1,
  'del': 0,
  'dfn': 0,
  'dir': 0,
  'div': 0,
  'dl': 0,
  'dt': 1,
  'em': 0,
  'fieldset': 0,
  'font': 0,
  'form': 0,
  'frame': 18,
  'frameset': 16,
  'h1': 0,
  'h2': 0,
  'h3': 0,
  'h4': 0,
  'h5': 0,
  'h6': 0,
  'head': 49,
  'hr': 2,
  'html': 49,
  'i': 0,
  'iframe': 4,
  'img': 2,
  'input': 2,
  'ins': 0,
  'isindex': 18,
  'kbd': 0,
  'label': 0,
  'legend': 0,
  'li': 1,
  'link': 18,
  'map': 0,
  'menu': 0,
  'meta': 18,
  'nobr': 0,
  'noframes': 20,
  'noscript': 20,
  'object': 16,
  'ol': 0,
  'optgroup': 0,
  'option': 1,
  'p': 1,
  'param': 18,
  'pre': 0,
  'q': 0,
  's': 0,
  'samp': 0,
  'script': 84,
  'select': 0,
  'small': 0,
  'span': 0,
  'strike': 0,
  'strong': 0,
  'style': 148,
  'sub': 0,
  'sup': 0,
  'table': 0,
  'tbody': 1,
  'td': 1,
  'textarea': 8,
  'tfoot': 1,
  'th': 1,
  'thead': 1,
  'title': 24,
  'tr': 1,
  'tt': 0,
  'u': 0,
  'ul': 0,
  'var': 0
};
html4 .ueffects = {
  'NOT_LOADED': 0,
  'SAME_DOCUMENT': 1,
  'NEW_DOCUMENT': 2
};
html4 .URIEFFECTS = {
  'a::href': 2,
  'area::href': 2,
  'blockquote::cite': 0,
  'body::background': 1,
  'del::cite': 0,
  'form::action': 2,
  'img::src': 1,
  'input::src': 1,
  'ins::cite': 0,
  'q::cite': 0
};
html4 .ltypes = {
  'UNSANDBOXED': 2,
  'SANDBOXED': 1,
  'DATA': 0
};
html4 .LOADERTYPES = {
  'a::href': 2,
  'area::href': 2,
  'blockquote::cite': 2,
  'body::background': 1,
  'del::cite': 2,
  'form::action': 2,
  'img::src': 1,
  'input::src': 1,
  'ins::cite': 2,
  'q::cite': 2
};;
// Copyright (C) 2006 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @fileoverview
 * An HTML sanitizer that can satisfy a variety of security policies.
 *
 * <p>
 * The HTML sanitizer is built around a SAX parser and HTML element and
 * attributes schemas.
 *
 * @author mikesamuel@gmail.com
 * @requires html4
 * @provides html, html_sanitize
 */

/**
 * @namespace
 */
var html = (function (html4) {
  var lcase;
  // The below may not be true on browsers in the Turkish locale.
  if ('script' === 'SCRIPT'.toLowerCase()) {
    lcase = function (s) { return s.toLowerCase(); };
  } else {
    /**
     * {@updoc
     * $ lcase('SCRIPT')
     * # 'script'
     * $ lcase('script')
     * # 'script'
     * }
     */
    lcase = function (s) {
      return s.replace(
          /[A-Z]/g,
          function (ch) {
            return String.fromCharCode(ch.charCodeAt(0) | 32);
          });
    };
  }

  var ENTITIES = {
    lt   : '<',
    gt   : '>',
    amp  : '&',
    nbsp : '\240',
    quot : '"',
    apos : '\''
  };

  var decimalEscapeRe = /^#(\d+)$/;
  var hexEscapeRe = /^#x([0-9A-Fa-f]+)$/;
  /**
   * Decodes an HTML entity.
   *
   * {@updoc
   * $ lookupEntity('lt')
   * # '<'
   * $ lookupEntity('GT')
   * # '>'
   * $ lookupEntity('amp')
   * # '&'
   * $ lookupEntity('nbsp')
   * # '\xA0'
   * $ lookupEntity('apos')
   * # "'"
   * $ lookupEntity('quot')
   * # '"'
   * $ lookupEntity('#xa')
   * # '\n'
   * $ lookupEntity('#10')
   * # '\n'
   * $ lookupEntity('#x0a')
   * # '\n'
   * $ lookupEntity('#010')
   * # '\n'
   * $ lookupEntity('#x00A')
   * # '\n'
   * $ lookupEntity('Pi')      // Known failure
   * # '\u03A0'
   * $ lookupEntity('pi')      // Known failure
   * # '\u03C0'
   * }
   *
   * @param name the content between the '&' and the ';'.
   * @return a single unicode code-point as a string.
   */
  function lookupEntity(name) {
    name = lcase(name);  // TODO: &pi; is different from &Pi;
    if (ENTITIES.hasOwnProperty(name)) { return ENTITIES[name]; }
    var m = name.match(decimalEscapeRe);
    if (m) {
      return String.fromCharCode(parseInt(m[1], 10));
    } else if (!!(m = name.match(hexEscapeRe))) {
      return String.fromCharCode(parseInt(m[1], 16));
    }
    return '';
  }

  function decodeOneEntity(_, name) {
    return lookupEntity(name);
  }

  var nulRe = /\0/g;
  function stripNULs(s) {
    return s.replace(nulRe, '');
  }

  var entityRe = /&(#\d+|#x[0-9A-Fa-f]+|\w+);/g;
  /**
   * The plain text of a chunk of HTML CDATA which possibly containing.
   *
   * {@updoc
   * $ unescapeEntities('')
   * # ''
   * $ unescapeEntities('hello World!')
   * # 'hello World!'
   * $ unescapeEntities('1 &lt; 2 &amp;&AMP; 4 &gt; 3&#10;')
   * # '1 < 2 && 4 > 3\n'
   * $ unescapeEntities('&lt;&lt <- unfinished entity&gt;')
   * # '<&lt <- unfinished entity>'
   * $ unescapeEntities('/foo?bar=baz&copy=true')  // & often unescaped in URLS
   * # '/foo?bar=baz&copy=true'
   * $ unescapeEntities('pi=&pi;&#x3c0;, Pi=&Pi;\u03A0') // FIXME: known failure
   * # 'pi=\u03C0\u03c0, Pi=\u03A0\u03A0'
   * }
   *
   * @param s a chunk of HTML CDATA.  It must not start or end inside an HTML
   *   entity.
   */
  function unescapeEntities(s) {
    return s.replace(entityRe, decodeOneEntity);
  }

  var ampRe = /&/g;
  var looseAmpRe = /&([^a-z#]|#(?:[^0-9x]|x(?:[^0-9a-f]|$)|$)|$)/gi;
  var ltRe = /</g;
  var gtRe = />/g;
  var quotRe = /\"/g;
  var eqRe = /\=/g;  // Backslash required on JScript.net

  /**
   * Escapes HTML special characters in attribute values as HTML entities.
   *
   * {@updoc
   * $ escapeAttrib('')
   * # ''
   * $ escapeAttrib('"<<&==&>>"')  // Do not just escape the first occurrence.
   * # '&#34;&lt;&lt;&amp;&#61;&#61;&amp;&gt;&gt;&#34;'
   * $ escapeAttrib('Hello <World>!')
   * # 'Hello &lt;World&gt;!'
   * }
   */
  function escapeAttrib(s) {
    // Escaping '=' defangs many UTF-7 and SGML short-tag attacks.
    return s.replace(ampRe, '&amp;').replace(ltRe, '&lt;').replace(gtRe, '&gt;')
        .replace(quotRe, '&#34;').replace(eqRe, '&#61;');
  }

  /**
   * Escape entities in RCDATA that can be escaped without changing the meaning.
   * {@updoc
   * $ normalizeRCData('1 < 2 &&amp; 3 > 4 &amp;& 5 &lt; 7&8')
   * # '1 &lt; 2 &amp;&amp; 3 &gt; 4 &amp;&amp; 5 &lt; 7&amp;8'
   * }
   */
  function normalizeRCData(rcdata) {
    return rcdata
        .replace(looseAmpRe, '&amp;$1')
        .replace(ltRe, '&lt;')
        .replace(gtRe, '&gt;');
  }


  // TODO(mikesamuel): validate sanitizer regexs against the HTML5 grammar at
  // http://www.whatwg.org/specs/web-apps/current-work/multipage/syntax.html
  // http://www.whatwg.org/specs/web-apps/current-work/multipage/parsing.html
  // http://www.whatwg.org/specs/web-apps/current-work/multipage/tokenization.html
  // http://www.whatwg.org/specs/web-apps/current-work/multipage/tree-construction.html

  /** token definitions. */
  var INSIDE_TAG_TOKEN = new RegExp(
      // Don't capture space.
      '^\\s*(?:'
      // Capture an attribute name in group 1, and value in group 3.
      // We capture the fact that there was an attribute in group 2, since
      // interpreters are inconsistent in whether a group that matches nothing
      // is null, undefined, or the empty string.
      + ('(?:'
         + '([a-z][a-z-]*)'                    // attribute name
         + ('('                                // optionally followed
            + '\\s*=\\s*'
            + ('('
               // A double quoted string.
               + '\"[^\"]*\"'
               // A single quoted string.
               + '|\'[^\']*\''
               // The positive lookahead is used to make sure that in
               // <foo bar= baz=boo>, the value for bar is blank, not "baz=boo".
               + '|(?=[a-z][a-z-]*\\s*=)'
               // An unquoted value that is not an attribute name.
               // We know it is not an attribute name because the previous
               // zero-width match would've eliminated that possibility.
               + '|[^>\"\'\\s]*'
               + ')'
               )
            + ')'
            ) + '?'
         + ')'
         )
      // End of tag captured in group 3.
      + '|(/?>)'
      // Don't capture cruft
      + '|[\\s\\S][^a-z\\s>]*)',
      'i');

  var OUTSIDE_TAG_TOKEN = new RegExp(
      '^(?:'
      // Entity captured in group 1.
      + '&(\\#[0-9]+|\\#[x][0-9a-f]+|\\w+);'
      // Comment, doctypes, and processing instructions not captured.
      + '|<\!--[\\s\\S]*?--\>|<!\\w[^>]*>|<\\?[^>*]*>'
      // '/' captured in group 2 for close tags, and name captured in group 3.
      + '|<(/)?([a-z][a-z0-9]*)'
      // Text captured in group 4.
      + '|([^<&>]+)'
      // Cruft captured in group 5.
      + '|([<&>]))',
      'i');

  /**
   * Given a SAX-like event handler, produce a function that feeds those
   * events and a parameter to the event handler.
   *
   * The event handler has the form:{@code
   * {
   *   // Name is an upper-case HTML tag name.  Attribs is an array of
   *   // alternating upper-case attribute names, and attribute values.  The
   *   // attribs array is reused by the parser.  Param is the value passed to
   *   // the saxParser.
   *   startTag: function (name, attribs, param) { ... },
   *   endTag:   function (name, param) { ... },
   *   pcdata:   function (text, param) { ... },
   *   rcdata:   function (text, param) { ... },
   *   cdata:    function (text, param) { ... },
   *   startDoc: function (param) { ... },
   *   endDoc:   function (param) { ... }
   * }}
   *
   * @param {Object} handler a record containing event handlers.
   * @return {Function} that takes a chunk of html and a parameter.
   *   The parameter is passed on to the handler methods.
   */
  function makeSaxParser(handler) {
    return function parse(htmlText, param) {
      htmlText = String(htmlText);
      var htmlLower = null;

      var inTag = false;  // True iff we're currently processing a tag.
      var attribs = [];  // Accumulates attribute names and values.
      var tagName = void 0;  // The name of the tag currently being processed.
      var eflags = void 0;  // The element flags for the current tag.
      var openTag = void 0;  // True if the current tag is an open tag.

      if (handler.startDoc) { handler.startDoc(param); }

      while (htmlText) {
        var m = htmlText.match(inTag ? INSIDE_TAG_TOKEN : OUTSIDE_TAG_TOKEN);
        htmlText = htmlText.substring(m[0].length);

        if (inTag) {
          if (m[1]) { // attribute
            // setAttribute with uppercase names doesn't work on IE6.
            var attribName = lcase(m[1]);
            var decodedValue;
            if (m[2]) {
              var encodedValue = m[3];
              switch (encodedValue.charCodeAt(0)) {  // Strip quotes
                case 34: case 39:
                  encodedValue = encodedValue.substring(
                      1, encodedValue.length - 1);
                  break;
              }
              decodedValue = unescapeEntities(stripNULs(encodedValue));
            } else {
              // Use name as value for valueless attribs, so
              //   <input type=checkbox checked>
              // gets attributes ['type', 'checkbox', 'checked', 'checked']
              decodedValue = attribName;
            }
            attribs.push(attribName, decodedValue);
          } else if (m[4]) {
            if (eflags !== void 0) {  // False if not in whitelist.
              if (openTag) {
                if (handler.startTag) {
                  handler.startTag(tagName, attribs, param);
                }
              } else {
                if (handler.endTag) {
                  handler.endTag(tagName, param);
                }
              }
            }

            if (openTag
                && (eflags & (html4.eflags.CDATA | html4.eflags.RCDATA))) {
              if (htmlLower === null) {
                htmlLower = lcase(htmlText);
              } else {
                htmlLower = htmlLower.substring(
                    htmlLower.length - htmlText.length);
              }
              var dataEnd = htmlLower.indexOf('</' + tagName);
              if (dataEnd < 0) { dataEnd = htmlText.length; }
              if (eflags & html4.eflags.CDATA) {
                if (handler.cdata) {
                  handler.cdata(htmlText.substring(0, dataEnd), param);
                }
              } else if (handler.rcdata) {
                handler.rcdata(
                    normalizeRCData(htmlText.substring(0, dataEnd)), param);
              }
              htmlText = htmlText.substring(dataEnd);
            }

            tagName = eflags = openTag = void 0;
            attribs.length = 0;
            inTag = false;
          }
        } else {
          if (m[1]) {  // Entity
            if (handler.pcdata) { handler.pcdata(m[0], param); }
          } else if (m[3]) {  // Tag
            openTag = !m[2];
            inTag = true;
            tagName = lcase(m[3]);
            eflags = html4.ELEMENTS.hasOwnProperty(tagName)
                ? html4.ELEMENTS[tagName] : void 0;
          } else if (m[4]) {  // Text
            if (handler.pcdata) { handler.pcdata(m[4], param); }
          } else if (m[5]) {  // Cruft
            if (handler.pcdata) {
              switch (m[5]) {
                case '<': handler.pcdata('&lt;', param); break;
                case '>': handler.pcdata('&gt;', param); break;
                default: handler.pcdata('&amp;', param); break;
              }
            }
          }
        }
      }

      if (handler.endDoc) { handler.endDoc(param); }
    };
  }

  /**
   * Returns a function that strips unsafe tags and attributes from html.
   * @param {Function} sanitizeAttributes
   *     maps from (tagName, attribs[]) to null or a sanitized attribute array.
   *     The attribs array can be arbitrarily modified, but the same array
   *     instance is reused, so should not be held.
   * @return {Function} from html to sanitized html
   */
  function makeHtmlSanitizer(sanitizeAttributes) {
    var stack;
    var ignoring;
    return makeSaxParser({
        startDoc: function (_) {
          stack = [];
          ignoring = false;
        },
        startTag: function (tagName, attribs, out) {
          if (ignoring) { return; }
          if (!html4.ELEMENTS.hasOwnProperty(tagName)) { return; }
          var eflags = html4.ELEMENTS[tagName];
          if (eflags & html4.eflags.FOLDABLE) {
            return;
          } else if (eflags & html4.eflags.UNSAFE) {
            ignoring = !(eflags & html4.eflags.EMPTY);
            return;
          }
          attribs = sanitizeAttributes(tagName, attribs);
          // TODO(mikesamuel): relying on sanitizeAttributes not to
          // insert unsafe attribute names.
          if (attribs) {
            if (!(eflags & html4.eflags.EMPTY)) {
              stack.push(tagName);
            }

            out.push('<', tagName);
            for (var i = 0, n = attribs.length; i < n; i += 2) {
              var attribName = attribs[i],
                  value = attribs[i + 1];
              if (value !== null && value !== void 0) {
                out.push(' ', attribName, '="', escapeAttrib(value), '"');
              }
            }
            out.push('>');
          }
        },
        endTag: function (tagName, out) {
          if (ignoring) {
            ignoring = false;
            return;
          }
          if (!html4.ELEMENTS.hasOwnProperty(tagName)) { return; }
          var eflags = html4.ELEMENTS[tagName];
          if (!(eflags & (html4.eflags.UNSAFE | html4.eflags.EMPTY
                          | html4.eflags.FOLDABLE))) {
            var index;
            if (eflags & html4.eflags.OPTIONAL_ENDTAG) {
              for (index = stack.length; --index >= 0;) {
                var stackEl = stack[index];
                if (stackEl === tagName) { break; }
                if (!(html4.ELEMENTS[stackEl]
                      & html4.eflags.OPTIONAL_ENDTAG)) {
                  // Don't pop non optional end tags looking for a match.
                  return;
                }
              }
            } else {
              for (index = stack.length; --index >= 0;) {
                if (stack[index] === tagName) { break; }
              }
            }
            if (index < 0) { return; }  // Not opened.
            for (var i = stack.length; --i > index;) {
              var stackEl = stack[i];
              if (!(html4.ELEMENTS[stackEl]
                    & html4.eflags.OPTIONAL_ENDTAG)) {
                out.push('</', stackEl, '>');
              }
            }
            stack.length = index;
            out.push('</', tagName, '>');
          }
        },
        pcdata: function (text, out) {
          if (!ignoring) { out.push(text); }
        },
        rcdata: function (text, out) {
          if (!ignoring) { out.push(text); }
        },
        cdata: function (text, out) {
          if (!ignoring) { out.push(text); }
        },
        endDoc: function (out) {
          for (var i = stack.length; --i >= 0;) {
            out.push('</', stack[i], '>');
          }
          stack.length = 0;
        }
      });
  }

  /**
   * Strips unsafe tags and attributes from html.
   * @param {string} htmlText to sanitize
   * @param {Function} opt_uriPolicy -- a transform to apply to uri/url
   *     attribute values.
   * @param {Function} opt_nmTokenPolicy : string -> string? -- a transform to
   *     apply to names, ids, and classes.
   * @return {string} html
   */
  function sanitize(htmlText, opt_uriPolicy, opt_nmTokenPolicy) {
    var out = [];
    makeHtmlSanitizer(
      function sanitizeAttribs(tagName, attribs) {
        for (var i = 0; i < attribs.length; i += 2) {
          var attribName = attribs[i];
          var value = attribs[i + 1];
          var atype = null, attribKey;
          if ((attribKey = tagName + '::' + attribName,
               html4.ATTRIBS.hasOwnProperty(attribKey))
              || (attribKey = '*::' + attribName,
                  html4.ATTRIBS.hasOwnProperty(attribKey))) {
            atype = html4.ATTRIBS[attribKey];
          }
          if (atype !== null) {
            switch (atype) {
              case html4.atype.NONE: break;
              case html4.atype.SCRIPT:
              case html4.atype.STYLE:
                value = null;
                break;
              case html4.atype.ID:
              case html4.atype.IDREF:
              case html4.atype.IDREFS:
              case html4.atype.GLOBAL_NAME:
              case html4.atype.LOCAL_NAME:
              case html4.atype.CLASSES:
                value = opt_nmTokenPolicy ? opt_nmTokenPolicy(value) : value;
                break;
              case html4.atype.URI:
                value = opt_uriPolicy && opt_uriPolicy(value);
                break;
              case html4.atype.URI_FRAGMENT:
                if (value && '#' === value.charAt(0)) {
                  value = opt_nmTokenPolicy ? opt_nmTokenPolicy(value) : value;
                  if (value) { value = '#' + value; }
                } else {
                  value = null;
                }
                break;
              default:
                value = null;
                break;
            }
          } else {
            value = null;
          }
          attribs[i + 1] = value;
        }
        return attribs;
      })(htmlText, out);
    return out.join('');
  }

  return {
    escapeAttrib: escapeAttrib,
    makeHtmlSanitizer: makeHtmlSanitizer,
    makeSaxParser: makeSaxParser,
    normalizeRCData: normalizeRCData,
    sanitize: sanitize,
    unescapeEntities: unescapeEntities
  };
})(html4);

var html_sanitize = html.sanitize;

;
// Copyright (C) 2008 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @fileoverview
 * JavaScript support for TemplateCompiler.java and for a tamed version of
 * <code>document.write{,ln}</code>.
 * <p>
 * This handles the problem of making sure that only the bits of a Gadget's
 * static HTML which should be visible to a script are visible, and provides
 * mechanisms to reliably find elements using dynamically generated unique IDs
 * in the face of DOM modifications by untrusted scripts.
 *
 * @author mikesamuel@gmail.com
 * @provides HtmlEmitter
 * @requires bridalMaker html html4 ___
 */

/**
 * @param base a node that is the ancestor of all statically generated HTML.
 * @param opt_tameDocument a tame document that will receive a load event
 *    when the html-emitter is closed, and which will have {@code write} and
 *    {@code writeln} members attached.
 */
function HtmlEmitter(base, opt_tameDocument) {
  if (!base) {
    throw new Error(
        'Host page error: Virtual document element was not provided');
  }
  var insertionPoint = base;
  var bridal = bridalMaker(base.ownerDocument);

  /**
   * Contiguous pairs of ex-descendants of base, and their ex-parent.
   * The detached elements (even indices) are ordered depth-first.
   */
  var detached = null;
  /** Makes sure IDs are accessible within removed detached nodes. */
  var idMap = null;

  var arraySplice = Array.prototype.splice;

  function buildIdMap() {
    idMap = {};
    var descs = base.getElementsByTagName('*');
    for (var i = 0, desc; (desc = descs[i]); ++i) {
      var id = desc.getAttributeNode('id');
      if (id && id.value) { idMap[id.value] = desc; }
    }
  }
  /**
   * Returns the element with the given ID under the base node.
   * @param id an auto-generated ID since we cannot rely on user supplied IDs
   *     to be unique.
   * @return {Element|null} null if no such element exists.
   */
  function byId(id) {
    if (!idMap) { buildIdMap(); }
    var node = idMap[id];
    if (node) { return node; }
    for (; (node = base.ownerDocument.getElementById(id));) {
      if (base.contains
          ? base.contains(node)
          : (base.compareDocumentPosition(node) & 0x10)) {
        idMap[id] = node;
        return node;
      } else {
        node.id = '';
      }
    }
    return null;
  }

  /**
   * emitStatic allows the caller to inject the static HTML from JavaScript,
   * if the gadget host page's usage pattern requires it.
   */
  function emitStatic(htmlString) {
    if (!base) {
      throw new Error('Host page error: HtmlEmitter.emitStatic called after' +
          ' document finish()ed');
    }
    // TODO: We could append the cajoled HTML to existing contents of the
    // 'base' element, thus allowing the host page to pre-populate it prior to
    // adding cajoled content. However, no clients need that yet.
    if (base.firstChild) {
      throw new Error('Host page error: Virtual document element is not empty');
    }
    base.innerHTML = htmlString;
  }
  
  // Below we define the attach, detach, and finish operations.
  // These obey the conventions that:
  //   (1) All detached nodes, along with their ex-parents are in detached,
  //       and they are ordered depth-first.
  //   (2) When a node is specified by an ID, after the operation is performed,
  //       it is in the tree.
  //   (3) Each node is attached to the same parent regardless of what the
  //       script does.  Even if a node is removed from the DOM by a script,
  //       any of its children that appear after the script, will be added.
  // As an example, consider this HTML which has the end-tags removed since
  // they don't correspond to actual nodes.
  //   <table>
  //     <script>
  //     <tr>
  //       <td>Foo<script>Bar
  //       <th>Baz
  //   <script>
  //   <p>The-End
  // There are two script elements, and we need to make sure that each only
  // sees the bits of the DOM that it is supposed to be aware of.
  //
  // To make sure that things work when javascript is off, we emit the whole
  // HTML tree, and then detach everything that shouldn't be present.
  // We represent the removed bits as pairs of (removedNode, parentItWasPartOf).
  // Including both makes us robust against changes scripts make to the DOM.
  // In this case, the detach operation results in the tree
  //   <table>
  // and the detached list
  //   [<tr><td>FooBar<th>Baz in <table>, <p>The-End in (base)]

  // After the first script executes, we reattach the bits needed by the second
  // script, which gives us the DOM
  //   <table><tr><td>Foo
  // and the detached list
  //   ['Bar' in <td>, <th>Baz in <tr>, <p>The-End in (base)]
  // Note that we did not simply remove items from the old detached list.  Since
  // the second script was deeper than the first, we had to add only a portion
  // of the <tr>'s content which required doing a separate mini-detach operation
  // and push its operation on to the front of the detached list.

  // After the second script executes, we reattach the bits needed by the third
  // script, which gives us the DOM
  //   <table><tr><td>FooBar<th>Baz
  // and the detached list
  //   [<p>The-End in (base)]

  // After the third script executes, we reattached the rest of the detached
  // nodes, and we're done.

  // To perform a detach or reattach operation, we impose a depth-first ordering
  // on HTML start tags, and text nodes:
  //   [0: <table>, 1: <tr>, 2: <td>, 3: 'Foo', 4: 'Bar', 5: <th>, 6: 'Baz',
  //    7: <p>, 8: 'The-End']
  // Then the detach operation simply removes the minimal number of nodes from
  // the DOM to make sure that only a prefix of those nodes are present.
  // In the case above, we are detaching everything after item 0.
  // Then the reattach operation advances the number.  In the example above, we
  // advance the index from 0 to 3, and then from 3 to 6.
  // The finish operation simply reattaches the rest, advancing the counter from
  // 6 to the end.

  // The minimal detached list from the node with DFS index I is the ordered
  // list such that a (node, parent) pair (N, P) is on the list if
  // dfs-index(N) > I and there is no pair (P, GP) on the list.

  // To calculate the minimal detached list given a node representing a point in
  // that ordering, we rely on the following observations:
  //    The minimal detached list after a node, is the concatenation of
  //    (1) that node's children in order
  //    (2) the next sibling of that node and its later siblings,
  //        the next sibling of that node's parent and its later siblings,
  //        the next sibling of that node's grandparent and its later siblings,
  //        etc., until base is reached.

  function detachOnto(limit, out) {
    // Set detached to be the minimal set of nodes that have to be removed
    // to make sure that limit is the last attached node in DFS order as
    // specified above.

    // First, store all the children.
    for (var child = limit.firstChild, next; child; child = next) {
      next = child.nextSibling;  // removeChild kills nextSibling.
      out.push(child, limit);
      limit.removeChild(child);
    }

    // Second, store your ancestor's next siblings and recurse.
    for (var anc = limit, greatAnc; anc && anc !== base; anc = greatAnc) {
      greatAnc = anc.parentNode;
      for (var sibling = anc.nextSibling, next; sibling; sibling = next) {
        next = sibling.nextSibling;
        out.push(sibling, greatAnc);
        greatAnc.removeChild(sibling);
      }
    }
  }
  /**
   * Make sure that everything up to and including the node with the given ID
   * is attached, and that nothing that follows the node is attached.
   */
  function attach(id) {
    var limit = byId(id);
    if (detached) {
      // Build an array of arguments to splice so we can replace the reattached
      // nodes with the nodes detached from limit.
      var newDetached = [0, 0];
      // Since limit has no parent, detachOnto will bottom out at its sibling.
      detachOnto(limit, newDetached);
      // Find the node containing limit that appears on detached.
      var limitAnc = limit;
      for (var parent; (parent = limitAnc.parentNode);) {
        limitAnc = parent;
      }
      // Reattach up to and including limit ancestor.
      // If some browser quirk causes us to miss limit in detached, we'll
      // reattach everything and try to continue.
      var nConsumed = 0;
      while (nConsumed < detached.length) {
        // in IE, some types of nodes can't be standalone, and detaching
        // one will create new parentNodes for them.  so at this point,
        // limitAnc might be an ancestor of the node on detached.
        var reattach = detached[nConsumed];
        var reattAnc = reattach;
        for (; reattAnc.parentNode; reattAnc = reattAnc.parentNode) {}
        (detached[nConsumed + 1] /* the parent */).appendChild(reattach);
        nConsumed += 2;
        if (reattAnc === limitAnc) { break; }
      }
      // Replace the reattached bits with the ones detached from limit.
      newDetached[1] = nConsumed;  // splice's second arg is the number removed
      arraySplice.apply(detached, newDetached);
    } else {
      // The first time attach is called, the limit is actually part of the DOM.
      // There's no point removing anything when all scripts are deferred.
      detached = [];
      detachOnto(limit, detached);
    }
    // Keep track of the insertion point for document.write.
    // The tag was closed if there is no child waiting to be added.
    // FIXME(mikesamuel): This is not technically correct, since the script
    // element could have been the only child.
    var isLimitClosed = detached[1] !== limit;
    insertionPoint = isLimitClosed ? limit.parentNode : limit;
    return limit;
  }
  /**
   * Removes a script place-holder.
   * When a text node immediately precedes a script block, the limit will be
   * a text node.  Text nodes can't be addressed by ID, so the TemplateCompiler
   * follows them with a {@code <span>} which must be removed to be semantics
   * preserving.
   */
  function discard(placeholder) {
    // An untrusted script block should not be able to access the wrapper before
    // it's removed since it won't be part of the DOM so there should be a
    // parentNode.
    placeholder.parentNode.removeChild(placeholder);
  }
  /**
   * Reattach any remaining detached bits, free resources.
   */
  function finish() {
    insertionPoint = null;
    if (detached) {
      for (var i = 0, n = detached.length; i < n; i += 2) {
        detached[i + 1].appendChild(detached[i]);
      }
    }
    // Release references so nodes can be garbage collected.
    idMap = detached = base = null;
    return this;
  }
  /**
   * Attach to the virtual document body classes that were extracted from the
   * body element.
   * @param {string} classes rewritten HTML classes.
   */
  function addBodyClasses(classes) {
    base.className += ' ' + classes;
  }

  function signalLoaded() {
    // Signals the close of the document and fires any window.onload event
    // handlers.
    var doc = opt_tameDocument;
    if (doc) { doc.signalLoaded___(); }
    return this;
  }

  this.byId = byId;
  this.attach = attach;
  this.discard = discard;
  this.emitStatic = emitStatic;
  this.finish = finish;
  this.signalLoaded = signalLoaded;
  this.setAttr = bridal.setAttribute;
  this.addBodyClasses = addBodyClasses;

  (function (tameDoc) {
    if (!tameDoc || tameDoc.write) { return; }

    function concat(items) {
      return Array.prototype.join.call(items, '');
    }

    var ucase;
    if ('script'.toUpperCase() === 'SCRIPT') {
      ucase = function (s) { return s.toUpperCase(); };
    } else {
      ucase = function (s) {
        return s.replace(
            /[a-z]/g,
            function (ch) {
              return String.fromCharCode(ch.charCodeAt(0) & ~32);
            });
      };
    }

    var documentWriter = {
      startTag: function (tagName, attribs) {
        var eltype = html4.ELEMENTS[tagName];
        if (!html4.ELEMENTS.hasOwnProperty(tagName)
            || (eltype & html4.eflags.UNSAFE) !== 0) {
          return;
        }
        tameDoc.sanitizeAttrs___(tagName, attribs);
        var el = bridal.createElement(tagName, attribs);
        if ((eltype & html4.eflags.OPTIONAL_ENDTAG)
            && el.tagName === insertionPoint.tagName) {
          documentWriter.endTag(el.tagName, true);
        }
        insertionPoint.appendChild(el);
        if (!(eltype & html4.eflags.EMPTY)) { insertionPoint = el; }
      },
      endTag: function (tagName, optional) {
        var anc = insertionPoint;
        tagName = ucase(tagName);
        while (anc !== base && !/\bvdoc-body___\b/.test(anc.className)) {
          var p = anc.parentNode;
          if (anc.tagName === tagName) {
            insertionPoint = p;
            return;
          }
          anc = p;
        }
      },
      pcdata: function (text) {
        insertionPoint.appendChild(insertionPoint.ownerDocument.createTextNode(
            html.unescapeEntities(text)));
      },
      cdata: function (text) {
        insertionPoint.appendChild(
            insertionPoint.ownerDocument.createTextNode(text));
      }
    };
    documentWriter.rcdata = documentWriter.pcdata;

    // Document.write and document.writeln behave as described at
    // http://www.w3.org/TR/2009/WD-html5-20090825/embedded-content-0.html#dom-document-write
    // but with a few differences:
    // (1) all HTML written is sanitized per the opt_tameDocument's HTML
    //     sanitizer
    // (2) HTML written cannot change where subsequent static HTML is emitted.
    // (3) if the document has been closed (insertion point is undefined) then
    //     the window will not be reopened.  Instead, execution will proceed at
    //     the end of the virtual document.  This is allowed by the spec but
    //     only if the onunload refuses to allow an unload, so we treat the
    //     virtual document as un-unloadable by document.write.
    // (4) document.write cannot be used to inject scripts, so the
    //     "if there is a pending external script" does not apply.
    /**
     * A tame version of document.write.
     * @param html_varargs according to HTML5, the input to document.write is
     *     varargs, and the HTML is the concatenation of all the arguments.
     */
    var tameDocWrite = function write(html_varargs) {
      var htmlText = concat(arguments);
      if (!insertionPoint) {
        // Handles case 3 where the document has been closed.
        insertionPoint = base;
      }
      var lexer = html.makeSaxParser(documentWriter);
      lexer(htmlText);
    };
    tameDoc.write = ___.markFuncFreeze(tameDocWrite, 'write');
    tameDoc.writeln = ___.markFuncFreeze(function writeln(html) {
      tameDocWrite(concat(arguments), '\n');
    }, 'writeln');
    ___.grantFunc(tameDoc, 'write');
    ___.grantFunc(tameDoc, 'writeln');
  })(opt_tameDocument);
}
;
// Copyright (C) 2008 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @fileoverview
 * A set of utility functions that implement browser feature testing to unify
 * certain DOM behaviors, and a set of recommendations about when to use these
 * functions as opposed to the native DOM functions.
 *
 * @author ihab.awad@gmail.com
 * @author jasvir@gmail.com
 * @provides bridalMaker, bridal
 * @requires ___, cajita, document, html, html4, navigator, window,
 *     XMLHttpRequest, ActiveXObject 
 */

var bridalMaker = function(document) {

  ////////////////////////////////////////////////////////////////////////////
  // Private section
  ////////////////////////////////////////////////////////////////////////////

  var isOpera = navigator.userAgent.indexOf('Opera') === 0;
  var isIE = !isOpera && navigator.userAgent.indexOf('MSIE') !== -1;
  var isWebkit = !isOpera && navigator.userAgent.indexOf('WebKit') !== -1;

  var featureAttachEvent = !!(window.attachEvent && !window.addEventListener);
  /**
   * Does the extended form of extendedCreateElement work?
   * From http://msdn.microsoft.com/en-us/library/ms536389.aspx :<blockquote>
   *     You can also specify all the attributes inside the createElement
   *     method by using an HTML string for the method argument.
   *     The following example demonstrates how to dynamically create two
   *     radio buttons utilizing this technique.
   *     <pre>
   *     ...
   *     var newRadioButton = document.createElement(
   *         "&lt;INPUT TYPE='RADIO' NAME='RADIOTEST' VALUE='First Choice'>")
   *     </pre>
   * </blockquote>
   */
  var featureExtendedCreateElement =
      (function () {
        try {
          return (
              document.createElement('<input type="radio">').type === 'radio');
        } catch (e) {
          return false;
        }
      })();

  // HTML5 compatibility on IE
  // Standard html5 but non-html4 tags cause IE to throw
  // Workaround from http://remysharp.com/html5-enabling-script
  function html5shim() {
    var html5_elements =["abbr", "article", "aside", "audio", "canvas",
        "details", "figcaption", "figure", "footer", "header", "hgroup", "mark",
        "meter", "nav", "output", "progress", "section", "summary", "time",
        "video"];
    var documentFragment = document.createDocumentFragment();
    for (var i = 0; i < html5_elements.length; i++) {
      try {
        document.createElement(html5_elements[i]);
        documentFragment.createElement(html5_elements[i]);
      } catch (e) {
        // failure in the shim is not a real failure
      }
    }
  }
  if (isIE) {
    html5shim();
  }

  var CUSTOM_EVENT_TYPE_SUFFIX = '_custom___';
  function tameEventType(type, opt_isCustom, opt_tagName) {
    type = String(type);
    if (endsWith__.test(type)) {
      throw new Error('Invalid event type ' + type);
    }
    var tagAttr = false;
    if (opt_tagName) {
      tagAttr = String(opt_tagName).toLowerCase() + '::on' + type;
    }
    if (!opt_isCustom
        && ((tagAttr && html4.atype.SCRIPT === html4.ATTRIBS[tagAttr])
            || html4.atype.SCRIPT === html4.ATTRIBS['*::on' + type])) {
      return type;
    }
    return type + CUSTOM_EVENT_TYPE_SUFFIX;
  }

  function eventHandlerTypeFilter(handler, tameType) {
    // This does not need to check that handler is callable by untrusted code
    // since the handler will invoke plugin_dispatchEvent which will do that
    // check on the untrusted function reference.
    return function (event) {
      if (tameType === event.eventType___) {
        return handler.call(this, event);
      }
    };
  }

  var endsWith__ = /__$/;
  var escapeAttrib = html.escapeAttrib;
  function constructClone(node, deep) {
    var clone;
    if (node.nodeType === 1 && featureExtendedCreateElement) {
      // From http://blog.pengoworks.com/index.cfm/2007/7/16/IE6--IE7-quirks-with-cloneNode-and-form-elements
      //     It turns out IE 6/7 doesn't properly clone some form elements
      //     when you use the cloneNode(true) and the form element is a
      //     checkbox, radio or select element.
      // JQuery provides a clone method which attempts to fix this and an issue
      // with event listeners.  According to the source code for JQuery's clone
      // method ( http://docs.jquery.com/Manipulation/clone#true ):
      //     IE copies events bound via attachEvent when
      //     using cloneNode. Calling detachEvent on the
      //     clone will also remove the events from the orignal
      // We do not need to deal with XHTML DOMs and so can skip the clean step
      // that jQuery does.
      var tagDesc = node.tagName;
      // Copying form state is not strictly mentioned in DOM2's spec of
      // cloneNode, but all implementations do it.  The value copying
      // can be interpreted as fixing implementations' failure to have
      // the value attribute "reflect" the input's value as determined by the
      // value property.
      switch (node.tagName) {
        case 'INPUT':
          tagDesc = '<input name="' + escapeAttrib(node.name)
              + '" type="' + escapeAttrib(node.type)
              + '" value="' + escapeAttrib(node.defaultValue) + '"'
              + (node.defaultChecked ? ' checked="checked">' : '>');
          break;
        case 'BUTTON':
          tagDesc = '<button name="' + escapeAttrib(node.name)
              + '" type="' + escapeAttrib(node.type)
              + '" value="' + escapeAttrib(node.value) + '">';
          break;
        case 'OPTION':
          tagDesc = '<option '
              + (node.defaultSelected ? ' selected="selected">' : '>');
          break;
        case 'TEXTAREA':
          tagDesc = '<textarea value="'
              + escapeAttrib(node.defaultValue) + '">';
          break;
      }

      clone = document.createElement(tagDesc);

      var attrs = node.attributes;
      for (var i = 0, attr; (attr = attrs[i]); ++i) {
        if (attr.specified && !endsWith__.test(attr.name)) {
          setAttribute(clone, attr.nodeName, attr.nodeValue);
        }
      }
    } else {
      clone = node.cloneNode(false);
    }
    if (deep) {
      // TODO(mikesamuel): should we whitelist nodes here, to e.g. prevent
      // untrusted code from reloading an already loaded script by cloning
      // a script node that somehow exists in a tree accessible to it?
      for (var child = node.firstChild; child; child = child.nextSibling) {
        var cloneChild = constructClone(child, deep);
        clone.appendChild(cloneChild);
      }
    }
    return clone;
  }

  function fixupClone(node, clone) {
    for (var child = node.firstChild, cloneChild = clone.firstChild; cloneChild;
         child = child.nextSibling, cloneChild = cloneChild.nextSibling) {
      fixupClone(child, cloneChild);
    }
    if (node.nodeType === 1) {
      switch (node.tagName) {
        case 'INPUT':
          clone.value = node.value;
          clone.checked = node.checked;
          break;
        case 'OPTION':
          clone.selected = node.selected;
          clone.value = node.value;
          break;
        case 'TEXTAREA':
          clone.value = node.value;
          break;
      }
    }

    // Do not copy listeners since DOM2 specifies that only attributes and
    // children are copied, and that children should only be copied if the
    // deep flag is set.
    // The children are handled in constructClone.
    var originalAttribs = node.attributes___;
    if (originalAttribs) {
      var attribs = {};
      clone.attributes___ = attribs;
      cajita.forOwnKeys(originalAttribs, ___.markFuncFreeze(function (k, v) {
        switch (typeof v) {
          case 'string': case 'number': case 'boolean':
            attribs[k] = v;
            break;
        }
      }));
    }
  }

  ////////////////////////////////////////////////////////////////////////////
  // Public section
  ////////////////////////////////////////////////////////////////////////////

  // Returns the window containing this element. 
  function getWindow(element) {
    var doc = element.ownerDocument;
    // IE
    if (doc.parentWindow) { return doc.parentWindow; }
    // Everything else
    // TODO: Safari 2's defaultView wasn't a window object :(
    // Safari 2 is not A-grade, though.
    if (doc.defaultView) { return doc.defaultView; }
    // Just in case
    var s = doc.createElement('script');
    s.innerHTML = "document.parentWindow = window;";
    doc.body.appendChild(s);
    doc.body.removeChild(s);
    return doc.parentWindow;
  }

  function untameEventType(type) {
    var suffix = CUSTOM_EVENT_TYPE_SUFFIX;
    var tlen = type.length, slen = suffix.length;
    var end = tlen - slen;
    if (end >= 0 && suffix === type.substring(end)) {
      type = type.substring(0, end);
    }
    return type;
  }

  function initEvent(event, type, bubbles, cancelable) {
    type = tameEventType(type, true);
    bubbles = Boolean(bubbles);
    cancelable = Boolean(cancelable);

    if (event.initEvent) {  // Non-IE
      event.initEvent(type, bubbles, cancelable);
    } else if (bubbles && cancelable) {  // IE
      event.eventType___ = type;
    } else {
      // TODO(mikesamuel): can bubbling and cancelable on events be simulated
      // via http://msdn.microsoft.com/en-us/library/ms533545(VS.85).aspx
      throw new Error(
          'Browser does not support non-bubbling/uncanceleable events');
    }
  }

  function dispatchEvent(element, event) {
    // TODO(mikesamuel): when we change event dispatching to happen
    // asynchronously, we should exempt custom events since those
    // need to return a useful value, and there may be code bracketing
    // them which could observe asynchronous dispatch.

    // "The return value of dispatchEvent indicates whether any of
    //  the listeners which handled the event called
    //  preventDefault. If preventDefault was called the value is
    //  false, else the value is true."
    if (element.dispatchEvent) {
      return Boolean(element.dispatchEvent(event));
    } else {
      // Only dispatches custom events as when tameEventType(t) !== t.
      element.fireEvent('ondataavailable', event);
      return Boolean(event.returnValue);
    }
  }

  /**
   * Add an event listener function to an element.
   *
   * <p>Replaces
   * W3C <code>Element::addEventListener</code> and
   * IE <code>Element::attachEvent</code>.
   *
   * @param {HTMLElement} element a native DOM element.
   * @param {string} type a string identifying the event type.
   * @param {boolean Element::function (event)} handler an event handler.
   * @param {boolean} useCapture whether the user wishes to initiate capture.
   * @return {boolean Element::function (event)} the handler added.  May be
   *     a wrapper around the input.
   */
  function addEventListener(element, type, handler, useCapture) {
    type = String(type);
    var tameType = tameEventType(type, false, element.tagName);
    if (featureAttachEvent) {
      // TODO(ihab.awad): How do we emulate 'useCapture' here?
      if (type !== tameType) {
        var wrapper = eventHandlerTypeFilter(handler, tameType);
        element.attachEvent('ondataavailable', wrapper);
        return wrapper;
      } else {
        element.attachEvent('on' + type, handler);
        return handler;
      }
    } else {
      // FF2 fails if useCapture not passed or is not a boolean.
      element.addEventListener(tameType, handler, useCapture);
      return handler;
    }
  }

  /**
   * Remove an event listener function from an element.
   *
   * <p>Replaces
   * W3C <code>Element::removeEventListener</code> and
   * IE <code>Element::detachEvent</code>.
   *
   * @param element a native DOM element.
   * @param type a string identifying the event type.
   * @param handler a function acting as an event handler.
   * @param useCapture whether the user wishes to initiate capture.
   */
  function removeEventListener(element, type, handler, useCapture) {
    type = String(type);
    var tameType = tameEventType(type, false, element.tagName);
    if (featureAttachEvent) {
      // TODO(ihab.awad): How do we emulate 'useCapture' here?
      if (tameType !== type) {
        element.detachEvent('ondataavailable', handler);
      } else {
        element.detachEvent('on' + type, handler);
      }
    } else {
      element.removeEventListener(tameType, handler, useCapture);
    }
  }

  /**
   * Clones a node per {@code Node.clone()}.
   * <p>
   * Returns a duplicate of this node, i.e., serves as a generic copy
   * constructor for nodes. The duplicate node has no parent;
   * (parentNode is null.).
   * <p>
   * Cloning an Element copies all attributes and their values,
   * including those generated by the XML processor to represent
   * defaulted attributes, but this method does not copy any text it
   * contains unless it is a deep clone, since the text is contained
   * in a child Text node. Cloning an Attribute directly, as opposed
   * to be cloned as part of an Element cloning operation, returns a
   * specified attribute (specified is true). Cloning any other type
   * of node simply returns a copy of this node.
   * <p>
   * Note that cloning an immutable subtree results in a mutable copy,
   * but the children of an EntityReference clone are readonly. In
   * addition, clones of unspecified Attr nodes are specified. And,
   * cloning Document, DocumentType, Entity, and Notation nodes is
   * implementation dependent.
   *
   * @param {boolean} deep If true, recursively clone the subtree
   * under the specified node; if false, clone only the node itself
   * (and its attributes, if it is an Element).
   *
   * @return {Node} The duplicate node.
   */
  function cloneNode(node, deep) {
    var clone;
    if (!document.all) {  // Not IE 6 or IE 7
      clone = node.cloneNode(deep);
    } else {
      clone = constructClone(node, deep);
    }
    fixupClone(node, clone);
    return clone;
  }

  function initCanvasElements(doc) {
    var els = doc.getElementsByTagName('canvas');
    for (var i = 0; i < els.length; i++) {
      initCanvasElement(els[i]);
    }
  }

  function initCanvasElement(el) {
    if (window.G_vmlCanvasManager) {
      window.G_vmlCanvasManager.initElement(el);
    }
  }

  function createElement(tagName, attribs) {
    if (featureExtendedCreateElement) {
      var tag = ['<', tagName];
      for (var i = 0, n = attribs.length; i < n; i += 2) {
        tag.push(' ', attribs[i], '="', escapeAttrib(attribs[i + 1]), '"');
      }
      tag.push('>');
      return document.createElement(tag.join(''));
    } else {
      var el = document.createElement(tagName);
      for (var i = 0, n = attribs.length; i < n; i += 2) {
        setAttribute(el, attribs[i], attribs[i + 1]);
      }
      return el;
    }
  }

  /**
   * Create a <code>style</code> element for a document containing some
   * specified CSS text. Does not add the element to the document: the client
   * may do this separately if desired.
   *
   * <p>Replaces directly creating the <code>style</code> element and
   * populating its contents.
   *
   * @param document a DOM document.
   * @param cssText a string containing a well-formed stylesheet production.
   * @return a <code>style</code> element for the specified document.
   */
  function createStylesheet(document, cssText) {
    // Courtesy Stoyan Stefanov who documents the derivation of this at
    // http://www.phpied.com/dynamic-script-and-style-elements-in-ie/ and
    // http://yuiblog.com/blog/2007/06/07/style/
    var styleSheet = document.createElement('style');
    styleSheet.setAttribute('type', 'text/css');
    if (styleSheet.styleSheet) {   // IE
      styleSheet.styleSheet.cssText = cssText;
    } else {                // the world
      styleSheet.appendChild(document.createTextNode(cssText));
    }
    return styleSheet;
  }

  /**
   * Set an attribute on a DOM node.
   *
   * <p>Replaces DOM <code>Node::setAttribute</code>.
   *
   * @param {HTMLElement} element a DOM element.
   * @param {string} name the name of an attribute.
   * @param {string} value the value of an attribute.
   */
  function setAttribute(element, name, value) {
    /*
      Hazards:

        - In IE[67], el.setAttribute doesn't work for attributes like
          'class' or 'for'.  IE[67] expects you to set 'className' or
          'htmlFor'.  Using setAttributeNode solves this problem.

        - In IE[67], <input> elements can shadow attributes.  If el is a
          form that contains an <input> named x, then el.setAttribute(x, y)
          will set x's value rather than setting el's attribute.  Using
          setAttributeNode solves this problem.

        - In IE[67], the style attribute can only be modified by setting
          el.style.cssText.  Neither setAttribute nor setAttributeNode will
          work.  el.style.cssText isn't bullet-proof, since it can be
          shadowed by <input> elements.

        - In IE[67], you can never change the type of an <button> element.
          setAttribute('type') silently fails, but setAttributeNode 
          throws an exception.  We want the silent failure.

        - In IE[67], you can never change the type of an <input> element.
          setAttribute('type') throws an exception.  We want the exception.

        - In IE[67], setAttribute is case-sensitive, unless you pass 0 as a
          3rd argument.  setAttributeNode is case-insensitive.

        - Trying to set an invalid name like ":" is supposed to throw an
          error.  In IE[678] and Opera 10, it fails without an error.
    */
    switch (name) {
      case 'style':
        element.style.cssText = value;
        return value;
      // Firefox will run javascript: URLs in the frame specified by target.
      // This can cause things to run in an unintended frame, so we make sure
      // that the target is effectively _self whenever a javascript: URL appears
      // on a node.
      case 'href':
        if (/^javascript:/i.test(value)) {
          element.stored_target___ = element.target;
          element.target = '';
        } else if (element.stored_target___) {
          element.target = element.stored_target___;
          delete element.stored_target___;
        }
        break;
      case 'target':
        if (element.href && /^javascript:/i.test(element.href)) {
          element.stored_target___ = value;
          return value;
        }
        break;
    }
    try {
      var attr = element.ownerDocument.createAttribute(name);
      attr.value = value;
      element.setAttributeNode(attr);
    } catch (e) {
      // It's a real failure only if setAttribute also fails.
      return element.setAttribute(name, value, 0);
    }
    return value;
  }

  /**
   * See <a href="http://www.w3.org/TR/cssom-view/#the-getclientrects"
   *      >ElementView.getBoundingClientRect()</a>.
   * @return {Object} duck types as a TextRectangle with numeric fields
   *    {@code left}, {@code right}, {@code top}, and {@code bottom}.
   */
  function getBoundingClientRect(el) {
    var doc = el.ownerDocument;
    // Use the native method if present.
    if (el.getBoundingClientRect) {
      var cRect = el.getBoundingClientRect();
      if (isIE) {
        // IE has an unnecessary border, which can be mucked with by styles, so
        // the amount of border is not predictable.
        // Depending on whether the document is in quirks or standards mode,
        // the border will be present on either the HTML or BODY elements.
        var fixupLeft = doc.documentElement.clientLeft + doc.body.clientLeft;
        cRect.left -= fixupLeft;
        cRect.right -= fixupLeft;
        var fixupTop = doc.documentElement.clientTop + doc.body.clientTop;
        cRect.top -= fixupTop;
        cRect.bottom -= fixupTop;
      }
      return ({
                top: +cRect.top,
                left: +cRect.left,
                right: +cRect.right,
                bottom: +cRect.bottom
              });
    }

    // Otherwise, try using the deprecated gecko method, or emulate it in
    // horribly inefficient ways.

    // http://code.google.com/p/doctype/wiki/ArticleClientViewportElement
    var viewport = (isIE && doc.compatMode === 'CSS1Compat')
        ? doc.body : doc.documentElement;

    // Figure out the position relative to the viewport.
    // From http://code.google.com/p/doctype/wiki/ArticlePageOffset
    var pageX = 0, pageY = 0;
    if (el === viewport) {
      // The viewport is the origin.
    } else if (doc.getBoxObjectFor) {  // Handles Firefox < 3
      var elBoxObject = doc.getBoxObjectFor(el);
      var viewPortBoxObject = doc.getBoxObjectFor(viewport);
      pageX = elBoxObject.screenX - viewPortBoxObject.screenX;
      pageY = elBoxObject.screenY - viewPortBoxObject.screenY;
    } else {
      // Walk the offsetParent chain adding up offsets.
      for (var op = el; (op && op !== el); op = op.offsetParent) {
        pageX += op.offsetLeft;
        pageY += op.offsetTop;
        if (op !== el) {
          pageX += op.clientLeft || 0;
          pageY += op.clientTop || 0;
        }
        if (isWebkit) {
          // On webkit the offsets for position:fixed elements are off by the
          // scroll offset.
          var opPosition = doc.defaultView.getComputedStyle(op, 'position');
          if (opPosition === 'fixed') {
            pageX += doc.body.scrollLeft;
            pageY += doc.body.scrollTop;
          }
          break;
        }
      }

      // Opera & (safari absolute) incorrectly account for body offsetTop
      if ((isWebkit
           && doc.defaultView.getComputedStyle(el, 'position') === 'absolute')
          || isOpera) {
        pageY -= doc.body.offsetTop;
      }

      // Accumulate the scroll positions for everything but the body element
      for (var op = el; (op = op.offsetParent) && op !== doc.body;) {
        pageX -= op.scrollLeft;
        // see https://bugs.opera.com/show_bug.cgi?id=249965
        if (!isOpera || op.tagName !== 'TR') {
          pageY -= op.scrollTop;
        }
      }
    }

    // Figure out the viewport container so we can subtract the window's
    // scroll offsets.
    var scrollEl = !isWebkit && doc.compatMode === 'CSS1Compat'
        ? doc.documentElement
        : doc.body;

    var left = pageX - scrollEl.scrollLeft, top = pageY - scrollEl.scrollTop;
    return ({
              top: top,
              left: left,
              right: left + el.clientWidth,
              bottom: top + el.clientHeight
            });
  }

  /**
   * Returns the value of the named attribute on element.
   * 
   * <p> In IE[67], if you have
   * <pre>
   *    <form id="f" foo="x"><input name="foo"></form>
   * </pre>
   * then f.foo is the input node,
   * and f.getAttribute('foo') is also the input node,
   * which is contrary to the DOM spec and the behavior of other browsers.
   * 
   * <p> This function tries to get a reliable value.
   *
   * <p> In IE[67], getting 'style' may be unreliable for form elements.
   *
   * @param {HTMLElement} element a DOM element.
   * @param {string} name the name of an attribute.
   */
  function getAttribute(element, name) {
    // In IE[67], element.style.cssText seems to be the only way to get the
    // value string.  This unfortunately fails when element.style is an
    // input element instead of a style object.
    if (name === 'style') {
      if (typeof element.style.cssText === 'string') {
        return element.style.cssText;
      }
    }
    var attr = element.getAttributeNode(name);
    if (attr && attr.specified) {
      return attr.value;
    } else {
      return null;
    }
  }

  function hasAttribute(element, name) {
    if (element.hasAttribute) {  // Non IE
      return element.hasAttribute(name);
    } else {
      var attr = element.getAttributeNode(name);
      return attr !== null && attr.specified;
    }
  }

  /**
   * Returns a "computed style" object for a DOM node.
   *
   * @param {HTMLElement element a DOM element.
   * @param {string} pseudoElement an optional pseudo-element selector,
   * such as ":first-child".
   */
  function getComputedStyle(element, pseudoElement) {
    if (element.currentStyle && pseudoElement === void 0) {
      return element.currentStyle;
    } else if (window.getComputedStyle) {
      return window.getComputedStyle(element, pseudoElement);
    } else {
      throw new Error(
          'Computed style not available for pseudo element '
          + pseudoElement);
    }
  }

  /**
   * Returns a new XMLHttpRequest object, hiding browser differences in the
   * method of construction.
   */
  function makeXhr() {
    if (typeof XMLHttpRequest === 'undefined') {
      var activeXClassIds = [
          'MSXML2.XMLHTTP.5.0', 'MSXML2.XMLHTTP.4.0', 'MSXML2.XMLHTTP.3.0',
          'MSXML2.XMLHTTP', 'MICROSOFT.XMLHTTP.1.0', 'MICROSOFT.XMLHTTP.1',
          'MICROSOFT.XMLHTTP'];
      for (var i = 0, n = activeXClassIds.length; i < n; i++) {
        var candidate = activeXClassIds[i];
        try {
          return new ActiveXObject(candidate);
        } catch (e) {}
      }
    }
    return new XMLHttpRequest;
  }

  return {
    addEventListener: addEventListener,
    removeEventListener: removeEventListener,
    initEvent: initEvent,
    dispatchEvent: dispatchEvent,
    cloneNode: cloneNode,
    createElement: createElement,
    createStylesheet: createStylesheet,
    setAttribute: setAttribute,
    getAttribute: getAttribute,
    hasAttribute: hasAttribute,
    getBoundingClientRect: getBoundingClientRect,
    getWindow: getWindow,
    untameEventType: untameEventType,
    extendedCreateElementFeature: featureExtendedCreateElement,
    getComputedStyle: getComputedStyle,
    makeXhr: makeXhr,
    initCanvasElement: initCanvasElement,
    initCanvasElements: initCanvasElements
  };
};

var bridal = bridalMaker(document);
;
// Copyright (C) 2010 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @fileoverview
 * Utilities for dealing with CSS source code.
 *
 * @author mikesamuel@gmail.com
 * @provides cssparser
 */

var cssparser = (function ()
{
  var ucaseLetter = /[A-Z]/g;
  function lcaseOne(ch) { return String.fromCharCode(ch.charCodeAt(0) | 32); };
  var LCASE = ('i' === 'I'.toLowerCase())
      ? function (s) { return s.toLowerCase(); }
      // Rhino's toLowerCase is broken.
      : function (s) { return s.replace(ucaseLetter, lcaseOne); };

  // CSS Lexical Grammar rules.
  // CSS lexical grammar from http://www.w3.org/TR/CSS21/grammar.html
  // The comments below are mostly copied verbatim from the grammar.

  // "@import"               {return IMPORT_SYM;}
  // "@page"                 {return PAGE_SYM;}
  // "@media"                {return MEDIA_SYM;}
  // "@charset"              {return CHARSET_SYM;}
  var KEYWORD = '(?:\\@(?:import|page|media|charset))';

  // nl                      \n|\r\n|\r|\f ; a newline
  var NEWLINE = '\\n|\\r\\n|\\r|\\f';

  // h                       [0-9a-f]      ; a hexadecimal digit
  var HEX = '[0-9a-f]';

  // nonascii                [\200-\377]
  var NON_ASCII = '[^\\0-\\177]';

  // unicode                 \\{h}{1,6}(\r\n|[ \t\r\n\f])?
  var UNICODE = '(?:(?:\\\\' + HEX + '{1,6})(?:\\r\\n|[ \t\\r\\n\\f])?)';

  // escape                  {unicode}|\\[^\r\n\f0-9a-f]
  var ESCAPE = '(?:' + UNICODE + '|\\\\[^\\r\\n\\f0-9a-f])';

  // nmstart                 [_a-z]|{nonascii}|{escape}
  var NMSTART = '(?:[_a-z]|' + NON_ASCII + '|' + ESCAPE + ')';

  // nmchar                  [_a-z0-9-]|{nonascii}|{escape}
  var NMCHAR = '(?:[_a-z0-9-]|' + NON_ASCII + '|' + ESCAPE + ')';

  // ident                   -?{nmstart}{nmchar}*
  var IDENT = '-?' + NMSTART + NMCHAR + '*';

  // name                    {nmchar}+
  var NAME = NMCHAR + '+';

  // hash
  var HASH = '#' + NAME;

  // string1                 \"([^\n\r\f\\"]|\\{nl}|{escape})*\"  ; "string"
  var STRING1 = '"(?:[^\\"\\\\]|\\\\[\\s\\S])*"';

  // string2                 \'([^\n\r\f\\']|\\{nl}|{escape})*\'  ; 'string'
  var STRING2 = "'(?:[^\\'\\\\]|\\\\[\\s\\S])*'";

  // string                  {string1}|{string2}
  var STRING = '(?:' + STRING1 + '|' + STRING2 + ')';

  // num                     [0-9]+|[0-9]*"."[0-9]+
  var NUM = '(?:[0-9]*\\.[0-9]+|[0-9]+)';

  // s                       [ \t\r\n\f]
  var SPACE = '[ \\t\\r\\n\\f]';

  // w                       {s}*
  var WHITESPACE = SPACE + '*';

  // url special chars
  var URL_SPECIAL_CHARS = '[!#$%&*-~]';

  // url chars               ({url_special_chars}|{nonascii}|{escape})*
  var URL_CHARS
      = '(?:' + URL_SPECIAL_CHARS + '|' + NON_ASCII + '|' + ESCAPE + ')*';

  // url
  var URL = (
      'url\\(' + WHITESPACE + '(?:' + STRING + '|' + URL_CHARS + ')'
      + WHITESPACE + '\\)');

  // comments
  // see http://www.w3.org/TR/CSS21/grammar.html
  var COMMENT = '/\\*[^*]*\\*+(?:[^/*][^*]*\\*+)*/';

  // {E}{M}             {return EMS;}
  // {E}{X}             {return EXS;}
  // {P}{X}             {return LENGTH;}
  // {C}{M}             {return LENGTH;}
  // {M}{M}             {return LENGTH;}
  // {I}{N}             {return LENGTH;}
  // {P}{T}             {return LENGTH;}
  // {P}{C}             {return LENGTH;}
  // {D}{E}{G}          {return ANGLE;}
  // {R}{A}{D}          {return ANGLE;}
  // {G}{R}{A}{D}       {return ANGLE;}
  // {M}{S}             {return TIME;}
  // {S}                {return TIME;}
  // {H}{Z}             {return FREQ;}
  // {K}{H}{Z}          {return FREQ;}
  // %                  {return PERCENTAGE;}
  var UNIT = '(?:em|ex|px|cm|mm|in|pt|pc|deg|rad|grad|ms|s|hz|khz|%)';

  // {num}{UNIT|IDENT}                   {return NUMBER;}
  var QUANTITY = NUM + '(?:' + WHITESPACE + UNIT + '|' + IDENT + ')?';

  // "<!--"                  {return CDO;}
  // "-->"                   {return CDC;}
  // "~="                    {return INCLUDES;}
  // "|="                    {return DASHMATCH;}
  // {w}"{"                  {return LBRACE;}
  // {w}"+"                  {return PLUS;}
  // {w}">"                  {return GREATER;}
  // {w}","                  {return COMMA;}
  var PUNC =  '<!--|-->|~=|[|=\\{\\+>,:;()]';

  var PROP_DECLS_TOKENS = new RegExp(
      '(?:'
      + [STRING, COMMENT, QUANTITY, URL, NAME, HASH, IDENT, SPACE + '+', PUNC]
          .join('|')
      + ')',
      'gi');

  var IDENT_RE = new RegExp('^(?:' + IDENT + ')$', 'i');
  var URL_RE = new RegExp('^(?:' + URL + ')$', 'i');
  var NON_HEX_ESC_RE = /\\(?:\r\n?|[^0-9A-Fa-f\r]|$)/g;
  var SPACE_RE = new RegExp(SPACE + '+', 'g');
  var BS = /\\/g;
  var DQ = /"/g;

  /** A replacer that deals with non hex backslashes. */
  function normEscs(x) {
    var out = '';
    // x could be '\\' in which case we return '' or it could be '\\\r\n' in
    // which case we escape both.
    // In the normal case where the length is 2 we end up turning any special
    // characters like \\, \", and \' into CSS escape sequences.
    for (var i = 1, n = x.length; i < n; ++i) {
      out += '\\' + x.charCodeAt(i).toString(16) + ' ';
    }
    return out;
  }

  function toCssStr(s) {
    return '"' + (s.replace(BS, '\\5c ').replace(DQ, '\\22 ')) + '"';
  }

  /**
   * Parser for CSS declaration groups that extracts property name, value
   * pairs.
   *
   * <p>
   * This method does not validate the CSS property value.  To do that, match
   * {@link css.properties} against the raw value in the handler.
   *
   * @param {string} cssText of CSS property declarations like
   *     {@code color:red}.
   * @param {function (string, Array.<string>) : void} handler
   *     receives each CSS property name and the tokenized value
   *     minus spaces and comments.
   */
  function parse(cssText, handler) {
    var toks = ('' + cssText).match(PROP_DECLS_TOKENS);
    if (!toks) { return; }
    var propName = null;
    var buf = [];
    var k = 0;
    for (var i = 0, n = toks.length; i < n; ++i) {
      var tok = toks[i];
      switch (tok.charCodeAt(0)) {
        // Skip spaces.  We can do this in properties even if they are
        // significant in rules.
        case 0x9: case 0xa: case 0xc: case 0xd: case 0x20: continue;
        case 0x27:  // Convert to double quoted string.
          tok = '"' + tok.substring(1, tok.length - 1).replace(DQ, '\\22 ')
              + '"';
          // $FALL-THROUGH$
        case 0x22: tok = tok.replace(NON_HEX_ESC_RE, normEscs); break;
        case 0x2f:  // slashes may start comments
          if ('*' === tok.charAt(1)) { continue; }
          break;
        // dot or digit
        case 0x2e:
        case 0x30: case 0x31: case 0x32: case 0x33: case 0x34:
        case 0x35: case 0x36: case 0x37: case 0x38: case 0x39:
          // 0.5 em  =>  0.5em
          tok = tok.replace(SPACE_RE, '');
          break;
        case 0x3a:  // colons separate property names from values
          // Remember the property name.
          if (k === 1 && IDENT_RE.test(buf[0])) {
            propName = LCASE(buf[0]);
          } else {
            propName = null;
          }
          k = buf.length = 0;
          continue;
        case 0x3b:  // semicolons separate name/value pairs
          if (propName) {
            if (buf.length) { handler(propName, buf.slice(0)); }
            propName = null;
          }
          k = buf.length = 0;
          continue;
        case 0x55: case 0x75:  // letter u
          var url = toUrl(tok);
          if (url !== null) { tok = 'url(' + toCssStr(url) + ')'; }
          break;
      }
      buf[k++] = tok;
    }
    if (propName && buf.length) { handler(propName, buf.slice(0)); }
  }

  var unicodeEscape
      = /\\(?:([0-9a-fA-F]{1,6})(?:\r\n?|[ \t\f\n])?|[^\r\n\f0-9a-f])/g;
  function decodeOne(_, hex) {
    return hex ? String.fromCharCode(parseInt(hex, 16)) : _.charAt(1);
  }
  /**
   * Given a css token, returns the URL contained therein or null.
   * @param {string} cssToken
   * @return {string|null}
   */
  function toUrl(cssToken) {
    if (!URL_RE.test(cssToken)) { return null; }
    cssToken = cssToken.replace(/^url[\s\(]+|[\s\)]+$/gi, '');
    switch (cssToken.charAt(0)) {
      case '"': case '\'':
        cssToken = cssToken.substring(1, cssToken.length - 1);
        break;
    }
    return cssToken.replace(unicodeEscape, decodeOne);
  }

  return {
    'parse': parse,
    'toUrl': toUrl,
    'toCssStr': toCssStr
  };
})();
;
// Copyright (C) 2008 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @fileoverview
 * A partially tamed browser object model based on
 * <a href="http://www.w3.org/TR/DOM-Level-2-HTML/Overview.html"
 * >DOM-Level-2-HTML</a> and specifically, the
 * <a href="http://www.w3.org/TR/DOM-Level-2-HTML/ecma-script-binding.html"
 * >ECMAScript Language Bindings</a>.
 *
 * Caveats:<ul>
 * <li>This is not a full implementation.
 * <li>Security Review is pending.
 * <li><code>===</code> and <code>!==</code> on node lists will not
 *   behave the same as with untamed node lists.  Specifically, it is
 *   not always true that {@code nodeA.childNodes === nodeA.childNodes}.
 * <li>Properties backed by setters/getters like {@code HTMLElement.innerHTML}
 *   will not appear to uncajoled code as DOM nodes do, since they are
 *   implemented using cajita property handlers.
 * </ul>
 *
 * <p>
 * TODO(ihab.awad): Our implementation of getAttribute (and friends)
 * is such that standard DOM attributes which we disallow for security
 * reasons (like 'form:enctype') are placed in the "virtual"
 * attributes map (this.node___.attributes___). They appear to be
 * settable and gettable, but their values are ignored and do not have
 * the expected semantics per the DOM API. This is because we do not
 * have a column in html4-defs.js stating that an attribute is valid
 * but explicitly blacklisted. Alternatives would be to always throw
 * upon access to these attributes; to make them always appear to be
 * null; etc. Revisit this decision if needed.
 *
 * <p>
 * TODO(ihab.awad): Come up with a uniform convention (and helper functions,
 * etc.) for checking that a user-supplied callback is a valid Cajita function
 * or Valija Disfunction.
 *
 * @author mikesamuel@gmail.com
 * @requires console
 * @requires clearInterval, clearTimeout, setInterval, setTimeout, cssparser
 * @requires ___, bridal, bridalMaker, css, html, html4, unicode
 * @provides attachDocumentStub, plugin_dispatchEvent___,
 *     plugin_dispatchToHandler___
 * @overrides domitaModules
 */

var domitaModules;
if (!domitaModules) { domitaModules = {}; }

domitaModules.classUtils = function() {
  function getterSetterSuffix(name) {
    return String.fromCharCode(name.charCodeAt(0) & ~32)
        + name.substring(1) + '___';
  }

  /**
   * Add setter and getter hooks so that the caja {@code node.innerHTML = '...'}
   * works as expected.
   */
  function exportFields(object, fields) {
    for (var i = fields.length; --i >= 0;) {
      var field = fields[i];
      var suffix = getterSetterSuffix(field);
      var getterName = 'get' + suffix;
      var setterName = 'set' + suffix;
      var count = 0;
      if (object[getterName]) {
        ++count;
        ___.useGetHandler(
           object, field, object[getterName]);
      }
      if (object[setterName]) {
        ++count;
        ___.useSetHandler(
           object, field, object[setterName]);
      }
      if (!count) {
        throw new Error('Failed to export field ' + field + ' on ' + object);
      }
    }
  }

  /**
   * Apply a supplied list of getter and setter functions to a given object.
   *
   * @param object an object to be decorated with getters and setters
   * implementing some properties.
   *
   * @param handlers an object containing the handler functions in the form:
   *
   *     {
   *       <propName> : { get: <getHandlerFcn>, set: <setHandlerFcn> },
   *       <propName> : { get: <getHandlerFcn>, set: <setHandlerFcn> },
   *       ...
   *     }
   *
   * For each <propName> entry, the "get" field is required, but the "set"
   * field may be empty; this implies that <propName> is a read-only property.
   */
  function applyAccessors(object, handlers) {
    function propertyOnlyHasGetter(_) {
      throw new TypeError('setting a property that only has a getter');
    }

    ___.forOwnKeys(handlers,
                   ___.markFuncFreeze(function (propertyName, def) {
      var setter = def.set || propertyOnlyHasGetter;
      ___.useGetHandler(object, propertyName, def.get);
      ___.useSetHandler(object, propertyName, setter);
    }));
  }

  /**
   * Checks that a user-supplied callback is either a Cajita function or a
   * Valija Disfuction. Return silently if the callback is valid; throw an
   * exception if it is not valid.
   *
   * @param aCallback some user-supplied "function-like" callback.
   */
  function ensureValidCallback(aCallback) {

    // ????????
    // ___.asFunc(___.readPub(aListener, 'call'))

    if ('function' !== typeof aCallback
        // Allow disfunctions
        && !('object' === (typeof aCallback) && aCallback !== null
             && ___.canCallPub(aCallback, 'call'))) {
      throw new Error('Expected function not ' + typeof aCallback);
    }
  }

  return {
    exportFields: exportFields,
    ensureValidCallback: ensureValidCallback,
    applyAccessors: applyAccessors,
    getterSetterSuffix: getterSetterSuffix
  };
};

/** XMLHttpRequest or an equivalent on IE 6. */
domitaModules.XMLHttpRequestCtor = function (XMLHttpRequest, ActiveXObject) {
  if (XMLHttpRequest) {
    return XMLHttpRequest;
  } else if (ActiveXObject) {
    // The first time the ctor is called, find an ActiveX class supported by
    // this version of IE.
    var activeXClassId;
    return function ActiveXObjectForIE() {
      if (activeXClassId === void 0) {
        activeXClassId = null;
        /** Candidate Active X types. */
        var activeXClassIds = [
            'MSXML2.XMLHTTP.5.0', 'MSXML2.XMLHTTP.4.0',
            'MSXML2.XMLHTTP.3.0', 'MSXML2.XMLHTTP',
            'MICROSOFT.XMLHTTP.1.0', 'MICROSOFT.XMLHTTP.1',
            'MICROSOFT.XMLHTTP'];
        for (var i = 0, n = activeXClassIds.length; i < n; i++) {
          var candidate = activeXClassIds[i];
          try {
            void new ActiveXObject(candidate);
            activeXClassId = candidate;
            break;
          } catch (e) {
            // do nothing; try next choice
          }
        }
        activeXClassIds = null;
      }
      return new ActiveXObject(activeXClassId);
    };
  } else {
    throw new Error('ActiveXObject not available');
  }
};

domitaModules.TameXMLHttpRequest = function(
    xmlHttpRequestMaker,
    uriCallback) {
  var classUtils = domitaModules.classUtils();

  // See http://www.w3.org/TR/XMLHttpRequest/

  // TODO(ihab.awad): Improve implementation (interleaving, memory leaks)
  // per http://www.ilinsky.com/articles/XMLHttpRequest/

  function TameXMLHttpRequest() {
    this.xhr___ = new xmlHttpRequestMaker();
    classUtils.exportFields(
        this,
        ['onreadystatechange', 'readyState', 'responseText', 'responseXML',
         'status', 'statusText']);
  }
  var FROZEN = "Object is frozen.";
  var INVALID_SUFFIX = "Property names may not end in '__'.";
  var endsWith__ = /__$/;
  TameXMLHttpRequest.prototype.handleRead___ = function (name) {
    name = '' + name;
    if (endsWith__.test(name)) { return void 0; }
    var handlerName = name + '_getter___';
    if (this[handlerName]) {
      return this[handlerName]();
    }
    if (___.hasOwnProp(this.xhr___.properties___, name)) {
      return this.xhr___.properties___[name];
    } else {
      return void 0;
    }
  };
  TameXMLHttpRequest.prototype.handleCall___ = function (name, args) {
    name = '' + name;
    if (endsWith__.test(name)) { throw new Error(INVALID_SUFFIX); }
    var handlerName = name + '_handler___';
    if (this[handlerName]) {
      return this[handlerName].call(this, args);
    }
    if (___.hasOwnProp(this.xhr___.properties___, name)) {
      return this.xhr___.properties___[name].call(this, args);
    } else {
      throw new TypeError(name + ' is not a function.');
    }
  };
  TameXMLHttpRequest.prototype.handleSet___ = function (name, val) {
    name = '' + name;
    if (endsWith__.test(name)) { throw new Error(INVALID_SUFFIX); }
    if (___.isFrozen(this)) { throw new Error(FROZEN); }
    var handlerName = name + '_setter___';
    if (this[handlerName]) {
      return this[handlerName](val);
    }
    if (!this.xhr___.properties___) {
      this.xhr___.properties___ = {};
    }
    this[name + '_canEnum___'] = true;
    return this.xhr___.properties___[name] = val;
  };
  TameXMLHttpRequest.prototype.handleDelete___ = function (name) {
    name = '' + name;
    if (endsWith__.test(name)) { throw new Error(INVALID_SUFFIX); }
    if (___.isFrozen(this)) { throw new Error(FROZEN); }
    var handlerName = name + '_deleter___';
    if (this[handlerName]) {
      return this[handlerName]();
    }
    if (this.xhr___.properties___) {
      return (
          delete this.xhr___.properties___[name]
          && delete this[name + '_canEnum___']);
    } else {
      return true;
    }
  };
  TameXMLHttpRequest.prototype.setOnreadystatechange___ = function (handler) {
    // TODO(ihab.awad): Do we need more attributes of the event than 'target'?
    // May need to implement full "tame event" wrapper similar to DOM events.
    var self = this;
    this.xhr___.onreadystatechange = function(event) {
      var evt = { target: self };
      return ___.callPub(handler, 'call', [void 0, evt]);
    };
    // Store for later direct invocation if need be
    this.handler___ = handler;
  };
  TameXMLHttpRequest.prototype.getReadyState___ = function () {
    // The ready state should be a number
    return Number(this.xhr___.readyState);
  };
  TameXMLHttpRequest.prototype.open = function (
      method, URL, opt_async, opt_userName, opt_password) {
    method = String(method);
    // The XHR interface does not tell us the MIME type in advance, so we
    // must assume the broadest possible.
    var safeUri = uriCallback.rewrite(
        String(URL), html4.ueffects.SAME_DOCUMENT, html4.ltypes.SANDBOXED);
    // If the uriCallback rejects the URL, we throw an exception, but we do not
    // put the URI in the exception so as not to put the caller at risk of some
    // code in its stack sniffing the URI.
    if ("string" !== typeof safeUri) { throw 'URI violates security policy'; }
    switch (arguments.length) {
    case 2:
      this.async___ = true;
      this.xhr___.open(method, safeUri);
      break;
    case 3:
      this.async___ = opt_async;
      this.xhr___.open(method, safeUri, Boolean(opt_async));
      break;
    case 4:
      this.async___ = opt_async;
      this.xhr___.open(
          method, safeUri, Boolean(opt_async), String(opt_userName));
      break;
    case 5:
      this.async___ = opt_async;
      this.xhr___.open(
          method, safeUri, Boolean(opt_async), String(opt_userName),
          String(opt_password));
      break;
    default:
      throw 'XMLHttpRequest cannot accept ' + arguments.length + ' arguments';
      break;
    }
  };
  TameXMLHttpRequest.prototype.setRequestHeader = function (label, value) {
    this.xhr___.setRequestHeader(String(label), String(value));
  };
  TameXMLHttpRequest.prototype.send = function(opt_data) {
    if (arguments.length === 0) {
      // TODO(ihab.awad): send()-ing an empty string because send() with no
      // args does not work on FF3, others?
      this.xhr___.send('');
    } else if (typeof opt_data === 'string') {
      this.xhr___.send(opt_data);
    } else /* if XML document */ {
      // TODO(ihab.awad): Expect tamed XML document; unwrap and send
      this.xhr___.send('');
    }

    // Firefox does not call the 'onreadystatechange' handler in
    // the case of a synchronous XHR. We simulate this behavior by
    // calling the handler explicitly.
    if (this.xhr___.overrideMimeType) {
      // This is Firefox
      if (!this.async___ && this.handler___) {
        var evt = { target: this };
        ___.callPub(this.handler___, 'call', [void 0, evt]);
      }
    }
  };
  TameXMLHttpRequest.prototype.abort = function () {
    this.xhr___.abort();
  };
  TameXMLHttpRequest.prototype.getAllResponseHeaders = function () {
    var result = this.xhr___.getAllResponseHeaders();
    return (result === undefined || result === null) ?
      result : String(result);
  };
  TameXMLHttpRequest.prototype.getResponseHeader = function (headerName) {
    var result = this.xhr___.getResponseHeader(String(headerName));
    return (result === undefined || result === null) ?
      result : String(result);
  };
  TameXMLHttpRequest.prototype.getResponseText___ = function () {
    var result = this.xhr___.responseText;
    return (result === undefined || result === null) ?
      result : String(result);
  };
  TameXMLHttpRequest.prototype.getResponseXML___ = function () {
    // TODO(ihab.awad): Implement a taming layer for XML. Requires generalizing
    // the HTML node hierarchy as well so we have a unified implementation.
    return {};
  };
  TameXMLHttpRequest.prototype.getStatus___ = function () {
    var result = this.xhr___.status;
    return (result === undefined || result === null) ?
      result : Number(result);
  };
  TameXMLHttpRequest.prototype.getStatusText___ = function () {
    var result = this.xhr___.statusText;
    return (result === undefined || result === null) ?
      result : String(result);
  };
  TameXMLHttpRequest.prototype.toString = ___.markFuncFreeze(function () {
    return 'Not a real XMLHttpRequest';
  });
  ___.markCtor(TameXMLHttpRequest, Object, 'TameXMLHttpRequest');
  ___.all2(___.grantTypedMethod, TameXMLHttpRequest.prototype,
           ['open', 'setRequestHeader', 'send', 'abort',
            'getAllResponseHeaders', 'getResponseHeader']);

  return TameXMLHttpRequest;
};

domitaModules.CssPropertiesCollection =
    function(cssPropertyNameCollection, anElement, css) {
  var canonicalStylePropertyNames = {};
  // Maps style property names, e.g. cssFloat, to property names, e.g. float.
  var cssPropertyNames = {};

  ___.forOwnKeys(cssPropertyNameCollection,
                 ___.markFuncFreeze(function (cssPropertyName) {
    var baseStylePropertyName = cssPropertyName.replace(
        /-([a-z])/g, function (_, letter) { return letter.toUpperCase(); });
    var canonStylePropertyName = baseStylePropertyName;
    cssPropertyNames[baseStylePropertyName]
        = cssPropertyNames[canonStylePropertyName]
        = cssPropertyName;
    if (css.alternates.hasOwnProperty(canonStylePropertyName)) {
      var alts = css.alternates[canonStylePropertyName];
      for (var i = alts.length; --i >= 0;) {
        cssPropertyNames[alts[i]] = cssPropertyName;
        // Handle oddities like cssFloat/styleFloat.
        if (alts[i] in anElement.style
            && !(canonStylePropertyName in anElement.style)) {
          canonStylePropertyName = alts[i];
        }
      }
    }
    canonicalStylePropertyNames[cssPropertyName] = canonStylePropertyName;
  }));

  return {
    isCanonicalProp: function (p) {
      return cssPropertyNames.hasOwnProperty(p);
    },
    isCssProp: function (p) {
      return canonicalStylePropertyNames.hasOwnProperty(p);
    },
    getCanonicalPropFromCss: function (p) {
      return canonicalStylePropertyNames[p];
    },
    getCssPropFromCanonical: function(p) {
      return cssPropertyNames[p];
    }
  };
};

/**
 * Add a tamed document implementation to a Gadget's global scope.
 *
 * Has the side effect of adding the classes "vdoc-body___" and
 * idSuffix.substring(1) to the pseudoBodyNode.
 *
 * @param {string} idSuffix a string suffix appended to all node IDs.
 *     It should begin with "-" and end with "___".
 * @param {Object} uriCallback an object like <pre>{
 *       rewrite: function (uri, uriEffect, loaderType, hints) { return safeUri }
 *     }</pre>.
 *       * uri: the uri to be rewritten
 *       * uriEffect: the effect that allowing a URI to load has (@see UriEffect.java).
 *       * loaderType: type of loader that would load the URI or the rewritten version.
 *       * hints: record that describes the context in which the URI appears.  If a hint is not 
 *         present it should not be relied upon.
 *     The rewrite function should be idempotent to allow rewritten HTML
 *     to be reinjected.
 * @param {Object} imports the gadget's global scope.
 * @param {Node} pseudoBodyNode an HTML node to act as the "body" of the
 *     virtual document provided to Cajoled code.
 * @param {Object} optPseudoWindowLocation a record containing the
 *     properties of the browser "window.location" object, which will
 *     be provided to the Cajoled code.
 */
var attachDocumentStub = (function () {
  // Array Remove - By John Resig (MIT Licensed)
  function arrayRemove(array, from, to) {
    var rest = array.slice((to || from) + 1 || array.length);
    array.length = from < 0 ? array.length + from : from;
    return array.push.apply(array, rest);
  }

  var TameNodeMark = ___.Trademark('TameNode');
  var TameNodeT = TameNodeMark.guard;
  var TameEventMark = ___.Trademark('TameEvent');
  var TameEventT = TameEventMark.guard;
  var TameImageDataMark = ___.Trademark('TameImageData');
  var TameImageDataT = TameImageDataMark.guard;
  var TameGradientMark = ___.Trademark('TameGradient');
  var TameGradientT = TameGradientMark.guard;

  // Define a wrapper type for known safe HTML, and a trademarker.
  // This does not actually use the trademarking functions since trademarks
  // cannot be applied to strings.
  function Html(htmlFragment) { this.html___ = String(htmlFragment || ''); }
  Html.prototype.valueOf = Html.prototype.toString =
      ___.markFuncFreeze(function () { return this.html___; });
  function safeHtml(htmlFragment) {
    return (htmlFragment instanceof Html)
        ? htmlFragment.html___
        : html.escapeAttrib(String(htmlFragment || ''));
  }
  function blessHtml(htmlFragment) {
    return (htmlFragment instanceof Html)
        ? htmlFragment
        : new Html(htmlFragment);
  }

  var XML_SPACE = '\t\n\r ';

  var JS_SPACE = '\t\n\r ';
  // An identifier that does not end with __.
  var JS_IDENT = '(?:[a-zA-Z_][a-zA-Z0-9$_]*[a-zA-Z0-9$]|[a-zA-Z])_?';
  var SIMPLE_HANDLER_PATTERN = new RegExp(
      '^[' + JS_SPACE + ']*'
      + '(return[' + JS_SPACE + ']+)?'  // Group 1 is present if it returns.
      + '(' + JS_IDENT + ')[' + JS_SPACE + ']*'  // Group 2 is a function name.
      // Which can be passed optionally this node, and optionally the event.
      + '\\((?:this'
        + '(?:[' + JS_SPACE + ']*,[' + JS_SPACE + ']*event)?'
        + '[' + JS_SPACE + ']*)?\\)'
      // And it can end with a semicolon.
      + '[' + JS_SPACE + ']*(?:;?[' + JS_SPACE + ']*)$');

  // These id patterns match the ones in HtmlAttributeRewriter.

  var VALID_ID_CHAR =
      unicode.LETTER + unicode.DIGIT + '_'
      + '$\\-.:;=()\\[\\]'
      + unicode.COMBINING_CHAR + unicode.EXTENDER;

  var VALID_ID_PATTERN = new RegExp(
      '^[' + VALID_ID_CHAR + ']+$');

  var VALID_ID_LIST_PATTERN = new RegExp(
      '^[' + XML_SPACE + VALID_ID_CHAR + ']*$');

  var FORBIDDEN_ID_PATTERN = new RegExp('__\\s*$');

  var FORBIDDEN_ID_LIST_PATTERN = new RegExp('__(?:\\s|$)');

  function isValidId(s) {
    return !FORBIDDEN_ID_PATTERN.test(s)
        && VALID_ID_PATTERN.test(s);
  }

  function isValidIdList(s) {
    return !FORBIDDEN_ID_LIST_PATTERN.test(s)
        && VALID_ID_LIST_PATTERN.test(s);
  }

  // Trim whitespace from the beginning and end of a CSS string.

  function trimCssSpaces(input) {
    return input.replace(/^[ \t\r\n\f]+|[ \t\r\n\f]+$/g, '');
  }

  /**
   * The plain text equivalent of a CSS string body.
   * @param {string} s the body of a CSS string literal w/o quotes
   *     or CSS identifier.
   * @return {string} plain text.
   * {@updoc
   * $ decodeCssString('')
   * # ''
   * $ decodeCssString('foo')
   * # 'foo'
   * $ decodeCssString('foo\\\nbar\\\r\nbaz\\\rboo\\\ffar')
   * # 'foobarbazboofar'
   * $ decodeCssString('foo\\000a bar\\000Abaz')
   * # 'foo' + '\n' + 'bar' + '\u0ABA' + 'z'
   * $ decodeCssString('foo\\\\bar\\\'baz')
   * # "foo\\bar'baz"
   * }
   */
  function decodeCssString(s) {
    // Decode a CSS String literal.
    // From http://www.w3.org/TR/CSS21/grammar.html
    //     string1    \"([^\n\r\f\\"]|\\{nl}|{escape})*\"
    //     unicode    \\{h}{1,6}(\r\n|[ \t\r\n\f])?
    //     escape     {unicode}|\\[^\r\n\f0-9a-f]
    //     s          [ \t\r\n\f]+
    //     nl         \n|\r\n|\r|\f
    return s.replace(
        /\\(?:(\r\n?|\n|\f)|([0-9a-f]{1,6})(?:\r\n?|[ \t\n\f])?|(.))/gi,
        function (_, nl, hex, esc) {
          return esc || (nl ? '' : String.fromCharCode(parseInt(hex, 16)));
        });
  }

  /**
   * Sanitize the 'style' attribute value of an HTML element.
   *
   * @param styleAttrValue the value of a 'style' attribute, which we
   * assume has already been checked by the caller to be a plain String.
   *
   * @return a sanitized version of the attribute value.
   */
  function sanitizeStyleAttrValue(styleAttrValue) {
    var sanitizedDeclarations = [];
    cssparser.parse(
        String(styleAttrValue),
        function (property, value) {
          property = property.toLowerCase();
          if (css.properties.hasOwnProperty(property)
              && css.properties[property].test(value + '')) {
            sanitizedDeclarations.push(property + ': ' + value);
          }
        });
    return sanitizedDeclarations.join(' ; ');
  }

  function mimeTypeForAttr(tagName, attribName) {
    if (attribName === 'src') {
      if (tagName === 'img') { return 'image/*'; }
      if (tagName === 'script') { return 'text/javascript'; }
    }
    return '*/*';
  }

  // TODO(ihab.awad): Does this work on IE, where console output
  // goes to a DOM node?
  function assert(cond) {
    if (!cond) {
      if (typeof console !== 'undefined') {
        console.error('domita assertion failed');
        console.trace();
      }
      throw new Error("Domita assertion failed");
    }
  }

  var classUtils = domitaModules.classUtils();

  var cssSealerUnsealerPair = ___.makeSealerUnsealerPair();

  // Implementations of setTimeout, setInterval, clearTimeout, and
  // clearInterval that only allow simple functions as timeouts and
  // that treat timeout ids as capabilities.
  // This is safe even if accessed across frame since the same
  // trademark value is never used with more than one version of
  // setTimeout.
  var TimeoutIdMark = ___.Trademark('TimeoutId');
  var TimeoutIdT = TimeoutIdMark.guard;

  function tameSetTimeout(timeout, delayMillis) {
    // Existing browsers treat a timeout of null or undefined as a noop.
    var timeoutId;
    if (timeout) {
      if (typeof timeout === 'string') {
        throw new Error(
            'setTimeout called with a string.'
            + '  Please pass a function instead of a string of javascript');
      }
      timeoutId = setTimeout(
          function () { ___.callPub(timeout, 'call', [___.USELESS]); },
          delayMillis | 0);
    } else {
      // tameClearTimeout checks for NaN and handles it specially.
      timeoutId = NaN;
    }
    return ___.stamp([TimeoutIdMark.stamp],
                     { timeoutId___: timeoutId });
  }
  ___.markFuncFreeze(tameSetTimeout);
  function tameClearTimeout(timeoutId) {
    if (timeoutId === null || timeoutId === (void 0)) { return; }
    try {
      timeoutId = TimeoutIdT.coerce(timeoutId);
    } catch (e) {
      // From https://developer.mozilla.org/en/DOM/window.clearTimeout says:
      // Notes:
      // Passing an invalid ID to clearTimeout does not have any effect
      // (and doesn't throw an exception).
      return;
    }
    var rawTimeoutId = timeoutId.timeoutId___;
    // Skip NaN values created for null timeouts above.
    if (rawTimeoutId === rawTimeoutId) { clearTimeout(rawTimeoutId); }
  }
  ___.markFuncFreeze(tameClearTimeout);

  var IntervalIdMark = ___.Trademark('IntervalId');
  var IntervalIdT = IntervalIdMark.guard;

  function tameSetInterval(interval, delayMillis) {
    // Existing browsers treat an interval of null or undefined as a noop.
    var intervalId;
    if (interval) {
      if (typeof interval === 'string') {
        throw new Error(
            'setInterval called with a string.'
            + '  Please pass a function instead of a string of javascript');
      }
      intervalId = setInterval(
          function () { ___.callPub(interval, 'call', [___.USELESS]); },
          delayMillis | 0);
    } else {
      intervalId = NaN;
    }
    return ___.stamp([IntervalIdMark.stamp],
                     { intervalId___: intervalId });
  }
  ___.markFuncFreeze(tameSetInterval);
  function tameClearInterval(intervalId) {
    if (intervalId === null || intervalId === (void 0)) { return; }
    try {
      intervalId = IntervalIdT.coerce(intervalId);
    } catch (e) {
      // See comment about corresponding error handling in clearTimeout.
      return;
    }
    var rawIntervalId = intervalId.intervalId___;
    if (rawIntervalId === rawIntervalId) { clearInterval(rawIntervalId); }
  }
  ___.markFuncFreeze(tameClearInterval);

  function makeScrollable(element) {
    var window = bridal.getWindow(element);
    var overflow = null;
    if (element.currentStyle) {
      overflow = element.currentStyle.overflow;
    } else if (window.getComputedStyle) {
      overflow = window.getComputedStyle(element, void 0).overflow;
    } else {
      overflow = null;
    }
    switch (overflow && overflow.toLowerCase()) {
      case 'visible':
      case 'hidden':
        element.style.overflow = 'auto';
        break;
    }
  }

  /**
   * Moves the given pixel within the element's frame of reference as close to
   * the top-left-most pixel of the element's viewport as possible without
   * moving the viewport beyond the bounds of the content.
   * @param {number} x x-coord of a pixel in the element's frame of reference.
   * @param {number} y y-coord of a pixel in the element's frame of reference.
   */
  function tameScrollTo(element, x, y) {
    if (x !== +x || y !== +y || x < 0 || y < 0) {
      throw new Error('Cannot scroll to ' + x + ':' + typeof x + ','
                      + y + ' : ' + typeof y);
    }
    element.scrollLeft = x;
    element.scrollTop = y;
  }

  /**
   * Moves the origin of the given element's view-port by the given offset.
   * @param {number} dx a delta in pixels.
   * @param {number} dy a delta in pixels.
   */
  function tameScrollBy(element, dx, dy) {
    if (dx !== +dx || dy !== +dy) {
      throw new Error('Cannot scroll by ' + dx + ':' + typeof dx + ', '
                      + dy + ':' + typeof dy);
    }
    element.scrollLeft += dx;
    element.scrollTop += dy;
  }

  function guessPixelsFromCss(cssStr) {
    if (!cssStr) { return 0; }
    var m = cssStr.match(/^([0-9]+)/);
    return m ? +m[1] : 0;
  }

  function tameResizeTo(element, w, h) {
    if (w !== +w || h !== +h) {
      throw new Error('Cannot resize to ' + w + ':' + typeof w + ', '
                      + h + ':' + typeof h);
    }
    element.style.width = w + 'px';
    element.style.height = h + 'px';
  }

  function tameResizeBy(element, dw, dh) {
    if (dw !== +dw || dh !== +dh) {
      throw new Error('Cannot resize by ' + dw + ':' + typeof dw + ', '
                      + dh + ':' + typeof dh);
    }
    if (!dw && !dh) { return; }

    // scrollWidth is width + padding + border.
    // offsetWidth is width + padding + border, but excluding the non-visible
    // area.
    // clientWidth iw width + padding, and like offsetWidth, clips to the
    // viewport.
    // margin does not count in any of these calculations.
    //
    // scrollWidth/offsetWidth
    //   +------------+
    //   |            |
    //
    // +----------------+
    // |                | Margin-top
    // | +------------+ |
    // | |############| | Border-top
    // | |#+--------+#| |
    // | |#|        |#| | Padding-top
    // | |#| +----+ |#| |
    // | |#| |    | |#| | Height
    // | |#| |    | |#| |
    // | |#| +----+ |#| |
    // | |#|        |#| |
    // | |#+--------+#| |
    // | |############| |
    // | +------------+ |
    // |                |
    // +----------------+
    //
    //     |        |
    //     +--------+
    //     clientWidth (but excludes content outside viewport)

    var style = element.currentStyle;
    if (!style) {
      style = bridal.getWindow(element).getComputedStyle(element, void 0);
    }

    // We guess the padding since it's not always expressed in px on IE
    var extraHeight = guessPixelsFromCss(style.paddingBottom)
        + guessPixelsFromCss(style.paddingTop);
    var extraWidth = guessPixelsFromCss(style.paddingLeft)
        + guessPixelsFromCss(style.paddingRight);

    var goalHeight = element.clientHeight + dh;
    var goalWidth = element.clientWidth + dw;

    var h = goalHeight - extraHeight;
    var w = goalWidth - extraWidth;

    if (dh) { element.style.height = Math.max(0, h) + 'px'; }
    if (dw) { element.style.width = Math.max(0, w) + 'px'; }

    // Correct if our guesses re padding and borders were wrong.
    // We may still not be able to resize if e.g. the deltas would take
    // a dimension negative.
    if (dh && element.clientHeight !== goalHeight) {
      var hError = element.clientHeight - goalHeight;
      element.style.height = Math.max(0, h - hError) + 'px';
    }
    if (dw && element.clientWidth !== goalWidth) {
      var wError = element.clientWidth - goalWidth;
      element.style.width = Math.max(0, w - wError) + 'px';
    }
  }

  // See above for a description of this function.
  function attachDocumentStub(
      idSuffix, uriCallback, imports, pseudoBodyNode, optPseudoWindowLocation) {
    var pluginId = ___.getId(imports);
    var document = pseudoBodyNode.ownerDocument;
    var bridal = bridalMaker(document);
    var window = bridal.getWindow(pseudoBodyNode);

    if (arguments.length < 4) {
      throw new Error('arity mismatch: ' + arguments.length);
    }
    if (!optPseudoWindowLocation) {
        optPseudoWindowLocation = {};
    }
    var elementPolicies = {};
    elementPolicies.form = function (attribs) {
      // Forms must have a gated onsubmit handler or they must have an
      // external target.
      var sawHandler = false;
      for (var i = 0, n = attribs.length; i < n; i += 2) {
        if (attribs[i] === 'onsubmit') {
          sawHandler = true;
        }
      }
      if (!sawHandler) {
        attribs.push('onsubmit', 'return false');
      }
      return attribs;
    };
    elementPolicies.a = elementPolicies.area = function (attribs) {
      // Anchor tags must always have the target '_blank'.
      attribs.push('target', '_blank');
      return attribs;
    };

    // On IE, turn an <canvas> tags into canvas elements that explorercanvas
    // will recognize 
    bridal.initCanvasElements(pseudoBodyNode);

    /** Sanitize HTML applying the appropriate transformations. */
    function sanitizeHtml(htmlText) {
      var out = [];
      htmlSanitizer(htmlText, out);
      return out.join('');
    }
    function sanitizeAttrs(tagName, attribs) {
      var n = attribs.length;
      for (var i = 0; i < n; i += 2) {
        var attribName = attribs[i];
        var value = attribs[i + 1];
        var atype = null, attribKey;
        if ((attribKey = tagName + '::' + attribName,
             html4.ATTRIBS.hasOwnProperty(attribKey))
            || (attribKey = '*::' + attribName,
                html4.ATTRIBS.hasOwnProperty(attribKey))) {
          atype = html4.ATTRIBS[attribKey];
          value = rewriteAttribute(tagName, attribName, atype, value);
        } else {
          value = null;
        }
        if (value !== null && value !== void 0) {
          attribs[i + 1] = value;
        } else {
          // Swap last attribute name/value pair in place, and reprocess here.
          // This could affect how user-agents deal with duplicate attributes.
          attribs[i + 1] = attribs[--n];
          attribs[i] = attribs[--n];
          i -= 2;
        }
      }
      attribs.length = n;
      var policy = elementPolicies[tagName];
      if (policy && elementPolicies.hasOwnProperty(tagName)) {
        return policy(attribs);
      }
      return attribs;
    }
    var htmlSanitizer = html.makeHtmlSanitizer(sanitizeAttrs);

    /**
     * If str ends with suffix,
     * and str is not identical to suffix,
     * then return the part of str before suffix.
     * Otherwise return fail.
     */
    function unsuffix(str, suffix, fail) {
      if (typeof str !== 'string') return fail;
      var n = str.length - suffix.length;
      if (0 < n && str.substring(n) === suffix) {
        return str.substring(0, n);
      } else {
        return fail;
      }
    }

    var ID_LIST_PARTS_PATTERN = new RegExp(
      '([^' + XML_SPACE + ']+)([' + XML_SPACE + ']+|$)', 'g');

    /** Convert a real attribute value to the value seen in a sandbox. */
    function virtualizeAttributeValue(attrType, realValue) {
      realValue = String(realValue);
      switch (attrType) {
        case html4.atype.GLOBAL_NAME:
        case html4.atype.ID:
        case html4.atype.IDREF:
          return unsuffix(realValue, idSuffix, null);
        case html4.atype.IDREFS:
          return realValue.replace(ID_LIST_PARTS_PATTERN,
              function(_, id, spaces) {
                return unsuffix(id, idSuffix, '') + (spaces ? ' ' : '');
              });
        case html4.atype.URI_FRAGMENT:
          if (realValue && '#' === realValue.charAt(0)) {
            realValue = unsuffix(realValue.substring(1), idSuffix, null);
            return realValue ? '#' + realValue : null;
          } else {
            return null;
          }
          break;
        default:
          return realValue;
      }
    }

    /**
     * Undoes some of the changes made by sanitizeHtml, e.g. stripping ID
     * prefixes.
     */
    function tameInnerHtml(htmlText) {
      var out = [];
      innerHtmlTamer(htmlText, out);
      return out.join('');
    }
    var innerHtmlTamer = html.makeSaxParser({
        startTag: function (tagName, attribs, out) {
          out.push('<', tagName);
          for (var i = 0; i < attribs.length; i += 2) {
            var aname = attribs[i];
            var atype = getAttributeType(tagName, aname);
            var value = attribs[i + 1];
            if (aname !== 'target' && atype !== void 0) {
              value = virtualizeAttributeValue(atype, value);
              if (typeof value === 'string') {
                out.push(' ', aname, '="', html.escapeAttrib(value), '"');
              }
            }
          }
          out.push('>');
        },
        endTag: function (name, out) { out.push('</', name, '>'); },
        pcdata: function (text, out) { out.push(text); },
        rcdata: function (text, out) { out.push(text); },
        cdata: function (text, out) { out.push(text); }
      });

    /**
     * Returns a normalized attribute value, or null if the attribute should
     * be omitted.
     * <p>This function satisfies the attribute rewriter interface defined in
     * {@link html-sanitizer.js}.  As such, the parameters are keys into
     * data structures defined in {@link html4-defs.js}.
     *
     * @param {string} tagName a canonical tag name.
     * @param {string} attribName a canonical tag name.
     * @param type as defined in html4-defs.js.
     *
     * @return {string|null} null to indicate that the attribute should not
     *   be set.
     */
    function rewriteAttribute(tagName, attribName, type, value) {
      switch (type) {
        case html4.atype.NONE:
          return String(value);
        case html4.atype.CLASSES:
          // note, className is arbitrary CDATA.
          value = String(value);
          if (!FORBIDDEN_ID_LIST_PATTERN.test(value)) {
            return value;
          }
          return null;
        case html4.atype.GLOBAL_NAME:
        case html4.atype.ID:
        case html4.atype.IDREF:
          value = String(value);
          if (value && isValidId(value)) {
            return value + idSuffix;
          }
          return null;
        case html4.atype.IDREFS:
          value = String(value);
          if (value && isValidIdList(value)) {
            return value.replace(ID_LIST_PARTS_PATTERN,
                function(_, id, spaces) {
                  return id + idSuffix + (spaces ? ' ' : '');
                });
          }
          return null;
        case html4.atype.LOCAL_NAME:
          value = String(value);
          if (value && isValidId(value)) {
            return value;
          }
          return null;
        case html4.atype.SCRIPT:
          value = String(value);
          // Translate a handler that calls a simple function like
          //   return foo(this, event)

          // TODO(mikesamuel): integrate cajita compiler to allow arbitrary
          // cajita in event handlers.
          var match = value.match(SIMPLE_HANDLER_PATTERN);
          if (!match) { return null; }
          var doesReturn = match[1];
          var fnName = match[2];
          value = (doesReturn ? 'return ' : '') + 'plugin_dispatchEvent___('
              + 'this, event, ' + pluginId + ', "'
              + fnName + '");';
          if (attribName === 'onsubmit') {
            value = 'try { ' + value + ' } finally { return false; }';
          }
          return value;
        case html4.atype.URI:
          value = String(value);
          if (!uriCallback) { return null; }
          return uriCallback.rewrite(
              value, getUriEffect(tagName, attribName), getLoaderType(tagName, attribName),
              { "XML_ATTR": attribName}) || null;
        case html4.atype.URI_FRAGMENT:
          value = String(value);
          if (value.charAt(0) === '#' && isValidId(value.substring(1))) {
            return value + idSuffix;
          }
          return null;
        case html4.atype.STYLE:
          if ('function' !== typeof value) {
            return sanitizeStyleAttrValue(String(value));
          }
          var cssPropertiesAndValues = cssSealerUnsealerPair.unseal(value);
          if (!cssPropertiesAndValues) { return null; }

          var css = [];
          for (var i = 0; i < cssPropertiesAndValues.length; i += 2) {
            var propName = cssPropertiesAndValues[i];
            var propValue = cssPropertiesAndValues[i + 1];
            // If the propertyName differs between DOM and CSS, there will
            // be a semicolon between the two.
            // E.g., 'background-color;backgroundColor'
            // See CssTemplate.toPropertyValueList.
            var semi = propName.indexOf(';');
            if (semi >= 0) { propName = propName.substring(0, semi); }
            css.push(propName + ' : ' + propValue);
          }
          return css.join(' ; ');
        // Frames are ambient, so disallow reference.
        case html4.atype.FRAME_TARGET:
        default:
          return null;
      }
    }

    function makeCache() {
      var cache = ___.newTable(false);
      cache.set(null, null);
      cache.set(void 0, null);
      return cache;
    }

    var editableTameNodeCache = makeCache();
    var readOnlyTameNodeCache = makeCache();

    /**
     * returns a tame DOM node.
     * @param {Node} node
     * @param {boolean} editable
     * @see <a href="http://www.w3.org/TR/DOM-Level-2-HTML/html.html"
     *       >DOM Level 2</a>
     */
    function defaultTameNode(node, editable) {
      if (node === null || node === void 0) { return null; }
      // TODO(mikesamuel): make sure it really is a DOM node

      var cache = editable ? editableTameNodeCache : readOnlyTameNodeCache;
      var tamed = cache.get(node);
      if (tamed !== void 0) {
        return tamed;
      }
      switch (node.nodeType) {
        case 1:  // Element
          var tagName = node.tagName.toLowerCase();
          switch (tagName) {
            case 'a':
              tamed = new TameAElement(node, editable);
              break;
            case 'canvas':
              tamed = new TameCanvasElement(node, editable);
              break;
            case 'form':
              tamed = new TameFormElement(node, editable);
              break;
            case 'select':
            case 'button':
            case 'option':
            case 'textarea':
            case 'input':
              tamed = new TameInputElement(node, editable);
              break;
            case 'iframe':
              tamed = new TameIFrameElement(node, editable);
              break;
            case 'img':
              tamed = new TameImageElement(node, editable);
              break;
            case 'label':
              tamed = new TameLabelElement(node, editable);
              break;
            case 'script':
              tamed = new TameScriptElement(node, editable);
              break;
            case 'td':
            case 'thead':
            case 'tfoot':
            case 'tbody':
            case 'th':
              tamed = new TameTableCompElement(node, editable);
              break;
            case 'tr':
              tamed = new TameTableRowElement(node, editable);
              break;
            case 'table':
              tamed = new TameTableElement(node, editable);
              break;
            default:
              if (!html4.ELEMENTS.hasOwnProperty(tagName)
                  || (html4.ELEMENTS[tagName] & html4.eflags.UNSAFE)) {
                // If an unrecognized or unsafe node, return a
                // placeholder that doesn't prevent tree navigation,
                // but that doesn't allow mutation or leak attribute
                // information.
                tamed = new TameOpaqueNode(node, editable);
              } else {
                tamed = new TameElement(node, editable, editable);
              }
              break;
          }
          break;
        case 2:  // Attr
          // Cannot generically wrap since we must have access to the
          // owner element
          throw 'Internal: Attr nodes cannot be generically wrapped';
          break;
        case 3:  // Text
        case 4:  // CDATA Section Node
          tamed = new TameTextNode(node, editable);
          break;
        case 8:  // Comment
          tamed = new TameCommentNode(node, editable);
          break;
        case 11: // Document Fragment
          tamed = new TameBackedNode(node, editable, editable);
          break;
        default:
          tamed = new TameOpaqueNode(node, editable);
          break;
      }

      if (node.nodeType === 1) {
        cache.set(node, tamed);
      }
      return tamed;
    }

    function tameRelatedNode(node, editable, tameNodeCtor) {
      if (node === null || node === void 0) { return null; }
      if (node === tameDocument.body___) {
        if (tameDocument.editable___ && !editable) {
          // FIXME: return a non-editable version of body.
          throw new Error(NOT_EDITABLE);
        }
        return tameDocument.getBody___();
      }

      // Catch errors because node might be from a different domain.
      try {
        var docElem = node.ownerDocument.documentElement;
        for (var ancestor = node; ancestor; ancestor = ancestor.parentNode) {
          if (idClassPattern.test(ancestor.className)) {
            return tameNodeCtor(node, editable);
          } else if (ancestor === docElem) {
            return null;
          }
        }
        return tameNodeCtor(node, editable);
      } catch (e) {}
      return null;
    }

    /**
     * Returns the length of a raw DOM Nodelist object, working around
     * NamedNodeMap bugs in IE, Opera, and Safari as discussed at
     * http://code.google.com/p/google-caja/issues/detail?id=935
     *
     * @param nodeList a DOM NodeList.
     *
     * @return the number of nodes in the NodeList.
     */
    function getNodeListLength(nodeList) {
      var limit = nodeList.length;
      if (limit !== +limit) { limit = 1/0; }
      return limit;
    }

    /**
     * Constructs a NodeList-like object.
     *
     * @param tamed a JavaScript array that will be populated and decorated
     *     with the DOM NodeList API.
     * @param nodeList an array-like object supporting a "length" property
     *     and "[]" numeric indexing, or a raw DOM NodeList;
     * @param editable whether the tame nodes wrapped by this object
     *     should permit editing.
     * @param opt_tameNodeCtor a function for constructing tame nodes
     *     out of raw DOM nodes.
     */
    function mixinNodeList(tamed, nodeList, editable, opt_tameNodeCtor) {
      var limit = getNodeListLength(nodeList);
      if (limit > 0 && !opt_tameNodeCtor) {
        throw 'Internal: Nonempty mixinNodeList() without a tameNodeCtor';
      }

      for (var i = 0; i < limit && nodeList[i]; ++i) {
        tamed[i] = opt_tameNodeCtor(nodeList[i], editable);
      }

      // Guard against accidental leakage of untamed nodes
      nodeList = null;

      tamed.item = ___.markFuncFreeze(function (k) {
        k &= 0x7fffffff;
        if (k !== k) { throw new Error(); }
        return tamed[k] || null;
      });

      return tamed;
    }

    function tameNodeList(nodeList, editable, opt_tameNodeCtor) {
      return ___.freeze(
          mixinNodeList([], nodeList, editable, opt_tameNodeCtor));
    }

    function tameOptionsList(nodeList, editable, opt_tameNodeCtor) {
      var nl = mixinNodeList([], nodeList, editable, opt_tameNodeCtor);
      nl.selectedIndex = +nodeList.selectedIndex;
      ___.grantRead(nl, 'selectedIndex');
      return ___.freeze(nl);
    }

    /**
     * Return a fake node list containing tamed nodes.
     * @param {Array.<TameNode>} array of tamed nodes.
     * @return an array that duck types to a node list.
     */
    function fakeNodeList(array) {
      array.item = ___.markFuncFreeze(function(i) { return array[i]; });
      return ___.freeze(array);
    }

    /**
     * Constructs an HTMLCollection-like object which indexes its elements
     * based on their NAME attribute.
     *
     * @param tamed a JavaScript array that will be populated and decorated
     *     with the DOM HTMLCollection API.
     * @param nodeList an array-like object supporting a "length" property
     *     and "[]" numeric indexing.
     * @param editable whether the tame nodes wrapped by this object
     *     should permit editing.
     * @param opt_tameNodeCtor a function for constructing tame nodes
     *     out of raw DOM nodes.
     */
    function mixinHTMLCollection(tamed, nodeList, editable, opt_tameNodeCtor) {
      mixinNodeList(tamed, nodeList, editable, opt_tameNodeCtor);

      var tameNodesByName = {};
      var tameNode;

      for (var i = 0; i < tamed.length && (tameNode = tamed[i]); ++i) {
        var name = tameNode.getAttribute('name');
        if (name && !(name.charAt(name.length - 1) === '_' || (name in tamed)
                     || name === String(name & 0x7fffffff))) {
          if (!tameNodesByName[name]) { tameNodesByName[name] = []; }
          tameNodesByName[name].push(tameNode);
        }
      }

      ___.forOwnKeys(
        tameNodesByName,
        ___.markFuncFreeze(function (name, tameNodes) {
          if (tameNodes.length > 1) {
            tamed[name] = fakeNodeList(tameNodes);
          } else {
            tamed[name] = tameNodes[0];
          }
        }));

      tamed.namedItem = ___.markFuncFreeze(function(name) {
        name = String(name);
        if (name.charAt(name.length - 1) === '_') {
          return null;
        }
        if (___.hasOwnProp(tamed, name)) {
          return ___.passesGuard(TameNodeT, tamed[name])
              ? tamed[name] : tamed[name][0];
        }
        return null;
      });

      return tamed;
    }

    function tameHTMLCollection(nodeList, editable, opt_tameNodeCtor) {
      return ___.freeze(
          mixinHTMLCollection([], nodeList, editable, opt_tameNodeCtor));
    }

    function tameGetElementsByTagName(rootNode, tagName, editable) {
      tagName = String(tagName);
      if (tagName !== '*') {
        tagName = tagName.toLowerCase();
        if (!___.hasOwnProp(html4.ELEMENTS, tagName)
            || html4.ELEMENTS[tagName] & html4.ELEMENTS.UNSAFE) {
          // Allowing getElementsByTagName to work for opaque element types
          // would leak information about those elements.
          return new fakeNodeList([]);
        }
      }
      return tameNodeList(
          rootNode.getElementsByTagName(tagName), editable, defaultTameNode);
    }

    /**
     * Implements http://www.whatwg.org/specs/web-apps/current-work/#dom-document-getelementsbyclassname
     * using an existing implementation on browsers that have one.
     */
    function tameGetElementsByClassName(rootNode, className, editable) {
      className = String(className);

      // The quotes below are taken from the HTML5 draft referenced above.

      // "having obtained the classes by splitting a string on spaces"
      // Instead of using split, we use match with the global modifier so that
      // we don't have to remove leading and trailing spaces.
      var classes = className.match(/[^\t\n\f\r ]+/g);

      // Filter out classnames in the restricted namespace.
      for (var i = classes ? classes.length : 0; --i >= 0;) {
        var classi = classes[i];
        if (FORBIDDEN_ID_PATTERN.test(classi)) {
          classes[i] = classes[classes.length - 1];
          --classes.length;
        }
      }

      if (!classes || classes.length === 0) {
        // "If there are no tokens specified in the argument, then the method
        //  must return an empty NodeList" [instead of all elements]
        // This means that
        //     htmlEl.ownerDocument.getElementsByClassName(htmlEl.className)
        // will return an HtmlCollection containing htmlElement iff
        // htmlEl.className contains a non-space character.
        return fakeNodeList([]);
      }

      // "unordered set of unique space-separated tokens representing classes"
      if (typeof rootNode.getElementsByClassName === 'function') {
        return tameNodeList(
            rootNode.getElementsByClassName(
                classes.join(' ')), editable, defaultTameNode);
      } else {
        // Add spaces around each class so that we can use indexOf later to find
        // a match.
        // This use of indexOf is strictly incorrect since
        // http://www.whatwg.org/specs/web-apps/current-work/#reflecting-content-attributes-in-dom-attributes
        // does not normalize spaces in unordered sets of unique space-separated
        // tokens.  This is not a problem since HTML5 compliant implementations
        // already have a getElementsByClassName implementation, and legacy
        // implementations do normalize according to comments on issue 935.

        // We assume standards mode, so the HTML5 requirement that
        //   "If the document is in quirks mode, then the comparisons for the
        //    classes must be done in an ASCII case-insensitive  manner,"
        // is not operative.
        var nClasses = classes.length;
        for (var i = nClasses; --i >= 0;) {
          classes[i] = ' ' + classes[i] + ' ';
        }

        // We comply with the requirement that the result is a list
        //   "containing all the elements in the document, in tree order,"
        // since the spec for getElementsByTagName has the same language.
        var candidates = rootNode.getElementsByTagName('*');
        var matches = [];
        var limit = candidates.length;
        if (limit !== +limit) { limit = 1/0; }  // See issue 935
        candidate_loop:
        for (var j = 0, candidate, k = -1;
             j < limit && (candidate = candidates[j]);
             ++j) {
          var candidateClass = ' ' + candidate.className + ' ';
          for (var i = nClasses; --i >= 0;) {
            if (-1 === candidateClass.indexOf(classes[i])) {
              continue candidate_loop;
            }
          }
          var tamed = defaultTameNode(candidate, editable);
          if (tamed) {
            matches[++k] = tamed;
          }
        }
        // "the method must return a live NodeList object"
        return fakeNodeList(matches);
      }
    }

    function makeEventHandlerWrapper(thisNode, listener) {
      classUtils.ensureValidCallback(listener);
      function wrapper(event) {
        return plugin_dispatchEvent___(
            thisNode, event, ___.getId(imports), listener);
      }
      return wrapper;
    }

    var NOT_EDITABLE = "Node not editable.";
    var INVALID_SUFFIX = "Property names may not end in '__'.";
    var UNSAFE_TAGNAME = "Unsafe tag name.";
    var UNKNOWN_TAGNAME = "Unknown tag name.";
    var INDEX_SIZE_ERROR = "Index size error.";

    /**
     * Define a property with the given name on the given ctor's prototype that
     * is accessible to untrusted code.
     * @param {Function} ctor the ctor for the class to modify.
     * @param {string} name the name of the property to define.
     * @param {boolean} useAttrGetter true if the getter should delegate to
     *     {@code ctor.prototype.getAttribute}.  That method is assumed to
     *     already be trusted though {@code toValue} is still called on the
     *     result.
     *     If false, then {@code toValue} is called on the result of accessing
     *     the name property on the underlying element, a possibly untrusted
     *     value.
     * @param {Function} toValue transforms the attribute or underlying property
     *     value retrieved according to the useAttrGetter flag above to the
     *     value of the defined property.
     * @param {boolean} useAttrSetter like useAttrGetter but for a setter.
     *     Switches between the name property on the underlying node
     *     (the false case) or using ctor's {@code setAttribute} method
     *     (the true case).
     * @param {Function} fromValue called on the input before it is passed
     *     through according to the flag above.  This receives untrusted values,
     *     and can do any vetting or transformation.  If {@code useAttrSetter}
     *     is true then it need not do much value vetting since the
     *     {@code setAttribute} method must do its own vetting.
     */
    function defProperty(
        ctor, name, useAttrGetter, toValue, useAttrSetter, fromValue) {
      // Given the name foo, the suffix is Foo___ so we end up defining
      // inaccessible methods getFoo___ and setFoo__ on ctor's prototype below.
      var getterSetterSuffix = classUtils.getterSetterSuffix(name);
      var proto = ctor.prototype;
      if (toValue) {
        proto['get' + getterSetterSuffix] = useAttrGetter
            ? function () {
                return toValue.call(this, this.getAttribute(name));
              }
            : function () {
                return toValue.call(this, this.node___[name]);
              };
      }
      if (fromValue) {
        proto['set' + getterSetterSuffix] = useAttrSetter
            ? function (value) {
                this.setAttribute(name, fromValue.call(this, value));
                return value;
              }
            : function (value) {
                if (!this.editable___) { throw new Error(NOT_EDITABLE); }
                this.node___[name] = fromValue.call(this, value);
                return value;
              };
      }
    }
    function defAttributeAlias(ctor, name, toValue, fromValue) {
      defProperty(ctor, name, true, toValue, true, fromValue);
    }

    // Implementation of EventTarget::addEventListener
    function tameAddEventListener(name, listener, useCapture) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      if (!this.wrappedListeners___) { this.wrappedListeners___ = []; }
      useCapture = Boolean(useCapture);
      var wrappedListener = makeEventHandlerWrapper(this.node___, listener);
      wrappedListener = bridal.addEventListener(
          this.node___, name, wrappedListener, useCapture);
      wrappedListener.originalListener___ = listener;
      this.wrappedListeners___.push(wrappedListener);
    }

    // Implementation of EventTarget::removeEventListener
    function tameRemoveEventListener(name, listener, useCapture) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      if (!this.wrappedListeners___) { return; }
      var wrappedListener = null;
      for (var i = this.wrappedListeners___.length; --i >= 0;) {
        if (this.wrappedListeners___[i].originalListener___ === listener) {
          wrappedListener = this.wrappedListeners___[i];
          arrayRemove(this.wrappedListeners___, i, i);
          break;
        }
      }
      if (!wrappedListener) { return; }
      bridal.removeEventListener(
          this.node___, name, wrappedListener, useCapture);
    }

    // A map of tamed node classes, keyed by DOM Level 2 standard name, which
    // will be exposed to the client.
    var nodeClasses = {};

    function inertCtor(tamedCtor, someSuper, name) {
      return nodeClasses[name] = ___.extend(tamedCtor, someSuper, name);
    }

    var tameNodeFields = [
        'nodeType', 'nodeValue', 'nodeName', 'firstChild',
        'lastChild', 'nextSibling', 'previousSibling', 'parentNode',
        'ownerDocument', 'childNodes', 'attributes'];

    /**
     * Base class for a Node wrapper.  Do not create directly -- use the
     * tameNode factory instead.
     * @param {boolean} editable true if the node's value, attributes, children,
     *     or custom properties are mutable.
     * @constructor
     */
    function TameNode(editable) {
      this.editable___ = editable;
      TameNodeMark.stamp.mark___(this);
      classUtils.exportFields(this, tameNodeFields);
    }
    inertCtor(TameNode, Object, 'Node');
    TameNode.prototype.getOwnerDocument___ = function () {
      // TODO(mikesamuel): upward navigation breaks capability discipline.
      if (!this.editable___ && tameDocument.editable___) {
        throw new Error(NOT_EDITABLE);
      }
      return tameDocument;
    };
    // abstract TameNode.prototype.getNodeType___
    // abstract TameNode.prototype.getNodeName___
    // abstract TameNode.prototype.getNodeValue___
    // abstract TameNode.prototype.cloneNode
    // abstract TameNode.prototype.appendChild
    // abstract TameNode.prototype.insertBefore
    // abstract TameNode.prototype.removeChild
    // abstract TameNode.prototype.replaceChild
    // abstract TameNode.prototype.getFirstChild___
    // abstract TameNode.prototype.getLastChild___
    // abstract TameNode.prototype.getNextSibling___
    // abstract TameNode.prototype.getPreviousSibling___
    // abstract TameNode.prototype.getParentNode___
    // abstract TameNode.prototype.getElementsByTagName
    // abstract TameNode.prototype.getElementsByClassName
    // abstract TameNode.prototype.getChildNodes___
    // abstract TameNode.prototype.getAttributes___
    var tameNodePublicMembers = [
        'cloneNode',
        'appendChild', 'insertBefore', 'removeChild', 'replaceChild',
        'getElementsByClassName', 'getElementsByTagName', 'dispatchEvent',
        'hasChildNodes'
        ];

    /**
     * A tame node that is backed by a real node.
     * @param {boolean} childrenEditable true iff the child list is mutable.
     * @constructor
     */
    function TameBackedNode(node, editable, childrenEditable) {
      if (!node) {
        throw new Error('Creating tame node with undefined native delegate');
      }
      this.node___ = node;
      this.childrenEditable___ = editable && childrenEditable;
      this.FERAL_TWIN___ = node;
      try {
        // bug 1330 IE<=8 doesn't allow arbitrary properties on text nodes,
        // so this can fail, but failures are mostly harmless.
        node.TAMED_TWIN___ = this;
      } catch (ex) {
        ___.log("Warning: couldn't set TAMED_TWIN___ for " + node);
      }
      TameNode.call(this, editable);
    }
    ___.extend(TameBackedNode, TameNode);
    TameBackedNode.prototype.getNodeType___ = function () {
      return this.node___.nodeType;
    };
    TameBackedNode.prototype.getNodeName___ = function () {
      return this.node___.nodeName;
    };
    TameBackedNode.prototype.getNodeValue___ = function () {
      return this.node___.nodeValue;
    };
    TameBackedNode.prototype.cloneNode = function (deep) {
      var clone = bridal.cloneNode(this.node___, Boolean(deep));
      // From http://www.w3.org/TR/DOM-Level-2-Core/core.html#ID-3A0ED0A4
      //     "Note that cloning an immutable subtree results in a mutable copy"
      return defaultTameNode(clone, true);
    };
    TameBackedNode.prototype.appendChild = function (child) {
      // Child must be editable since appendChild can remove it from its parent.
      child = TameNodeT.coerce(child);
      if (!this.childrenEditable___ || !child.editable___) {
        throw new Error(NOT_EDITABLE);
      }
      this.node___.appendChild(child.node___);
      return child;
    };
    TameBackedNode.prototype.insertBefore = function (toInsert, child) {
      toInsert = TameNodeT.coerce(toInsert);
      if (child === void 0) { child = null; }
      if (child !== null) {
        child = TameNodeT.coerce(child);
        if (!child.editable___) {
          throw new Error(NOT_EDITABLE);
        }
      }
      if (!this.childrenEditable___ || !toInsert.editable___) {
        throw new Error(NOT_EDITABLE);
      }
      this.node___.insertBefore(
          toInsert.node___, child !== null ? child.node___ : null);
      return toInsert;
    };
    TameBackedNode.prototype.removeChild = function (child) {
      child = TameNodeT.coerce(child);
      if (!this.childrenEditable___ || !child.editable___) {
        throw new Error(NOT_EDITABLE);
      }
      this.node___.removeChild(child.node___);
      return child;
    };
    TameBackedNode.prototype.replaceChild = function (newChild, oldChild) {
      newChild = TameNodeT.coerce(newChild);
      oldChild = TameNodeT.coerce(oldChild);
      if (!this.childrenEditable___ || !newChild.editable___
          || !oldChild.editable___) {
        throw new Error(NOT_EDITABLE);
      }
      this.node___.replaceChild(newChild.node___, oldChild.node___);
      return oldChild;
    };
    TameBackedNode.prototype.getFirstChild___ = function () {
      return defaultTameNode(this.node___.firstChild, this.childrenEditable___);
    };
    TameBackedNode.prototype.getLastChild___ = function () {
      return defaultTameNode(this.node___.lastChild, this.childrenEditable___);
    };
    TameBackedNode.prototype.getNextSibling___ = function () {
      return tameRelatedNode(this.node___.nextSibling, this.editable___,
                             defaultTameNode);
    };
    TameBackedNode.prototype.getPreviousSibling___ = function () {
      return tameRelatedNode(this.node___.previousSibling, this.editable___,
                             defaultTameNode);
    };
    TameBackedNode.prototype.getParentNode___ = function () {
      return tameRelatedNode(
          this.node___.parentNode, this.editable___, defaultTameNode);
    };
    TameBackedNode.prototype.getElementsByTagName = function (tagName) {
      return tameGetElementsByTagName(
          this.node___, tagName, this.childrenEditable___);
    };
    TameBackedNode.prototype.getElementsByClassName = function (className) {
      return tameGetElementsByClassName(
          this.node___, className, this.childrenEditable___);
    };
    TameBackedNode.prototype.getChildNodes___ = function () {
      return tameNodeList(
          this.node___.childNodes, this.childrenEditable___, defaultTameNode);
    };
    TameBackedNode.prototype.getAttributes___ = function () {
      var thisNode = this.node___;
      var tameNodeCtor = function(node, editable) {
        return new TameBackedAttributeNode(node, editable, thisNode);
      };
      return tameNodeList(
          this.node___.attributes, this.editable___, tameNodeCtor);
    };
    var endsWith__ = /__$/;
    // TODO(erights): Come up with some notion of a keeper chain so we can
    // say, "let every other keeper try to handle this first".
    TameBackedNode.prototype.handleRead___ = function (name) {
      name = String(name);
      if (endsWith__.test(name)) { return void 0; }
      var handlerName = name + '_getter___';
      if (this[handlerName]) {
        return this[handlerName]();
      }
      handlerName = handlerName.toLowerCase();
      if (this[handlerName]) {
        return this[handlerName]();
      }
      if (___.hasOwnProp(this.node___.properties___, name)) {
        return this.node___.properties___[name];
      } else {
        return void 0;
      }
    };
    TameBackedNode.prototype.handleCall___ = function (name, args) {
      name = String(name);
      if (endsWith__.test(name)) { throw new Error(INVALID_SUFFIX); }
      var handlerName = name + '_handler___';
      if (this[handlerName]) {
        return this[handlerName].call(this, args);
      }
      handlerName = handlerName.toLowerCase();
      if (this[handlerName]) {
        return this[handlerName].call(this, args);
      }
      if (___.hasOwnProp(this.node___.properties___, name)) {
        return this.node___.properties___[name].call(this, args);
      } else {
        throw new TypeError(name + ' is not a function.');
      }
    };
    TameBackedNode.prototype.handleSet___ = function (name, val) {
      name = String(name);
      if (endsWith__.test(name)) { throw new Error(INVALID_SUFFIX); }
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      var handlerName = name + '_setter___';
      if (this[handlerName]) {
        return this[handlerName](val);
      }
      handlerName = handlerName.toLowerCase();
      if (this[handlerName]) {
        return this[handlerName](val);
      }
      if (!this.node___.properties___) {
        this.node___.properties___ = {};
      }
      this[name + '_canEnum___'] = true;
      return this.node___.properties___[name] = val;
    };
    TameBackedNode.prototype.handleDelete___ = function (name) {
      name = String(name);
      if (endsWith__.test(name)) { throw new Error(INVALID_SUFFIX); }
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      var handlerName = name + '_deleter___';
      if (this[handlerName]) {
        return this[handlerName]();
      }
      handlerName = handlerName.toLowerCase();
      if (this[handlerName]) {
        return this[handlerName]();
      }
      if (this.node___.properties___) {
        return (
            delete this.node___.properties___[name]
            && delete this[name + '_canEnum___']);
      } else {
        return true;
      }
    };
    /**
     * @param {boolean} ownFlag ignored
     */
    TameBackedNode.prototype.handleEnum___ = function (ownFlag) {
      // TODO(metaweta): Add code to list all the other handled stuff we know
      // about.
      if (this.node___.properties___) {
        return ___.allKeys(this.node___.properties___);
      }
      return [];
    };
    TameBackedNode.prototype.hasChildNodes = function () {
      return !!this.node___.hasChildNodes();
    };
    // http://www.w3.org/TR/DOM-Level-2-Events/events.html#Events-EventTarget :
    // "The EventTarget interface is implemented by all Nodes"
    TameBackedNode.prototype.dispatchEvent = function dispatchEvent(evt) {
      evt = TameEventT.coerce(evt);
      bridal.dispatchEvent(this.node___, evt.event___);
    };
    ___.all2(
        ___.grantTypedMethod, TameBackedNode.prototype, tameNodePublicMembers);
    if (document.documentElement.contains) {  // typeof is 'object' on IE
      TameBackedNode.prototype.contains = function (other) {
        other = TameNodeT.coerce(other);
        var otherNode = other.node___;
        return this.node___.contains(otherNode);
      };
    }
    if ('function' ===
        typeof document.documentElement.compareDocumentPosition) {
      /**
       * Speced in <a href="http://www.w3.org/TR/DOM-Level-3-Core/core.html#Node3-compareDocumentPosition">DOM-Level-3</a>.
       */
      TameBackedNode.prototype.compareDocumentPosition = function (other) {
        other = TameNodeT.coerce(other);
        var otherNode = other.node___;
        if (!otherNode) { return 0; }
        var bitmask = +this.node___.compareDocumentPosition(otherNode);
        // To avoid leaking information about the relative positioning of
        // different roots, if neither contains the other, then we mask out
        // the preceding/following bits.
        // 0x18 is (CONTAINS | CONTAINED)
        // 0x1f is all the bits documented at
        //     http://www.w3.org/TR/DOM-Level-3-Core/core.html#DocumentPosition
        //     except IMPLEMENTATION_SPECIFIC
        // 0x01 is DISCONNECTED
        /*
        if (!(bitmask & 0x18)) {
          // TODO: If they are not under the same virtual doc root, return
          // DOCUMENT_POSITION_DISCONNECTED instead of leaking information
          // about PRECEDING | FOLLOWING.
        }
        */
        // Firefox3 returns spurious PRECEDING and FOLLOWING bits for
        // disconnected trees.
        // https://bugzilla.mozilla.org/show_bug.cgi?id=486002
        if (bitmask & 1) {
          bitmask &= ~6;
        }
        return bitmask & 0x1f;
      };
      if (!___.hasOwnProp(TameBackedNode.prototype, 'contains')) {
        // http://www.quirksmode.org/blog/archives/2006/01/contains_for_mo.html
        TameBackedNode.prototype.contains = function (other) {
          var docPos = this.compareDocumentPosition(other);
          return !(!(docPos & 0x10) && docPos);
        };
      }
    }
    ___.all2(function (o, k) {
               if (___.hasOwnProp(o, k)) { ___.grantTypedMethod(o, k);  }
             }, TameBackedNode.prototype,
             ['contains', 'compareDocumentPosition']);

    /**
     * A fake node that is not backed by a real DOM node.
     * @constructor
     */
    function TamePseudoNode(editable) {
      TameNode.call(this, editable);
      this.properties___ = {};
      this.TAMED_TWIN___ = this.FERAL_TWIN___ = this;
    }
    ___.extend(TamePseudoNode, TameNode);
    TamePseudoNode.prototype.appendChild =
    TamePseudoNode.prototype.insertBefore =
    TamePseudoNode.prototype.removeChild =
    TamePseudoNode.prototype.replaceChild = function () {
      ___.log("Node not editable; no action performed.");
      return void 0;
    };
    TamePseudoNode.prototype.getFirstChild___ = function () {
      var children = this.getChildNodes___();
      return children.length ? children[0] : null;
    };
    TamePseudoNode.prototype.getLastChild___ = function () {
      var children = this.getChildNodes___();
      return children.length ? children[children.length - 1] : null;
    };
    TamePseudoNode.prototype.getNextSibling___ = function () {
      var parentNode = this.getParentNode___();
      if (!parentNode) { return null; }
      var siblings = parentNode.getChildNodes___();
      for (var i = siblings.length - 1; --i >= 0;) {
        if (siblings[i] === this) { return siblings[i + 1]; }
      }
      return null;
    };
    TamePseudoNode.prototype.getPreviousSibling___ = function () {
      var parentNode = this.getParentNode___();
      if (!parentNode) { return null; }
      var siblings = parentNode.getChildNodes___();
      for (var i = siblings.length; --i >= 1;) {
        if (siblings[i] === this) { return siblings[i - 1]; }
      }
      return null;
    };
    TamePseudoNode.prototype.handleRead___ = function (name) {
      name = String(name);
      if (endsWith__.test(name)) { return void 0; }
      var handlerName = name + '_getter___';
      if (this[handlerName]) {
        return this[handlerName]();
      }
      handlerName = handlerName.toLowerCase();
      if (this[handlerName]) {
        return this[handlerName]();
      }
      if (___.hasOwnProp(this.properties___, name)) {
        return this.properties___[name];
      } else {
        return void 0;
      }
    };
    TamePseudoNode.prototype.handleCall___ = function (name, args) {
      name = String(name);
      if (endsWith__.test(name)) { throw new Error(INVALID_SUFFIX); }
      var handlerName = name + '_handler___';
      if (this[handlerName]) {
        return this[handlerName].call(this, args);
      }
      handlerName = handlerName.toLowerCase();
      if (this[handlerName]) {
        return this[handlerName].call(this, args);
      }
      if (___.hasOwnProp(this.properties___, name)) {
        return this.properties___[name].call(this, args);
      } else {
        throw new TypeError(name + ' is not a function.');
      }
    };
    TamePseudoNode.prototype.handleSet___ = function (name, val) {
      name = String(name);
      if (endsWith__.test(name)) { throw new Error(INVALID_SUFFIX); }
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      var handlerName = name + '_setter___';
      if (this[handlerName]) {
        return this[handlerName](val);
      }
      handlerName = handlerName.toLowerCase();
      if (this[handlerName]) {
        return this[handlerName](val);
      }
      if (!this.properties___) {
        this.properties___ = {};
      }
      this[name + '_canEnum___'] = true;
      return this.properties___[name] = val;
    };
    TamePseudoNode.prototype.handleDelete___ = function (name) {
      name = String(name);
      if (endsWith__.test(name)) { throw new Error(INVALID_SUFFIX); }
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      var handlerName = name + '_deleter___';
      if (this[handlerName]) {
        return this[handlerName]();
      }
      handlerName = handlerName.toLowerCase();
      if (this[handlerName]) {
        return this[handlerName]();
      }
      if (this.properties___) {
        return (
            delete this.properties___[name]
            && delete this[name + '_canEnum___']);
      } else {
        return true;
      }
    };
    TamePseudoNode.prototype.handleEnum___ = function (ownFlag) {
      // TODO(metaweta): Add code to list all the other handled stuff we know
      // about.
      if (this.properties___) {
        return ___.allKeys(this.properties___);
      }
      return [];
    };
    TamePseudoNode.prototype.hasChildNodes = function () {
      return this.getFirstChild___() != null;
    };
    ___.all2(
        ___.grantTypedMethod, TamePseudoNode.prototype, tameNodePublicMembers);

    var commonElementPropertyHandlers = {
      clientWidth: {
        get: function () { return this.getGeometryDelegate___().clientWidth; }
      },
      clientHeight: {
        get: function () { return this.getGeometryDelegate___().clientHeight; }
      },
      offsetLeft: {
        get: function () { return this.getGeometryDelegate___().offsetLeft; }
      },
      offsetTop: {
        get: function () { return this.getGeometryDelegate___().offsetTop; }
      },
      offsetWidth: {
        get: function () { return this.getGeometryDelegate___().offsetWidth; }
      },
      offsetHeight: {
        get: function () { return this.getGeometryDelegate___().offsetHeight; }
      },
      scrollLeft: {
        get: function () { return this.getGeometryDelegate___().scrollLeft; },
        set: function (x) {
          if (!this.editable___) { throw new Error(NOT_EDITABLE); }
          this.getGeometryDelegate___().scrollLeft = +x;
          return x;
        }
      },
      scrollTop: {
        get: function () { return this.getGeometryDelegate___().scrollTop; },
        set: function (y) {
          if (!this.editable___) { throw new Error(NOT_EDITABLE); }
          this.getGeometryDelegate___().scrollTop = +y;
          return y;
        }
      },
      scrollWidth: {
        get: function () { return this.getGeometryDelegate___().scrollWidth; }
      },
      scrollHeight: {
        get: function () { return this.getGeometryDelegate___().scrollHeight; }
      }
    };

    function TamePseudoElement(
        tagName, tameDoc, childNodesGetter, parentNodeGetter, innerHTMLGetter,
        geometryDelegate, editable) {
      TamePseudoNode.call(this, editable);
      this.tagName___ = tagName;
      this.tameDoc___ = tameDoc;
      this.childNodesGetter___ = childNodesGetter;
      this.parentNodeGetter___ = parentNodeGetter;
      this.innerHTMLGetter___ = innerHTMLGetter;
      this.geometryDelegate___ = geometryDelegate;
      classUtils.exportFields(this, ['tagName', 'innerHTML']);
      classUtils.applyAccessors(this, commonElementPropertyHandlers);
    }
    ___.extend(TamePseudoElement, TamePseudoNode);
    // TODO(mikesamuel): make nodeClasses work.
    TamePseudoElement.prototype.getNodeType___ = function () { return 1; };
    TamePseudoElement.prototype.getNodeName___
        = TamePseudoElement.prototype.getTagName___
        = function () { return this.tagName___; };
    TamePseudoElement.prototype.getNodeValue___ = function () { return null; };
    TamePseudoElement.prototype.getAttribute
        = function (attribName) { return null; };
    TamePseudoElement.prototype.setAttribute
        = function (attribName, value) { };
    TamePseudoElement.prototype.hasAttribute
        = function (attribName) { return false; };
    TamePseudoElement.prototype.removeAttribute
        = function (attribName) { };
    TamePseudoElement.prototype.getOwnerDocument___
        = function () { return this.tameDoc___; };
    TamePseudoElement.prototype.getChildNodes___
        = function () { return this.childNodesGetter___(); };
    TamePseudoElement.prototype.getAttributes___
        = function () { return tameNodeList([], false, undefined); };
    TamePseudoElement.prototype.getParentNode___
        = function () { return this.parentNodeGetter___(); };
    TamePseudoElement.prototype.getInnerHTML___
        = function () { return this.innerHTMLGetter___(); };
    TamePseudoElement.prototype.getElementsByTagName = function (tagName) {
      tagName = String(tagName).toLowerCase();
      if (tagName === this.tagName___) {
        // Works since html, head, body, and title can't contain themselves.
        return fakeNodeList([]);
      }
      return this.getOwnerDocument___().getElementsByTagName(tagName);
    };
    TamePseudoElement.prototype.getElementsByClassName = function (className) {
      return this.getOwnerDocument___().getElementsByClassName(className);
    };
    TamePseudoElement.prototype.getBoundingClientRect = function () {
      return this.geometryDelegate___.getBoundingClientRect();
    };
    TamePseudoElement.prototype.getGeometryDelegate___ = function () {
      return this.geometryDelegate___;
    };
    TamePseudoElement.prototype.toString = ___.markFuncFreeze(function () {
      return '<' + this.tagName___ + '>';
    });
    ___.all2(___.grantTypedMethod, TamePseudoElement.prototype,
             ['getAttribute', 'setAttribute',
              'hasAttribute', 'removeAttribute',
              'getBoundingClientRect', 'getElementsByTagName']);

    function TameOpaqueNode(node, editable) {
      TameBackedNode.call(this, node, editable, editable);
    }
    ___.extend(TameOpaqueNode, TameBackedNode);
    TameOpaqueNode.prototype.getNodeValue___
        = TameBackedNode.prototype.getNodeValue___;
    TameOpaqueNode.prototype.getNodeType___
        = TameBackedNode.prototype.getNodeType___;
    TameOpaqueNode.prototype.getNodeName___
        = TameBackedNode.prototype.getNodeName___;
    TameOpaqueNode.prototype.getNextSibling___
        = TameBackedNode.prototype.getNextSibling___;
    TameOpaqueNode.prototype.getPreviousSibling___
        = TameBackedNode.prototype.getPreviousSibling___;
    TameOpaqueNode.prototype.getFirstChild___
        = TameBackedNode.prototype.getFirstChild___;
    TameOpaqueNode.prototype.getLastChild___
        = TameBackedNode.prototype.getLastChild___;
    TameOpaqueNode.prototype.getParentNode___
        = TameBackedNode.prototype.getParentNode___;
    TameOpaqueNode.prototype.getChildNodes___
        = TameBackedNode.prototype.getChildNodes___;
    TameOpaqueNode.prototype.getOwnerDocument___
        = TameBackedNode.prototype.getOwnerDocument___;
    TameOpaqueNode.prototype.getElementsByTagName
        = TameBackedNode.prototype.getElementsByTagName;
    TameOpaqueNode.prototype.getElementsByClassName
        = TameBackedNode.prototype.getElementsByClassName;
    TameOpaqueNode.prototype.hasChildNodes
        = TameBackedNode.prototype.hasChildNodes;
    TameOpaqueNode.prototype.getAttributes___
        = function () { return tameNodeList([], false, undefined); };
    for (var i = tameNodePublicMembers.length; --i >= 0;) {
      var k = tameNodePublicMembers[i];
      if (!TameOpaqueNode.prototype.hasOwnProperty(k)) {
        TameOpaqueNode.prototype[k] = ___.markFuncFreeze(function () {
          throw new Error('Node is opaque');
        });
      }
    }
    ___.all2(
        ___.grantTypedMethod, TameOpaqueNode.prototype, tameNodePublicMembers);

    function TameTextNode(node, editable) {
      assert(node.nodeType === 3);

      // The below should not be strictly necessary since childrenEditable for
      // TameScriptElements is always false, but it protects against tameNode
      // being called naively on a text node from container code.
      var pn = node.parentNode;
      if (editable && pn) {
        if (1 === pn.nodeType
            && (html4.ELEMENTS[pn.tagName.toLowerCase()]
                & html4.eflags.UNSAFE)) {
          // Do not allow mutation of text inside script elements.
          // See the testScriptLoading testcase for examples of exploits.
          editable = false;
        }
      }

      TameBackedNode.call(this, node, editable, editable);
      classUtils.exportFields(
          this, ['nodeValue', 'data', 'textContent', 'innerText']);
    }
    inertCtor(TameTextNode, TameBackedNode, 'Text');
    TameTextNode.prototype.setNodeValue___ = function (value) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      this.node___.nodeValue = String(value || '');
      return value;
    };
    TameTextNode.prototype.getTextContent___
        = TameTextNode.prototype.getInnerText___
        = TameTextNode.prototype.getData___
        = TameTextNode.prototype.getNodeValue___;
    TameTextNode.prototype.setTextContent___
        = TameTextNode.prototype.setInnerText___
        = TameTextNode.prototype.setData___
        = TameTextNode.prototype.setNodeValue___;
    TameTextNode.prototype.toString = ___.markFuncFreeze(function () {
      return '#text';
    });

    function TameCommentNode(node, editable) {
      assert(node.nodeType === 8);
      TameBackedNode.call(this, node, editable, editable);
    }
    inertCtor(TameCommentNode, TameBackedNode, 'CommentNode');
    TameCommentNode.prototype.toString = ___.markFuncFreeze(function () {
      return '#comment';
    });

    function lookupAttribute(map, tagName, attribName) {
      var attribKey;
      attribKey = tagName + '::' + attribName;
      if (map.hasOwnProperty(attribKey)) {
        return map[attribKey];
      }
      attribKey = '*::' + attribName;
      if (map.hasOwnProperty(attribKey)) {
        return map[attribKey];
      }
      return void 0;
    }
    function getAttributeType(tagName, attribName) {
      return lookupAttribute(html4.ATTRIBS, tagName, attribName);
    }
    function getLoaderType(tagName, attribName) {
      return lookupAttribute(html4.LOADERTYPES, tagName, attribName);
    }
    function getUriEffect(tagName, attribName) {
      return lookupAttribute(html4.URIEFFECTS, tagName, attribName);
    }

    /**
     * Plays the role of an Attr node for TameElement objects.
     */
    function TameBackedAttributeNode(node, editable, ownerElement) {
      TameBackedNode.call(this, node, editable);
      this.ownerElement___ = ownerElement;
      classUtils.exportFields(this,
          ['name', 'specified', 'value', 'ownerElement']);
    }
    inertCtor(TameBackedAttributeNode, TameBackedNode, 'Attr');
    TameBackedAttributeNode.prototype.getNodeName___ =
    TameBackedAttributeNode.prototype.getName___ =
        function () { return String(this.node___.name); };
    TameBackedAttributeNode.prototype.getSpecified___ = function () {
      return defaultTameNode(this.ownerElement___, this.editable___)
          .hasAttribute(this.getName___());
    };
    TameBackedAttributeNode.prototype.getNodeValue___ =
    TameBackedAttributeNode.prototype.getValue___ = function () {
      return defaultTameNode(this.ownerElement___, this.editable___)
          .getAttribute(this.getName___());
    };
    TameBackedAttributeNode.prototype.setNodeValue___ =
    TameBackedAttributeNode.prototype.setValue___ = function (value) {
      return defaultTameNode(this.ownerElement___, this.editable___)
          .setAttribute(this.getName___(), value);
    };
    TameBackedAttributeNode.prototype.getOwnerElement___ = function () {
      return defaultTameNode(this.ownerElement___, this.editable___);
    };
    TameBackedAttributeNode.prototype.getNodeType___ = function () {
      return 2;
    };
    TameBackedAttributeNode.prototype.cloneNode = function (deep) {
      var clone = bridal.cloneNode(this.node___, Boolean(deep));
      // From http://www.w3.org/TR/DOM-Level-2-Core/core.html#ID-3A0ED0A4
      //     "Note that cloning an immutable subtree results in a mutable copy"
      return new TameBackedAttributeNode(clone, true, this.ownerElement____);
    };
    TameBackedAttributeNode.prototype.appendChild =
    TameBackedAttributeNode.prototype.insertBefore =
    TameBackedAttributeNode.prototype.removeChild =
    TameBackedAttributeNode.prototype.replaceChild =
    TameBackedAttributeNode.prototype.getFirstChild___ =
    TameBackedAttributeNode.prototype.getLastChild___ =
    TameBackedAttributeNode.prototype.getNextSibling___ =
    TameBackedAttributeNode.prototype.getPreviousSibling___ =
    TameBackedAttributeNode.prototype.getParentNode___ =
    TameBackedAttributeNode.prototype.getElementsByTagName =
    TameBackedAttributeNode.prototype.getElementsByClassName =
    TameBackedAttributeNode.prototype.getChildNodes___ =
    TameBackedAttributeNode.prototype.getAttributes___ = function () {
      throw new Error ("Not implemented.");
    };
    TameBackedAttributeNode.prototype.toString = 
        ___.markFuncFreeze(function () {
          return '[Fake attribute node]';
        });

    // Register set handlers for onclick, onmouseover, etc.
    function registerElementScriptAttributeHandlers(aTameElement) {
      var attrNameRe = /::(.*)/;
      for (var html4Attrib in html4.ATTRIBS) {
        if (html4.atype.SCRIPT === html4.ATTRIBS[html4Attrib]) {
          (function (attribName) {
            ___.useSetHandler(
                aTameElement,
                attribName,
                function eventHandlerSetter(listener) {
                  if (!this.editable___) { throw new Error(NOT_EDITABLE); }
                  if (!listener) {  // Clear the current handler
                    this.node___[attribName] = null;
                  } else {
                    // This handler cannot be copied from one node to another
                    // which is why getters are not yet supported.
                    this.node___[attribName] = makeEventHandlerWrapper(
                        this.node___, listener);
                  }
                  return listener;
                });
           })(html4Attrib.match(attrNameRe)[1]);
        }
      }
    }

    function TameElement(node, editable, childrenEditable) {
      assert(node.nodeType === 1);
      TameBackedNode.call(this, node, editable, childrenEditable);
      classUtils.exportFields(
          this,
          ['className', 'id', 'innerHTML', 'tagName', 'style',
           'offsetParent', 'title', 'dir', 'innerText', 'textContent']);
      classUtils.applyAccessors(this, commonElementPropertyHandlers);
      registerElementScriptAttributeHandlers(this);
    }
    nodeClasses.Element = inertCtor(TameElement, TameBackedNode, 'HTMLElement');
    TameElement.prototype.blur = function () {
      this.node___.blur();
    };
    TameElement.prototype.focus = function () {
      if (imports.isProcessingEvent___) {
        this.node___.focus();
      }
    };
    // IE-specific method.  Sets the element that will have focus when the
    // window has focus, without focusing the window.
    if (document.documentElement.setActive) {
      TameElement.prototype.setActive = function () {
        if (imports.isProcessingEvent___) {
          this.node___.setActive();
        }
      };
      ___.grantTypedMethod(TameElement.prototype, 'setActive');
    }
    // IE-specific method.
    if (document.documentElement.hasFocus) {
      TameElement.prototype.hasFocus = function () {
        return this.node___.hasFocus();
      };
      ___.grantTypedMethod(TameElement.prototype, 'hasFocus');
    }
    defAttributeAlias(TameElement, 'id', defaultToEmptyStr, identity);
    TameElement.prototype.getAttribute = function (attribName) {
      attribName = String(attribName).toLowerCase();
      var tagName = this.node___.tagName.toLowerCase();
      var atype = getAttributeType(tagName, attribName);
      if (atype === void 0) {
        // Unrecognized attribute; use virtual map
        if (this.node___.attributes___) {
          return this.node___.attributes___[attribName] || null;
        }
        return null;
      }
      var value = bridal.getAttribute(this.node___, attribName);
      if ('string' !== typeof value) { return value; }
      return virtualizeAttributeValue(atype, value);
    };
    TameElement.prototype.getAttributeNode = function (name) {
      var hostDomNode = this.node___.getAttributeNode(name);
      if (hostDomNode === null) { return null; }
      return new TameBackedAttributeNode(
          hostDomNode, this.editable___, this.node___);
    };
    TameElement.prototype.hasAttribute = function (attribName) {
      attribName = String(attribName).toLowerCase();
      var tagName = this.node___.tagName.toLowerCase();
      var atype = getAttributeType(tagName, attribName);
      if (atype === void 0) {
        // Unrecognized attribute; use virtual map
        return !!(
            this.node___.attributes___ &&
            ___.hasOwnProp(this.node___.attributes___, attribName));
      } else {
        return bridal.hasAttribute(this.node___, attribName);
      }
    };
    TameElement.prototype.setAttribute = function (attribName, value) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      attribName = String(attribName).toLowerCase();
      var tagName = this.node___.tagName.toLowerCase();
      var atype = getAttributeType(tagName, attribName);
      if (atype === void 0) {
        // Unrecognized attribute; use virtual map
        if (!this.node___.attributes___) { this.node___.attributes___ = {}; }
        this.node___.attributes___[attribName] = String(value);
      } else {
        var sanitizedValue = rewriteAttribute(
            tagName, attribName, atype, value);
        if (sanitizedValue !== null) {
          bridal.setAttribute(this.node___, attribName, sanitizedValue);
        }
      }
      return value;
    };
    TameElement.prototype.removeAttribute = function (attribName) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      attribName = String(attribName).toLowerCase();
      var tagName = this.node___.tagName.toLowerCase();
      var atype = getAttributeType(tagName, attribName);
      if (atype === void 0) {
        // Unrecognized attribute; use virtual map
        if (this.node___.attributes___) {
          delete this.node___.attributes___[attribName];
        }
      } else {
        this.node___.removeAttribute(attribName);
      }
    };
    TameElement.prototype.getBoundingClientRect = function () {
      var elRect = bridal.getBoundingClientRect(this.node___);
      var vbody = bridal.getBoundingClientRect(
          this.getOwnerDocument___().body___);
      var vbodyLeft = vbody.left, vbodyTop = vbody.top;
      return ({
                top: elRect.top - vbodyTop,
                left: elRect.left - vbodyLeft,
                right: elRect.right - vbodyLeft,
                bottom: elRect.bottom - vbodyTop
              });
    };
    TameElement.prototype.getClassName___ = function () {
      return this.getAttribute('class') || '';
    };
    TameElement.prototype.setClassName___ = function (classes) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      return this.setAttribute('class', String(classes));
    };
    function defaultToEmptyStr(x) { return x || ''; }
    defAttributeAlias(TameElement, 'title', defaultToEmptyStr, String);
    defAttributeAlias(TameElement, 'dir', defaultToEmptyStr, String);
    function innerTextOf(rawNode, out) {
      switch (rawNode.nodeType) {
        case 1:  // Element
          var tagName = rawNode.tagName.toLowerCase();
          if (html4.ELEMENTS.hasOwnProperty(tagName)
              && !(html4.ELEMENTS[tagName] & html4.eflags.UNSAFE)) {
            // Not an opaque node.
            for (var c = rawNode.firstChild; c; c = c.nextSibling) {
              innerTextOf(c, out);
            }
          }
          break;
        case 3:  // Text Node
        case 4:  // CDATA Section Node
          out[out.length] = rawNode.data;
          break;
        case 11:  // Document Fragment
          for (var c = rawNode.firstChild; c; c = c.nextSibling) {
            innerTextOf(c, out);
          }
          break;
      }
    }
    TameElement.prototype.getTextContent___
        = TameElement.prototype.getInnerText___
        = function () {
      var text = [];
      innerTextOf(this.node___, text);
      return text.join('');
    };
    TameElement.prototype.setTextContent___
        = TameElement.prototype.setInnerText___
        = function (newText) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      var newTextStr = newText != null ? String(newText) : '';
      var el = this.node___;
      for (var c; (c = el.firstChild);) { el.removeChild(c); }
      if (newTextStr) {
        el.appendChild(el.ownerDocument.createTextNode(newTextStr));
      }
      return newText;
    };
    TameElement.prototype.getTagName___
        = TameBackedNode.prototype.getNodeName___;
    TameElement.prototype.getInnerHTML___ = function () {
      var tagName = this.node___.tagName.toLowerCase();
      if (!html4.ELEMENTS.hasOwnProperty(tagName)) {
        return '';  // unknown node
      }
      var flags = html4.ELEMENTS[tagName];
      var innerHtml = this.node___.innerHTML;
      if (flags & html4.eflags.CDATA) {
        innerHtml = html.escapeAttrib(innerHtml);
      } else if (flags & html4.eflags.RCDATA) {
        // Make sure we return PCDATA.
        // For RCDATA we only need to escape & if they're not part of an entity.
        innerHtml = html.normalizeRCData(innerHtml);
      } else {
        // If we blessed the resulting HTML, then this would round trip better
        // but it would still not survive appending, and it would propagate
        // event handlers where the setter of innerHTML does not expect it to.
        innerHtml = tameInnerHtml(innerHtml);
      }
      return innerHtml;
    };
    TameElement.prototype.setInnerHTML___ = function (htmlFragment) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      var tagName = this.node___.tagName.toLowerCase();
      if (!html4.ELEMENTS.hasOwnProperty(tagName)) { throw new Error(); }
      var flags = html4.ELEMENTS[tagName];
      if (flags & html4.eflags.UNSAFE) { throw new Error(); }
      var isRcData = flags & html4.eflags.RCDATA;
      var htmlFragmentString;
      if (!isRcData && htmlFragment instanceof Html) {
        htmlFragmentString = '' + safeHtml(htmlFragment);
      } else if (htmlFragment === null) {
        htmlFragmentString = '';
      } else {
        htmlFragmentString = '' + htmlFragment;
      }
      var sanitizedHtml;
      if (isRcData) {
        sanitizedHtml = html.normalizeRCData(htmlFragmentString);
      } else {
        sanitizedHtml = sanitizeHtml(htmlFragmentString);
      }
      this.node___.innerHTML = sanitizedHtml;
      return htmlFragment;
    };
    function identity(x) { return x; }
    defProperty(
        TameElement, 'style',
        false, 
        function (styleNode) {
          return new TameStyle(styleNode, this.editable___, this);
        },
        true, identity);
    TameElement.prototype.updateStyle = function (style) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      var cssPropertiesAndValues = cssSealerUnsealerPair.unseal(style);
      if (!cssPropertiesAndValues) { throw new Error(); }

      var styleNode = this.node___.style;
      for (var i = 0; i < cssPropertiesAndValues.length; i += 2) {
        var propName = cssPropertiesAndValues[i];
        var propValue = cssPropertiesAndValues[i + 1];
        // If the propertyName differs between DOM and CSS, there will
        // be a semicolon between the two.
        // E.g., 'background-color;backgroundColor'
        // See CssTemplate.toPropertyValueList.
        var semi = propName.indexOf(';');
        if (semi >= 0) { propName = propName.substring(semi + 1); }
        styleNode[propName] = propValue;
      }
    };

    TameElement.prototype.getOffsetParent___ = function () {
      return tameRelatedNode(
          this.node___.offsetParent, this.editable___, defaultTameNode);
    };
    TameElement.prototype.getGeometryDelegate___ = function () {
      return this.node___;
    };
    TameElement.prototype.toString = ___.markFuncFreeze(function () {
      return '<' + this.node___.tagName + '>';
    });
    TameElement.prototype.addEventListener = tameAddEventListener;
    TameElement.prototype.removeEventListener = tameRemoveEventListener;
    ___.all2(
       ___.grantTypedMethod, TameElement.prototype,
       ['addEventListener', 'removeEventListener',
        'blur', 'focus',
        'getAttribute', 'setAttribute',
        'removeAttribute', 'hasAttribute',
        'getAttributeNode',
        'getBoundingClientRect',
        'updateStyle']);

    function TameAElement(node, editable) {
      TameElement.call(this, node, editable, editable);
      classUtils.exportFields(this, ['href']);
    }
    inertCtor(TameAElement, TameElement, 'HTMLAnchorElement');
    defProperty(TameAElement, 'href', false, identity, true, identity);

    // http://dev.w3.org/html5/spec/Overview.html#the-canvas-element
    var TameCanvasElement = (function() {
      function isFont(value) {
        return typeof(value) == "string" && css.properties.font.test(value);
      }
      function isColor(value) {
        // Note: we're testing against the pattern for the CSS "color:"
        // property, but what is actually referenced by the draft canvas spec is
        // the CSS syntactic element <color>, which is why we need to
        // specifically exclude "inherit".
        return typeof(value) == "string" && css.properties.color.test(value) &&
            value !== "inherit";
      }
      var colorNameTable = {
        // http://dev.w3.org/csswg/css3-color/#html4 as cited by
        // http://dev.w3.org/html5/2dcontext/#dom-context-2d-fillstyle
        // TODO(kpreid): avoid duplication with table in CssRewriter.java
        " black":   "#000000",
        " silver":  "#c0c0c0",
        " gray":    "#808080",
        " white":   "#ffffff",
        " maroon":  "#800000",
        " red":     "#ff0000",
        " purple":  "#800080",
        " fuchsia": "#ff00ff",
        " green":   "#008000",
        " lime":    "#00ff00",
        " olive":   "#808000",
        " yellow":  "#ffff00",
        " navy":    "#000080",
        " blue":    "#0000ff",
        " teal":    "#008080",
        " aqua":    "#00ffff"
      };
      function StringTest(strings) {
        var table = {};
        // The table itself as a value is a marker to avoid running into
        // Object.prototype properties.
        for (var i = strings.length; --i >= 0;) { table[strings[i]] = table; }
        return ___.markFuncFreeze(function (string) {
          return typeof string === 'string' && table[string] === table;
        });
      }
      function canonColor(colorString) {
        // http://dev.w3.org/html5/2dcontext/ says the color shall be returned
        // only as #hhhhhh, not as names.
        return colorNameTable[" " + colorString] || colorString;
      }
      function TameImageData(imageData) {
        // Since we can't interpose indexing, we can't wrap the CanvasPixelArray
        // so we have to copy the pixel data. This is horrible, bad, and awful.
        var tameImageData = {
          toString: ___.markFuncFreeze(function () {
              return "[Domita Canvas ImageData]"; }),
          width: ___.enforceType(imageData.width, "number"),
          height: ___.enforceType(imageData.height, "number"),
        
          // used to unwrap for passing to putImageData
          bare___: imageData,
        
           // lazily constructed tame copy, backs .data accessor; also used to
           // test whether we need to write-back the copy before a putImageData
          tamePixelArray___: undefined
        };
        ___.tamesTo(imageData, tameImageData);
        TameImageDataMark.stamp.mark___(tameImageData);
        classUtils.applyAccessors(tameImageData, {
          data: {
            // Accessor used so we don't need to copy if the client is just
            // blitting (getImageData -> putImageData) rather than inspecting
            // the pixels.
            get: ___.markFuncFreeze(function () {
              if (!tameImageData.tamePixelArray___) {
                var bareArray = imageData.data;
                var length = bareArray.length;
                var tamePixelArray = { // not frozen, user-modifiable
                  // TODO: Investigate whether it would be an optimization to
                  // make this an array with properties added.
                  toString: ___.markFuncFreeze(function () {
                      return "[Domita CanvasPixelArray]"; }),
                  canvas_writeback___: function () {
                    // This is invoked just before each putImageData
                    for (var i = length-1; i >= 0; i--) {
                      bareArray[i] = tamePixelArray[i];
                    }
                  }
                };
                for (var i = length-1; i >= 0; i--) {
                  tamePixelArray[i] = bareArray[i];
                }
                tameImageData.tamePixelArray___ = tamePixelArray;
              }
              return tameImageData.tamePixelArray___;
            })
          }
        });
        return ___.primFreeze(tameImageData);
      }
      function TameGradient(gradient) {
        var tameGradient = {
          toString: ___.markFuncFreeze(function () {
              return "[Domita CanvasGradient]"; }),
          addColorStop: ___.markFuncFreeze(function (offset, color) {
            ___.enforceType(offset, 'number', 'color stop offset');
            if (!(0 <= offset && offset <= 1)) {
              throw new Error(INDEX_SIZE_ERROR);
              // TODO(kpreid): should be a DOMException per spec
            }
            if (!isColor(color)) {
              throw new Error("SYNTAX_ERR");
              // TODO(kpreid): should be a DOMException per spec
            }
            gradient.addColorStop(offset, color);
          })
        };
        ___.tamesTo(gradient, tameGradient);
        TameGradientMark.stamp.mark___(tameGradient);
        return ___.primFreeze(tameGradient);
      }
      function enforceFinite(value, name) {
        ___.enforceType(value, 'number', name);
        if (!isFinite(value)) {
          throw new Error("NOT_SUPPORTED_ERR");
          // TODO(kpreid): should be a DOMException per spec
        }
      }

      function TameCanvasElement(node, editable) {
        TameElement.call(this, node, editable, editable);
        classUtils.exportFields(
            this,
            ['height', 'width']);
      
        // helpers for tame context
        var context = node.getContext('2d');
        function tameFloatsOp(count, verb) {
          return ___.markFuncFreeze(function () {
            if (arguments.length !== count) {
              throw new Error(verb + ' takes ' + count + ' args, not ' +
                              arguments.length);
            }
            for (var i = 0; i < count; i++) {
              ___.enforceType(arguments[i], 'number', verb + ' argument ' + i);
            }
            context[verb].apply(context, arguments);
          });
        }
        function tameRectMethod(m, hasResult) {
          return ___.markFuncFreeze(function (x, y, w, h) {
            if (arguments.length !== 4) {
              throw new Error(m + ' takes 4 args, not ' +
                              arguments.length);
            }
            ___.enforceType(x, 'number', 'x');
            ___.enforceType(y, 'number', 'y');
            ___.enforceType(w, 'number', 'width');
            ___.enforceType(h, 'number', 'height');
            if (hasResult) {
              return m.call(context, x, y, w, h);            
            } else {
              m.call(context, x, y, w, h);
            }
          });
        }
        function tameDrawText(m) {
          return ___.markFuncFreeze(function (text, x, y, maxWidth) {
            ___.enforceType(text, 'string', 'text');
            ___.enforceType(x, 'number', 'x');
            ___.enforceType(y, 'number', 'y');
            switch (arguments.length) {
            case 3:
              m.apply(context, arguments);
              return;
            case 4:
              ___.enforceType(maxWidth, 'number', 'maxWidth');
              m.apply(context, arguments);
              return;
            default:
              throw new Error(m + ' cannot accept ' + arguments.length +
                                  ' arguments');
            }
          });
        }
        function tameGetMethod(prop) {
          return ___.markFuncFreeze(function () { return context[prop]; });
        }
        function tameSetMethod(prop, validator) {
          return ___.markFuncFreeze(function (newValue) {
            if (validator(newValue)) {
              context[prop] = newValue;
            }
            return newValue;
          });
        }
        function tameGetStyleMethod(prop) {
          return ___.markFuncFreeze(function () {
            var value = context[prop];
            if (typeof(value) == "string") {
              return canonColor(value);
            } else if (___.passesGuard(TameGradientT, ___.tame(value))) {
              return ___.tame(value);
            } else {
              throw("Internal: Can't tame value " + value + " of " + prop);
            }
          });
        }
        function tameSetStyleMethod(prop) {
          return ___.markFuncFreeze(function (newValue) {
            if (isColor(newValue)) {
              context[prop] = newValue;
            } else if (typeof(newValue) === "object" &&
                       ___.passesGuard(TameGradientT, newValue)) {
              context[prop] = ___.untame(newValue);
            } // else do nothing
            return newValue;
          });
        }
        function tameSimpleOp(m) {  // no return value
          return ___.markFuncFreeze(function () {
            if (arguments.length !== 0) {
              throw new Error(m + ' takes no args, not ' + arguments.length);
            }
            m.call(context);
          });
        }
      
        // Design note: We generally reject the wrong number of arguments,
        // unlike default JS behavior. This is because we are just passing data
        // through to the underlying implementation, but we don't want to pass
        // on anything which might be an extension we don't know about, and it
        // is better to fail explicitly than to leave the client wondering about
        // why their extension usage isn't working.
      
        // http://dev.w3.org/html5/2dcontext/
        this.tameContext2d___ = {
          toString: ___.markFuncFreeze(function () {
              return "[Domita CanvasRenderingContext2D]"; }),

          save: tameSimpleOp(context.save),
          restore: tameSimpleOp(context.restore),
        
          scale: tameFloatsOp(2, 'scale'),
          rotate: tameFloatsOp(1, 'rotate'),
          translate: tameFloatsOp(2, 'translate'),
          transform: tameFloatsOp(6, 'transform'),
          setTransform: tameFloatsOp(6, 'setTransform'),
        
          createLinearGradient: ___.markFuncFreeze(function (x0, y0, x1, y1) {
            if (arguments.length !== 4) {
              throw new Error('createLinearGradient takes 4 args, not ' +
                              arguments.length);
            }
            ___.enforceType(x0, 'number', 'x0');
            ___.enforceType(y0, 'number', 'y0');
            ___.enforceType(x1, 'number', 'x1');
            ___.enforceType(y1, 'number', 'y1');
            return TameGradient(context.createLinearGradient(x0, y0, x1, y1));
          }),
          createRadialGradient: ___.markFuncFreeze(function (x0, y0, r0, x1, y1, r1) {
            if (arguments.length !== 6) {
              throw new Error('createRadialGradient takes 6 args, not ' +
                              arguments.length);
            }
            ___.enforceType(x0, 'number', 'x0');
            ___.enforceType(y0, 'number', 'y0');
            ___.enforceType(r0, 'number', 'r0');
            ___.enforceType(x1, 'number', 'x1');
            ___.enforceType(y1, 'number', 'y1');
            ___.enforceType(r1, 'number', 'r1');
            return TameGradient(context.createRadialGradient(x0, y0, r0, x1, y1, r1));
          }),

          createPattern: ___.markFuncFreeze(function (imageElement, repetition) {
            // Consider what policy to have wrt reading the pixels from image
            // elements before implementing this.
            throw new Error('Domita: canvas createPattern not yet implemented');
          }),
        
                
          clearRect:  tameRectMethod(context.clearRect,  false),
          fillRect:   tameRectMethod(context.fillRect,   false),
          strokeRect: tameRectMethod(context.strokeRect, false),

          beginPath: tameSimpleOp(context.beginPath),
          closePath: tameSimpleOp(context.closePath),
          moveTo: tameFloatsOp(2, 'moveTo'),
          lineTo: tameFloatsOp(2, 'lineTo'),
          quadraticCurveTo: tameFloatsOp(4, 'quadraticCurveTo'),
          bezierCurveTo: tameFloatsOp(6, 'bezierCurveTo'),
          arcTo: tameFloatsOp(5, 'arcTo'),
          rect: tameFloatsOp(4, 'rect'),
          arc: ___.markFuncFreeze(function (x, y, radius, startAngle, endAngle,
                                            anticlockwise) {
            if (arguments.length !== 6) {
              throw new Error('arc takes 6 args, not ' + arguments.length);
            }
            ___.enforceType(x, 'number', 'x');
            ___.enforceType(y, 'number', 'y');
            ___.enforceType(radius, 'number', 'radius');
            ___.enforceType(startAngle, 'number', 'startAngle');
            ___.enforceType(endAngle, 'number', 'endAngle');
            ___.enforceType(anticlockwise, 'boolean', 'anticlockwise');
            if (radius < 0) {
              throw new Error(INDEX_SIZE_ERROR);
              // TODO(kpreid): should be a DOMException per spec
            }
            context.arc(x, y, radius, startAngle, endAngle, anticlockwise);
          }),
          fill: tameSimpleOp(context.fill),
          stroke: tameSimpleOp(context.stroke),
          clip: tameSimpleOp(context.clip),
        
          isPointInPath: ___.markFuncFreeze(function (x, y) {
            ___.enforceType(x, 'number', 'x');
            ___.enforceType(y, 'number', 'y');
            return ___.enforceType(context.isPointInPath(x, y), 'boolean');
          }),
        
          fillText: tameDrawText(context.fillText),
          strokeText: tameDrawText(context.strokeText),
          measureText: ___.markFuncFreeze(function (string) {
            if (arguments.length !== 1) {
              throw new Error('measureText takes 1 arg, not ' + arguments.length);
            }
            ___.enforceType(string, 'string', 'measureText argument');
            return context.measureText(string);
          }),
        
          drawImage: ___.markFuncFreeze(function (imageElement) {
            // Consider what policy to have wrt reading the pixels from image
            // elements before implementing this.
            throw new Error('Domita: canvas drawImage not yet implemented');
          }),
        
          createImageData: ___.markFuncFreeze(function (sw, sh) {
            if (arguments.length !== 2) {
              throw new Error('createImageData takes 2 args, not ' +
                              arguments.length);
            }
            ___.enforceType(sw, 'number', 'sw');
            ___.enforceType(sh, 'number', 'sh');
            return TameImageData(context.createImageData(sw, sh));
          }),
          getImageData: tameRectMethod(function (sx, sy, sw, sh) {
            return TameImageData(context.getImageData(sx, sy, sw, sh));
          }, true),
          putImageData: ___.markFuncFreeze(function 
              (tameImageData, dx, dy, dirtyX, dirtyY, dirtyWidth, dirtyHeight) {
            tameImageData = TameImageDataT.coerce(tameImageData);
            enforceFinite(dx, 'dx');
            enforceFinite(dy, 'dy');
            switch (arguments.length) {
            case 3:
              dirtyX = 0;
              dirtyY = 0;
              dirtyWidth = tameImageData.width;
              dirtyHeight = tameImageData.height;
              break;
            case 7:
              enforceFinite(dirtyX, 'dirtyX');
              enforceFinite(dirtyY, 'dirtyY');
              enforceFinite(dirtyWidth, 'dirtyWidth');
              enforceFinite(dirtyHeight, 'dirtyHeight');
              break;
            default:
              throw 'putImageData cannot accept ' + arguments.length +
                  ' arguments';
            }
            if (tameImageData.tamePixelArray___) {
              tameImageData.tamePixelArray___.canvas_writeback___();
            }
            context.putImageData(tameImageData.bare___, dx, dy, dirtyX, dirtyY,
                                 dirtyWidth, dirtyHeight);
          })
        };
      
        if ("drawFocusRing" in context) {
          this.tameContext2d___.drawFocusRing = ___.markFuncFreeze(function
              (tameElement, x, y, canDrawCustom) {
            switch (arguments.length) {
            case 3:
              canDrawCustom = false;
              break;
            case 4:
              break;
            default:
              throw 'drawFocusRing cannot accept ' + arguments.length +
                  ' arguments';
            }
            tameElement = TameNodeT.coerce(tameElement);
            ___.enforceType(x, 'number', 'x');
            ___.enforceType(y, 'number', 'y');
            ___.enforceType(canDrawCustom, 'boolean', 'canDrawCustom');
          
            // On safety of using the untamed node here: The only information
            // drawFocusRing takes from the node is whether it is focused.
            return ___.enforceType(
                context.drawFocusRing(tameElement.node___, x, y, canDrawCustom),
                'boolean');
          });
        }
      
        classUtils.applyAccessors(this.tameContext2d___, {
          // We filter the values supplied to setters in case some browser
          // extension makes them more powerful, e.g. containing scripting or a
          // URL.
          // TODO(kpreid): Do we want to filter the *getters* as well? Scenarios:
          // (a) canvas shared with innocent code, (b) browser quirks?? If we do,
          // then what should be done with a bad value?
          globalAlpha: {
            get: tameGetMethod('globalAlpha'),
            set: tameSetMethod('globalAlpha',
                function (v) { return typeof v === "number" &&
                                      0.0 <= v && v <= 1.0;     })
          },
          globalCompositeOperation: {
            get: tameGetMethod('globalCompositeOperation'),
            set: tameSetMethod('globalCompositeOperation',
                StringTest([
                  "source-atop",
                  "source-in",
                  "source-out",
                  "source-over",
                  "destination-atop",
                  "destination-in",
                  "destination-out",
                  "destination-over",
                  "lighter",
                  "copy",
                  "xor"
                ]))
          },
          strokeStyle: {
            get: tameGetStyleMethod('strokeStyle'),
            set: tameSetStyleMethod('strokeStyle')
          },
          fillStyle: {
            get: tameGetStyleMethod('fillStyle'),
            set: tameSetStyleMethod('fillStyle')
          },
          lineWidth: {
            get: tameGetMethod('lineWidth'),
            set: tameSetMethod('lineWidth',
                function (v) { return typeof v === "number" &&
                                      0.0 < v && v !== Infinity; })
          },
          lineCap: {
            get: tameGetMethod('lineCap'),
            set: tameSetMethod('lineCap', StringTest([
              "butt",
              "round",
              "square"
            ]))
          },
          lineJoin: {
            get: tameGetMethod('lineJoin'),
            set: tameSetMethod('lineJoin', StringTest([
              "bevel",
              "round",
              "miter"
            ]))
          },
          miterLimit: {
            get: tameGetMethod('miterLimit'),
            set: tameSetMethod('miterLimit',
                function (v) { return typeof v === "number" &&
                                      0 < v && v !== Infinity; })
          },
          shadowOffsetX: {
            get: tameGetMethod('shadowOffsetX'),
            set: tameSetMethod('shadowOffsetX',
                function (v) { return typeof v === "number" && isFinite(v); })
          },
          shadowOffsetY: {
            get: tameGetMethod('shadowOffsetY'),
            set: tameSetMethod('shadowOffsetY',
                function (v) { return typeof v === "number" && isFinite(v); })
          },
          shadowBlur: {
            get: tameGetMethod('shadowBlur'),
            set: tameSetMethod('shadowBlur',
                function (v) { return typeof v === "number" &&
                                      0.0 <= v && v !== Infinity; })
          },
          shadowColor: {
            get: tameGetStyleMethod('shadowColor'),
            set: tameSetMethod('shadowColor', isColor)
          },
        
          font: {
            get: tameGetMethod('font'),
            set: tameSetMethod('font', isFont)
          },
          textAlign: {
            get: tameGetMethod('textAlign'),
            set: tameSetMethod('textAlign', StringTest([
              "start",
              "end",
              "left",
              "right",
              "center"
            ]))
          },
          textBaseline: {
            get: tameGetMethod('textBaseline'),
            set: tameSetMethod('textBaseline', StringTest([
              "top",
              "hanging",
              "middle",
              "alphabetic",
              "ideographic",
              "bottom"
            ]))
          }
        });
        ___.primFreeze(this.tameContext2d___);
      }  // end of TameCanvasElement
      inertCtor(TameCanvasElement, TameElement, 'HTMLCanvasElement');
      TameCanvasElement.prototype.getContext = function (contextId) {
      
        // TODO(kpreid): We can refine this by inventing a ReadOnlyCanvas object
        // to return in this situation, which allows getImageData and so on but
        // not any drawing. Not bothering to do that for now; if you have a use
        // for it let us know.
        if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      
        ___.enforceType(contextId, 'string', 'contextId');
        switch (contextId) {
          case '2d':
            return this.tameContext2d___;
          default:
            // http://dev.w3.org/html5/spec/the-canvas-element.html#the-canvas-element
            // says: The getContext(contextId, args...) method of the canvas
            // element, when invoked, must run the following steps:
            // [...]
            //     If contextId is not the name of a context supported by the
            //     user agent, return null and abort these steps.
            //
            // However, Mozilla throws and WebKit returns undefined instead.
            // Returning undefined rather than null is closer to the spec than
            // throwing.
            return undefined;
            throw new Error('Unapproved canvas contextId');
        }
      };
      defProperty(TameCanvasElement, 'height', false, identity, false, Number);
      defProperty(TameCanvasElement, 'width', false, identity, false, Number);
      ___.all2(___.grantTypedMethod, TameCanvasElement.prototype,
               ['getContext']);
      
      return TameCanvasElement;
    })();

    // http://www.w3.org/TR/DOM-Level-2-HTML/html.html#ID-40002357
    function TameFormElement(node, editable) {
      TameElement.call(this, node, editable, editable);
      this.length = node.length;
      classUtils.exportFields(
          this, ['action', 'elements', 'enctype', 'method', 'target']);
    }
    inertCtor(TameFormElement, TameElement, 'HTMLFormElement');
    TameFormElement.prototype.handleRead___ = function (name) {
      name = String(name);
      if (endsWith__.test(name)) { return void 0; }
      // TODO(ihab.awad): Due to the following bug:
      //     http://code.google.com/p/google-caja/issues/detail?id=997
      // the read handlers get called on the *prototypes* as well as the
      // instances on which they are installed. In that case, we just
      // defer to the super handler, which works for now.
      if (___.passesGuard(TameNodeT, this)) {
        var tameElements = this.getElements___();
        if (___.hasOwnProp(tameElements, name)) { return tameElements[name]; }
      }
      return TameBackedNode.prototype.handleRead___.call(this, name);
    };
    TameFormElement.prototype.submit = function () {
      return this.node___.submit();
    };
    TameFormElement.prototype.reset = function () {
      return this.node___.reset();
    };
    defAttributeAlias(TameFormElement, 'action', defaultToEmptyStr, String);
    TameFormElement.prototype.getElements___ = function () {
      return tameHTMLCollection(
          this.node___.elements, this.editable___, defaultTameNode);
    };
    defAttributeAlias(TameFormElement, 'enctype', defaultToEmptyStr, String);
    defAttributeAlias(TameFormElement, 'method', defaultToEmptyStr, String);
    defAttributeAlias(TameFormElement, 'target', defaultToEmptyStr, String);
    TameFormElement.prototype.reset = function () {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      this.node___.reset();
    };
    TameFormElement.prototype.submit = function () {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      this.node___.submit();
    };
    ___.all2(___.grantTypedMethod, TameFormElement.prototype,
             ['reset', 'submit']);


    function TameInputElement(node, editable) {
      TameElement.call(this, node, editable, editable);
      classUtils.exportFields(
          this,
          ['form', 'value', 'defaultValue',
           'checked', 'disabled', 'readOnly',
           'options', 'selected', 'selectedIndex',
           'name', 'accessKey', 'tabIndex', 'text',
           'defaultChecked', 'defaultSelected', 'maxLength',
           'size', 'type', 'index', 'label',
           'multiple', 'cols', 'rows']);
    }
    inertCtor(TameInputElement, TameElement, 'HTMLInputElement');
    defProperty(TameInputElement, 'checked', false, identity, false, Boolean);
    defProperty(
        TameInputElement, 'defaultChecked', false, identity, false, identity);
    defProperty(
        TameInputElement, 'value',
        false, function (x) { return x == null ? null : String(x); },
        false, function (x) { return x == null ? '' : '' + x; });
    defProperty(
        TameInputElement, 'defaultValue',
        false, function (x) { return x == null ? null : String(x); },
        false, function (x) { return x == null ? '' : '' + x; });
    TameInputElement.prototype.select = function () {
      this.node___.select();
    };
    TameInputElement.prototype.getForm___ = function () {
      return tameRelatedNode(
          this.node___.form, this.editable___, defaultTameNode);
    };
    defProperty(TameInputElement, 'disabled', false, identity, false, identity);
    defProperty(TameInputElement, 'readOnly', false, identity, false, identity);
    TameInputElement.prototype.getOptions___ = function () {
      return tameOptionsList(
          this.node___.options, this.editable___, defaultTameNode, 'name');
    };
    defProperty(
        TameInputElement, 'selected', false, identity, false, identity);
    defProperty(
        TameInputElement, 'defaultSelected', false, identity, false, Boolean);
    function toInt(x) { return x | 0; }
    defProperty(
        TameInputElement, 'selectedIndex', false, identity, false, toInt);
    defProperty(TameInputElement, 'name', false, identity, false, identity);
    defProperty(
        TameInputElement, 'accessKey', false, identity, false, identity);
    defProperty(TameInputElement, 'tabIndex', false, identity, false, identity);
    defProperty(TameInputElement, 'text', false, String);
    defProperty(
        TameInputElement, 'maxLength', false, identity, false, identity);
    defProperty(TameInputElement, 'size', false, identity, false, identity);
    defProperty(TameInputElement, 'type', false, identity, false, identity);
    defProperty(TameInputElement, 'index', false, identity, false, identity);
    defProperty(TameInputElement, 'label', false, identity, false, identity);
    defProperty(TameInputElement, 'multiple', false, identity, false, identity);
    defProperty(TameInputElement, 'cols', false, identity, false, identity);
    defProperty(TameInputElement, 'rows', false, identity, false, identity);
    ___.all2(___.grantTypedMethod, TameInputElement.prototype, ['select']);


    function TameImageElement(node, editable) {
      TameElement.call(this, node, editable, editable);
      classUtils.exportFields(this, ['src', 'alt']);
    }
    inertCtor(TameImageElement, TameElement, 'HTMLImageElement');
    defProperty(TameImageElement, 'src', false, identity, true, identity);
    defProperty(TameImageElement, 'alt', false, identity, false, String);

    function TameLabelElement(node, editable) {
      TameElement.call(this, node, editable, editable);
      classUtils.exportFields(this, ['htmlFor']);
    }
    inertCtor(TameLabelElement, TameElement, 'HTMLLabelElement');
    TameLabelElement.prototype.getHtmlFor___ = function () {
      return this.getAttribute('for');
    };
    TameLabelElement.prototype.setHtmlFor___ = function (id) {
      this.setAttribute('for', id);
      return id;
    };

    /**
     * A script element wrapper that allows setting of a src that has been
     * rewritten by a URI policy, but not modifying of textual content.
     */
    function TameScriptElement(node, editable) {
      // Make the child list immutable so that text content can't be added
      // or removed.
      TameElement.call(this, node, editable, false);
      classUtils.exportFields(this, ['src']);
    }
    inertCtor(TameScriptElement, TameElement, 'HTMLScriptElement');
    defProperty(TameScriptElement, 'src', false, identity, true, identity);

    function TameIFrameElement(node, editable) {
      // Make the child list immutable so that text content can't be added
      // or removed.
      TameElement.call(this, node, editable, false);
      classUtils.exportFields(
          this,
          ['align', 'frameBorder', 'height', 'width']);
    }
    inertCtor(TameIFrameElement, TameElement, "HTMLIFrameElement");
    TameIFrameElement.prototype.getAlign___ = function () {
      return this.node___.align;
    };
    TameIFrameElement.prototype.setAlign___ = function (alignment) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      alignment = String(alignment);
      if (alignment === 'left' ||
          alignment === 'right' ||
          alignment === 'center') {
        this.node___.align = alignment;
      }
    };
    TameIFrameElement.prototype.getAttribute = function(attr) {
      var attrLc = String(attr).toLowerCase();
      if (attrLc !== 'name' && attrLc !== 'src') {
        return TameElement.prototype.getAttribute.call(this, attr);
      }
      return null;
    };
    TameIFrameElement.prototype.setAttribute = function(attr, value) {
      var attrLc = String(attr).toLowerCase();
      // The 'name' and 'src' attributes are whitelisted for all tags in
      // html4-attributes-whitelist.json, since they're needed on tags
      // like <img>.  Because there's currently no way to filter attributes
      // based on the tag, we have to blacklist these two here.
      if (attrLc !== 'name' && attrLc !== 'src') {
        return TameElement.prototype.setAttribute.call(this, attr, value);
      }
      ___.log('Cannot set the [' + attrLc + '] attribute of an iframe.');
      return value;
    };
    TameIFrameElement.prototype.getFrameBorder___ = function () {
      return this.node___.frameBorder;
    };
    TameIFrameElement.prototype.setFrameBorder___ = function (border) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      border = String(border).toLowerCase();
      if (border === '0' || border === '1' ||
          border === 'no' || border === 'yes') {
        this.node___.frameBorder = border;
      }
    };
    defProperty(TameIFrameElement, 'height', false, identity, false, Number);
    defProperty(TameIFrameElement, 'width', false, identity, false, Number);
    TameIFrameElement.prototype.handleRead___ = function (name) {
      var nameLc = String(name).toLowerCase();
      if (nameLc !== 'src' && nameLc !== 'name') {
        return TameElement.prototype.handleRead___.call(this, name);
      }
      return undefined;
    };
    TameIFrameElement.prototype.handleSet___ = function (name, value) {
      var nameLc = String(name).toLowerCase();
      if (nameLc !== 'src' && nameLc !== 'name') {
        return TameElement.prototype.handleSet___.call(this, name, value);
      }
      ___.log('Cannot set the [' + nameLc + '] property of an iframe.');
      return value;
    };
    ___.all2(___.grantTypedMethod, TameIFrameElement.prototype,
             ['getAttribute', 'setAttribute']);



    function TameTableCompElement(node, editable) {
      TameElement.call(this, node, editable, editable);
      classUtils.exportFields(
          this,
          ['colSpan', 'cells', 'cellIndex', 'rowSpan', 'rows', 'rowIndex',
           'align', 'vAlign', 'nowrap', 'sectionRowIndex']);
    }
    ___.extend(TameTableCompElement, TameElement);
    defProperty(
        TameTableCompElement, 'colSpan', false, identity, false, identity);
    TameTableCompElement.prototype.getCells___ = function () {
      return tameNodeList(
          this.node___.cells, this.editable___, defaultTameNode);
    };
    TameTableCompElement.prototype.getCellIndex___ = function () {
      return this.node___.cellIndex;
    };
    defProperty(
        TameTableCompElement, 'rowSpan', false, identity, false, identity);
    TameTableCompElement.prototype.getRows___ = function () {
      return tameNodeList(this.node___.rows, this.editable___, defaultTameNode);
    };
    TameTableCompElement.prototype.getRowIndex___ = function () {
      return this.node___.rowIndex;
    };
    TameTableCompElement.prototype.getSectionRowIndex___ = function () {
      return this.node___.sectionRowIndex;
    };
    defProperty(
        TameTableCompElement, 'align', false, identity, false, identity);
    defProperty(
        TameTableCompElement, 'vAlign', false, identity, false, identity);
    defProperty(
        TameTableCompElement, 'nowrap', false, identity, false, identity);
    TameTableCompElement.prototype.insertRow = function (index) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      requireIntIn(index, -1, this.node___.rows.length);
      return defaultTameNode(this.node___.insertRow(index), this.editable___);
    };
    TameTableCompElement.prototype.deleteRow = function (index) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      requireIntIn(index, -1, this.node___.rows.length);
      this.node___.deleteRow(index);
    };
    ___.all2(___.grantTypedMethod, TameTableCompElement.prototype,
             ['insertRow', 'deleteRow']);

    function requireIntIn(idx, min, max) {
      if (idx !== (idx | 0) || idx < min || idx > max) {
        throw new Error(INDEX_SIZE_ERROR);
      }
    }

    function TameTableRowElement(node, editable) {
      TameTableCompElement.call(this, node, editable);
    }
    inertCtor(TameTableRowElement, TameTableCompElement, 'HTMLTableRowElement');
    TameTableRowElement.prototype.insertCell = function (index) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      requireIntIn(index, -1, this.node___.cells.length);
      return defaultTameNode(
          this.node___.insertCell(index),
          this.editable___);
    };
    TameTableRowElement.prototype.deleteCell = function (index) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      requireIntIn(index, -1, this.node___.cells.length);
      this.node___.deleteCell(index);
    };
    ___.all2(___.grantTypedMethod, TameTableRowElement.prototype,
             ['insertCell', 'deleteCell']);

    function TameTableElement(node, editable) {
      TameTableCompElement.call(this, node, editable);
      classUtils.exportFields(
          this,
          ['tBodies', 'tHead', 'tFoot', 'cellPadding', 'cellSpacing', 'border']
          );
    }
    inertCtor(TameTableElement, TameTableCompElement, 'HTMLTableElement');
    TameTableElement.prototype.getTBodies___ = function () {
      return tameNodeList(
          this.node___.tBodies, this.editable___, defaultTameNode);
    };
    TameTableElement.prototype.getTHead___ = function () {
      return defaultTameNode(this.node___.tHead, this.editable___);
    };
    TameTableElement.prototype.getTFoot___ = function () {
      return defaultTameNode(this.node___.tFoot, this.editable___);
    };
    TameTableElement.prototype.createTHead = function () {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      return defaultTameNode(this.node___.createTHead(), this.editable___);
    };
    TameTableElement.prototype.deleteTHead = function () {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      this.node___.deleteTHead();
    };
    TameTableElement.prototype.createTFoot = function () {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      return defaultTameNode(this.node___.createTFoot(), this.editable___);
    };
    TameTableElement.prototype.deleteTFoot = function () {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      this.node___.deleteTFoot();
    };
    TameTableElement.prototype.createCaption = function () {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      return defaultTameNode(this.node___.createCaption(), this.editable___);
    };
    TameTableElement.prototype.deleteCaption = function () {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      this.node___.deleteCaption();
    };
    TameTableElement.prototype.insertRow = function (index) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      requireIntIn(index, -1, this.node___.rows.length);
      return defaultTameNode(this.node___.insertRow(index), this.editable___);
    };
    TameTableElement.prototype.deleteRow = function (index) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      requireIntIn(index, -1, this.node___.rows.length);
      this.node___.deleteRow(index);
    };
    function fromInt(x) { return '' + (x | 0); }  // coerce null and false to 0
    defAttributeAlias(TameTableElement, 'cellPadding', Number, fromInt);
    defAttributeAlias(TameTableElement, 'cellSpacing', Number, fromInt);
    defAttributeAlias(TameTableElement, 'border', Number, fromInt);

    ___.all2(___.grantTypedMethod, TameTableElement.prototype,
             ['createTHead', 'deleteTHead', 'createTFoot', 'deleteTFoot',
              'createCaption', 'deleteCaption', 'insertRow', 'deleteRow']);

    function tameEvent(event) {
      if (event.tamed___) { return event.tamed___; }
      return event.tamed___ = new TameEvent(event);
    }

    function TameEvent(event) {
      assert(!!event);
      this.event___ = event;
      TameEventMark.stamp.mark___(this);
      classUtils.exportFields(
          this,
          ['type', 'target', 'pageX', 'pageY', 'altKey',
           'ctrlKey', 'metaKey', 'shiftKey', 'button',
           'screenX', 'screenY',
           'currentTarget', 'relatedTarget',
           'fromElement', 'toElement',
           'srcElement',
           'clientX', 'clientY', 'keyCode', 'which']);
    }
    inertCtor(TameEvent, Object, 'Event');
    TameEvent.prototype.getType___ = function () {
      return bridal.untameEventType(String(this.event___.type));
    };
    TameEvent.prototype.getTarget___ = function () {
      var event = this.event___;
      return tameRelatedNode(
          event.target || event.srcElement, true, defaultTameNode);
    };
    TameEvent.prototype.getSrcElement___ = function () {
      return tameRelatedNode(this.event___.srcElement, true, defaultTameNode);
    };
    TameEvent.prototype.getCurrentTarget___ = function () {
      var e = this.event___;
      return tameRelatedNode(e.currentTarget, true, defaultTameNode);
    };
    TameEvent.prototype.getRelatedTarget___ = function () {
      var e = this.event___;
      var t = e.relatedTarget;
      if (!t) {
        if (e.type === 'mouseout') {
          t = e.toElement;
        } else if (e.type === 'mouseover') {
          t = e.fromElement;
        }
      }
      return tameRelatedNode(t, true, defaultTameNode);
    };
    // relatedTarget is read-only.  this dummy setter is because some code
    // tries to workaround IE by setting a relatedTarget when it's not set.
    // code in a sandbox can't tell the difference between "falsey because
    // relatedTarget is not supported" and "falsey because relatedTarget is
    // outside sandbox".
    TameEvent.prototype.setRelatedTarget___ = function (newValue) {
      return newValue;
    };
    TameEvent.prototype.getFromElement___ = function () {
      return tameRelatedNode(this.event___.fromElement, true, defaultTameNode);
    };
    TameEvent.prototype.getToElement___ = function () {
      return tameRelatedNode(this.event___.toElement, true, defaultTameNode);
    };
    TameEvent.prototype.getPageX___ = function () {
      return Number(this.event___.pageX);
    };
    TameEvent.prototype.getPageY___ = function () {
      return Number(this.event___.pageY);
    };
    TameEvent.prototype.stopPropagation = function () {
      // TODO(mikesamuel): make sure event doesn't propagate to dispatched
      // events for this gadget only.
      // But don't allow it to stop propagation to the container.
      if (this.event___.stopPropagation) {
        this.event___.stopPropagation();
      } else {
        this.event___.cancelBubble = true;
      }
    };
    TameEvent.prototype.preventDefault = function () {
      // TODO(mikesamuel): make sure event doesn't propagate to dispatched
      // events for this gadget only.
      // But don't allow it to stop propagation to the container.
      if (this.event___.preventDefault) {
        this.event___.preventDefault();
      } else {
        this.event___.returnValue = false;
      }
    };
    TameEvent.prototype.getAltKey___ = function () {
      return Boolean(this.event___.altKey);
    };
    TameEvent.prototype.getCtrlKey___ = function () {
      return Boolean(this.event___.ctrlKey);
    };
    TameEvent.prototype.getMetaKey___ = function () {
      return Boolean(this.event___.metaKey);
    };
    TameEvent.prototype.getShiftKey___ = function () {
      return Boolean(this.event___.shiftKey);
    };
    TameEvent.prototype.getButton___ = function () {
      var e = this.event___;
      return e.button && Number(e.button);
    };
    TameEvent.prototype.getClientX___ = function () {
      return Number(this.event___.clientX);
    };
    TameEvent.prototype.getClientY___ = function () {
      return Number(this.event___.clientY);
    };
    TameEvent.prototype.getScreenX___ = function () {
      return Number(this.event___.screenX);
    };
    TameEvent.prototype.getScreenY___ = function () {
      return Number(this.event___.screenY);
    };
    TameEvent.prototype.getWhich___ = function () {
      var w = this.event___.which;
      return w && Number(w);
    };
    TameEvent.prototype.getKeyCode___ = function () {
      var kc = this.event___.keyCode;
      return kc && Number(kc);
    };
    TameEvent.prototype.toString = 
        ___.markFuncFreeze(function () { return '[Fake Event]'; });
    ___.all2(___.grantTypedMethod, TameEvent.prototype,
             ['stopPropagation', 'preventDefault']);

    function TameCustomHTMLEvent(event) {
      TameEvent.call(this, event);
      this.properties___ = {};
    }
    ___.extend(TameCustomHTMLEvent, TameEvent);
    TameCustomHTMLEvent.prototype.initEvent
        = function (type, bubbles, cancelable) {
      bridal.initEvent(this.event___, type, bubbles, cancelable);
    };
    TameCustomHTMLEvent.prototype.handleRead___ = function (name) {
      name = String(name);
      if (endsWith__.test(name)) { return void 0; }
      var handlerName = name + '_getter___';
      if (this[handlerName]) {
        return this[handlerName]();
      }
      if (___.hasOwnProp(this.event___.properties___, name)) {
        return this.event___.properties___[name];
      } else {
        return void 0;
      }
    };
    TameCustomHTMLEvent.prototype.handleCall___ = function (name, args) {
      name = String(name);
      if (endsWith__.test(name)) { throw new Error(INVALID_SUFFIX); }
      var handlerName = name + '_handler___';
      if (this[handlerName]) {
        return this[handlerName].call(this, args);
      }
      if (___.hasOwnProp(this.event___.properties___, name)) {
        return this.event___.properties___[name].call(this, args);
      } else {
        throw new TypeError(name + ' is not a function.');
      }
    };
    TameCustomHTMLEvent.prototype.handleSet___ = function (name, val) {
      name = String(name);
      if (endsWith__.test(name)) { throw new Error(INVALID_SUFFIX); }
      var handlerName = name + '_setter___';
      if (this[handlerName]) {
        return this[handlerName](val);
      }
      if (!this.event___.properties___) {
        this.event___.properties___ = {};
      }
      this[name + '_canEnum___'] = true;
      return this.event___.properties___[name] = val;
    };
    TameCustomHTMLEvent.prototype.handleDelete___ = function (name) {
      name = String(name);
      if (endsWith__.test(name)) { throw new Error(INVALID_SUFFIX); }
      var handlerName = name + '_deleter___';
      if (this[handlerName]) {
        return this[handlerName]();
      }
      if (this.event___.properties___) {
        return (
            delete this.event___.properties___[name]
            && delete this[name + '_canEnum___']);
      } else {
        return true;
      }
    };
    TameCustomHTMLEvent.prototype.handleEnum___ = function (ownFlag) {
      // TODO(metaweta): Add code to list all the other handled stuff we know
      // about.
      if (this.event___.properties___) {
        return ___.allKeys(this.event___.properties___);
      }
      return [];
    };
    TameCustomHTMLEvent.prototype.toString = ___.markFuncFreeze(function () {
      return '[Fake CustomEvent]';
    });
    ___.grantTypedMethod(TameCustomHTMLEvent.prototype, 'initEvent');

    function TameHTMLDocument(doc, body, domain, editable) {
      TamePseudoNode.call(this, editable);
      this.doc___ = doc;
      this.body___ = body;
      this.domain___ = domain;
      this.onLoadListeners___ = [];
      var tameDoc = this;

      var tameBody = defaultTameNode(body, editable);
      this.tameBody___ = tameBody;
      // TODO(mikesamuel): create a proper class for BODY, HEAD, and HTML along
      // with all the other specialized node types.
      var tameBodyElement = new TamePseudoElement(
          'BODY',
          this,
          function () {
            return tameNodeList(body.childNodes, editable, defaultTameNode);
          },
          function () { return tameHtmlElement; },
          function () { return tameInnerHtml(body.innerHTML); },
          tameBody,
          editable);
      ___.forOwnKeys(
          { appendChild: 0, removeChild: 0, insertBefore: 0, replaceChild: 0 },
          ___.markFuncFreeze(function (k) {
            tameBodyElement[k] = tameBody[k].bind(tameBody);
            ___.grantFunc(tameBodyElement, k);
          }));

      var title = doc.createTextNode(body.getAttribute('title') || '');
      var tameTitleElement = new TamePseudoElement(
          'TITLE',
          this,
          function () { return [defaultTameNode(title, false)]; },
          function () { return tameHeadElement; },
          function () { return html.escapeAttrib(title.nodeValue); },
          null,
          editable);
      var tameHeadElement = new TamePseudoElement(
          'HEAD',
          this,
          function () { return [tameTitleElement]; },
          function () { return tameHtmlElement; },
          function () {
            return '<title>' + tameTitleElement.getInnerHTML___() + '</title>';
          },
          null,
          editable);
      var tameHtmlElement = new TamePseudoElement(
          'HTML',
          this,
          function () { return [tameHeadElement, tameBodyElement]; },
          function () { return tameDoc; },
          function () {
            return ('<head>' + tameHeadElement.getInnerHTML___()
                    + '<\/head><body>'
                    + tameBodyElement.getInnerHTML___() + '<\/body>');
          },
          tameBody,
          editable);
      if (body.contains) {  // typeof is 'object' on IE
        tameHtmlElement.contains = function (other) {
          other = TameNodeT.coerce(other);
          var otherNode = other.node___;
          return body.contains(otherNode);
        };
        ___.grantFunc(tameHtmlElement, 'contains');
      }
      if ('function' === typeof body.compareDocumentPosition) {
        /**
         * Speced in <a href="http://www.w3.org/TR/DOM-Level-3-Core/core.html#Node3-compareDocumentPosition">DOM-Level-3</a>.
         */
        tameHtmlElement.compareDocumentPosition = function (other) {
          other = TameNodeT.coerce(other);
          var otherNode = other.node___;
          if (!otherNode) { return 0; }
          var bitmask = +body.compareDocumentPosition(otherNode);
          // To avoid leaking information about the relative positioning of
          // different roots, if neither contains the other, then we mask out
          // the preceding/following bits.
          // 0x18 is (CONTAINS | CONTAINED).
          // 0x1f is all the bits documented at
          // http://www.w3.org/TR/DOM-Level-3-Core/core.html#DocumentPosition
          // except IMPLEMENTATION_SPECIFIC.
          // 0x01 is DISCONNECTED.
          /*
          if (!(bitmask & 0x18)) {
            // TODO: If they are not under the same virtual doc root, return
            // DOCUMENT_POSITION_DISCONNECTED instead of leaking information
            // about PRECEEDED | FOLLOWING.
          }
          */
          return bitmask & 0x1f;
        };
        if (!___.hasOwnProp(tameHtmlElement, 'contains')) {
          // http://www.quirksmode.org/blog/archives/2006/01/contains_for_mo.html
          tameHtmlElement.contains = (function (other) {
            var docPos = this.compareDocumentPosition(other);
            return !(!(docPos & 0x10) && docPos);
          }).bind(tameHtmlElement);
          ___.grantFunc(tameHtmlElement, 'contains');
        }
        ___.grantFunc(tameHtmlElement, 'compareDocumentPosition');
      }
      this.documentElement___ = tameHtmlElement;
      classUtils.exportFields(
          this, ['documentElement', 'body', 'title', 'domain', 'forms',
                 'compatMode']);
    }
    inertCtor(TameHTMLDocument, TamePseudoNode, 'HTMLDocument');
    TameHTMLDocument.prototype.getNodeType___ = function () { return 9; };
    TameHTMLDocument.prototype.getNodeName___
        = function () { return '#document'; };
    TameHTMLDocument.prototype.getNodeValue___ = function () { return null; };
    TameHTMLDocument.prototype.getChildNodes___
        = function () { return [this.documentElement___]; };
    TameHTMLDocument.prototype.getAttributes___ = function () { return []; };
    TameHTMLDocument.prototype.getParentNode___ = function () { return null; };
    TameHTMLDocument.prototype.getElementsByTagName = function (tagName) {
      tagName = String(tagName).toLowerCase();
      switch (tagName) {
        case 'body': return fakeNodeList([ this.getBody___() ]);
        case 'head': return fakeNodeList([ this.getHead___() ]);
        case 'title': return fakeNodeList([ this.getTitle___() ]);
        case 'html': return fakeNodeList([ this.getDocumentElement___() ]);
        default:
          var nodes = tameGetElementsByTagName(
              this.body___, tagName, this.editable___);
          if (tagName === '*') {
            nodes.unshift(this.getBody___());
            nodes.unshift(this.getTitle___());
            nodes.unshift(this.getHead___());
            nodes.unshift(this.getDocumentElement___());
          }
          return nodes;
      }
    };
    TameHTMLDocument.prototype.getDocumentElement___ = function () {
      return this.documentElement___;
    };
    TameHTMLDocument.prototype.getBody___ = function () {
      return this.documentElement___.getLastChild___();
    };
    TameHTMLDocument.prototype.getHead___ = function () {
      return this.documentElement___.getFirstChild___();
    };
    TameHTMLDocument.prototype.getTitle___ = function () {
      return this.getHead___().getFirstChild___();
    };
    TameHTMLDocument.prototype.getDomain___ = function () {
      return this.domain___;
    };
    TameHTMLDocument.prototype.getElementsByClassName = function (className) {
      return tameGetElementsByClassName(
          this.body___, className, this.editable___);
    };
    TameHTMLDocument.prototype.addEventListener =
        function (name, listener, useCapture) {
          return this.tameBody___.addEventListener(name, listener, useCapture);
        };
    TameHTMLDocument.prototype.removeEventListener =
        function (name, listener, useCapture) {
          return this.tameBody___.removeEventListener(
              name, listener, useCapture);
        };
    TameHTMLDocument.prototype.createComment = function (text) {
      return defaultTameNode(this.doc___.createComment(" "), true);
    };
    TameHTMLDocument.prototype.createDocumentFragment = function () {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      return defaultTameNode(this.doc___.createDocumentFragment(), true);
    };
    TameHTMLDocument.prototype.createElement = function (tagName) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      tagName = String(tagName).toLowerCase();
      if (!html4.ELEMENTS.hasOwnProperty(tagName)) {
        throw new Error(UNKNOWN_TAGNAME + "[" + tagName + "]");
      }
      var flags = html4.ELEMENTS[tagName];
      // Script exemption allows dynamic loading of proxied scripts.
      if ((flags & html4.eflags.UNSAFE) && !(flags & html4.eflags.SCRIPT)) {
         ___.log(UNSAFE_TAGNAME + "[" + tagName + "]: no action performed");
        return null;
      }
      var newEl = this.doc___.createElement(tagName);
      if ("canvas" == tagName) {
        bridal.initCanvasElement(newEl);
      }
      if (elementPolicies.hasOwnProperty(tagName)) {
        var attribs = elementPolicies[tagName]([]);
        if (attribs) {
          for (var i = 0; i < attribs.length; i += 2) {
            bridal.setAttribute(newEl, attribs[i], attribs[i + 1]);
          }
        }
      }
      return defaultTameNode(newEl, true);
    };
    TameHTMLDocument.prototype.createTextNode = function (text) {
      if (!this.editable___) { throw new Error(NOT_EDITABLE); }
      return defaultTameNode(this.doc___.createTextNode(
          text !== null && text !== void 0 ? '' + text : ''), true);
    };
    TameHTMLDocument.prototype.getElementById = function (id) {
      id += idSuffix;
      var node = this.doc___.getElementById(id);
      return defaultTameNode(node, this.editable___);
    };
    TameHTMLDocument.prototype.getForms___ = function () {
      var tameForms = [];
      for (var i = 0; i < this.doc___.forms.length; i++) {
        var tameForm = tameRelatedNode(
          this.doc___.forms.item(i), this.editable___, defaultTameNode);
        // tameRelatedNode returns null if the node is not part of
        // this node's virtual document.
        if (tameForm !== null) { tameForms.push(tameForm); }
      }
      return fakeNodeList(tameForms);
    };
    TameHTMLDocument.prototype.getCompatMode___ = function () {
      return 'CSS1Compat';
    };
    TameHTMLDocument.prototype.toString = ___.markFuncFreeze(function () {
      return '[Fake Document]';
    });
    // http://www.w3.org/TR/DOM-Level-2-Events/events.html
    // #Events-DocumentEvent-createEvent
    TameHTMLDocument.prototype.createEvent = function (type) {
      type = String(type);
      if (type !== 'HTMLEvents') {
        // See https://developer.mozilla.org/en/DOM/document.createEvent#Notes
        // for a long list of event ypes.
        // See http://www.w3.org/TR/DOM-Level-2-Events/events.html
        // #Events-eventgroupings
        // for the DOM2 list.
        throw new Error('Unrecognized event type ' + type);
      }
      var document = this.doc___;
      var rawEvent;
      if (document.createEvent) {
        rawEvent = document.createEvent(type);
      } else {
        rawEvent = document.createEventObject();
        rawEvent.eventType = 'ondataavailable';
      }
      var tamedEvent = new TameCustomHTMLEvent(rawEvent);
      rawEvent.tamed___ = tamedEvent;
      return tamedEvent;
    };
    TameHTMLDocument.prototype.getOwnerDocument___ = function () {
      return null;
    };
    // Called by the html-emitter when the virtual document has been loaded.
    TameHTMLDocument.prototype.signalLoaded___ = function () {
      var onload = ((___.canRead(imports, '$v')
                     && ___.canCallPub(imports.$v, 'ros')
                     && imports.$v.ros('onload'))
                    || (imports.window &&
                        ___.readPub(imports.window, 'onload')));
      if (onload) {
        setTimeout(
            function () { ___.callPub(onload, 'call', [___.USELESS]); },
            0);
      }
      var listeners = this.onLoadListeners___;
      this.onLoadListeners___ = [];
      for (var i = 0, n = listeners.length; i < n; ++i) {
        (function (listener) {
          setTimeout(
              function () { ___.callPub(listener, 'call', [___.USELESS]); },
              0);
        })(listeners[i]);
      }
    };

    ___.all2(___.grantTypedMethod, TameHTMLDocument.prototype,
             ['addEventListener', 'removeEventListener',
              'createComment', 'createDocumentFragment',
              'createElement', 'createEvent', 'createTextNode',
              'getElementById', 'getElementsByClassName',
              'getElementsByTagName']);


    // For JavaScript handlers.  See plugin_dispatchEvent___ below
    imports.handlers___ = [];
    imports.TameHTMLDocument___ = TameHTMLDocument;  // Exposed for testing
    imports.tameNode___ = defaultTameNode;
    imports.feralNode___ = ___.markFuncFreeze(function(tameNode) {
      tameNode = TameNodeT.coerce(tameNode);
      return tameNode.node___;
    });
    imports.tameEvent___ = tameEvent;
    imports.blessHtml___ = blessHtml;
    imports.blessCss___ = function (var_args) {
      var arr = [];
      for (var i = 0, n = arguments.length; i < n; ++i) {
        arr[i] = arguments[i];
      }
      return cssSealerUnsealerPair.seal(arr);
    };
    imports.htmlAttr___ = function (s) {
      return html.escapeAttrib(String(s || ''));
    };
    imports.html___ = safeHtml;
    imports.rewriteUri___ = function (uri, mimeType) {
      var s = rewriteAttribute(null, null, html4.atype.URI, uri);
      if (!s) { throw new Error(); }
      return s;
    };
    imports.suffix___ = function (nmtokens) {
      var p = String(nmtokens).replace(/^\s+|\s+$/g, '').split(/\s+/g);
      var out = [];
      for (var i = 0; i < p.length; ++i) {
        var nmtoken = rewriteAttribute(null, null, html4.atype.ID, p[i]);
        if (!nmtoken) { throw new Error(nmtokens); }
        out.push(nmtoken);
      }
      return out.join(' ');
    };
    imports.ident___ = function (nmtokens) {
      var p = String(nmtokens).replace(/^\s+|\s+$/g, '').split(/\s+/g);
      var out = [];
      for (var i = 0; i < p.length; ++i) {
        var nmtoken = rewriteAttribute(null, null, html4.atype.CLASSES, p[i]);
        if (!nmtoken) { throw new Error(nmtokens); }
        out.push(nmtoken);
      }
      return out.join(' ');
    };
    imports.rewriteUriInCss___ = function(value) {
      return value
        ? uriCallback.rewrite(value, html4.ueffects.SAME_DOCUMENT,
              html4.ltypes.SANDBOXED, {})
        : void 0;
    };
    imports.rewriteUriInAttribute___ = function(value, tagName, attribName) {
      return value
        ? uriCallback.rewrite(value, getUriEffect(tagName, attribName),
              getLoaderType(tagName, attribName), {"XML_ATTR": attribName})
        : void 0;
    };

    var allCssProperties = domitaModules.CssPropertiesCollection(
        css.properties, document.documentElement, css);
    var historyInsensitiveCssProperties = domitaModules.CssPropertiesCollection(
        css.HISTORY_INSENSITIVE_STYLE_WHITELIST, document.documentElement, css);

    /**
     * http://www.w3.org/TR/DOM-Level-2-Style/css.html#CSS-CSSStyleDeclaration
     */
    function TameStyle(style, editable, tameEl) {
      this.style___ = style;
      this.editable___ = editable;
      this.tameEl___ = tameEl;
    }
    inertCtor(TameStyle, Object, 'Style');
    TameStyle.prototype.readByCanonicalName___ = function(canonName) {
      return String(this.style___[canonName] || '');
    };
    TameStyle.prototype.writeByCanonicalName___ = function(canonName, val) {
      this.style___[canonName] = val;
    };
    TameStyle.prototype.allowProperty___ = function (cssPropertyName) {
      return allCssProperties.isCssProp(cssPropertyName);
    };
    TameStyle.prototype.handleRead___ = function (stylePropertyName) {
      var self = this;
      if (String(stylePropertyName) === 'getPropertyValue') {
        return ___.markFuncFreeze(function(args) {
          return TameStyle.prototype.getPropertyValue.call(self, args);
        });
      }
      if (!this.style___
          || !allCssProperties.isCanonicalProp(stylePropertyName)) {
        return void 0;
      }
      var cssPropertyName =
          allCssProperties.getCssPropFromCanonical(stylePropertyName);
      if (!this.allowProperty___(cssPropertyName)) { return void 0; }
      var canonName = allCssProperties.getCanonicalPropFromCss(cssPropertyName);
      return this.readByCanonicalName___(canonName);
    };
    TameStyle.prototype.handleCall___ = function(name, args) {
      if (String(name) === 'getPropertyValue') {
        return TameStyle.prototype.getPropertyValue.call(this, args);
      }
      throw 'Cannot handle method ' + String(name);
    };
    TameStyle.prototype.getPropertyValue = function (cssPropertyName) {
      cssPropertyName = String(cssPropertyName || '').toLowerCase();
      if (!this.allowProperty___(cssPropertyName)) { return ''; }
      var canonName = allCssProperties.getCanonicalPropFromCss(cssPropertyName);
      return this.readByCanonicalName___(canonName);
    };
    TameStyle.prototype.handleSet___ = function (stylePropertyName, value) {
      if (!this.editable___) { throw new Error('style not editable'); }
      stylePropertyName = String(stylePropertyName);
      if (stylePropertyName === 'cssText') {
        if (typeof this.style___.cssText === 'string') {
          this.style___.cssText = sanitizeStyleAttrValue(value);
        } else {
          // If the browser doesn't support setting cssText, then fall back to
          // setting the style attribute of the containing element.  This won't
          // work for style declarations that are part of stylesheets and not
          // attached to elements.
          this.tameEl___.setAttribute('style', value);
        }
        return value;
      }
      if (!allCssProperties.isCanonicalProp(stylePropertyName)) {
        throw new Error('Unknown CSS property name ' + stylePropertyName);
      }
      var cssPropertyName =
          allCssProperties.getCssPropFromCanonical(stylePropertyName);
      if (!this.allowProperty___(cssPropertyName)) { return void 0; }
      var pattern = css.properties[cssPropertyName];
      if (!pattern) { throw new Error('style not editable'); }
      var val = '' + (value || '');
      // CssPropertyPatterns.java only allows styles of the form
      // url("...").  See the BUILTINS definition for the "uri" symbol.
      val = val.replace(
          /\burl\s*\(\s*\"([^\"]*)\"\s*\)/gi,
          function (_, url) {
            var decodedUrl = decodeCssString(url);
            var rewrittenUrl = uriCallback
                ? uriCallback.rewrite(
                    decodedUrl, html4.ueffects.SAME_DOCUMENT, html4.ltypes.SANDBOXED,
                    { "CSS_PROP": cssPropertyName})
                : null;
            if (!rewrittenUrl) {
              rewrittenUrl = 'about:blank';
            }
            return 'url("'
                + rewrittenUrl.replace(
                    /[\"\'\{\}\(\):\\]/g,
                    function (ch) {
                      return '\\' + ch.charCodeAt(0).toString(16) + ' ';
                    })
                + '")';
          });
      if (val && !pattern.test(val + ' ')) {
        throw new Error('bad value `' + val + '` for CSS property '
                        + stylePropertyName);
      }
      var canonName = allCssProperties.getCanonicalPropFromCss(cssPropertyName);
      this.writeByCanonicalName___(canonName, val);
      return value;
    };
    TameStyle.prototype.toString =
        ___.markFuncFreeze(function () { return '[Fake Style]'; });

    function isNestedInAnchor(rawElement) {
      for ( ; rawElement && rawElement != pseudoBodyNode;
           rawElement = rawElement.parentNode) {
        if (rawElement.tagName.toLowerCase() === 'a') { return true; }
      }
      return false;
    }

    function TameComputedStyle(rawElement, pseudoElement) {
      rawElement = rawElement || document.createElement('div');
      TameStyle.call(
          this,
          bridal.getComputedStyle(rawElement, pseudoElement),
          false);
      this.rawElement___ = rawElement;
      this.pseudoElement___ = pseudoElement;
    }
    ___.extend(TameComputedStyle, TameStyle);
    TameComputedStyle.prototype.readByCanonicalName___ = function(canonName) {
      var canReturnDirectValue =
          historyInsensitiveCssProperties.isCanonicalProp(canonName)
          || !isNestedInAnchor(this.rawElement___);
      if (canReturnDirectValue) {
        return TameStyle.prototype.readByCanonicalName___.call(this, canonName);
      } else {
        return new TameComputedStyle(pseudoBodyNode, this.pseudoElement___)
            .readByCanonicalName___(canonName);
      }
    };
    TameComputedStyle.prototype.writeByCanonicalName___ = function(canonName) {
      throw 'Computed styles not editable: This code should be unreachable';
    };
    TameComputedStyle.prototype.toString = ___.markFuncFreeze(function () {
      return '[Fake Computed Style]';
    });

    // Note: nodeClasses.XMLHttpRequest is a ctor that *can* be directly
    // called by cajoled code, so we do not use inertCtor().
    nodeClasses.XMLHttpRequest = domitaModules.TameXMLHttpRequest(
        domitaModules.XMLHttpRequestCtor(
            window.XMLHttpRequest,
            window.ActiveXObject),
        uriCallback);

    /**
     * given a number, outputs the equivalent css text.
     * @param {number} num
     * @return {string} an CSS representation of a number suitable for both html
     *    attribs and plain text.
     */
    imports.cssNumber___ = function (num) {
      if ('number' === typeof num && isFinite(num) && !isNaN(num)) {
        return '' + num;
      }
      throw new Error(num);
    };
    /**
     * given a number as 24 bits of RRGGBB, outputs a properly formatted CSS
     * color.
     * @param {number} num
     * @return {string} a CSS representation of num suitable for both html
     *    attribs and plain text.
     */
    imports.cssColor___ = function (color) {
      // TODO: maybe whitelist the color names defined for CSS if the arg is a
      // string.
      if ('number' !== typeof color || (color != (color | 0))) {
        throw new Error(color);
      }
      var hex = '0123456789abcdef';
      return '#' + hex.charAt((color >> 20) & 0xf)
          + hex.charAt((color >> 16) & 0xf)
          + hex.charAt((color >> 12) & 0xf)
          + hex.charAt((color >> 8) & 0xf)
          + hex.charAt((color >> 4) & 0xf)
          + hex.charAt(color & 0xf);
    };
    imports.cssUri___ = function (uri, mimeType) {
      var s = rewriteAttribute(null, null, html4.atype.URI, uri);
      if (!s) { throw new Error(); }
      return s;
    };

    /**
     * Create a CSS stylesheet with the given text and append it to the DOM.
     * @param {string} cssText a well-formed stylesheet production.
     */
    imports.emitCss___ = function (cssText) {
      this.getCssContainer___().appendChild(
          bridal.createStylesheet(document, cssText));
    };
    /** The node to which gadget stylesheets should be added. */
    imports.getCssContainer___ = function () {
      return document.getElementsByTagName('head')[0];
    };

    if (!/^-/.test(idSuffix)) {
      throw new Error('id suffix "' + idSuffix + '" must start with "-"');
    }
    if (!/___$/.test(idSuffix)) {
      throw new Error('id suffix "' + idSuffix + '" must end with "___"');
    }
    var idClass = idSuffix.substring(1);
    var idClassPattern = new RegExp(
        '(?:^|\\s)' + idClass.replace(/[\.$]/g, '\\$&') + '(?:\\s|$)');
    /** A per-gadget class used to separate style rules. */
    imports.getIdClass___ = function () {
      return idClass;
    };
    // enforce id class on element
    bridal.setAttribute(pseudoBodyNode, "class",
        bridal.getAttribute(pseudoBodyNode, "class")
        + " " + idClass + " vdoc-body___");

    // bitmask of trace points
    //    0x0001 plugin_dispatchEvent
    imports.domitaTrace___ = 0;
    imports.getDomitaTrace = ___.markFuncFreeze(
        function () { return imports.domitaTrace___; }
    );
    imports.setDomitaTrace = ___.markFuncFreeze(
        function (x) { imports.domitaTrace___ = x; }
    );

   function assignToImports(target, name, value) {
     if (target.DefineOwnProperty___) {
        target.DefineOwnProperty___(name, {
          value: value,
          writable: true,
          enumerable: true,
          configurable: true
        });
     } else {
       target[name] =  value;
     }
   }

    // TODO(mikesamuel): remove these, and only expose them via window once
    // Valija works
    imports.setTimeout = tameSetTimeout;
    imports.setInterval = tameSetInterval;
    imports.clearTimeout = tameClearTimeout;
    imports.clearInterval = tameClearInterval;

    var tameDocument = new TameHTMLDocument(
        document,
        pseudoBodyNode,
        String(optPseudoWindowLocation.hostname || 'nosuchhost.fake'),
        true);
    assignToImports(imports, 'document', tameDocument);
    imports.document.tameNode___ = imports.tameNode___;
    imports.document.feralNode___ = imports.feralNode___;
    imports.document.tameNode___ = imports.tameNode___;
    imports.document.feralNode___ = imports.feralNode___;

    // TODO(mikesamuel): figure out a mechanism by which the container can
    // specify the gadget's apparent URL.
    // See http://www.whatwg.org/specs/web-apps/current-work/multipage/history.html#location0
    var tameLocation = ___.primFreeze({
      toString: ___.markFuncFreeze(function () { return tameLocation.href; }),
      href: String(optPseudoWindowLocation.href || 'http://nosuchhost.fake/'),
      hash: String(optPseudoWindowLocation.hash || ''),
      host: String(optPseudoWindowLocation.host || 'nosuchhost.fake'),
      hostname: String(optPseudoWindowLocation.hostname || 'nosuchhost.fake'),
      pathname: String(optPseudoWindowLocation.pathname || '/'),
      port: String(optPseudoWindowLocation.port || ''),
      protocol: String(optPseudoWindowLocation.protocol || 'http:'),
      search: String(optPseudoWindowLocation.search || '')
      });

    // See spec at http://www.whatwg.org/specs/web-apps/current-work/multipage/browsers.html#navigator
    // We don't attempt to hide or abstract userAgent details since
    // they are discoverable via side-channels we don't control.
    var tameNavigator = ___.primFreeze({
      appName: String(window.navigator.appName),
      appVersion: String(window.navigator.appVersion),
      platform: String(window.navigator.platform),
      // userAgent should equal the string sent in the User-Agent HTTP header.
      userAgent: String(window.navigator.userAgent),
      // Custom attribute indicating Caja is active.
      cajaVersion: '1.0'
      });

    /**
     * Set of allowed pseudo elements as described at
     * http://www.w3.org/TR/CSS2/selector.html#q20
     */
    var PSEUDO_ELEMENT_WHITELIST = {
      // after and before disallowed since they can leak information about
      // arbitrary ancestor nodes.
      'first-letter': true,
      'first-line': true
    };

    /**
     * See http://www.whatwg.org/specs/web-apps/current-work/multipage/browsers.html#window for the full API.
     */
    function TameWindow() {
      this.properties___ = {};
      this.FERAL_TWIN___ = this.TAMED_TWIN___ = this;
    }

    /**
     * An <a href=
     * href=http://www.w3.org/TR/DOM-Level-2-Views/views.html#Views-AbstractView
     * >AbstractView</a> implementation that exposes styling, positioning, and
     * sizing information about the current document's pseudo-body.
     * <p>
     * The AbstractView spec specifies very little in its IDL description, but
     * mozilla defines it thusly:<blockquote>
     *   document.defaultView is generally a reference to the window object
     *   for the document, however that is not defined in the specification
     *   and can't be relied upon for all host environments, particularly as
     *   not all browsers implement it.
     * </blockquote>
     * <p>
     * We can't provide access to the tamed window directly from document
     * since it is the global scope of valija code, and so access to another
     * module's tamed window provides an unbounded amount of authority.
     * <p>
     * Instead, we expose styling, positioning, and sizing properties
     * via this class.  All of this authority is already available from the
     * document.
     */
    function TameDefaultView() {
      // TODO(mikesamuel): Implement in terms of
      //     http://www.w3.org/TR/cssom-view/#the-windowview-interface
      // TODO: expose a read-only version of the document
      this.document = tameDocument;
      // Exposing an editable default view that pointed to a read-only
      // tameDocument via document.defaultView would allow escalation of
      // authority.
      assert(tameDocument.editable___);
      ___.grantRead(this, 'document');
    }

    ___.forOwnKeys({
      document: tameDocument,
      location: tameLocation,
      navigator: tameNavigator,
      setTimeout: tameSetTimeout,
      setInterval: tameSetInterval,
      clearTimeout: tameClearTimeout,
      clearInterval: tameClearInterval,
      addEventListener: ___.markFuncFreeze(
          function (name, listener, useCapture) {
            if (name === 'load') {
              classUtils.ensureValidCallback(listener);
              tameDocument.onLoadListeners___.push(listener);
            } else {
              // TODO: need a testcase for this
              tameDocument.addEventListener(name, listener, useCapture);
            }
          }),
      removeEventListener: ___.markFuncFreeze(
          function (name, listener, useCapture) {
            if (name === 'load') {
              var listeners = tameDocument.onLoadListeners___;
              var k = 0;
              for (var i = 0, n = listeners.length; i < n; ++i) {
                listeners[i - k] = listeners[i];
                if (listeners[i] === listener) {
                  ++k;
                }
              }
              listeners.length -= k;
            } else {
              tameDocument.removeEventListener(name, listener, useCapture);
            }
          }),
      dispatchEvent: ___.markFuncFreeze(function (evt) {
        // TODO(ihab.awad): Implement
      })
    }, ___.markFuncFreeze(function (propertyName, value) {
      TameWindow.prototype[propertyName] = value;
      ___.grantRead(TameWindow.prototype, propertyName);
    }));
    ___.forOwnKeys({
      scrollBy: ___.markFuncFreeze(
          function (dx, dy) {
            // The window is always auto scrollable, so make the apparent window
            // body scrollable if the gadget tries to scroll it.
            if (dx || dy) { makeScrollable(tameDocument.body___); }
            tameScrollBy(tameDocument.body___, dx, dy);
          }),
      scrollTo: ___.markFuncFreeze(
          function (x, y) {
            // The window is always auto scrollable, so make the apparent window
            // body scrollable if the gadget tries to scroll it.
            makeScrollable(tameDocument.body___);
            tameScrollTo(tameDocument.body___, x, y);
          }),
      resizeTo: ___.markFuncFreeze(
          function (w, h) {
            tameResizeTo(tameDocument.body___, w, h);
          }),
      resizeBy: ___.markFuncFreeze(
          function (dw, dh) {
            tameResizeBy(tameDocument.body___, dw, dh);
          }),
      /** A partial implementation of getComputedStyle. */
      getComputedStyle: ___.markFuncFreeze(
          // Pseudo elements are suffixes like :first-line which constrain to
          // a portion of the element's content as defined at
          // http://www.w3.org/TR/CSS2/selector.html#q20
          function (tameElement, pseudoElement) {
            tameElement = TameNodeT.coerce(tameElement);
            // Coerce all nullish values to undefined, since that is the value
            // for unspecified parameters.
            // Per bug 973: pseudoElement should be null according to the
            // spec, but mozilla docs contradict this.
            // From https://developer.mozilla.org/En/DOM:window.getComputedStyle
            //     pseudoElt is a string specifying the pseudo-element to match.
            //     Should be an empty string for regular elements.
            pseudoElement = (pseudoElement === null || pseudoElement === void 0
                             || '' === pseudoElement)
                ? void 0 : String(pseudoElement).toLowerCase();
            if (pseudoElement !== void 0
                && !PSEUDO_ELEMENT_WHITELIST.hasOwnProperty(pseudoElement)) {
              throw new Error('Bad pseudo element ' + pseudoElement);
            }
            // No need to check editable since computed styles are readonly.
            return new TameComputedStyle(
                tameElement.node___,
                pseudoElement);
          })

      // NOT PROVIDED
      // event: a global on IE.  We always define it in scopes that can handle
      //        events.
      // opera: defined only on Opera.
    }, ___.markFuncFreeze(function (propertyName, value) {
      TameWindow.prototype[propertyName] = value;
      ___.grantRead(TameWindow.prototype, propertyName);
      TameDefaultView.prototype[propertyName] = value;
      ___.grantRead(TameDefaultView.prototype, propertyName);
    }));
    if (!Object.prototype.DefineOwnProperty___) {
      TameWindow.prototype.handleRead___ = function (name) {
        name = String(name);
        if (endsWith__.test(name)) { return void 0; }
        var handlerName = name + '_getter___';
        if (this[handlerName]) {
          return this[handlerName]();
        }
        if (___.hasOwnProp(this, name)) {
          return this[name];
        } else {
          return void 0;
        }
      };
      TameWindow.prototype.handleSet___ = function (name, val) {
        name = String(name);
        if (endsWith__.test(name)) { throw new Error(INVALID_SUFFIX); }
        var handlerName = name + '_setter___';
        if (this[handlerName]) {
          return this[handlerName](val);
        }
        this[name + '_canEnum___'] = true;
        this[name + '_canRead___'] = true;
        return this[name] = val;
      };
      TameWindow.prototype.handleDelete___ = function (name) {
        name = String(name);
        if (endsWith__.test(name)) { throw new Error(INVALID_SUFFIX); }
        var handlerName = name + '_deleter___';
        if (this[handlerName]) {
          return this[handlerName]();
        }
        return ___.deleteFieldEntirely(this, name);
      };
    }

    var tameWindow = new TameWindow();
    var tameDefaultView = new TameDefaultView(tameDocument.editable___);

    function propertyOnlyHasGetter(_) {
      throw new TypeError('setting a property that only has a getter');
    }
    ___.forOwnKeys({
      // We define all the window positional properties relative to
      // the fake body element to maintain the illusion that the fake
      // document is completely defined by the nodes under the fake body.
      clientLeft: {
        get: function () { return tameDocument.body___.clientLeft; }
      },
      clientTop: {
        get: function () { return tameDocument.body___.clientTop; }
      },
      clientHeight: {
        get: function () { return tameDocument.body___.clientHeight; }
      },
      clientWidth: {
        get: function () { return tameDocument.body___.clientWidth; }
      },
      offsetLeft: {
        get: function () { return tameDocument.body___.offsetLeft; }
      },
      offsetTop: {
        get: function () { return tameDocument.body___.offsetTop; }
      },
      offsetHeight: {
        get: function () { return tameDocument.body___.offsetHeight; }
      },
      offsetWidth: {
        get: function () { return tameDocument.body___.offsetWidth; }
      },
      // page{X,Y}Offset appear only as members of window, not on all elements
      // but http://www.howtocreate.co.uk/tutorials/javascript/browserwindow
      // says that they are identical to the scrollTop/Left on all browsers but
      // old versions of Safari.
      pageXOffset: {
        get: function () { return tameDocument.body___.scrollLeft; }
      },
      pageYOffset: {
        get: function () { return tameDocument.body___.scrollTop; }
      },
      scrollLeft: {
        get: function () { return tameDocument.body___.scrollLeft; },
        set: function (x) { tameDocument.body___.scrollLeft = +x; return x; }
      },
      scrollTop: {
        get: function () { return tameDocument.body___.scrollTop; },
        set: function (y) { tameDocument.body___.scrollTop = +y; return y; }
      },
      scrollHeight: {
        get: function () { return tameDocument.body___.scrollHeight; }
      },
      scrollWidth: {
        get: function () { return tameDocument.body___.scrollWidth; }
      }
    }, ___.markFuncFreeze(function (propertyName, def) {
      var views = [tameWindow, tameDefaultView, tameDocument.getBody___(),
                   tameDocument.getDocumentElement___()];
      var setter = def.set || propertyOnlyHasGetter, getter = def.get;
      for (var i = views.length; --i >= 0;) {
        var view = views[i];
        ___.useGetHandler(view, propertyName, getter);
        ___.useSetHandler(view, propertyName, setter);
      }
    }));

    ___.forOwnKeys({
      innerHeight: function () { return tameDocument.body___.clientHeight; },
      innerWidth: function () { return tameDocument.body___.clientWidth; },
      outerHeight: function () { return tameDocument.body___.clientHeight; },
      outerWidth: function () { return tameDocument.body___.clientWidth; }
    }, ___.markFuncFreeze(function (propertyName, handler) {
      // TODO(mikesamuel): define on prototype.
      ___.useGetHandler(tameWindow, propertyName, handler);
      ___.useGetHandler(tameDefaultView, propertyName, handler);
    }));

    // Attach reflexive properties to 'window' object
    var windowProps = ['top', 'self', 'opener', 'parent', 'window'];
    var wpLen = windowProps.length;
    for (var i = 0; i < wpLen; ++i) {
      var prop = windowProps[i];
      if (Object.prototype.DefineOwnProperty___) {
        tameWindow.DefineOwnProperty___(prop, {
          value: tameWindow,
          writable: true,
          enumerable: true,
          configurable: true
        });
      } else {
        tameWindow[prop] = tameWindow;
        ___.grantRead(tameWindow, prop);
      }
    }

    if (tameDocument.editable___) {
      tameDocument.defaultView = tameDefaultView;
      ___.grantRead(tameDocument, 'defaultView');
      // Hook for document.write support.
      tameDocument.sanitizeAttrs___ = sanitizeAttrs;
    }

    // Iterate over all node classes, assigning them to the Window object
    // under their DOM Level 2 standard name.
    ___.forOwnKeys(nodeClasses, ___.markFuncFreeze(function(name, ctor) {
      ___.primFreeze(ctor);
      if (Object.prototype.DefineOwnProperty___) {
        tameWindow.DefineOwnProperty___(name, {
          value: ctor,
          writable: true,
          enumerable: true,
          configurable: true
        });
      } else {
        tameWindow[name] = ctor;
        ___.grantRead(tameWindow, name);
      }
    }));

    // TODO(ihab.awad): Build a more sophisticated virtual class hierarchy by
    // creating a table of actual subclasses and instantiating tame nodes by
    // table lookups. This will allow the client code to see a truly consistent
    // DOM class hierarchy.
    var defaultNodeClasses = [
      'HTMLAppletElement',
      'HTMLAreaElement',
      'HTMLBaseElement',
      'HTMLBaseFontElement',
      'HTMLBodyElement',
      'HTMLBRElement',
      'HTMLButtonElement',
      'HTMLDirectoryElement',
      'HTMLDivElement',
      'HTMLDListElement',
      'HTMLFieldSetElement',
      'HTMLFontElement',
      'HTMLFrameElement',
      'HTMLFrameSetElement',
      'HTMLHeadElement',
      'HTMLHeadingElement',
      'HTMLHRElement',
      'HTMLHtmlElement',
      'HTMLIFrameElement',
      'HTMLIsIndexElement',
      'HTMLLabelElement',
      'HTMLLegendElement',
      'HTMLLIElement',
      'HTMLLinkElement',
      'HTMLMapElement',
      'HTMLMenuElement',
      'HTMLMetaElement',
      'HTMLModElement',
      'HTMLObjectElement',
      'HTMLOListElement',
      'HTMLOptGroupElement',
      'HTMLOptionElement',
      'HTMLParagraphElement',
      'HTMLParamElement',
      'HTMLPreElement',
      'HTMLQuoteElement',
      'HTMLScriptElement',
      'HTMLSelectElement',
      'HTMLStyleElement',
      'HTMLTableCaptionElement',
      'HTMLTableCellElement',
      'HTMLTableColElement',
      'HTMLTableElement',
      'HTMLTableRowElement',
      'HTMLTableSectionElement',
      'HTMLTextAreaElement',
      'HTMLTitleElement',
      'HTMLUListElement'
    ];

    var defaultNodeClassCtor = nodeClasses.Element;
    for (var i = 0; i < defaultNodeClasses.length; i++) {
      if (Object.prototype.DefineOwnProperty___) {
        TameWindow.DefineOwnProperty___(defaultNodeClasses[i], {
          value: defaultNodeClassCtor,
          writable: true,
          enumerable: true,
          configurable: true
        });
      } else {
        tameWindow[defaultNodeClasses[i]] = defaultNodeClassCtor;
        ___.grantRead(tameWindow, defaultNodeClasses[i]);
      }
    }

    var outers = imports.outers;
    if (___.isJSONContainer(outers)) {
      // For Valija, use the window object as outers.
      ___.forOwnKeys(outers, ___.markFuncFreeze(function(k, v) {
        if (!(k in tameWindow)) {
          // No need to check for DefineOwnProperty___; this case
          // occurs iff we are in Valija, not ES53.
          tameWindow[k] = v;
          ___.grantRead(tameWindow, k);
        }
      }));
      imports.outers = tameWindow;
    } else {
      assignToImports(imports, 'window', tameWindow);
    }
  }

  return attachDocumentStub;
})();

/**
 * Function called from rewritten event handlers to dispatch an event safely.
 */
function plugin_dispatchEvent___(thisNode, event, pluginId, handler) {
  event = (event || bridal.getWindow(thisNode).event);
  // support currentTarget on IE[678]
  if (!event.currentTarget) {
    event.currentTarget = thisNode;
  }
  var imports = ___.getImports(pluginId);
  var node = imports.tameNode___(thisNode, true);
  return plugin_dispatchToHandler___(
      pluginId, handler, [ node, imports.tameEvent___(event), node ]);
}

function plugin_dispatchToHandler___(pluginId, handler, args) {
  var sig = ('' + handler).match(/^function\b[^\)]*\)/);
  var imports = ___.getImports(pluginId);
  if (imports.domitaTrace___ & 0x1) {
    ___.log(
        'Dispatch pluginId=' + pluginId +
        ', handler=' + (sig ? sig[0] : handler) +
        ', args=' + args);
  }
  switch (typeof handler) {
    case 'number':
      handler = imports.handlers___[handler];
      break;
    case 'string':
      var fn = void 0;
      var tameWin = void 0;
      var $v = ___.readPub(imports, '$v');
      if ($v) {
        fn = ___.callPub($v, 'ros', [handler]);
        if (!fn) { tameWin = ___.callPub($v, 'ros', ['window']); }
      }
      if (!fn) {
        fn = ___.readPub(imports, handler);
        if (!fn) {
          if (!tameWin) { tameWin = ___.readPub(imports, 'window'); }
          if (tameWin) { fn = ___.readPub(tameWin, handler); }
        }
      }
      handler = fn && typeof fn.call === 'function' ? fn : void 0;
      break;
    case 'function': case 'object': break;
    default:
      throw new Error(
          'Expected function as event handler, not ' + typeof handler);
  }
  if (___.startCallerStack) { ___.startCallerStack(); }
  imports.isProcessingEvent___ = true;
  try {
    return ___.callPub(handler, 'call', args);
  } catch (ex) {
    if (ex && ex.cajitaStack___ && 'undefined' !== (typeof console)) {
      console.error(
          'Event dispatch %s: %s', handler, ex.cajitaStack___.join('\n'));
    }
    throw ex;
  } finally {
    imports.isProcessingEvent___ = false;
  }
}
;
/**
 * @fileoverview
 * Given that the next script executed is the cajoled Valija module, make
 * valijaMaker available in the global scope.
 *
 * @author jasvir@gmail.com
 * @author kpreid@switchb.org
 * @requires ___
 * @provides valijaMaker
 */

var valijaMaker = undefined;
(function(){
  // Save and restore
  var originalNewModuleHandler = ___.getNewModuleHandler();
  
  // Set up a fresh handler
  var ourHandler = ___.makeNormalNewModuleHandler();

  // ... which captures Valija
  var imports = ourHandler.getImports();
  imports.loader = {
    provide: ___.markFuncFreeze(function (v) {
      valijaMaker = v;
    })
  };
  imports.outers = imports;
  
  // ... and removes itself.
  var normalHandle = ourHandler.handle;
  ourHandler.handle = ___.markFuncFreeze(function (module) {
    ___.setNewModuleHandler(originalNewModuleHandler);
    normalHandle.call(ourHandler, module);
  });
  
  ___.setNewModuleHandler(ourHandler);
})();
;
{
  ___.loadModule({
      'instantiate': function (___, IMPORTS___) {
        var Array = ___.readImport(IMPORTS___, 'Array');
        var Object = ___.readImport(IMPORTS___, 'Object');
        var ReferenceError = ___.readImport(IMPORTS___, 'ReferenceError', {});
        var RegExp = ___.readImport(IMPORTS___, 'RegExp', {});
        var TypeError = ___.readImport(IMPORTS___, 'TypeError', {});
        var cajita = ___.readImport(IMPORTS___, 'cajita', {
            'beget': { '()': {} },
            'freeze': { '()': {} },
            'newTable': { '()': {} },
            'getFuncCategory': { '()': {} },
            'enforceType': { '()': {} },
            'getSuperCtor': { '()': {} },
            'getOwnPropertyNames': { '()': {} },
            'getProtoPropertyNames': { '()': {} },
            'getProtoPropertyValue': { '()': {} },
            'inheritsFrom': { '()': {} },
            'readOwn': { '()': {} },
            'directConstructor': { '()': {} },
            'USELESS': {},
            'construct': { '()': {} },
            'forAllKeys': { '()': {} },
            'Token': { '()': {} },
            'args': { '()': {} }
          });
        var loader = ___.readImport(IMPORTS___, 'loader');
        var outers = ___.readImport(IMPORTS___, 'outers');
        var moduleResult___, valijaMaker;
        moduleResult___ = ___.NO_RESULT;
        valijaMaker = (function () {
            function valijaMaker$_var(outers) {
              var ObjectPrototype, DisfunctionPrototype, Disfunction,
              ObjectShadow, x0___, FuncHeader, x1___, myPOE, x2___, pumpkin,
              x3___, x4___, x5___, t, undefIndicator;
              function disfuncToString($dis) {
                var callFn, printRep, match, name;
                callFn = $dis.call_canRead___? $dis.call: ___.readPub($dis,
                  'call');
                if (callFn) {
                  printRep = callFn.toString_canCall___? callFn.toString():
                  ___.callPub(callFn, 'toString', [ ]);
                  match = FuncHeader.exec_canCall___? FuncHeader.exec(printRep)
                    : ___.callPub(FuncHeader, 'exec', [ printRep ]);
                  if (null !== match) {
                    name = $dis.name_canRead___? $dis.name: ___.readPub($dis,
                      'name');
                    if (name === void 0) {
                      name = match[ 1 ];
                    }
                    return 'function ' + name + '(' + match[ 2 ] +
                      ') {\n  [cajoled code]\n}';
                  }
                  return printRep;
                }
                return 'disfunction(var_args){\n   [cajoled code]\n}';
              }
              disfuncToString.FUNC___ = 'disfuncToString';
              function getShadow(func) {
                var cat, result, parentFunc, parentShadow, proto, statics, i,
                k, meths, i, k, v;
                cajita.enforceType(func, 'function');
                cat = cajita.getFuncCategory(func);
                result = myPOE.get_canCall___? myPOE.get(cat):
                ___.callPub(myPOE, 'get', [ cat ]);
                if (void 0 === result) {
                  result = cajita.beget(DisfunctionPrototype);
                  parentFunc = cajita.getSuperCtor(func);
                  if (___.typeOf(parentFunc) === 'function') {
                    parentShadow = getShadow.CALL___(parentFunc);
                  } else {
                    parentShadow = ObjectShadow;
                  }
                  proto = cajita.beget(parentShadow.prototype_canRead___?
                    parentShadow.prototype: ___.readPub(parentShadow,
                      'prototype'));
                  result.prototype_canSet___ === result? (result.prototype =
                    proto): ___.setPub(result, 'prototype', proto);
                  proto.constructor_canSet___ === proto? (proto.constructor =
                    func): ___.setPub(proto, 'constructor', func);
                  statics = cajita.getOwnPropertyNames(func);
                  for (i = 0; i < statics.length; i++) {
                    k = ___.readPub(statics, i);
                    if (k !== 'valueOf') {
                      ___.setPub(result, k, ___.readPub(func, k));
                    }
                  }
                  meths = cajita.getProtoPropertyNames(func);
                  for (i = 0; i < meths.length; i++) {
                    k = ___.readPub(meths, i);
                    if (k !== 'valueOf') {
                      v = cajita.getProtoPropertyValue(func, k);
                      if (___.typeOf(v) === 'object' && v !== null &&
                        ___.typeOf(v.call_canRead___? v.call: ___.readPub(v,
                            'call')) === 'function') {
                        v = dis.CALL___(v.call_canRead___? v.call:
                          ___.readPub(v, 'call'), k);
                      }
                      ___.setPub(proto, k, v);
                    }
                  }
                  myPOE.set_canCall___? myPOE.set(cat, result):
                  ___.callPub(myPOE, 'set', [ cat, result ]);
                }
                return result;
              }
              getShadow.FUNC___ = 'getShadow';
              function getFakeProtoOf(func) {
                var shadow;
                if (___.typeOf(func) === 'function') {
                  shadow = getShadow.CALL___(func);
                  return shadow.prototype_canRead___? shadow.prototype:
                  ___.readPub(shadow, 'prototype');
                } else if (___.typeOf(func) === 'object' && func !== null) {
                  return func.prototype_canRead___? func.prototype:
                  ___.readPub(func, 'prototype');
                } else { return void 0; }
              }
              getFakeProtoOf.FUNC___ = 'getFakeProtoOf';
              function typeOf(obj) {
                var result;
                result = ___.typeOf(obj);
                if (result !== 'object') { return result; }
                if (cajita.isPseudoFunc_canCall___? cajita.isPseudoFunc(obj):
                  ___.callPub(cajita, 'isPseudoFunc', [ obj ])) { return 'function'; }
                return result;
              }
              typeOf.FUNC___ = 'typeOf';
              function instanceOf(obj, func) {
                if (___.typeOf(func) === 'function' && obj instanceof func) {
                  return true; } else {
                  return cajita.inheritsFrom(obj, getFakeProtoOf.CALL___(func))
                    ;
                }
              }
              instanceOf.FUNC___ = 'instanceOf';
              function read(obj, name) {
                var result, stepParent;
                result = cajita.readOwn(obj, name, pumpkin);
                if (result !== pumpkin) { return result; }
                if (___.typeOf(obj) === 'function') {
                  return ___.readPub(getShadow.CALL___(obj), name);
                }
                if (obj === null || obj === void 0) {
                  throw ___.construct(TypeError, [ 'Cannot read property \"' +
                        name + '\" from ' + obj ]);
                }
                if (___.inPub(name, ___.construct(Object, [ obj ]))) {
                  return ___.readPub(obj, name);
                }
                stepParent =
                  getFakeProtoOf.CALL___(cajita.directConstructor(obj));
                if (stepParent !== void 0 && ___.inPub(name,
                    ___.construct(Object, [ stepParent ])) && name !==
                  'valueOf') {
                  return ___.readPub(stepParent, name);
                }
                return ___.readPub(obj, name);
              }
              read.FUNC___ = 'read';
              function set(obj, name, newValue) {
                if (___.typeOf(obj) === 'function') {
                  ___.setPub(getShadow.CALL___(obj), name, newValue);
                } else {
                  ___.setPub(obj, name, newValue);
                }
                return newValue;
              }
              set.FUNC___ = 'set';
              function callFunc(func, args) {
                var x0___;
                return x0___ = cajita.USELESS, func.apply_canCall___?
                  func.apply(x0___, args): ___.callPub(func, 'apply', [ x0___,
                    args ]);
              }
              callFunc.FUNC___ = 'callFunc';
              function callMethod(obj, name, args) {
                var m;
                m = read.CALL___(obj, name);
                if (!m) {
                  throw ___.construct(TypeError, [ 'callMethod: ' + obj +
                        ' has no method ' + name ]);
                }
                return m.apply_canCall___? m.apply(obj, args): ___.callPub(m,
                  'apply', [ obj, args ]);
              }
              callMethod.FUNC___ = 'callMethod';
              function construct(ctor, args) {
                var result, altResult;
                if (___.typeOf(ctor) === 'function') {
                  return cajita.construct(ctor, args);
                }
                result = cajita.beget(ctor.prototype_canRead___?
                  ctor.prototype: ___.readPub(ctor, 'prototype'));
                altResult = ctor.apply_canCall___? ctor.apply(result, args):
                ___.callPub(ctor, 'apply', [ result, args ]);
                switch (___.typeOf(altResult)) {
                case 'object':
                  if (null !== altResult) { return altResult; }
                  break;
                case 'function':
                  return altResult;
                }
                return result;
              }
              construct.FUNC___ = 'construct';
              function dis(callFn, opt_name) {
                var template, result, x0___, disproto, x1___;
                template = cajita.PseudoFunction_canCall___?
                  cajita.PseudoFunction(callFn): ___.callPub(cajita,
                  'PseudoFunction', [ callFn ]);
                result = cajita.beget(DisfunctionPrototype);
                result.call_canSet___ === result? (result.call = callFn):
                ___.setPub(result, 'call', callFn);
                x0___ = template.apply_canRead___? template.apply:
                ___.readPub(template, 'apply'), result.apply_canSet___ ===
                  result? (result.apply = x0___): ___.setPub(result, 'apply',
                  x0___);
                disproto = cajita.beget(ObjectPrototype);
                result.prototype_canSet___ === result? (result.prototype =
                  disproto): ___.setPub(result, 'prototype', disproto);
                disproto.constructor_canSet___ === disproto?
                  (disproto.constructor = result): ___.setPub(disproto,
                  'constructor', result);
                x1___ = template.length, result.length_canSet___ === result?
                  (result.length = x1___): ___.setPub(result, 'length', x1___);
                if (opt_name !== void 0 && opt_name !== '') {
                  result.name_canSet___ === result? (result.name = opt_name):
                  ___.setPub(result, 'name', opt_name);
                }
                return result;
              }
              dis.FUNC___ = 'dis';
              function disfuncCall($dis, self, var_args) {
                var a___ = ___.args(arguments);
                var x0___;
                return x0___ = Array.slice(a___, 2), $dis.apply_canCall___?
                  $dis.apply(self, x0___): ___.callPub($dis, 'apply', [ self,
                    x0___ ]);
              }
              disfuncCall.FUNC___ = 'disfuncCall';
              function disfuncApply($dis, self, args) {
                return $dis.apply_canCall___? $dis.apply(self, args):
                ___.callPub($dis, 'apply', [ self, args ]);
              }
              disfuncApply.FUNC___ = 'disfuncApply';
              function disfuncBind($dis, self, var_args) {
                var a___ = ___.args(arguments);
                var leftArgs;
                function disfuncBound(var_args) {
                  var a___ = ___.args(arguments);
                  var x0___, x1___;
                  return x1___ = (x0___ = Array.slice(a___, 0),
                    leftArgs.concat_canCall___? leftArgs.concat(x0___):
                    ___.callPub(leftArgs, 'concat', [ x0___ ])),
                  $dis.apply_canCall___? $dis.apply(self, x1___):
                  ___.callPub($dis, 'apply', [ self, x1___ ]);
                }
                disfuncBound.FUNC___ = 'disfuncBound';
                leftArgs = Array.slice(a___, 2);
                return ___.primFreeze(disfuncBound);
              }
              disfuncBind.FUNC___ = 'disfuncBind';
              function getOuters() {
                cajita.enforceType(outers, 'object');
                return outers;
              }
              getOuters.FUNC___ = 'getOuters';
              function readOuter(name) {
                var result;
                result = cajita.readOwn(outers, name, pumpkin);
                if (result !== pumpkin) { return result; }
                if (canReadRev.CALL___(name, outers)) {
                  return read.CALL___(outers, name);
                } else {
                  throw ___.construct(ReferenceError, [
                      'Outer variable not found: ' + name ]);
                }
              }
              readOuter.FUNC___ = 'readOuter';
              function readOuterSilent(name) {
                if (canReadRev.CALL___(name, outers)) {
                  return read.CALL___(outers, name);
                } else { return void 0; }
              }
              readOuterSilent.FUNC___ = 'readOuterSilent';
              function setOuter(name, val) {
                return ___.setPub(outers, name, val);
              }
              setOuter.FUNC___ = 'setOuter';
              function initOuter(name) {
                if (canReadRev.CALL___(name, outers)) { return; }
                set.CALL___(outers, name, void 0);
              }
              initOuter.FUNC___ = 'initOuter';
              function remove(obj, name) {
                var shadow;
                if (___.typeOf(obj) === 'function') {
                  shadow = getShadow.CALL___(obj);
                  return ___.deletePub(shadow, name);
                } else {
                  return ___.deletePub(obj, name);
                }
              }
              remove.FUNC___ = 'remove';
              function keys(obj) {
                var result;
                result = [ ];
                cajita.forAllKeys(obj, ___.markFuncFreeze(function (name) {
                      result.push_canCall___? result.push(name):
                      ___.callPub(result, 'push', [ name ]);
                    }));
                cajita.forAllKeys(getSupplement.CALL___(obj),
                  ___.markFuncFreeze(function (name) {
                      if (!___.inPub(name, obj) && name !== 'constructor') {
                        result.push_canCall___? result.push(name):
                        ___.callPub(result, 'push', [ name ]);
                      }
                    }));
                return result;
              }
              keys.FUNC___ = 'keys';
              function canReadRev(name, obj) {
                if (___.inPub(name, obj)) { return true; }
                return ___.inPub(name, getSupplement.CALL___(obj));
              }
              canReadRev.FUNC___ = 'canReadRev';
              function getSupplement(obj) {
                var ctor;
                if (___.typeOf(obj) === 'function') {
                  return getShadow.CALL___(obj);
                } else {
                  ctor = cajita.directConstructor(obj);
                  return getFakeProtoOf.CALL___(ctor);
                }
              }
              getSupplement.FUNC___ = 'getSupplement';
              function exceptionTableSet(ex) {
                var result, x0___;
                result = cajita.Token('' + ex);
                x0___ = ex === void 0? undefIndicator: ex, t.set_canCall___?
                  t.set(result, x0___): ___.callPub(t, 'set', [ result, x0___ ]
                );
                return result;
              }
              exceptionTableSet.FUNC___ = 'exceptionTableSet';
              function exceptionTableRead(key) {
                var v, x0___;
                v = t.get_canCall___? t.get(key): ___.callPub(t, 'get', [ key ]
                );
                x0___ = void 0, t.set_canCall___? t.set(key, x0___):
                ___.callPub(t, 'set', [ key, x0___ ]);
                return v === void 0? key: v === undefIndicator? void 0: v;
              }
              exceptionTableRead.FUNC___ = 'exceptionTableRead';
              function disArgs(original) {
                return cajita.args(Array.slice(original, 1));
              }
              disArgs.FUNC___ = 'disArgs';
              ObjectPrototype = ___.iM([ 'constructor', Object ]);
              DisfunctionPrototype =
                cajita.beget(cajita.PseudoFunctionProto_canRead___?
                cajita.PseudoFunctionProto: ___.readPub(cajita,
                  'PseudoFunctionProto'));
              Disfunction = cajita.beget(DisfunctionPrototype);
              Disfunction.prototype_canSet___ === Disfunction?
                (Disfunction.prototype = DisfunctionPrototype):
              ___.setPub(Disfunction, 'prototype', DisfunctionPrototype);
              Disfunction.length_canSet___ === Disfunction? (Disfunction.length
                = 1): ___.setPub(Disfunction, 'length', 1);
              DisfunctionPrototype.constructor_canSet___ ===
                DisfunctionPrototype? (DisfunctionPrototype.constructor =
                Disfunction): ___.setPub(DisfunctionPrototype, 'constructor',
                Disfunction);
              outers.Function_canSet___ === outers? (outers.Function =
                Disfunction): ___.setPub(outers, 'Function', Disfunction);
              ObjectShadow = cajita.beget(DisfunctionPrototype);
              ObjectShadow.prototype_canSet___ === ObjectShadow?
                (ObjectShadow.prototype = ObjectPrototype):
              ___.setPub(ObjectShadow, 'prototype', ObjectPrototype);
              x0___ = (function () {
                  function freeze$_meth(obj) {
                    if (___.typeOf(obj) === 'function') {
                      cajita.freeze(getShadow.CALL___(obj));
                    } else {
                      cajita.freeze(obj);
                    }
                    return obj;
                  }
                  return ___.markFuncFreeze(freeze$_meth, 'freeze$_meth');
                })(), ObjectShadow.freeze_canSet___ === ObjectShadow?
                (ObjectShadow.freeze = x0___): ___.setPub(ObjectShadow,
                'freeze', x0___);
              FuncHeader = ___.construct(RegExp, [
                  '^\\s*function\\s*([^\\s\\(]*)\\s*\\(' + '(?:\\$dis,?\\s*)?'
                    + '([^\\)]*)\\)' ]);
              x1___ = dis.CALL___(___.primFreeze(disfuncToString), 'toString'),
              DisfunctionPrototype.toString_canSet___ === DisfunctionPrototype?
                (DisfunctionPrototype.toString = x1___):
              ___.setPub(DisfunctionPrototype, 'toString', x1___);
              outers.Function_canSet___ === outers? (outers.Function =
                Disfunction): ___.setPub(outers, 'Function', Disfunction);
              myPOE = cajita.newTable();
              x2___ = cajita.getFuncCategory(Object), myPOE.set_canCall___?
                myPOE.set(x2___, ObjectShadow): ___.callPub(myPOE, 'set', [
                  x2___, ObjectShadow ]);
              pumpkin = ___.iM([ ]);
              x3___ = dis.CALL___(___.primFreeze(disfuncCall), 'call'),
              DisfunctionPrototype.call_canSet___ === DisfunctionPrototype?
                (DisfunctionPrototype.call = x3___):
              ___.setPub(DisfunctionPrototype, 'call', x3___);
              x4___ = dis.CALL___(___.primFreeze(disfuncApply), 'apply'),
              DisfunctionPrototype.apply_canSet___ === DisfunctionPrototype?
                (DisfunctionPrototype.apply = x4___):
              ___.setPub(DisfunctionPrototype, 'apply', x4___);
              x5___ = dis.CALL___(___.primFreeze(disfuncBind), 'bind'),
              DisfunctionPrototype.bind_canSet___ === DisfunctionPrototype?
                (DisfunctionPrototype.bind = x5___):
              ___.setPub(DisfunctionPrototype, 'bind', x5___);
              t = cajita.newTable();
              undefIndicator = ___.iM([ ]);
              return cajita.freeze(___.iM([ 'typeOf', ___.primFreeze(typeOf),
                    'instanceOf', ___.primFreeze(instanceOf), 'tr',
                    ___.primFreeze(exceptionTableRead), 'ts',
                    ___.primFreeze(exceptionTableSet), 'r',
                    ___.primFreeze(read), 's', ___.primFreeze(set), 'cf',
                    ___.primFreeze(callFunc), 'cm', ___.primFreeze(callMethod),
                    'construct', ___.primFreeze(construct), 'getOuters',
                    ___.primFreeze(getOuters), 'ro', ___.primFreeze(readOuter),
                    'ros', ___.primFreeze(readOuterSilent), 'so',
                    ___.primFreeze(setOuter), 'initOuter',
                    ___.primFreeze(initOuter), 'remove', ___.primFreeze(remove)
                      , 'keys', ___.primFreeze(keys), 'canReadRev',
                    ___.primFreeze(canReadRev), 'disArgs',
                    ___.primFreeze(disArgs), 'dis', ___.primFreeze(dis) ]));
            }
            return ___.markFuncFreeze(valijaMaker$_var, 'valijaMaker$_var');
          })();
        if (___.typeOf(loader) !== 'undefined') {
          loader.provide_canCall___? loader.provide(valijaMaker):
          ___.callPub(loader, 'provide', [ valijaMaker ]);
        }
        if (___.typeOf(outers) !== 'undefined') {
          moduleResult___ = valijaMaker.CALL___(outers);
        }
        return moduleResult___;
      },
      'cajolerName': 'com.google.caja',
      'cajolerVersion': '4486:4487',
      'cajoledDate': 1306533523240
});
};
// Copyright (C) 2010 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @fileoverview
 * Implements RFC 3986 for parsing/formatting URIs.
 *
 * @author mikesamuel@gmail.com
 * @provides URI
 */

var URI = (function () {

/**
 * creates a uri from the string form.  The parser is relaxed, so special
 * characters that aren't escaped but don't cause ambiguities will not cause
 * parse failures.
 *
 * @return {URI|null}
 */
function parse(uriStr) {
  var m = ('' + uriStr).match(URI_RE_);
  if (!m) { return null; }
  return new URI(
      nullIfAbsent(m[1]),
      nullIfAbsent(m[2]),
      nullIfAbsent(m[3]),
      nullIfAbsent(m[4]),
      nullIfAbsent(m[5]),
      nullIfAbsent(m[6]),
      nullIfAbsent(m[7]));
}


/**
 * creates a uri from the given parts.
 *
 * @param scheme {string} an unencoded scheme such as "http" or null
 * @param credentials {string} unencoded user credentials or null
 * @param domain {string} an unencoded domain name or null
 * @param port {number} a port number in [1, 32768].
 *    -1 indicates no port, as does null.
 * @param path {string} an unencoded path
 * @param query {Array.<string>|string|null} a list of unencoded cgi
 *   parameters where even values are keys and odds the corresponding values
 *   or an unencoded query.
 * @param fragment {string} an unencoded fragment without the "#" or null.
 * @return {URI}
 */
function create(scheme, credentials, domain, port, path, query, fragment) {
  var uri = new URI(
      encodeIfExists2(scheme, URI_DISALLOWED_IN_SCHEME_OR_CREDENTIALS_),
      encodeIfExists2(
          credentials, URI_DISALLOWED_IN_SCHEME_OR_CREDENTIALS_),
      encodeIfExists(domain),
      port > 0 ? port.toString() : null,
      encodeIfExists2(path, URI_DISALLOWED_IN_PATH_),
      null,
      encodeIfExists(fragment));
  if (query) {
    if ('string' === typeof query) {
      uri.setRawQuery(query.replace(/[^?&=0-9A-Za-z_\-~.%]/g, encodeOne));
    } else {
      uri.setAllParameters(query);
    }
  }
  return uri;
}
function encodeIfExists(unescapedPart) {
  if ('string' == typeof unescapedPart) {
    return encodeURIComponent(unescapedPart);
  }
  return null;
};
/**
 * if unescapedPart is non null, then escapes any characters in it that aren't
 * valid characters in a url and also escapes any special characters that
 * appear in extra.
 *
 * @param unescapedPart {string}
 * @param extra {RegExp} a character set of characters in [\01-\177].
 * @return {string|null} null iff unescapedPart == null.
 */
function encodeIfExists2(unescapedPart, extra) {
  if ('string' == typeof unescapedPart) {
    return encodeURI(unescapedPart).replace(extra, encodeOne);
  }
  return null;
};
/** converts a character in [\01-\177] to its url encoded equivalent. */
function encodeOne(ch) {
  var n = ch.charCodeAt(0);
  return '%' + '0123456789ABCDEF'.charAt((n >> 4) & 0xf) +
      '0123456789ABCDEF'.charAt(n & 0xf);
}

/**
 * {@updoc
 *  $ normPath('foo/./bar')
 *  # 'foo/bar'
 *  $ normPath('./foo')
 *  # 'foo'
 *  $ normPath('foo/.')
 *  # 'foo'
 *  $ normPath('foo//bar')
 *  # 'foo/bar'
 * }
 */
function normPath(path) {
  return path.replace(/(^|\/)\.(?:\/|$)/g, '$1').replace(/\/{2,}/g, '/');
}

var PARENT_DIRECTORY_HANDLER = new RegExp(
    ''
    // A path break
    + '(/|^)'
    // followed by a non .. path element
    // (cannot be . because normPath is used prior to this RegExp)
    + '(?:[^./][^/]*|\\.{2,}(?:[^./][^/]*)|\\.{3,}[^/]*)'
    // followed by .. followed by a path break.
    + '/\\.\\.(?:/|$)');

var PARENT_DIRECTORY_HANDLER_RE = new RegExp(PARENT_DIRECTORY_HANDLER);

var EXTRA_PARENT_PATHS_RE = /^(?:\.\.\/)*(?:\.\.$)?/;

/**
 * Normalizes its input path and collapses all . and .. sequences except for
 * .. sequences that would take it above the root of the current parent
 * directory.
 * {@updoc
 *  $ collapse_dots('foo/../bar')
 *  # 'bar'
 *  $ collapse_dots('foo/./bar')
 *  # 'foo/bar'
 *  $ collapse_dots('foo/../bar/./../../baz')
 *  # 'baz'
 *  $ collapse_dots('../foo')
 *  # '../foo'
 *  $ collapse_dots('../foo').replace(EXTRA_PARENT_PATHS_RE, '')
 *  # 'foo'
 * }
 */
function collapse_dots(path) {
  if (path === null) { return null; }
  var p = normPath(path);
  // Only /../ left to flatten
  var r = PARENT_DIRECTORY_HANDLER_RE;
  // We replace with $1 which matches a / before the .. because this
  // guarantees that:
  // (1) we have at most 1 / between the adjacent place,
  // (2) always have a slash if there is a preceding path section, and
  // (3) we never turn a relative path into an absolute path.
  for (var q; (q = p.replace(r, '$1')) != p; p = q);
  return p;
}

/**
 * resolves a relative url string to a base uri.
 * @return {URI}
 */
function resolve(baseUri, relativeUri) {
  // there are several kinds of relative urls:
  // 1. //foo - replaces everything from the domain on.  foo is a domain name
  // 2. foo - replaces the last part of the path, the whole query and fragment
  // 3. /foo - replaces the the path, the query and fragment
  // 4. ?foo - replace the query and fragment
  // 5. #foo - replace the fragment only

  var absoluteUri = baseUri.clone();
  // we satisfy these conditions by looking for the first part of relativeUri
  // that is not blank and applying defaults to the rest

  var overridden = relativeUri.hasScheme();

  if (overridden) {
    absoluteUri.setRawScheme(relativeUri.getRawScheme());
  } else {
    overridden = relativeUri.hasCredentials();
  }

  if (overridden) {
    absoluteUri.setRawCredentials(relativeUri.getRawCredentials());
  } else {
    overridden = relativeUri.hasDomain();
  }

  if (overridden) {
    absoluteUri.setRawDomain(relativeUri.getRawDomain());
  } else {
    overridden = relativeUri.hasPort();
  }

  var rawPath = relativeUri.getRawPath();
  var simplifiedPath = collapse_dots(rawPath);
  if (overridden) {
    absoluteUri.setPort(relativeUri.getPort());
    simplifiedPath = simplifiedPath
        && simplifiedPath.replace(EXTRA_PARENT_PATHS_RE, '');
  } else {
    overridden = !!rawPath;
    if (overridden) {
      // resolve path properly
      if (simplifiedPath.charCodeAt(0) !== 0x2f /* / */) {  // path is relative
        var absRawPath = collapse_dots(absoluteUri.getRawPath() || '')
            .replace(EXTRA_PARENT_PATHS_RE, '');
        var slash = absRawPath.lastIndexOf('/') + 1;
        simplifiedPath = collapse_dots(
            (slash ? absRawPath.substring(0, slash) : '')
            + collapse_dots(rawPath))
            .replace(EXTRA_PARENT_PATHS_RE, '');
      }
    } else {
      simplifiedPath = simplifiedPath
          && simplifiedPath.replace(EXTRA_PARENT_PATHS_RE, '');
      if (simplifiedPath !== rawPath) {
        absoluteUri.setRawPath(simplifiedPath);
      }
    }
  }

  if (overridden) {
    absoluteUri.setRawPath(simplifiedPath);
  } else {
    overridden = relativeUri.hasQuery();
  }

  if (overridden) {
    absoluteUri.setRawQuery(relativeUri.getRawQuery());
  } else {
    overridden = relativeUri.hasFragment();
  }

  if (overridden) {
    absoluteUri.setRawFragment(relativeUri.getRawFragment());
  }

  return absoluteUri;
}

/**
 * a mutable URI.
 *
 * This class contains setters and getters for the parts of the URI.
 * The <tt>getXYZ</tt>/<tt>setXYZ</tt> methods return the decoded part -- so
 * <code>uri.parse('/foo%20bar').getPath()</code> will return the decoded path,
 * <tt>/foo bar</tt>.
 *
 * <p>The raw versions of fields are available too.
 * <code>uri.parse('/foo%20bar').getRawPath()</code> will return the raw path,
 * <tt>/foo%20bar</tt>.  Use the raw setters with care, since
 * <code>URI::toString</code> is not guaranteed to return a valid url if a
 * raw setter was used.
 *
 * <p>All setters return <tt>this</tt> and so may be chained, a la
 * <code>uri.parse('/foo').setFragment('part').toString()</code>.
 *
 * <p>You should not use this constructor directly -- please prefer the factory
 * functions {@link uri.parse}, {@link uri.create}, {@link uri.resolve}
 * instead.</p>
 *
 * <p>The parameters are all raw (assumed to be properly escaped) parts, and
 * any (but not all) may be null.  Undefined is not allowed.</p>
 *
 * @constructor
 */
function URI(
    rawScheme,
    rawCredentials, rawDomain, port,
    rawPath, rawQuery, rawFragment) {
  this.scheme_ = rawScheme;
  this.credentials_ = rawCredentials;
  this.domain_ = rawDomain;
  this.port_ = port;
  this.path_ = rawPath;
  this.query_ = rawQuery;
  this.fragment_ = rawFragment;
  /**
   * @type {Array|null}
   */
  this.paramCache_ = null;
}

/** returns the string form of the url. */
URI.prototype.toString = function () {
  var out = [];
  if (null !== this.scheme_) { out.push(this.scheme_, ':'); }
  if (null !== this.domain_) {
    out.push('//');
    if (null !== this.credentials_) { out.push(this.credentials_, '@'); }
    out.push(this.domain_);
    if (null !== this.port_) { out.push(':', this.port_.toString()); }
  }
  if (null !== this.path_) { out.push(this.path_); }
  if (null !== this.query_) { out.push('?', this.query_); }
  if (null !== this.fragment_) { out.push('#', this.fragment_); }
  return out.join('');
};

URI.prototype.clone = function () {
  return new URI(this.scheme_, this.credentials_, this.domain_, this.port_,
                 this.path_, this.query_, this.fragment_);
};

URI.prototype.getScheme = function () {
  return this.scheme_ && decodeURIComponent(this.scheme_);
};
URI.prototype.getRawScheme = function () {
  return this.scheme_;
};
URI.prototype.setScheme = function (newScheme) {
  this.scheme_ = encodeIfExists2(
      newScheme, URI_DISALLOWED_IN_SCHEME_OR_CREDENTIALS_);
  return this;
};
URI.prototype.setRawScheme = function (newScheme) {
  this.scheme_ = newScheme ? newScheme : null;
  return this;
};
URI.prototype.hasScheme = function () {
  return null !== this.scheme_;
};


URI.prototype.getCredentials = function () {
  return this.credentials_ && decodeURIComponent(this.credentials_);
};
URI.prototype.getRawCredentials = function () {
  return this.credentials_;
};
URI.prototype.setCredentials = function (newCredentials) {
  this.credentials_ = encodeIfExists2(
      newCredentials, URI_DISALLOWED_IN_SCHEME_OR_CREDENTIALS_);

  return this;
};
URI.prototype.setRawCredentials = function (newCredentials) {
  this.credentials_ = newCredentials ? newCredentials : null;
  return this;
};
URI.prototype.hasCredentials = function () {
  return null !== this.credentials_;
};


URI.prototype.getDomain = function () {
  return this.domain_ && decodeURIComponent(this.domain_);
};
URI.prototype.getRawDomain = function () {
  return this.domain_;
};
URI.prototype.setDomain = function (newDomain) {
  this.domain_ = newDomain ? encodeURIComponent(newDomain) : null;
  return this;
};
URI.prototype.setRawDomain = function (newDomain) {
  this.domain_ = newDomain ? newDomain : null;
  return this;
};
URI.prototype.hasDomain = function () {
  return null !== this.domain_;
};


URI.prototype.getPort = function () {
  return this.port_ && decodeURIComponent(this.port_);
};
URI.prototype.setPort = function (newPort) {
  if (newPort) {
    newPort = Number(newPort);
    if (newPort !== (newPort & 0xffff)) {
      throw new Error('Bad port number ' + newPort);
    }
    this.port_ = '' + newPort;
  } else {
    this.port_ = null;
  }
  return this;
};
URI.prototype.hasPort = function () {
  return null !== this.port_;
};


URI.prototype.getPath = function () {
  return this.path_ && decodeURIComponent(this.path_);
};
URI.prototype.getRawPath = function () {
  return this.path_;
};
URI.prototype.setPath = function (newPath) {
  this.path_ = encodeIfExists2(newPath, URI_DISALLOWED_IN_PATH_);
  return this;
};
URI.prototype.setRawPath = function (newPath) {
  this.path_ = newPath ? newPath : null;
  return this;
};
URI.prototype.hasPath = function () {
  return null !== this.path_;
};


URI.prototype.getQuery = function () {
  // From http://www.w3.org/Addressing/URL/4_URI_Recommentations.html
  // Within the query string, the plus sign is reserved as shorthand notation
  // for a space.
  return this.query_ && decodeURIComponent(this.query_).replace(/\+/g, ' ');
};
URI.prototype.getRawQuery = function () {
  return this.query_;
};
URI.prototype.setQuery = function (newQuery) {
  this.paramCache_ = null;
  this.query_ = encodeIfExists(newQuery);
  return this;
};
URI.prototype.setRawQuery = function (newQuery) {
  this.paramCache_ = null;
  this.query_ = newQuery ? newQuery : null;
  return this;
};
URI.prototype.hasQuery = function () {
  return null !== this.query_;
};

/**
 * sets the query given a list of strings of the form
 * [ key0, value0, key1, value1, ... ].
 *
 * <p><code>uri.setAllParameters(['a', 'b', 'c', 'd']).getQuery()</code>
 * will yield <code>'a=b&c=d'</code>.
 */
URI.prototype.setAllParameters = function (params) {
  if (typeof params === 'object') {
    if (!(params instanceof Array)
        && (params instanceof Object
            || Object.prototype.toString.call(params) !== '[object Array]')) {
      var newParams = [];
      var i = -1;
      for (var k in params) {
        var v = params[k];
        if ('string' === typeof v) {
          newParams[++i] = k;
          newParams[++i] = v;
        }
      }
      params = newParams;
    }
  }
  this.paramCache_ = null;
  var queryBuf = [];
  var separator = '';
  for (var j = 0; j < params.length;) {
    var k = params[j++];
    var v = params[j++];
    queryBuf.push(separator, encodeURIComponent(k.toString()));
    separator = '&';
    if (v) {
      queryBuf.push('=', encodeURIComponent(v.toString()));
    }
  }
  this.query_ = queryBuf.join('');
  return this;
};
URI.prototype.checkParameterCache_ = function () {
  if (!this.paramCache_) {
    var q = this.query_;
    if (!q) {
      this.paramCache_ = [];
    } else {
      var cgiParams = q.split(/[&\?]/);
      var out = [];
      var k = -1;
      for (var i = 0; i < cgiParams.length; ++i) {
        var m = cgiParams[i].match(/^([^=]*)(?:=(.*))?$/);
        // From http://www.w3.org/Addressing/URL/4_URI_Recommentations.html
        // Within the query string, the plus sign is reserved as shorthand
        // notation for a space.
        out[++k] = decodeURIComponent(m[1]).replace(/\+/g, ' ');
        out[++k] = decodeURIComponent(m[2] || '').replace(/\+/g, ' ');
      }
      this.paramCache_ = out;
    }
  }
};
/**
 * sets the values of the named cgi parameters.
 *
 * <p>So, <code>uri.parse('foo?a=b&c=d&e=f').setParameterValues('c', ['new'])
 * </code> yields <tt>foo?a=b&c=new&e=f</tt>.</p>
 *
 * @param key {string}
 * @param values {Array.<string>} the new values.  If values is a single string
 *   then it will be treated as the sole value.
 */
URI.prototype.setParameterValues = function (key, values) {
  // be nice and avoid subtle bugs where [] operator on string performs charAt
  // on some browsers and crashes on IE
  if (typeof values === 'string') {
    values = [ values ];
  }

  this.checkParameterCache_();
  var newValueIndex = 0;
  var pc = this.paramCache_;
  var params = [];
  for (var i = 0, k = 0; i < pc.length; i += 2) {
    if (key === pc[i]) {
      if (newValueIndex < values.length) {
        params.push(key, values[newValueIndex++]);
      }
    } else {
      params.push(pc[i], pc[i + 1]);
    }
  }
  while (newValueIndex < values.length) {
    params.push(key, values[newValueIndex++]);
  }
  this.setAllParameters(params);
  return this;
};
URI.prototype.removeParameter = function (key) {
  return this.setParameterValues(key, []);
};
/**
 * returns the parameters specified in the query part of the uri as a list of
 * keys and values like [ key0, value0, key1, value1, ... ].
 *
 * @return {Array.<string>}
 */
URI.prototype.getAllParameters = function () {
  this.checkParameterCache_();
  return this.paramCache_.slice(0, this.paramCache_.length);
};
/**
 * returns the value<b>s</b> for a given cgi parameter as a list of decoded
 * query parameter values.
 * @return {Array.<string>}
 */
URI.prototype.getParameterValues = function (paramNameUnescaped) {
  this.checkParameterCache_();
  var values = [];
  for (var i = 0; i < this.paramCache_.length; i += 2) {
    if (paramNameUnescaped === this.paramCache_[i]) {
      values.push(this.paramCache_[i + 1]);
    }
  }
  return values;
};
/**
 * returns a map of cgi parameter names to (non-empty) lists of values.
 * @return {Object.<string,Array.<string>>}
 */
URI.prototype.getParameterMap = function (paramNameUnescaped) {
  this.checkParameterCache_();
  var paramMap = {};
  for (var i = 0; i < this.paramCache_.length; i += 2) {
    var key = this.paramCache_[i++],
      value = this.paramCache_[i++];
    if (!(key in paramMap)) {
      paramMap[key] = [value];
    } else {
      paramMap[key].push(value);
    }
  }
  return paramMap;
};
/**
 * returns the first value for a given cgi parameter or null if the given
 * parameter name does not appear in the query string.
 * If the given parameter name does appear, but has no '<tt>=</tt>' following
 * it, then the empty string will be returned.
 * @return {string|null}
 */
URI.prototype.getParameterValue = function (paramNameUnescaped) {
  this.checkParameterCache_();
  for (var i = 0; i < this.paramCache_.length; i += 2) {
    if (paramNameUnescaped === this.paramCache_[i]) {
      return this.paramCache_[i + 1];
    }
  }
  return null;
};

URI.prototype.getFragment = function () {
  return this.fragment_ && decodeURIComponent(this.fragment_);
};
URI.prototype.getRawFragment = function () {
  return this.fragment_;
};
URI.prototype.setFragment = function (newFragment) {
  this.fragment_ = newFragment ? encodeURIComponent(newFragment) : null;
  return this;
};
URI.prototype.setRawFragment = function (newFragment) {
  this.fragment_ = newFragment ? newFragment : null;
  return this;
};
URI.prototype.hasFragment = function () {
  return null !== this.fragment_;
};

function nullIfAbsent(matchPart) {
  return ('string' == typeof matchPart) && (matchPart.length > 0)
         ? matchPart
         : null;
}




/**
 * a regular expression for breaking a URI into its component parts.
 *
 * <p>http://www.gbiv.com/protocols/uri/rfc/rfc3986.html#RFC2234 says
 * As the "first-match-wins" algorithm is identical to the "greedy"
 * disambiguation method used by POSIX regular expressions, it is natural and
 * commonplace to use a regular expression for parsing the potential five
 * components of a URI reference.
 *
 * <p>The following line is the regular expression for breaking-down a
 * well-formed URI reference into its components.
 *
 * <pre>
 * ^(([^:/?#]+):)?(//([^/?#]*))?([^?#]*)(\?([^#]*))?(#(.*))?
 *  12            3  4          5       6  7        8 9
 * </pre>
 *
 * <p>The numbers in the second line above are only to assist readability; they
 * indicate the reference points for each subexpression (i.e., each paired
 * parenthesis). We refer to the value matched for subexpression <n> as $<n>.
 * For example, matching the above expression to
 * <pre>
 *     http://www.ics.uci.edu/pub/ietf/uri/#Related
 * </pre>
 * results in the following subexpression matches:
 * <pre>
 *    $1 = http:
 *    $2 = http
 *    $3 = //www.ics.uci.edu
 *    $4 = www.ics.uci.edu
 *    $5 = /pub/ietf/uri/
 *    $6 = <undefined>
 *    $7 = <undefined>
 *    $8 = #Related
 *    $9 = Related
 * </pre>
 * where <undefined> indicates that the component is not present, as is the
 * case for the query component in the above example. Therefore, we can
 * determine the value of the five components as
 * <pre>
 *    scheme    = $2
 *    authority = $4
 *    path      = $5
 *    query     = $7
 *    fragment  = $9
 * </pre>
 *
 * <p>msamuel: I have modified the regular expression slightly to expose the
 * credentials, domain, and port separately from the authority.
 * The modified version yields
 * <pre>
 *    $1 = http              scheme
 *    $2 = <undefined>       credentials -\
 *    $3 = www.ics.uci.edu   domain       | authority
 *    $4 = <undefined>       port        -/
 *    $5 = /pub/ietf/uri/    path
 *    $6 = <undefined>       query without ?
 *    $7 = Related           fragment without #
 * </pre>
 */
var URI_RE_ = new RegExp(
      "^" +
      "(?:" +
        "([^:/?#]+)" +         // scheme
      ":)?" +
      "(?://" +
        "(?:([^/?#]*)@)?" +    // credentials
        "([^/?#:@]*)" +        // domain
        "(?::([0-9]+))?" +     // port
      ")?" +
      "([^?#]+)?" +            // path
      "(?:\\?([^#]*))?" +      // query
      "(?:#(.*))?" +           // fragment
      "$"
      );

var URI_DISALLOWED_IN_SCHEME_OR_CREDENTIALS_ = /[#\/\?@]/g;
var URI_DISALLOWED_IN_PATH_ = /[\#\?]/g;

URI.parse = parse;
URI.create = create;
URI.resolve = resolve;
URI.collapse_dots = collapse_dots;  // Visible for testing.

return URI;
})();
;
// Copyright (C) 2009 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * Each load maker object, given the absolute URL of the current module, an
 * identifier resolver, and a cajoler finder, returns a load object.
 *
 * A load object is a function object; load() returns a module object,
 * given its module identifier.
 * load.async() returns a promise to the module, given the module identifier.
 * 
 * What a module identifier is is entirely up to the identifier resolver. The
 * identifier resolver is a function of two parameters, (current module 
 * absolute URL, module identifier), returning the absolute URL for the
 * identified module. The default module identifier resolver considers the
 * module identifier to be a relative URL.
 * 
 * The cajoler finder is a function which should take an absolute module URL
 * and return a URL for cajoled code to load as if by a <script> element. (It
 * need not point to an actual cajoling service and could instead use static
 * .out.js files, depending on the application.)
 *
 * Note that this system never actually fetches the module absolute URL, only
 * passes it to the cajoler. But it *is* used as a key in the cache of loaded
 * modules, so a module absolute URL should always have the same module.
 * 
 * To obtain the dependencies of this file, load:
 *   cajita.js, bridal.js, uri.js, cajita-promise.js
 * 
 * TODO(kpreid): explain static (sync) loading module id semantics.
 * @author maoziqing@gmail.com, kpreid@switchb.org, ihab.awad@gmail.com
 * @requires eval, document, ___, bridal, Q, URI, cajita, window
 * @provides xhrModuleLoadMaker, scriptModuleLoadMaker, clearModuleCache,
 *           defaultModuleIdResolver, defaultCajolerFinder,
 *           CajolingServiceFinder
 */
var xhrModuleLoadMaker;
var scriptModuleLoadMaker;
var defaultModuleIdResolver;
var defaultCajolerFinder;
var CajolingServiceFinder;
var clearModuleCache;

(function() {
  // Map from absolute module URLs to module objects.
  var cache = {};

  defaultModuleIdResolver = function(thisModURL, mid) {
    if (!/\.js$/.test(mid) && !/\.html$/.test(mid)) {
      mid = mid + '.js';
    }
    return URI.resolve(URI.parse(thisModURL), URI.parse(mid)).toString();
  };

  /**
   * Constructor for a cajoler finder given the URL of a cajoling service.
   */
  CajolingServiceFinder = function(serviceURL) {
    return  function cajolingServiceFinder(uncajoledSourceURL, jsonpCallback) {
      var inputMimeType;
      if (/\.js$/.test(uncajoledSourceURL)) {
        inputMimeType = 'application/javascript';
      } else if (/\.html$/.test(uncajoledSourceURL)) {
        inputMimeType = 'text/html';
      } else {
        inputMimeType = 'application/javascript';
      }

      return jsonpCallback
          ? serviceURL +
              '?url=' + encodeURIComponent(uncajoledSourceURL) +
              '&renderer=pretty' +
              '&input-mime-type=' + inputMimeType +
              '&output-mime-type=application/json' +
              '&emit-html-in-js=true' +
              '&callback=' + jsonpCallback +
              '&alt=json-in-script'
          : serviceURL +
              '?url=' + encodeURIComponent(uncajoledSourceURL) +
              '&renderer=pretty' +
              '&input-mime-type=' + inputMimeType +
              '&output-mime-type=application/javascript' +
              '&emit-html-in-js=true' +
              '&alt=json';
    };
  };
  
  defaultCajolerFinder = new CajolingServiceFinder(
      'http://caja.appspot.com/cajole');

  function syncLoad(modURL) {
    if (cache[modURL] === undefined || Q.near(cache[modURL]).isPromise___) {
      var msg = "The static module " + modURL + " cannot be resolved.";
      cajita.log(msg);
      throw new Error(msg);
    }
    return Q.near(cache[modURL]);
  }

  function loadMaker(thisModURL, midResolver, cajolerFinder, asyncLoad) {
    var load = function(mid) {
      return syncLoad(midResolver(thisModURL, mid));
    };

    var async = function(mid) {
      return asyncLoad(midResolver(thisModURL, mid),
                       midResolver, cajolerFinder);
    };

    var asyncAll = function(moduleNames) {
      var r = Q.defer();
      var i;
      var modulePromises = [];
      var modules = {};

      for (i = 0; i < moduleNames.length; ++i) {
        modulePromises[i] = async(moduleNames[i]);
      }

      var waitNext = function(idx) {
        if (idx === moduleNames.length) {
          r.resolve(modules);
        } else {
          Q.when( modulePromises[idx], function(theModule) {
            modules[moduleNames[idx]] = theModule;
            waitNext(idx + 1);
          }, function(reason) {
            r.resolve(Q.reject(reason));
          });
        }
      };
      waitNext(0);
      return r.promise();
    };

    load.FUNC___ = 'load';
    ___.setStatic(load, 'async', ___.markFuncFreeze(async));
    ___.setStatic(load, 'asyncAll', ___.markFuncFreeze(asyncAll));
    return ___.primFreeze(load);
  }

  function resolveDependency(module, load) {
    var r = Q.defer();
    if (module.includedModules !== undefined
        && module.includedModules.length !== 0) {
      var size = module.includedModules.length;
      var count = 0;
      for (var i = 0; i < size; i++) {
        var mid = module.includedModules[i];
        var m = load.async(mid);
        Q.when(m, function(childModule) {
                    count++;
                    if (count === size) {
                      r.resolve(true);
                    }
                  },
                  function(reason) {
                    r.resolve(Q.reject(
                        "Retrieving the module " + mid + " failed."));
                  });
      }
    } else {
      r.resolve(true);
    }
    return r.promise;
  }

  function noop() {}
  
  /** 
   * Given a method of async loading, produce the load-maker that the client
   * will use.
   */
  function makeConcreteLoadMaker(asyncLoadFunction) {
    return ___.markFuncFreeze(function(mid, midResolver, cajolerFinder) {
      if (midResolver === undefined) {
        midResolver = defaultModuleIdResolver;
      }
      if (cajolerFinder === undefined) {
        cajolerFinder = defaultCajolerFinder;
      }

      return loadMaker(mid, midResolver, cajolerFinder, asyncLoadFunction);
    });
  }

  function messagesToLog(moduleURL, cajolerMessages) {
    if (!cajolerMessages) { return; }
    cajita.log("Messages cajoling " + moduleURL);
    var msg;
    for (var i = 0; i < cajolerMessages.length; i++) {
      msg = cajolerMessages[i];
      cajita.log(
          msg.name + '(' + msg.level + ') '
          + msg.type + ': ' + msg.message);
    }
  }

  function xhrAsyncLoad(modURL, midResolver, cajolerFinder) {
    if (cache[modURL] !== undefined) {
      return cache[modURL];
    }

    var r = Q.defer();
    cache[modURL] = r.promise;

    var xhr = bridal.makeXhr();
    xhr.onreadystatechange = function() {
      if (xhr.readyState === 4) {
        xhr.onreadystatechange = noop;  // avoid memory leak
        if (xhr.status === 200) {
          var savedModuleHandler = ___.getNewModuleHandler();
          ___.setNewModuleHandler(___.primFreeze({
            handle: ___.markFuncFreeze(function theHandler(module) {
              try {
                var load = loadMaker(modURL, midResolver, cajolerFinder,
                                     xhrAsyncLoad);
                module.moduleURL = modURL;
                var securedModule = ___.prepareModule(module, load);
                var dependency = resolveDependency(module, load);
                Q.when(dependency, function(result) {
                                     r.resolve(securedModule);
                                   },
                                   function(reason) {
                                     r.resolve(Q.reject(
                                         "Resolving dependency for the"
                                         + "module " + modURL + " failed."));
                                   });
              } catch (e) {
                r.resolve(Q.reject(e));
              }
            }),
            handleUncaughtException: savedModuleHandler.handleUncaughtException
          }));
          //TODO: validate the response before eval it
          try {
            eval(xhr.responseText);
          } finally {
            ___.setNewModuleHandler(savedModuleHandler);
          }
        } else {
          r.resolve(Q.reject(
              "Retrieving the module " + modURL + " failed, "
              + "status code = " + xhr.status));
        }
      }
    };

    var cajoledModURL = cajolerFinder(modURL);

    xhr.open("GET", cajoledModURL, true);
    if (typeof xhr.overrideMimeType === 'function') {
      xhr.overrideMimeType("application/javascript");
    }
    xhr.send(null);
    return r.promise;
  }

  xhrModuleLoadMaker = makeConcreteLoadMaker(xhrAsyncLoad);

  var jsonpCallbackCount = 0;

  function scriptAsyncLoad(modURL, midResolver, cajolerFinder) {
    if (cache[modURL] !== undefined) {
      return cache[modURL];
    }

    var r = Q.defer();
    cache[modURL] = r.promise;

    var load = loadMaker(modURL, midResolver, cajolerFinder, scriptAsyncLoad);

    var jsonpCallbackName = '___caja_mod_' + jsonpCallbackCount++ + '___';

    function prepareModule(moduleText) {
      var rawModule = undefined;
      (function() {
        var ___ = { loadModule: function(m) { rawModule = m; } };
        eval(moduleText);
      })();
      return ___.prepareModule(rawModule, load);
    }

    var w = window;  // Caja linter rejects direct assignment to 'window'

    w[jsonpCallbackName] = function(moduleJson) {

      delete w[jsonpCallbackName];
      if (moduleJson.js) {
        var preparedModule = prepareModule(moduleJson.js);
        Q.when(resolveDependency(preparedModule, load),
            function(result) { r.resolve(preparedModule); },
            function(reason) { r.resolve(Q.reject(reason)); });
      } else {
        r.resolve(Q.reject('Cajoling module ' + modURL + ' failed'));
      }
      messagesToLog(modURL, moduleJson.messages);
    };

    var script = document.createElement('script');
    script.src = cajolerFinder(modURL, jsonpCallbackName);
    script.onerror = function() {
      r.resolve(Q.reject('Error loading cajoled module ' + modURL));
    };
    document.getElementsByTagName('head')[0].appendChild(script);

    return r.promise;
  }

  scriptModuleLoadMaker = makeConcreteLoadMaker(scriptAsyncLoad);

  clearModuleCache = ___.markFuncFreeze(function() {
    ___.forOwnKeys(cache, ___.markFuncFreeze(function(k, v) {
      delete cache[k];
    }));
  });
})();
;
// Copyright 2007-2009 Tyler Close 
// under the terms of the MIT X license found at 
// http://www.opensource.org/licenses/mit-license.html


/** 
 * Implementation of promise for Cajita
 * Export Q to the global scope
 * 
 * Mostly taken from the ref_send implementation by Tyler Close
 * Add the isPromise___ flag to support function promise
 * 
 * @contributor: maoziqing@gmail.com
 * @requires setTimeout, ___
 * @provides Q
 */

var Q;

(function() {
  function reject(reason) {
    function rejected(op, arg1, arg2, arg3) {
      if (undefined === op) { return rejected; }
        if ('WHEN' === op) { return arg2 ? arg2(reason) : reject(reason); }
          return arg1 ? arg1(reject(reason)) : reject(reason);
    }
    rejected.reason = reason;
    rejected.isPromise___ = true;
    return rejected;
  }

  function ref(value) {
    if (null === value || undefined === value) {
      return reject({ 'class': [ 'NaO' ] });
    }
    if ('number' === typeof value && !isFinite(value)) {
      return reject({ 'class': [ 'NaN' ] });
    }
    function fulfilled(op, arg1, arg2, arg3) {
      if (undefined === op) { return value; }
      var r;
      switch (op) {
        case 'WHEN':
          r = value;
          break;
        case 'GET':
          if (undefined === arg2 || null === arg2) {
            r = value;
          } else {
            r = value[arg2];
          }
          break;
        case 'POST':
          if (undefined === arg2 || null === arg2) {
            r = reject({});
          } else {
            r = value[arg2].apply(value, arg3);
          }
          break;
        case 'PUT':
          if (undefined === arg2 || null === arg2) {
            r = reject({});
          } else {
            value[arg2] = arg3;
            r = {};
          }
          break;
        case 'DELETE':
          if (undefined === arg2 || null === arg2) {
            r = reject({});
          } else {
            delete value[arg2];
            r = {};
          }
          break;
        default:
          r = reject({});
      }
      return arg1 ? arg1.apply(null, [r]) : r;
    }
    fulfilled.isPromise___ = true;
    return fulfilled;
  }
 	
  var enqueue = (function () {
    var active = false;
    var pending = [];
    var run = function () {
      var task = pending.shift();
      if (0 === pending.length) {
        active = false;
      } else {
        setTimeout(run, 0);
      }
      task();
    };
    return function (task) {
      pending.push(task);
      if (!active) {
        setTimeout(run, 0);
        active = true;
      }
    };
  }());
 	
  /**
   * Enqueues a promise operation.
   *
   * The above functions, reject() and ref(), each construct a kind of
   * promise. Other libraries can provide other kinds of promises by
   * implementing the same API. A promise is a function with signature:
   * function (op, arg1, arg2, arg3). The first argument determines the
   * interpretation of the remaining arguments. The following cases exist:
   *
   * 'op' is undefined:
   *  Return the most resolved current value of the promise.
   *
   * 'op' is "WHEN":
   *  'arg1': callback to invoke with the fulfilled value of the promise
   *  'arg2': callback to invoke with the rejection reason for the promise
   *
   * 'op' is "GET":
   *  'arg1': callback to invoke with the value of the named property
   *  'arg2': name of the property to read
   *
   * 'op' is "POST":
   *  'arg1': callback to invoke with the return value from the invocation
   *  'arg2': name of the method to invoke
   *  'arg3': array of invocation arguments
   *
   * 'op' is "PUT":
   *  'arg1': callback to invoke with the return value from the operation
   *  'arg2': name of the property to set
   *  'arg3': new value of property
   *
   * 'op' is "DELETE":
   *  'arg1': callback to invoke with the return value from the operation
   *  'arg2': name of the property to delete
   *
   * 'op' is unrecognized:
   *  'arg1': callback to invoke with a rejected promise
   */
  function forward(p, op, arg1, arg2, arg3) {
    enqueue(function () { p(op, arg1, arg2, arg3); });
  }

  /**
   * Gets the corresponding promise for a given reference.
   */
  function promised(value) {
    return ('function' === typeof value && value.isPromise___ === true) 
        ? value : ref(value);
  }

  function defer() {
    var value;
    var pending = [];
    function promise(op, arg1, arg2, arg3) {
      if (undefined === op) { return pending ? promise : value(); }
      if (pending) {
        pending.push({ op: op, arg1: arg1, arg2: arg2, arg3: arg3 });
      } else {
        forward(value, op, arg1, arg2, arg3);
      }
    }
    promise.isPromise___ = true;
    return ___.primFreeze({
      promise: ___.markFuncFreeze(promise),
      resolve: ___.markFuncFreeze(function (p) {
        if (!pending) { return; }

        var todo = pending;
        pending = null;
        value = promised(p);
        for (var i = 0; i !== todo.length; i += 1) {
          var x = todo[+i];
          forward(value, x.op, x.arg1, x.arg2, x.arg3);
        }
      })
    });
  }

  Q = {	
    /**
     * Enqueues a task to be run in a future turn.
     * @param task  function to invoke later
     */
    run: ___.markFuncFreeze(enqueue),
  
    /**
     * Constructs a rejected promise.
     * @param reason    value describing the failure
     */
    reject: ___.markFuncFreeze(reject),
  
    /**
     * Constructs a promise for an immediate reference.
     * @param value immediate reference
     */
    ref: ___.markFuncFreeze(ref),
  
    /**
     * Constructs a ( promise, resolver ) pair.
     *
     * The resolver is a callback to invoke with a more resolved value for
     * the promise. To fulfill the promise, simply invoke the resolver with
     * an immediate reference. To reject the promise, invoke the resolver
     * with the return from a call to reject(). To put the promise in the
     * same state as another promise, invoke the resolver with that other
     * promise.
     */
    defer: ___.markFuncFreeze(defer),

    /**
     * Gets the current value of a promise.
     * @param value promise or immediate reference to evaluate
     */
    near: ___.markFuncFreeze(function (value) {
      return ('function' === typeof value && value.isPromise___ === true)
          ? value() : value;
    }),

    /**
     * Registers an observer on a promise.
     * @param value     promise or immediate reference to observe
     * @param fulfilled function to be called with the resolved value
     * @param rejected  function to be called with the rejection reason
     * @return promise for the return value from the invoked callback
     */
    when: ___.markFuncFreeze(function (value, fulfilled, rejected) {
      var r = defer();
      var done = false;   // ensure the untrusted promise makes at most a
                          // single call to one of the callbacks
      forward(promised(value), 'WHEN', function (x) {
        if (done) { throw new Error(); }
        done = true;
        r.resolve(ref(x)('WHEN', fulfilled, rejected));
      }, function (reason) {
        if (done) { throw new Error(); }
        done = true;
        r.resolve(rejected ? rejected.apply(null, [reason]) : reject(reason));
      });
      return r.promise;
    }),

    /**
     * Gets the value of a property in a future turn.
     * @param target    promise or immediate reference for target object
     * @param noun      name of property to get
     * @return promise for the property value
     */
    get: ___.markFuncFreeze(function (target, noun) {
      var r = defer();
      forward(promised(target), 'GET', r.resolve, noun);
      return r.promise;
    }),

    /**
     * Invokes a method in a future turn.
     * @param target    promise or immediate reference for target object
     * @param verb      name of method to invoke
     * @param argv      array of invocation arguments
     * @return promise for the return value
     */
    post: ___.markFuncFreeze(function (target, verb, argv) {
      var r = defer();
      forward(promised(target), 'POST', r.resolve, verb, argv);
      return r.promise;
    }),

    /**
     * Sets the value of a property in a future turn.
     * @param target    promise or immediate reference for target object
     * @param noun      name of property to set
     * @param value     new value of property
     * @return promise for the return value
     */
    put: ___.markFuncFreeze(function (target, noun, value) {
      var r = defer();
      forward(promised(target), 'PUT', r.resolve, noun, value);
      return r.promise;
    }),

    /**
     * Deletes a property in a future turn.
     * @param target    promise or immediate reference for target object
     * @param noun      name of property to delete
     * @return promise for the return value
     */
    remove: ___.markFuncFreeze(function (target, noun) {
      var r = defer();
      forward(promised(target), 'DELETE', r.resolve, noun);
      return r.promise;
    })
  };

  ___.primFreeze(Q);
})();
;
// Copyright (C) 2010 Google Inc.
//      
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @fileoverview Utilities for common patterns in host pages.
 *
 * (HostTools can also work without cajita-module, i.e. without
 * scriptModuleLoadMaker, at the cost of sandbox.run() functionality.)
 *
 * @author kpreid@switchb.org
 * @requires eval, window, document, ___, cajita, attachDocumentStub,
 *           valijaMaker, Q,
 *           scriptModuleLoadMaker, defaultModuleIdResolver,
 *           CajolingServiceFinder,
 *           HtmlEmitter
 * @provides HostTools
 */

var HostTools;
(function () {
  var hasModuleLoader = "scriptModuleLoadMaker" in window;
  var toolsInstanceCounter = 0;
  
  HostTools = function () {
    var toolsInstanceIndex = ++toolsInstanceCounter;
    var gadgetInstanceCounter = 0;

    // user-modifiable state
    var cajolingService = "http://caja.appspot.com/cajole";
    var baseURL = document.location.toString();
    // TODO(kpreid): the above probably does the wrong thing in the case where
    // the document has a <base>; fix.
    
    // cache
    var load;
    
    // internal functions
    function updateLoad() {
      if (!hasModuleLoader) {
        load = null;
      } else {
        // TODO(kpreid): allow subbing module id resolver
        // TODO(kpreid): Using XHR loader didn't work; why?
        load = scriptModuleLoadMaker(
          baseURL,
          defaultModuleIdResolver,
          new CajolingServiceFinder(cajolingService));
      }
    }
    updateLoad();
    
    // public methods
    
    function setCajolerService(url) {
      cajolingService = url;
      updateLoad();
    }
    
    function setBaseURL(url) {
      baseURL = url;
      updateLoad();
    }
    
    function Sandbox() {
      var attached = false;

      // user-modifiable state
      var imports = ___.copy(___.sharedImports);
      var uriPolicy = cajita.freeze({
        rewrite: function (uri, uriEffect, loaderType, hints) {
          // TODO(kpreid): This needs to be redefined in terms of effect/loader;
          // we don't necessarily know the actual specific mime type before
          // making the request.
          // TODO(ihab.awad): Hack to make MIME types both '*/*' since we are
          // going to deprecate this file anyway. Breakage introduced at r4372.
          return cajolingService +
              '?url=' + encodeURIComponent(uri) +
              '&input-mime-type=*%2F*' +
              '&output-mime-type=*%2F*';
        }
      });
  
      // public methods
      
      function setURIPolicy(newPolicy) {
        if (attached) {
          throw("setURIPolicy() must be used before attach()");
        }
        uriPolicy = newPolicy;
      }

      function run(mid) {
        if (load == null) {
          throw new Error("HostTools: Loaded without cajita-module-orig.js, "
              + "so cannot dynamically load modules.");
        }
        
        // TODO: review error handling -- are errors propagated and descriptive?
        return Q.when(load.async(mid), function (moduleFunc) {
          //console.log("got load Q callback");
          return moduleFunc(imports);
        });
      }
      
      function runCajoledModuleString(js) {
        // This is primarily useful when used from caja.js (Caja in an iframe),
        // since the host page doesn't have eval of the frame

        var pumpkin = {};
        var moduleFunc = pumpkin;
        var oldModuleHandler = ___.getNewModuleHandler();
        var newModuleHandler = ___.makeNormalNewModuleHandler();
        newModuleHandler.handle = ___.markFuncFreeze(
            function theHandler(module) {
              // TODO: does not support dependencies. Needs to tie in to
              // cajita-module-orig.js for that and to give the module a proper
              // relative load function.
              moduleFunc = ___.prepareModule(module, load);
            });
        try {
          ___.setNewModuleHandler(newModuleHandler);
          eval(js);
        } finally {
          ___.setNewModuleHandler(oldModuleHandler);
        }
        if (moduleFunc === pumpkin) {
          throw new Error("runCajoledModuleString: the provided code did not " +
            "invoke the new module handler");
        }
        return moduleFunc(imports);
      }
      
      function attach(vdocBody, options) {
        attached = true;

        // Generate unique element id suffix for Domita.
        // There are two counters just to make them a little more decodable.
        var gadgetInstanceIndex = ++gadgetInstanceCounter;
        var idSuffix = "-HostToolsGadget-" + toolsInstanceIndex + "-" +
            gadgetInstanceIndex + "___";
        
        options = ___.copy(options ? options : {});
        if (options.valija === undefined) { options.valija = true; }
        
        // TODO(kpreid): do we want to reject multiple attach attempts?
        
        if (options.valija) { imports.outers = imports; }
        
        attachDocumentStub(idSuffix, uriPolicy, imports, vdocBody);
        imports.htmlEmitter___ = new HtmlEmitter(vdocBody, imports.document);
        
        if (options.valija) { imports.$v = valijaMaker.CALL___(imports.outers); }
      }
      
      return cajita.freeze({
        attach: attach,
        setURIPolicy: setURIPolicy,
        imports: imports,
        run: run,
        runCajoledModuleString: runCajoledModuleString
      });
    }
    
    return cajita.freeze({
      getLoad: function () { return load; },
      setBaseURL: setBaseURL,
      setCajolerService: setCajolerService,
      Sandbox: Sandbox
    });
  };
})();
;
// Copyright (C) 2010 Google Inc.
//      
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @fileoverview
 * This file exists to be concatenated into the single file that caja.js (the
 * iframed-Caja-runtime loader) loads as the very last thing to give an on-load
 * callback.
 *
 * @author kpreid@switchb.org
 * @requires cajaIframeDone___
 */

cajaIframeDone___();
