/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.io;

import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import org.apache.lucene.util.XIOUtils;
import org.elasticsearch.common.collect.Iterators;
import org.elasticsearch.common.logging.ESLogger;

public class FileSystemUtils {
    public static boolean mkdirs(File dir) {
        return dir.mkdirs();
    }

    public static boolean hasExtensions(File root, String ... extensions) {
        File[] children;
        if (root != null && (children = root.listFiles()) != null) {
            for (File child : children) {
                if (child.isDirectory()) {
                    boolean has = FileSystemUtils.hasExtensions(child, extensions);
                    if (!has) continue;
                    return true;
                }
                for (String extension : extensions) {
                    if (!child.getName().endsWith(extension)) continue;
                    return true;
                }
            }
        }
        return false;
    }

    public static boolean exists(File ... files) {
        for (File file : files) {
            if (!file.exists()) continue;
            return true;
        }
        return false;
    }

    public static boolean exists(Path ... files) {
        for (Path file : files) {
            if (!Files.exists(file, new LinkOption[0])) continue;
            return true;
        }
        return false;
    }

    public static boolean deleteRecursively(File[] roots, boolean deleteRoots) {
        boolean deleted = true;
        for (File root : roots) {
            deleted &= FileSystemUtils.deleteRecursively(root, deleteRoots);
        }
        return deleted;
    }

    public static void deleteSubDirectories(Path ... paths) throws IOException {
        for (Path path : paths) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(path);){
                for (Path subPath : stream) {
                    if (!Files.isDirectory(subPath, new LinkOption[0])) continue;
                    XIOUtils.rm(subPath);
                }
            }
        }
    }

    public static boolean deleteRecursively(File ... roots) {
        return FileSystemUtils.deleteRecursively(roots, true);
    }

    public static boolean deleteRecursively(File root, boolean deleteRoot) {
        if (root != null) {
            File[] children = root.listFiles();
            if (children != null) {
                for (File aChildren : children) {
                    FileSystemUtils.deleteRecursively(aChildren, true);
                }
            }
            if (deleteRoot) {
                return root.delete();
            }
            return true;
        }
        return false;
    }

    public static boolean isAccessibleDirectory(File directory, ESLogger logger) {
        assert (directory != null && logger != null);
        if (!directory.exists()) {
            logger.debug("[{}] directory does not exist.", directory.getAbsolutePath());
            return false;
        }
        if (!directory.isDirectory()) {
            logger.debug("[{}] should be a directory but is not.", directory.getAbsolutePath());
            return false;
        }
        if (!directory.canRead()) {
            logger.debug("[{}] directory is not readable.", directory.getAbsolutePath());
            return false;
        }
        return true;
    }

    private FileSystemUtils() {
    }

    public static void moveFilesWithoutOverwriting(File source, final File destination, final String suffix) throws IOException {
        FileSystemUtils.mkdirs(destination);
        final int configPathRootLevel = source.toPath().getNameCount();
        Files.walkFileTree(source.toPath(), (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            private Path buildPath(Path path) {
                return destination.toPath().resolve(path);
            }

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                Path subpath;
                Path path;
                if (configPathRootLevel != dir.getNameCount() && !Files.exists(path = this.buildPath(subpath = dir.subpath(configPathRootLevel, dir.getNameCount())), new LinkOption[0])) {
                    FileSystemUtils.move(dir, path);
                    return FileVisitResult.SKIP_SUBTREE;
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Path path;
                Path subpath = null;
                if (configPathRootLevel != file.getNameCount()) {
                    subpath = file.subpath(configPathRootLevel, file.getNameCount());
                }
                if (!Files.exists(path = this.buildPath(subpath), new LinkOption[0])) {
                    Files.move(file, path, new CopyOption[0]);
                } else if (suffix != null) {
                    path = Paths.get(path.toString().concat(suffix), new String[0]);
                    Files.move(file, path, StandardCopyOption.REPLACE_EXISTING);
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static void copyDirectoryRecursively(File source, File destination) throws IOException {
        Files.walkFileTree(source.toPath(), new TreeCopier(source.toPath(), destination.toPath(), false));
    }

    public static void move(Path source, Path destination) throws IOException {
        try {
            Files.move(source, destination, new CopyOption[0]);
        }
        catch (DirectoryNotEmptyException e) {
            Files.walkFileTree(source, new TreeCopier(source, destination, true));
        }
    }

    public static Path[] files(Path directory) throws IOException {
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(directory);){
            Path[] pathArray = Iterators.toArray(stream.iterator(), Path.class);
            return pathArray;
        }
    }

    static class TreeCopier
    extends SimpleFileVisitor<Path> {
        private final Path source;
        private final Path target;
        private final boolean delete;

        TreeCopier(Path source, Path target, boolean delete) {
            this.source = source;
            this.target = target;
            this.delete = delete;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
            Path newDir = this.target.resolve(this.source.relativize(dir));
            try {
                Files.copy(dir, newDir, new CopyOption[0]);
            }
            catch (FileAlreadyExistsException fileAlreadyExistsException) {
            }
            catch (IOException x) {
                return FileVisitResult.SKIP_SUBTREE;
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
            if (this.delete) {
                FileSystemUtils.deleteRecursively(dir.toFile(), true);
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            Path newFile = this.target.resolve(this.source.relativize(file));
            try {
                Files.copy(file, newFile, new CopyOption[0]);
                if (this.delete) {
                    Files.deleteIfExists(file);
                }
            }
            catch (IOException iOException) {
                // empty catch block
            }
            return FileVisitResult.CONTINUE;
        }
    }
}

