/*
 * Copyright (c) 2004 - 2007, Tranql project contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.tranql.connector.jdbc;

import java.sql.PreparedStatement;

public class PreparedStatementKey
{
    private final int stmtFormat;
    static final int PREPARED_STMT_FORMAT_1 = 1;
    static final int PREPARED_STMT_FORMAT_2 = 2;
    static final int PREPARED_STMT_FORMAT_3 = 3;
    static final int PREPARED_STMT_FORMAT_4 = 4;
    static final int PREPARED_STMT_FORMAT_5 = 5;
    static final int PREPARED_STMT_FORMAT_6 = 6;
    String sql = null;
    protected int parm0 = 0;
    protected int parm1 = 0;
    protected int parm2 = 0;
    private static final int nullIntArray[] = { 0 };
    private static final String nullStringArray[] = { "" };
    private int columnIndexes[] = nullIntArray;
    private String columnNames[] = nullStringArray;
    private int preparedStatementKeyHashcode = 0;
    PreparedStatement ps = null;
    ConnectionWrapper c = null;
    PreparedStatementWrapper psw = null;

    public PreparedStatementKey(ConnectionWrapper c, String sql)
    {
        this.c = c;
        this.sql = sql;
        stmtFormat = PREPARED_STMT_FORMAT_1;
        setHashCode();
    }

    public PreparedStatementKey(ConnectionWrapper c, String sql, int autoGeneratedKeys)
    {
        this.c = c;
        this.sql = sql;
        parm0 = autoGeneratedKeys;
        stmtFormat = PREPARED_STMT_FORMAT_2;
        setHashCode();
    }

    public PreparedStatementKey(ConnectionWrapper c, String sql, int resultSetType, int resultSetConcurrency)
    {
        this.c = c;
        this.sql = sql;
        parm0 = resultSetType;
        parm1 = resultSetConcurrency;
        stmtFormat = PREPARED_STMT_FORMAT_3;
        setHashCode();
    }

    public PreparedStatementKey(ConnectionWrapper c, String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability)
    {
        this.c = c;
        this.sql = sql;
        parm0 = resultSetType;
        parm1 = resultSetConcurrency;
        parm2 = resultSetHoldability;
        stmtFormat = PREPARED_STMT_FORMAT_4;
        setHashCode();
    }

    public PreparedStatementKey(ConnectionWrapper c, String sql, int columnIndexes[])
    {
        this.c = c;
        this.sql = sql;
        this.columnIndexes = columnIndexes.clone();
        stmtFormat = PREPARED_STMT_FORMAT_5;
        setHashCode();
    }

    public PreparedStatementKey(ConnectionWrapper c, String sql, String columnNames[])
    {
        this.c = c;
        this.sql = sql;
        this.columnNames = columnNames.clone();
        stmtFormat = PREPARED_STMT_FORMAT_6;
        setHashCode();
    }

    @Override
	public int hashCode()
    {
        return preparedStatementKeyHashcode;
    }

    void setHashCode()
    {
        preparedStatementKeyHashcode = sql.hashCode() + parm0 * 10000 + parm1 * 1000 + parm2 * 100 + columnIndexes.hashCode() + columnNames.hashCode();
    }

    @Override
	public boolean equals(Object o)
    {
        PreparedStatementKey psk = (PreparedStatementKey)o;
        if(psk.stmtFormat != stmtFormat)
            return false;
        if(!psk.sql.equals(sql))
            return false;
        
        switch(stmtFormat)
        {
        case 1: // PREPARED_STMT_FORMAT_1 only has SQL...we're good
            return true;

        case 2: // PREPARED_STMT_FORMAT_2 check additional parms
            return psk.parm0 == parm0;

        case 3: // PREPARED_STMT_FORMAT_3 check two additional parms.
            return psk.parm0 == parm0 && psk.parm1 == parm1;

        case 4: // PREPARED_STMT_FORMAT_4 check three additional parms
            return psk.parm0 == parm0 && psk.parm1 == parm1 && psk.parm2 == parm2;

        case 5: // PREPARED_STMT_FORMAT_5 check for column indexes.
            if(psk.columnIndexes.length != columnIndexes.length)
                return false;
            for(int y = 0; y < psk.columnIndexes.length; y++)
                if(psk.columnIndexes[y] != columnIndexes[y])
                    return false;

            return true;

        case 6: // PREPARED_STMT_FORMAT_6 check column names
            if(psk.columnNames.length != columnNames.length)
                return false;
            for(int y = 0; y < psk.columnNames.length; y++)
                if(!psk.columnNames[y].equals(columnNames[y]))
                    return false;

            return true;
        }
        throw new RuntimeException("Invalid Prepared Statment Format of " + stmtFormat + " detected in compare." + 
        		"Statement may be corrupted.\n" +
        		"  Query = "+sql+"\n"+
        		"  PreparedStatementWrapper Information:\n"+psw.toString() );
    }

    public void setPreparedStatementWrapper(PreparedStatementWrapper psw)
    {
        this.psw = psw;
    }

    public PreparedStatementWrapper getPreparedStatementWrapper()
    {
        return psw;
    }
}
