/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *  
 *    http://www.apache.org/licenses/LICENSE-2.0
 *  
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License. 
 *  
 */
package org.apache.directory.server.core.jndi;


import java.util.Hashtable;

import javax.naming.Context;
import javax.naming.NamingException;
import javax.naming.ldap.Control;
import javax.naming.ldap.ExtendedRequest;
import javax.naming.ldap.ExtendedResponse;
import javax.naming.ldap.LdapContext;

import org.apache.directory.server.core.DirectoryService;
import org.apache.directory.server.core.authn.LdapPrincipal;
import org.apache.directory.server.core.configuration.StartupConfiguration;
import org.apache.directory.server.core.interceptor.context.CompareOperationContext;
import org.apache.directory.server.core.interceptor.context.UnbindOperationContext;
import org.apache.directory.server.core.referral.ReferralService;
import org.apache.directory.shared.ldap.NotImplementedException;
import org.apache.directory.shared.ldap.name.LdapDN;


/**
 * An implementation of a JNDI LdapContext.
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 * @version $Rev: 569237 $
 */
public class ServerLdapContext extends ServerDirContext implements LdapContext
{
    /** A reference to the RTeferralService interceptor */
    private transient ReferralService refService = null; 
    

    /**
     * Creates an instance of an ServerLdapContext.
     *
     * @param service the parent service that manages this context
     * @param env the JNDI environment parameters
     * @throws NamingException the context cannot be created
     */
    public ServerLdapContext( DirectoryService service, Hashtable env ) throws NamingException
    {
        super( service, env );
        refService = (( ReferralService )service.getConfiguration()
            .getInterceptorChain().get( StartupConfiguration.REFERRAL_SERVICE_NAME ) );        
    }


    /**
     * Creates a new ServerDirContext with a distinguished name which is used to
     * set the PROVIDER_URL to the distinguished name for this context.
     *
     * @param principal the directory user principal that is propagated
     * @param dn the distinguished name of this context
     */
    ServerLdapContext( DirectoryService service, LdapPrincipal principal, LdapDN dn ) throws NamingException
    {
        super( service, principal, dn );
        refService = (( ReferralService )service.getConfiguration()
            .getInterceptorChain().get( StartupConfiguration.REFERRAL_SERVICE_NAME ) );        
    }


    /**
     * @see javax.naming.ldap.LdapContext#extendedOperation(
     * javax.naming.ldap.ExtendedRequest)
     */
    public ExtendedResponse extendedOperation( ExtendedRequest request )
    {
        throw new NotImplementedException();
    }


    /**
     * @see javax.naming.ldap.LdapContext#newInstance(
     * javax.naming.ldap.Control[])
     */
    public LdapContext newInstance( Control[] requestControls ) throws NamingException
    {
        ServerLdapContext ctx = new ServerLdapContext( getService(), getPrincipal(), ( LdapDN ) getDn() );
        ctx.setRequestControls( requestControls );
        return ctx;
    }


    /**
     * @see javax.naming.ldap.LdapContext#reconnect(javax.naming.ldap.Control[])
     */
    public void reconnect( Control[] connCtls ) throws NamingException
    {
        this.connectControls = connCtls;
    }


    /**
     * @see javax.naming.ldap.LdapContext#getConnectControls()
     */
    public Control[] getConnectControls() throws NamingException
    {
        return this.connectControls;
    }


    /**
     * @see javax.naming.ldap.LdapContext#setRequestControls(
     * javax.naming.ldap.Control[])
     */
    public void setRequestControls( Control[] requestControls ) throws NamingException
    {
        this.requestControls = requestControls;
    }


    /**
     * @see javax.naming.ldap.LdapContext#getRequestControls()
     */
    public Control[] getRequestControls() throws NamingException
    {
        return requestControls;
    }


    /**
     * @see javax.naming.ldap.LdapContext#getResponseControls()
     */
    public Control[] getResponseControls() throws NamingException
    {
        return responseControls;
    }


    // ------------------------------------------------------------------------
    // Additional ApacheDS Specific JNDI Functionality
    // ------------------------------------------------------------------------

    /**
     * Explicitly exposes an LDAP compare operation which JNDI does not
     * directly provide.  All normalization and schema checking etcetera
     * is handled by this call.
     *
     * @param name the name of the entri
     * @param oid the name or object identifier for the attribute to compare
     * @param value the value to compare the attribute to
     * @return true if the entry has the value for the attribute, false otherwise
     * @throws NamingException if the backing store cannot be accessed, or
     * permission is not allowed for this operation or the oid is not recognized,
     * or the attribute is not present in the entry ... you get the picture.
     */
    public boolean compare( LdapDN name, String oid, Object value ) throws NamingException
    {
        // make sure we add the request controls to operation
        CompareOperationContext opCtx = new CompareOperationContext( name, oid, value );
        opCtx.addRequestControls( requestControls );

        // execute operation
        boolean result = super.getNexusProxy().compare( opCtx );
        
        // extract the response controls from the operation and return
        responseControls = getResponseControls();
        requestControls = EMPTY_CONTROLS;
        return result;
    }


    /**
     * Calling this method tunnels an unbind call down into the partition holding 
     * the bindDn.  The bind() counter part is not exposed because it is automatically
     * called when you create a new initial context for a new connection (on wire) or 
     * (programatic) caller.
     * 
     * @throws NamingException
     */
    public void ldapUnbind() throws NamingException
    {
        LdapDN principalDn = null;
        Object principalDnValue = getEnvironment().get( Context.SECURITY_PRINCIPAL );
        
        if ( principalDnValue instanceof LdapDN )
        {
            principalDn = ( LdapDN ) principalDnValue;
        }
        else
        {
            String bindDn = ( String ) principalDnValue;
            principalDn = new LdapDN( bindDn );
        }

        UnbindOperationContext opCtx = new UnbindOperationContext( principalDn );
        opCtx.addRequestControls( requestControls );
        super.getNexusProxy().unbind( opCtx );
        responseControls = opCtx.getResponseControls();
        requestControls = EMPTY_CONTROLS;
    }


    /**
     * Check if a Name is a referral
     * @param name The Name to check
     * @return <code>true</code> if the Name is a referral.
     * @throws NamingException If the Name is incorrect
     */
    public boolean isReferral( String name ) throws NamingException
    {
        return refService.isReferral( name );
    }

    /**
     * Check if a Name is a referral
     * @param name The Name to check
     * @return <code>true</code> if the Name is a referral.
     * @throws NamingException If the Name is incorrect
     */
    public boolean isReferral( LdapDN name ) throws NamingException
    {
        return refService.isReferral( name );
    }

    public ServerContext getRootContext() throws NamingException
    {
        return new ServerLdapContext( getService(), getPrincipal(), new LdapDN() );
    }
}
