/*
 * (C) Copyright 2006-2007 Nuxeo SA (http://nuxeo.com/) and others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 *     Nuxeo - initial API and implementation
 *
 * $Id$
 */

package org.nuxeo.runtime.deployment.preprocessor.install.commands;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;

import org.nuxeo.common.utils.FileNamePattern;
import org.nuxeo.common.utils.FileUtils;
import org.nuxeo.common.utils.Path;
import org.nuxeo.runtime.deployment.preprocessor.install.Command;
import org.nuxeo.runtime.deployment.preprocessor.install.CommandContext;

/**
 * @author <a href="mailto:bs@nuxeo.com">Bogdan Stefanescu</a>
 */
public class AppendCommand implements Command {

    protected final Path src;

    protected final Path dst;

    protected final boolean addNewLine;

    protected final FileNamePattern pattern;

    public AppendCommand(Path src, Path dst, boolean addNewLine, FileNamePattern pattern) {
        this.src = src;
        this.dst = dst;
        this.addNewLine = addNewLine;
        this.pattern = pattern;
    }

    public AppendCommand(Path src, Path dst) {
        this(src, dst, true, null);
    }

    @Override
    public void exec(CommandContext ctx) throws IOException {
        File baseDir = ctx.getBaseDir();
        File srcFile = new File(baseDir, ctx.expandVars(src.toString()));
        File dstFile = new File(baseDir, ctx.expandVars(dst.toString()));

        if (pattern == null && !srcFile.exists()) {
            throw new FileNotFoundException("Could not find the file " + srcFile.getAbsolutePath() + " to append.");
        }

        if (!dstFile.isFile()) {
            try {
                dstFile.createNewFile();
            } catch (IOException e) {
                throw new IOException("Could not create " + dstFile, e);
            }
        }
        if (pattern == null) {
            FileUtils.append(srcFile, dstFile, addNewLine);
        } else {
            ArrayList<File> files = new ArrayList<File>();
            FileUtils.collectFiles(srcFile, pattern, files);
            for (File file : files) {
                FileUtils.append(file, dstFile);
            }
        }
    }

    @Override
    public String toString() {
        return "append " + src.toString() + " > " + dst.toString();
    }

    @Override
    public String toString(CommandContext ctx) {
        return "append " + ctx.expandVars(src.toString()) + " > " + ctx.expandVars(dst.toString());
    }

}
