/*
 * (C) Copyright 2015 Nuxeo SA (http://nuxeo.com/) and others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 *     Antoine Taillefer <ataillefer@nuxeo.com>
 */

package org.nuxeo.drive.listener;

import java.util.Collections;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuxeo.drive.service.NuxeoDriveEvents;
import org.nuxeo.ecm.core.event.Event;
import org.nuxeo.ecm.core.event.EventBundle;
import org.nuxeo.ecm.core.event.EventContext;
import org.nuxeo.ecm.core.event.PostCommitFilteringEventListener;
import org.nuxeo.ecm.platform.audit.api.AuditLogger;
import org.nuxeo.ecm.platform.audit.api.LogEntry;
import org.nuxeo.runtime.api.Framework;

/**
 * Post-commit asynchronous listener that pushes the virtual events generated by
 * {@link NuxeoDriveFileSystemDeletionListener} to the Audit log.
 *
 * @since 7.4
 */
public class NuxeoDriveVirtualEventLogger implements PostCommitFilteringEventListener {

    private static final Log log = LogFactory.getLog(NuxeoDriveVirtualEventLogger.class);

    @Override
    public boolean acceptEvent(Event event) {
        return NuxeoDriveEvents.VIRTUAL_EVENT_CREATED.equals(event.getName());
    }

    @Override
    public void handleEvent(EventBundle events) {
        AuditLogger logger = Framework.getService(AuditLogger.class);
        if (logger != null) {
            for (Event event : events) {
                EventContext ctx = event.getContext();
                Object[] args = ctx.getArguments();
                if (ArrayUtils.isEmpty(args)) {
                    return;
                }
                Object arg = args[0];
                if (!(arg instanceof LogEntry)) {
                    return;
                }
                LogEntry entry = (LogEntry) arg;
                logger.addLogEntries(Collections.singletonList(entry));
            }
        } else {
            log.error("Can not reach AuditLogger");
        }
    }

}
