/*
 * (C) Copyright 2014 Nuxeo SA (http://nuxeo.com/) and others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 *     Vladimir Pasquier <vpasquier@nuxeo.com>
 */

package org.nuxeo.binary.metadata.internals.listeners;

import static org.nuxeo.ecm.core.api.event.DocumentEventTypes.ABOUT_TO_CREATE;
import static org.nuxeo.ecm.core.api.event.DocumentEventTypes.BEFORE_DOC_UPDATE;

import org.nuxeo.binary.metadata.api.BinaryMetadataConstants;
import org.nuxeo.binary.metadata.api.BinaryMetadataService;
import org.nuxeo.ecm.core.api.DocumentModel;
import org.nuxeo.ecm.core.event.Event;
import org.nuxeo.ecm.core.event.EventContext;
import org.nuxeo.ecm.core.event.EventListener;
import org.nuxeo.ecm.core.event.impl.DocumentEventContext;
import org.nuxeo.runtime.api.Framework;

/**
 * Handle document and blob updates according to following rules in an event context: - Define if rule should be
 * executed in async or sync mode. - If Blob dirty and document metadata dirty, write metadata from doc to Blob. - If
 * Blob dirty and document metadata not dirty, write metadata from Blob to doc. - If Blob not dirty and document
 * metadata dirty, write metadata from doc to Blob.
 *
 * @since 7.1
 */
public class BinaryMetadataSyncListener implements EventListener {

    @Override
    public void handleEvent(Event event) {
        EventContext ctx = event.getContext();
        if (!(ctx instanceof DocumentEventContext)) {
            return;
        }
        BinaryMetadataService binaryMetadataService = Framework.getLocalService(BinaryMetadataService.class);
        DocumentEventContext docCtx = (DocumentEventContext) ctx;
        DocumentModel doc = docCtx.getSourceDocument();
        Boolean disable = (Boolean) event.getContext().getProperty(
                BinaryMetadataConstants.DISABLE_BINARY_METADATA_LISTENER);
        if (ABOUT_TO_CREATE.equals(event.getName()) && !doc.isProxy() && disable == null) {
            binaryMetadataService.writeMetadata(doc, docCtx.getCoreSession());
        } else if (BEFORE_DOC_UPDATE.equals(event.getName()) && !doc.isProxy() && disable == null) {
            doc.putContextData(BinaryMetadataConstants.DISABLE_BINARY_METADATA_LISTENER, Boolean.TRUE);
            binaryMetadataService.handleSyncUpdate(doc, docCtx);
        }
    }

}
