/*
 * (C) Copyright 2006-2010 Nuxeo SAS (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     bstefanescu
 */
package org.nuxeo.gwt.habyt.upload.client;

import java.io.Serializable;

import com.google.gwt.user.client.Random;

/**
 * @author <a href="mailto:bs@nuxeo.com">Bogdan Stefanescu</a>
 * 
 */
@SuppressWarnings("serial")
public class FileRef implements Serializable {

    /**
     * A new file was queued for upload
     */
    public final static int QUEUED = 0;

    /**
     * A file is uploading
     */
    public final static int UPLOADING = 1;

    /**
     * The file was uploaded
     */
    public final static int UPLOADED = 2;

    /**
     * A stored file was marked as removed (will be when submitting changes).
     */
    public final static int REMOVED = 3;

    /**
     * The file upload failed
     */
    public final static int ERROR = 4;

    /**
     * The file is already stored on the server
     */
    public final static int STORED = 5;

    /**
     * A number between 0 - 100 that gives the upload status in percent.
     */
    protected int progress;

    protected int status;

    protected String name;

    protected String id;

    protected String ctype;

    protected String length;

    public FileRef() {
        this(Uploader.UPLOADED_FILE_ID_PREFIX
                + Integer.toHexString(Random.nextInt()), null);
    }

    public FileRef(String name) {
        this(Uploader.UPLOADED_FILE_ID_PREFIX
                + Integer.toHexString(Random.nextInt()), name);
    }

    public FileRef(String id, String name) {
        this.id = id;
        this.name = name;
        this.status = STORED;
    }

    public String getId() {
        return id;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public int getStatus() {
        return status;
    }

    public void setStatus(int status) {
        this.status = status;
    }

    public int getProgress() {
        return progress;
    }

    public void setProgress(int progress) {
        this.progress = progress;
        if (progress == 100 && status == UPLOADING) {
            this.status = UPLOADED;
        }
    }

    public void setContentType(String ctype) {
        this.ctype = ctype;
    }

    public String getContentType() {
        return ctype;
    }

    public String getLength() {
        return length;
    }

    public String getDescription() {
        String desc = "";
        if (ctype != null && ctype.length() > 0) {
            desc = "(" + ctype + ") ";
        }
        desc += getPrettyLength();
        return desc;
    }

    public String getPrettyLength() {
        if (length == null) {
            return "";
        }
        try {
            int len = Integer.parseInt(length);
            return formatLength(len);
        } catch (Exception e) {
            return "";
        }
    }

    public String formatLength(int len) {
        int k = len / 1024;
        if (k == 0) {
            return length + " B";
        } else if (k < 1024) {
            return k + " K";
        } else {
            return (k / 1024) + " M";
        }
    }

    public void setLength(String length) {
        this.length = length;
    }

    @Override
    public int hashCode() {
        return getId().hashCode();
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (obj instanceof FileRef) {
            return id.equals(((FileRef) obj).id);
        }
        return super.equals(obj);
    }

    @Override
    public String toString() {
        return id + ":" + status;
    }
}
