/*
 * (C) Copyright 2014 Nuxeo SA (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-2.1.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     nuxeo.io Team
 */

package org.nuxeo.io.listener;

import static org.nuxeo.ecm.core.api.event.DocumentEventTypes.ABOUT_TO_CREATE;
import static org.nuxeo.io.Constants.IO_ENVIRONMENT_DOCUMENT_TYPE;

import java.io.Serializable;
import java.util.Map;
import java.util.regex.Pattern;

import org.nuxeo.ecm.core.api.ClientException;
import org.nuxeo.ecm.core.api.CoreSession;
import org.nuxeo.ecm.core.api.DocumentModel;
import org.nuxeo.ecm.core.api.DocumentRef;
import org.nuxeo.ecm.core.api.IterableQueryResult;
import org.nuxeo.ecm.core.api.event.CoreEventConstants;
import org.nuxeo.ecm.core.event.Event;
import org.nuxeo.ecm.core.event.EventContext;
import org.nuxeo.ecm.core.event.EventListener;
import org.nuxeo.ecm.core.event.impl.DocumentEventContext;
import org.nuxeo.ecm.core.query.sql.NXQL;
import org.nuxeo.io.Constants;
import org.nuxeo.io.NameValidationException;

/**
 * Listener validating the name of the nuxeo.io environment and application
 * documents.
 *
 * @since 1.0
 */
public class EnvironmentNameValidationListener implements EventListener {

    public static Pattern REGEXP = Pattern.compile("^[a-z][a-z0-9\\-]*$");

    @Override
    public void handleEvent(Event event) throws ClientException {
        if (!ABOUT_TO_CREATE.equals(event.getName())) {
            return;
        }

        EventContext ctx = event.getContext();
        if (ctx instanceof DocumentEventContext) {
            DocumentEventContext docCtx = (DocumentEventContext) ctx;
            DocumentModel doc = docCtx.getSourceDocument();
            if (IO_ENVIRONMENT_DOCUMENT_TYPE.equals(doc.getType())) {
                String title = doc.getTitle();
                // check the name itself
                if (!REGEXP.matcher(title).matches()) {
                    event.markBubbleException();
                    String message = String.format(
                            "Invalid name '%s'. Name must start with a letter and can only contain lowercase letters, numbers, and dashes.",
                            title);
                    throw new NameValidationException(message);
                }

                // check that there is no environment with the same name
                IterableQueryResult result = null;
                try {
                    CoreSession session = docCtx.getCoreSession();
                    DocumentModel parent = session.getDocument((DocumentRef) docCtx.getProperty(CoreEventConstants.DESTINATION_REF));
                    String query = String.format(
                            "SELECT COUNT(ecm:uuid) From Document "
                                    + "WHERE ecm:primaryType = '%s' "
                                    + "AND dc:title = '%s' AND ecm:isCheckedInVersion = 0 "
                                    + "AND ecm:parentId = '%s' "
                                    + "AND ecm:isProxy = 0",
                            Constants.IO_ENVIRONMENT_DOCUMENT_TYPE, title,
                            parent.getId());
                    result = session.queryAndFetch(query, NXQL.NXQL);
                    if (result.size() > 0) {
                        Map<String, Serializable> r = result.iterator().next();
                        long count = (long) r.get("COUNT(ecm:uuid)");
                        if (count > 0) {
                            event.markBubbleException();
                            String message = String.format(
                                    "The name '%s' is already taken.", title);
                            throw new NameValidationException(message);
                        }
                    }
                } finally {
                    if (result != null) {
                        result.close();
                    }
                }
            }
        }
    }

}
