/*
 * (C) Copyright 2014 Nuxeo SA (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-2.1.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     nuxeo.io Team
 */
package org.nuxeo.io.adapter;

import static org.nuxeo.io.Constants.IO_ENVIRONMENT_APPLICATION_ID_PROPERTY;
import static org.nuxeo.io.Constants.IO_ENVIRONMENT_APPLICATION_PROPERTY;
import static org.nuxeo.io.Constants.IO_ENVIRONMENT_DESCRIPTION_PROPERTY;
import static org.nuxeo.io.Constants.IO_ENVIRONMENT_DOMAIN_PROPERTY;
import static org.nuxeo.io.Constants.IO_ENVIRONMENT_TECH_ID_PROPERTY;
import static org.nuxeo.io.Constants.IO_ENVIRONMENT_TYPE_PROPERTY;

import java.util.Calendar;

import org.codehaus.jackson.annotate.JsonIgnore;
import org.codehaus.jackson.annotate.JsonIgnoreProperties;
import org.nuxeo.connect.connector.NuxeoClientInstanceType;
import org.nuxeo.connect.data.ConnectProject;
import org.nuxeo.connect.identity.LogicalInstanceIdentifier;
import org.nuxeo.ecm.core.api.ClientException;
import org.nuxeo.ecm.core.api.ClientRuntimeException;
import org.nuxeo.ecm.core.api.DocumentModel;
import org.nuxeo.io.connect.InstanceRegistrator;
import org.nuxeo.io.connect.TargetPlatformInformation;
import org.nuxeo.io.service.IoEtcdService;
import org.nuxeo.io.service.IoEtcdServiceImpl;
import org.nuxeo.io.service.IoServiceImpl;
import org.nuxeo.io.service.Statuses;
import org.nuxeo.runtime.api.Framework;

/**
 * Adapter for ioEnvironment document type
 *
 * @since 1.0
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public class IoEnvironment extends AbstractAdapter {

    public IoEnvironment() {
        super();
    }

    public IoEnvironment(DocumentModel doc) {
        super(doc);
    }

    public String getName() {
        try {
            return doc.getTitle();
        } catch (ClientException e) {
            throw new ClientRuntimeException(e);
        }
    }

    public void setName(String name) {
        setPropertyValue("dc:title", name);
    }

    public String getPath() {
        return doc.getPathAsString();
    }

    public Calendar getCreationDate() {
        return getPropertyValue("dc:created", Calendar.class);
    }

    public Calendar getModificationDate() {
        return getPropertyValue("dc:modified", Calendar.class);
    }

    public String getLastContributor() {
        return getPropertyValue("dc:lastContributor", String.class);
    }

    public String getVersion() {
        return doc.getVersionLabel();
    }

    public String getDescription() {
        return getPropertyValue(IO_ENVIRONMENT_DESCRIPTION_PROPERTY, String.class);
    }

    public void setDescription(String description) {
        setPropertyValue(IO_ENVIRONMENT_DESCRIPTION_PROPERTY, description);
    }

    public String getTechId() {
        return getPropertyValue(IO_ENVIRONMENT_TECH_ID_PROPERTY, String.class);
    }

    public String getEnvironmentType() {
        return getPropertyValue(IO_ENVIRONMENT_TYPE_PROPERTY, String.class);
    }

    public void setEnvironmentType(String type) {
        setPropertyValue(IO_ENVIRONMENT_TYPE_PROPERTY, type);
    }

    public String getDomain() {
        return getPropertyValue(IO_ENVIRONMENT_DOMAIN_PROPERTY, String.class);
    }

    public void setDomain(String domain) {
        setPropertyValue(IO_ENVIRONMENT_DOMAIN_PROPERTY, domain);
    }

    public String getApplication() {
        return getPropertyValue(IO_ENVIRONMENT_APPLICATION_PROPERTY, String.class);
    }

    public String getApplicationId() {
        return getPropertyValue(IO_ENVIRONMENT_APPLICATION_ID_PROPERTY, String.class);
    }

    public void setProject(ConnectProject project) {
        if (project == null) {
            setApplicationId(null);
            setApplication(null);
        } else {
            setApplication(project.getSymbolicName());
            setApplicationId(project.getUuid());
        }
    }

    public void setApplication(String application) {
        setPropertyValue(IO_ENVIRONMENT_APPLICATION_PROPERTY, application);
    }

    public void setApplicationId(String applicationId) {
        setPropertyValue(IO_ENVIRONMENT_APPLICATION_ID_PROPERTY, applicationId);
    }

    @JsonIgnore
    public void setCurrentPlatform(TargetPlatformInformation targetPlatform) {
        IoEtcdService etcdService = Framework.getLocalService(IoEtcdService.class);
        String key = IoEtcdServiceImpl.getPlatformTargetKey(this);
        etcdService.setValue(key, targetPlatform.getVersion());
    }

    @JsonIgnore
    public void setExpectedStatus(Statuses status) {
        IoEtcdService etcdService = Framework.getLocalService(IoEtcdService.class);
        String expectedStatusKey = IoEtcdServiceImpl.getExpectedStatusKeyFor(this, IoServiceImpl._1);
        etcdService.setValue(expectedStatusKey, status.toString());
    }

    @JsonIgnore
    public void setCurrentStatus(Statuses status, int ttl) {
        IoEtcdService etcdService = Framework.getLocalService(IoEtcdService.class);
        String currentStatusKey = IoEtcdServiceImpl.getCurrentStatusKeyFor(this, IoServiceImpl._1);
        etcdService.setValue(currentStatusKey, status.toString(), ttl);
    }

    @JsonIgnore
    public String getClid() {
        return Framework.getLocalService(IoEtcdService.class).getClid(this);
    }

    @JsonIgnore
    public void setClid(String clid) {
        IoEtcdService etcdService = Framework.getLocalService(IoEtcdService.class);
        String clidsKey = IoEtcdServiceImpl.getClidKey(this, IoServiceImpl._1);
        etcdService.setValue(clidsKey, clid);
    }

    @JsonIgnore
    public void updateClid() {
        setClid(requestNewClid());
    }

    @JsonIgnore
    protected String requestNewClid() {
        LogicalInstanceIdentifier CLIDs = new InstanceRegistrator(getApplicationId(), getName(),
                NuxeoClientInstanceType.DEV).getCLIDs();
        if (CLIDs != null) {
            return String.format("%s\\n%s\\n%s", CLIDs.getCLID1(), CLIDs.getCLID2(), CLIDs.getInstanceDescription());
        }
        return null;
    }
}
