/*
 * (C) Copyright 2014 Nuxeo SA (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-2.1.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     nuxeo.io Team
 */

package org.nuxeo.io.adapter;

import static org.nuxeo.io.Constants.BLOCK_DOMAIN_SUFFIX_KEY;
import static org.nuxeo.io.Constants.CONNECT_URL_KEY;
import static org.nuxeo.io.Constants.DEFAULT_DOMAIN_SUFFIX_KEY;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.nuxeo.connect.data.ConnectProject;
import org.nuxeo.ecm.core.api.NuxeoPrincipal;
import org.nuxeo.runtime.api.Framework;

/**
 * Object representing an user of nuxeo.io.
 * <p>
 * Contains:
 * <ul>
 * <li>the needed user information</li>
 * <li>the list of clients this user has access to</li>
 * </ul>
 *
 * @since 1.0
 */
public class IoUser {

    protected String username;

    protected String firstName;

    protected String lastName;

    protected String company;

    protected String email;

    protected boolean isAdministrator;

    protected List<IoClient> clients;

    protected List<ConnectProject> projects;

    public IoUser(NuxeoPrincipal principal, List<IoClient> clients,
            List<ConnectProject> projects) {
        username = principal.getName();
        firstName = principal.getFirstName();
        lastName = principal.getLastName();
        company = principal.getCompany();
        email = principal.getEmail();
        isAdministrator = principal.isAdministrator();

        this.clients = new ArrayList<>();
        this.clients.addAll(clients);

        Collections.sort(projects, new Comparator<ConnectProject>() {
            @Override
            public int compare(ConnectProject o1, ConnectProject o2) {
                return o1.getName().toLowerCase().compareTo(
                        o2.getName().toLowerCase());
            }
        });
        this.projects = new ArrayList<>();
        this.projects.addAll(projects);
    }

    public String getUsername() {
        return username;
    }

    public String getFirstName() {
        return firstName;
    }

    public String getLastName() {
        return lastName;
    }

    public String getCompany() {
        return company;
    }

    public String getEmail() {
        return email;
    }

    public boolean isAdministrator() {
        return isAdministrator;
    }

    public List<IoClient> getClients() {
        return clients;
    }

    public List<ConnectProject> getProjects() {
        return projects;
    }

    public String getDefaultDomainSuffix() {
        return Framework.getProperty(DEFAULT_DOMAIN_SUFFIX_KEY);
    }

    public boolean isBlockDomainSuffix() {
        return Framework.isBooleanPropertyTrue(BLOCK_DOMAIN_SUFFIX_KEY);
    }

    public String getConnectURL() {
        return Framework.getProperty(CONNECT_URL_KEY);
    }
}
