/*
 * (C) Copyright 2014 Nuxeo SA (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-2.1.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     nuxeo.io Team
 */

package org.nuxeo.io.listener;

import static org.nuxeo.ecm.core.api.event.DocumentEventTypes.ABOUT_TO_CREATE;
import static org.nuxeo.ecm.core.api.event.DocumentEventTypes.ABOUT_TO_REMOVE;
import static org.nuxeo.io.Constants.IO_CLIENT_DOCUMENT_TYPE;

import org.nuxeo.common.utils.IdUtils;
import org.nuxeo.ecm.core.api.ClientException;
import org.nuxeo.ecm.core.api.CoreSession;
import org.nuxeo.ecm.core.api.DocumentModel;
import org.nuxeo.ecm.core.api.IterableQueryResult;
import org.nuxeo.ecm.core.api.event.CoreEventConstants;
import org.nuxeo.ecm.core.event.EventContext;
import org.nuxeo.ecm.core.event.EventListener;
import org.nuxeo.ecm.core.event.impl.DocumentEventContext;
import org.nuxeo.ecm.core.query.sql.NXQL;
import org.nuxeo.ecm.platform.uidgen.UIDSequencer;
import org.nuxeo.io.Constants;
import org.nuxeo.io.NameValidationException;
import org.nuxeo.io.adapter.IoClient;
import org.nuxeo.io.adapter.IoEnvironment;
import org.nuxeo.io.service.IoService;
import org.nuxeo.runtime.api.Framework;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * Listener computing a prettier name for ioClient documents.
 *
 * @since 1.0
 */
public class IoClientListener implements EventListener {

    protected final List<String> EVENTS = Arrays.asList(ABOUT_TO_CREATE, ABOUT_TO_REMOVE);

    @Override
    public void handleEvent(org.nuxeo.ecm.core.event.Event event)
            throws ClientException {
        if (!EVENTS.contains(event.getName())) {
            return;
        }

        EventContext ctx = event.getContext();
        if (ctx instanceof DocumentEventContext) {
            DocumentEventContext docCtx = (DocumentEventContext) ctx;
            DocumentModel doc = docCtx.getSourceDocument();
            if (!IO_CLIENT_DOCUMENT_TYPE.equals(doc.getType())) {
                return;
            }

            switch (event.getName()) {
                case ABOUT_TO_CREATE:
                    handleAboutCreate(docCtx, doc);
                    break;
                case ABOUT_TO_REMOVE:
                    handleAboutRemove(docCtx, doc);
                    break;
            }

        }
    }

    protected void handleAboutRemove(DocumentEventContext docCtx, DocumentModel doc) {
        // Client Environement do not fire ABOUT_TO_REMOVE event cascade
        IoClient client = doc.getAdapter(IoClient.class);
        // @see org.nuxeo.io.listener.IoEnvironmentListener.handleEvent()
        for (IoEnvironment env : client.getEnvironments()) {
            Framework.getLocalService(IoService.class).deleteEnvironment(env);
        }
    }

    protected void handleAboutCreate(DocumentEventContext docCtx, DocumentModel doc) {
        String title = doc.getTitle();
        String name = IdUtils.generateId(title, "-", true, 24);

        // check if there is already a client with the same name
        IterableQueryResult result = null;
        try {
            String query = String.format(
                    "SELECT COUNT(ecm:uuid) From Document "
                            + "WHERE ecm:primaryType = '%s' "
                            + "AND ecm:name = '%s' AND ecm:isCheckedInVersion = 0 "
                            + "AND ecm:isProxy = 0",
                    IO_CLIENT_DOCUMENT_TYPE, name);
            result = docCtx.getCoreSession().queryAndFetch(query,
                    NXQL.NXQL);
            if (result.size() > 0) {
                Map<String, Serializable> r = result.iterator().next();
                long count = (long) r.get("COUNT(ecm:uuid)");
                if (count > 0) {
                    // increment the client name
                    UIDSequencer uidSequencer = Framework.getLocalService(UIDSequencer.class);
                    int id = uidSequencer.getNext(name);
                    name += ("-" + id);
                }
            }
        } finally {
            if (result != null) {
                result.close();
            }
        }

        docCtx.setProperty(CoreEventConstants.DESTINATION_NAME, name);
    }

}
