/*
 * (C) Copyright 2014 Nuxeo SA (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-2.1.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     nuxeo.io Team
 */

package org.nuxeo.io.listener;

import static org.apache.commons.lang.StringUtils.isBlank;
import static org.nuxeo.ecm.core.api.event.DocumentEventTypes.ABOUT_TO_REMOVE;
import static org.nuxeo.ecm.core.api.event.DocumentEventTypes.BEFORE_DOC_UPDATE;
import static org.nuxeo.ecm.core.api.event.DocumentEventTypes.DOCUMENT_CREATED;
import static org.nuxeo.io.Constants.IO_ENVIRONMENT_APPLICATION_ID_PROPERTY;
import static org.nuxeo.io.Constants.IO_ENVIRONMENT_DOCUMENT_TYPE;
import static org.nuxeo.io.Constants.IO_ENVIRONMENT_DOMAIN_PROPERTY;
import static org.nuxeo.io.Constants.IO_ENVIRONMENT_TECH_ID_PROPERTY;

import java.io.Serializable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuxeo.ecm.core.api.ClientException;
import org.nuxeo.ecm.core.api.CoreSession;
import org.nuxeo.ecm.core.api.DocumentModel;
import org.nuxeo.ecm.core.api.model.Property;
import org.nuxeo.ecm.core.event.Event;
import org.nuxeo.ecm.core.event.EventContext;
import org.nuxeo.ecm.core.event.EventListener;
import org.nuxeo.ecm.core.event.impl.DocumentEventContext;
import org.nuxeo.ecm.platform.uidgen.UIDSequencer;
import org.nuxeo.io.adapter.IoEnvironment;
import org.nuxeo.io.service.IoService;
import org.nuxeo.runtime.api.Framework;

/**
 * Listener handling IoEnvironment creation, modification and deletion.
 *
 * @since 1.0
 */
public class IoEnvironmentListener implements EventListener {

    public static final String ENVIRONMENT_TECH_ID_KEY = "environmentTechId";

    public static final String ENV_TECH_ID_FORMAT = "nxio_%06d";

    @Override
    public void handleEvent(Event event) throws ClientException {
        String eventName = event.getName();

        EventContext ctx = event.getContext();
        if (ctx instanceof DocumentEventContext) {
            DocumentEventContext docCtx = (DocumentEventContext) ctx;
            DocumentModel doc = docCtx.getSourceDocument();
            if (IO_ENVIRONMENT_DOCUMENT_TYPE.equals(doc.getType())) {
                CoreSession session = docCtx.getCoreSession();
                IoService ioService = Framework.getLocalService(IoService.class);
                if (DOCUMENT_CREATED.equals(eventName)) {
                    // compute new environment tech id
                    UIDSequencer uidSequencer = Framework.getLocalService(UIDSequencer.class);
                    int id = uidSequencer.getNext(ENVIRONMENT_TECH_ID_KEY);
                    String techId = String.format(ENV_TECH_ID_FORMAT, id);
                    doc.setPropertyValue(IO_ENVIRONMENT_TECH_ID_PROPERTY, techId);

                    ioService.createEnvironment(doc.getAdapter(IoEnvironment.class));
                } else if (BEFORE_DOC_UPDATE.equals(eventName)) {
                    Property property = doc.getProperty(IO_ENVIRONMENT_DOMAIN_PROPERTY);

                    // XXX test about appId : Must be removed for manager 0.6 release
                    String appId = (String) doc.getPropertyValue(IO_ENVIRONMENT_APPLICATION_ID_PROPERTY);
                    if (property.isDirty() || isBlank(appId)) {
                        // get the old env (with old domain)
                        DocumentModel oldEnvironment = session.getDocument(doc.getRef());
                        ioService.updateEnvironment(oldEnvironment.getAdapter(IoEnvironment.class),
                                doc.getAdapter(IoEnvironment.class));
                    }
                } else if (ABOUT_TO_REMOVE.equals(eventName)) {
                    // @see org.nuxeo.io.listener.IoClientListener.handleAboutRemove()
                    ioService.deleteEnvironment(doc.getAdapter(IoEnvironment.class));
                }
            }
        }
    }

}
