package org.nuxeo.io.operation;

import static org.apache.commons.lang.StringUtils.isBlank;
import static org.apache.commons.lang.StringUtils.isNotBlank;
import static org.nuxeo.ecm.core.api.security.ACL.LOCAL_ACL;
import static org.nuxeo.ecm.core.api.security.SecurityConstants.READ_WRITE;
import static org.nuxeo.io.Constants.IO_CLIENT_DOCUMENT_TYPE;
import static org.nuxeo.io.Constants.IO_ENVIRONMENT_DOCUMENT_TYPE;
import static org.nuxeo.io.Constants.NUXEO_IO_CLIENTS_PATH;
import static org.nuxeo.io.Constants.NUXEO_IO_ENVIRONMENTS_PATH;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuxeo.ecm.automation.core.annotations.Context;
import org.nuxeo.ecm.automation.core.annotations.Operation;
import org.nuxeo.ecm.automation.core.annotations.OperationMethod;
import org.nuxeo.ecm.automation.core.annotations.Param;
import org.nuxeo.ecm.core.api.ClientException;
import org.nuxeo.ecm.core.api.CoreSession;
import org.nuxeo.ecm.core.api.DocumentModel;
import org.nuxeo.ecm.core.api.DocumentRef;
import org.nuxeo.ecm.core.api.IdRef;
import org.nuxeo.ecm.core.api.UnrestrictedSessionRunner;
import org.nuxeo.ecm.core.api.security.ACE;
import org.nuxeo.ecm.core.api.security.ACP;
import org.nuxeo.ecm.platform.usermanager.UserManager;
import org.nuxeo.io.NameValidationException;
import org.nuxeo.io.listener.EnvironmentNameValidationListener;
import org.nuxeo.io.service.IoService;
import org.nuxeo.runtime.transaction.TransactionHelper;

/**
 * Operation that allow to create a new Client for a dedicated user, with or
 * without an associated environment.
 *
 * @author <a href="mailto:ak@nuxeo.com">Arnaud Kervern</a>
 * @since 1.0
 */
@Operation(id = CreateIoClient.ID, category = "Io", label = "Create an Io Client", description = "Create a new Client for a dedicated user.")
public class CreateIoClient {
    public static final String ID = "Io.CreateClient";

    public static final String ENV_REGEX = "[^a-z0-9\\-]";

    public static final Log log = LogFactory.getLog(CreateIoClient.class);

    @Context
    protected CoreSession session;

    @Context
    protected IoService ioService;

    @Context
    protected UserManager usrMg;

    @Param(name = "clientName")
    protected String clientName;

    @Param(name = "username")
    protected String username;

    @Param(name = "email", required = false)
    protected String email;

    @Param(name = "projectName", required = false)
    protected String projectName;

    @Param(name = "createEnvironment", required = false)
    protected boolean createEnvironment = false;

    @OperationMethod
    public DocumentModel run() throws ClientException {
        final DocumentRef[] clientRef = { null };

        createUserIfNeeded();

        final String repositoryName = session.getRepositoryName();
        new UnrestrictedSessionRunner(repositoryName, username) {
            @Override
            public void run() throws ClientException {
                clientRef[0] = createClient(session).getRef();
            }
        }.runUnrestricted();

        // Env creation is handled in a separate SessionRunner with a new transaction
        // to Ensure that the client is correctly created and some name validation
        // error bubbled do not rollback the whole client registration
        TransactionHelper.commitOrRollbackTransaction();
        TransactionHelper.startTransaction();

        final DocumentModel client = session.getDocument(clientRef[0]);
        if (createEnvironment) {
            new UnrestrictedSessionRunner(repositoryName, username) {
                @Override
                public void run() throws ClientException {
                    createEnvironment(session, client.getName());
                }
            }.runUnrestricted();
        }

        return client;
    }

    protected void createUserIfNeeded() throws ClientException {
        DocumentModel user = usrMg.getUserModel(username);
        if (user == null) {
            user = usrMg.getBareUserModel();

            user.setPropertyValue(usrMg.getUserIdField(), username);
            user.setPropertyValue("user:company", clientName);
            user.setPropertyValue("user:email", email);

            user = usrMg.createUser(user);
            assert isNotBlank(user.getId());
        }
    }

    protected DocumentModel createClient(CoreSession session)
            throws ClientException {
        DocumentModel doc = session.createDocumentModel(NUXEO_IO_CLIENTS_PATH,
                clientName, IO_CLIENT_DOCUMENT_TYPE);
        doc.setPropertyValue("dc:title", clientName);
        doc = session.createDocument(doc);

        ACE ace = new ACE(username, READ_WRITE, true);
        ACP acp = doc.getACP();

        acp.getOrCreateACL(LOCAL_ACL).add(ace);
        session.setACP(doc.getRef(), acp, true);

        return doc;
    }

    protected void createEnvironment(CoreSession session, String clientName) {
        String envName = projectName;
        if (isBlank(envName)) {
            envName = username;
        }
        envName = envName.toLowerCase().replaceAll(ENV_REGEX, "");

        try {
            DocumentModel environment = session.createDocumentModel(
                    String.format(NUXEO_IO_ENVIRONMENTS_PATH, clientName), envName, IO_ENVIRONMENT_DOCUMENT_TYPE);
            environment.setPropertyValue("dc:title", envName);

            session.createDocument(environment);
        } catch (RuntimeException e) {
            if (e instanceof NameValidationException) {
                log.trace(String.format("Is transaction rollback: %s", TransactionHelper.isTransactionMarkedRollback()));
            } else {
                throw e;
            }
        }
    }
}
