/*
 * (C) Copyright 2014 Nuxeo SA (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     Nuxeo
 */

package org.nuxeo.io.service;

import static org.nuxeo.io.service.IoServiceImpl._1;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuxeo.etcd.EtcdService;
import org.nuxeo.io.adapter.IoEnvironment;
import org.nuxeo.runtime.api.Framework;

/**
 * @author <a href="mailto:ak@nuxeo.com">Arnaud Kervern</a>
 * @since 0.6
 */
public class IoEtcdServiceImpl implements IoEtcdService {

    public static final String SERVICE_KEY_PATTERN = "/services/%s";

    public static final String SERVICE_CONFIG_KEY_PATTERN = SERVICE_KEY_PATTERN + "/%d/config";

    public static final String SERVICE_CONFIG_CLIDS = SERVICE_CONFIG_KEY_PATTERN + "/instance.clid";

    public static final String SERVICE_GLOBAL_CONFIG_KEY_PATTERN = SERVICE_KEY_PATTERN + "/config";

    public static final String SERVICE_STATUS_KEY_PATTERN = SERVICE_KEY_PATTERN + "/%d/status";

    public static final String SERVICE_CURRENT_STATUS_KEY_PATTERN = SERVICE_STATUS_KEY_PATTERN + "/current";

    public static final String SERVICE_EXPECTED_STATUS_KEY_PATTERN = SERVICE_STATUS_KEY_PATTERN + "/expected";

    public static final String SERVICE_ALIVE_STATUS_KEY_PATTERN = SERVICE_STATUS_KEY_PATTERN + "/alive";

    public static final String SERVICE_DOMAIN_KEY_PATTERN = SERVICE_KEY_PATTERN + "/%d/domain";

    public static final String SERVICE_TARGET_PLATFORM_KEY_PATTERN = SERVICE_GLOBAL_CONFIG_KEY_PATTERN + "/platform";

    public static final String DOMAIN_KEY_PATTERN = "/domains/%s";

    public static final String DOMAIN_VALUE_KEY_PATTERN = DOMAIN_KEY_PATTERN + "/value";

    public static final String DOMAIN_TYPE_KEY_PATTERN = DOMAIN_KEY_PATTERN + "/type";

    public static final String IO_CONTAINER_TYPE = "service";

    private static final Log log = LogFactory.getLog(IoEtcdService.class);

    public static String getCurrentStatusKeyFor(IoEnvironment env, int index) {
        return String.format(SERVICE_CURRENT_STATUS_KEY_PATTERN, env.getTechId(), index);
    }

    public static String getExpectedStatusKeyFor(IoEnvironment env, int index) {
        return String.format(SERVICE_EXPECTED_STATUS_KEY_PATTERN, env.getTechId(), index);
    }

    public static String getAliveStatusKeyFor(IoEnvironment env, int index) {
        return String.format(SERVICE_ALIVE_STATUS_KEY_PATTERN, env.getTechId(), index);
    }

    public static String getNXDomainKeyFor(IoEnvironment env, int index) {
        return String.format(SERVICE_DOMAIN_KEY_PATTERN, env.getTechId(), index);
    }

    public static String getPlatformTargetKey(IoEnvironment env) {
        return String.format(SERVICE_TARGET_PLATFORM_KEY_PATTERN, env.getTechId());
    }

    public static String getClidKey(IoEnvironment env, int index) {
        return String.format(IoEtcdServiceImpl.SERVICE_CONFIG_CLIDS, env.getTechId(), index);
    }

    @Override
    public void createEnvironment(IoEnvironment environment) {
        String domain = environment.getDomain();
        String techId = environment.getTechId();
        String typeKey = String.format(IoEtcdServiceImpl.DOMAIN_TYPE_KEY_PATTERN, domain);
        String valueKey = String.format(IoEtcdServiceImpl.DOMAIN_VALUE_KEY_PATTERN, domain);
        String domainKey = String.format(IoEtcdServiceImpl.SERVICE_DOMAIN_KEY_PATTERN, techId, _1);

        setValue(typeKey, IO_CONTAINER_TYPE);
        setValue(valueKey, techId);
        setValue(domainKey, domain);

        environment.updateClid();

        environment.setCurrentStatus(Statuses.STOPPED, -1);
        environment.setExpectedStatus(Statuses.STOPPED);
    }

    @Override
    public String getEnvironmentStatus(IoEnvironment environment) {
        EtcdService etcd = Framework.getLocalService(EtcdService.class);
        String currentStatus = etcd.getValue(IoEtcdServiceImpl.getCurrentStatusKeyFor(environment, _1));
        String expectedStatus = etcd.getValue(IoEtcdServiceImpl.getExpectedStatusKeyFor(environment, _1));
        String alive = etcd.getValue(IoEtcdServiceImpl.getAliveStatusKeyFor(environment, _1));

        return new EnvironmentStatusComputer(expectedStatus, currentStatus, alive).computeStatus();
    }

    @Override
    public void deleteDomain(IoEnvironment environment) {
        String domain = environment.getDomain();
        if (StringUtils.isNotBlank(domain)) {
            String key = String.format(IoEtcdServiceImpl.DOMAIN_KEY_PATTERN, domain);
            EtcdService etcd = Framework.getLocalService(EtcdService.class);
            etcd.delete(key, true);
        }
    }

    @Override
    public void setDomain(IoEnvironment environment) {
        String domain = environment.getDomain();
        String techId = environment.getTechId();

        String typeKey = String.format(IoEtcdServiceImpl.DOMAIN_TYPE_KEY_PATTERN, domain);
        String valueKey = String.format(IoEtcdServiceImpl.DOMAIN_VALUE_KEY_PATTERN, domain);
        setValue(typeKey, IoEtcdServiceImpl.IO_CONTAINER_TYPE);
        setValue(valueKey, techId);

        String domainKey = String.format(IoEtcdServiceImpl.SERVICE_DOMAIN_KEY_PATTERN, techId, _1);
        setValue(domainKey, domain);
    }

    @Override
    public String getClid(IoEnvironment environment) {
        EtcdService etcd = Framework.getLocalService(EtcdService.class);
        return etcd.getValue(getClidKey(environment, _1));
    }

    @Override
    public void setValue(String key, String value) {
        setValue(key, value, -1);
    }

    /**
     * Try to set the value several times
     */
    @Override
    public void setValue(String key, String value, int ttl) {
        EtcdService etcd = Framework.getLocalService(EtcdService.class);
        if (etcd.set(key, value, ttl) == null) {
            log.debug(String.format("SetValue failed (queued) %s %s", key, value));
        }
    }

    @Override
    public String getValue(String key) {
        return Framework.getLocalService(EtcdService.class).getValue(key);
    }
}
