/*
 * (C) Copyright 2014 Nuxeo SA (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-2.1.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     nuxeo.io Team
 */

package org.nuxeo.io.service;

import static org.apache.commons.lang.StringUtils.isBlank;
import static org.nuxeo.io.Constants.DOCKER_CONTAINER_NAME;
import static org.nuxeo.io.Constants.DOCKER_REGISTRY_RESOURCE;
import static org.nuxeo.io.connect.IoConnectClient.TARGET_PLATFORM_ENDPOINT;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.ws.rs.core.Response;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.codehaus.jackson.Version;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.map.module.SimpleModule;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.nuxeo.connect.connector.http.ConnectUrlConfig;
import org.nuxeo.connect.data.AbstractJSONSerializableData;
import org.nuxeo.connect.data.ConnectProject;
import org.nuxeo.connect.registration.RegistrationHelper;
import org.nuxeo.ecm.core.api.ClientException;
import org.nuxeo.ecm.platform.commandline.executor.api.CmdParameters;
import org.nuxeo.ecm.platform.commandline.executor.api.CommandLineExecutorService;
import org.nuxeo.ecm.platform.commandline.executor.api.CommandNotAvailable;
import org.nuxeo.io.adapter.IoEnvironment;
import org.nuxeo.io.adapter.IoEnvironmentStatus;
import org.nuxeo.io.connect.IoConnectClient;
import org.nuxeo.io.connect.TargetPlatformInformation;
import org.nuxeo.runtime.api.Framework;
import org.nuxeo.targetplatforms.api.TargetPackage;
import org.nuxeo.targetplatforms.api.TargetPlatformInstance;
import org.nuxeo.targetplatforms.api.impl.TargetPackageImpl;
import org.nuxeo.targetplatforms.api.impl.TargetPlatformInstanceImpl;

import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.UniformInterfaceException;
import com.sun.jersey.api.client.WebResource;

/**
 * @since 1.0
 */
public class IoServiceImpl implements IoService {

    public static final int _1 = 1;

    private static final Log log = LogFactory.getLog(IoServiceImpl.class);

    public static final String ETCD_ENDPOINT_PROPERTY_NAME = "org.nuxeo.io.etcd.endpoint";

    public static final String START_ENV_COMMAND = "startEnvironment";

    public static final String STOP_ENV_COMMAND = "stopEnvironment";

    public static final String START_CLEANER_COMMAND = "startCleaner";

    public static final String VERSION_REGEX = "^(?:(?:[^12345]{1}|\\d{2,})\\.(?:\\d+)(?:\\.(?:\\d+))?|5\\.9\\.5)$";

    protected static ObjectMapper objectMapper;

    @Override
    public void createEnvironment(IoEnvironment environment) {
        List<ConnectProject> projects = getAvailableConnectProjects();
        if (projects.size() > 0) {
            environment.setProject(projects.get(0));
        }

        Framework.getLocalService(IoEtcdService.class).createEnvironment(environment);
    }

    protected CmdParameters computeUnitNameCmdParameters(IoEnvironment environment) {
        String techId = environment.getTechId();
        String unitName = techId.substring(techId.indexOf("_") + 1);

        CmdParameters parameters = new CmdParameters();
        parameters.addNamedParameter("unitName", unitName);
        parameters.addNamedParameter("etcdEndpoint", Framework.getProperty(ETCD_ENDPOINT_PROPERTY_NAME));
        return parameters;
    }

    @Override
    public void startEnvironment(IoEnvironment environment) {
        // Ensure that we are trying to start a good environment
        TargetPlatformInformation tp = getTargetPlatform(environment.getApplication());
        if (!tp.getVersion().matches(VERSION_REGEX)) {
            throw new ClientException("Target platform version is lower than 5.9.5: " + tp.getVersion());
        }

        if (!checkPlatformAvailability(tp.getVersion())) {
            throw new ClientException("Target platform " + tp.getVersion() + " is not available.");
        }

        if (isBlank(environment.getClid())) {
            throw new ClientException(
                    "Something went wrong with your instance registration. Edit it to solve the problem.");
        }

        try {
            CommandLineExecutorService cles = Framework.getLocalService(CommandLineExecutorService.class);
            CmdParameters parameters = computeUnitNameCmdParameters(environment);
            // Stop env while starting; to ensure that Fleet unit is ready to
            // start
            cles.execCommand(STOP_ENV_COMMAND, parameters);

            environment.setCurrentPlatform(tp);
            environment.setExpectedStatus(Statuses.STARTED);
            // Set a temporary status as starting to prevent the short error state before the service is launched on a
            // node.
            environment.setCurrentStatus(Statuses.STARTING, 20);
            cles.execCommand(START_ENV_COMMAND, parameters);
        } catch (CommandNotAvailable e) {
            log.error(e, e);
        }
    }

    @Override
    public void deleteEnvironment(IoEnvironment environment) {
        CommandLineExecutorService cles = Framework.getLocalService(CommandLineExecutorService.class);
        CmdParameters parameters = computeUnitNameCmdParameters(environment);
        try {
            cles.execCommand(START_CLEANER_COMMAND, parameters);
        } catch (CommandNotAvailable e) {
            log.error(e, e);
        }
    }

    @Override
    public void stopEnvironment(IoEnvironment environment) {
        try {
            environment.setExpectedStatus(Statuses.STOPPED);

            CommandLineExecutorService cles = Framework.getLocalService(CommandLineExecutorService.class);
            CmdParameters parameters = computeUnitNameCmdParameters(environment);
            cles.execCommand(STOP_ENV_COMMAND, parameters);
        } catch (CommandNotAvailable e) {
            log.error(e, e);
        }
    }

    @Override
    public void updateEnvironment(IoEnvironment oldEnvironment, IoEnvironment newEnvironment) {
        IoEtcdService etcdService = Framework.getLocalService(IoEtcdService.class);

        String oldDomain = oldEnvironment.getDomain();
        String oldApplication = oldEnvironment.getApplication();
        String oldApplicationId = oldEnvironment.getApplicationId();
        String newDomain = newEnvironment.getDomain();
        String newApplication = newEnvironment.getApplication();
        String techId = newEnvironment.getTechId();

        log.debug("Updating environment: " + techId + "\nDomain: " + newDomain);
        if (!oldDomain.equals(newDomain)) {
            etcdService.setDomain(newEnvironment);
            etcdService.deleteDomain(oldEnvironment);
        }

        boolean hasBlankField = isBlank(oldApplication) || isBlank(oldApplicationId)
                || isBlank(newEnvironment.getClid());
        if (!oldApplication.equals(newApplication) || hasBlankField) {
            log.debug("Regenerate CLID for environment: " + techId);
            List<ConnectProject> projects = getAvailableConnectProjects();
            newEnvironment.setApplicationId(null);

            // find corresponding project studio id
            for (ConnectProject project : projects) {
                if (project.getSymbolicName().equals(newApplication)) {
                    newEnvironment.setProject(project);
                }
            }

            newEnvironment.updateClid();
        }
    }

    @Override
    public IoEnvironmentStatus getEnvironmentStatus(IoEnvironment environment) {
        IoEtcdService etcdService = Framework.getLocalService(IoEtcdService.class);
        String environmentStatus = etcdService.getEnvironmentStatus(environment);
        return new IoEnvironmentStatus(environment.getDocument(), environmentStatus);
    }

    @Override
    public List<ConnectProject> getAvailableConnectProjects() {
        if (Framework.isTestModeSet()) {
            return Collections.emptyList();
        }

        WebResource r = IoConnectClient.resource(ConnectUrlConfig.getRegistrationBaseUrl()
                + RegistrationHelper.GET_PROJECTS_SUFFIX);

        List<ConnectProject> result = new ArrayList<>();
        try {
            JSONArray array = new JSONArray(r.get(String.class));
            for (int i = 0; i < array.length(); i++) {
                JSONObject ob = (JSONObject) array.get(i);

                result.add(AbstractJSONSerializableData.loadFromJSON(ConnectProject.class, ob));
            }
        } catch (JSONException | UniformInterfaceException e) {
            log.info("Unable to get available connect project (" + r.getURI().toString() + "): " + e.getMessage());
            log.debug(e, e);
        }

        return result;
    }

    @Override
    public TargetPlatformInformation getTargetPlatform(String projectId) {
        WebResource r = IoConnectClient.resource(String.format(TARGET_PLATFORM_ENDPOINT, projectId));

        TargetPlatformInformation res = null;
        ClientResponse response = r.get(ClientResponse.class);
        try {
            res = getObjectMapper().readValue(response.getEntityInputStream(), TargetPlatformInformation.class);
        } catch (IOException e) {
            log.warn(response, e);
        }
        return res;
    }

    @Override
    public boolean checkPlatformAvailability(String targetPlatformVersion) {
        Client client = Client.create();
        String resource = String.format(DOCKER_REGISTRY_RESOURCE, DOCKER_CONTAINER_NAME, targetPlatformVersion);

        ClientResponse res = client.resource(resource).get(ClientResponse.class);
        if (res.getStatus() != Response.Status.OK.getStatusCode()) {
            log.info("Trying to get an unknown container: " + targetPlatformVersion);
            return false;
        }
        return true;
    }

    protected static ObjectMapper getObjectMapper() {
        if (objectMapper == null) {
            SimpleModule module = new SimpleModule(IoServiceImpl.class.getName(), Version.unknownVersion());
            module.addAbstractTypeMapping(TargetPlatformInstance.class, TargetPlatformInstanceImpl.class);
            module.addAbstractTypeMapping(TargetPackage.class, TargetPackageImpl.class);

            objectMapper = new ObjectMapper();
            objectMapper.registerModule(module);
        }
        return objectMapper;
    }
}
