package org.nuxeo.io.connect;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MultivaluedMap;

import net.oauth.OAuth;
import net.oauth.OAuthAccessor;
import net.oauth.OAuthConsumer;
import net.oauth.OAuthException;
import net.oauth.OAuthMessage;

import org.nuxeo.runtime.api.Framework;

import com.sun.jersey.api.client.ClientHandlerException;
import com.sun.jersey.api.client.ClientRequest;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.filter.ClientFilter;

/**
 * @author <a href="mailto:ak@nuxeo.com">Arnaud Kervern</a>
 * @since 5.9.4
 */
public class OauthAuthFilter extends ClientFilter {

    public static final String CONSUMER_KEY_KEY = "io.oauth.consumer.key";

    public static final String CONSUMER_SECRET_KEY = "io.oauth.consumer.secret";

    public static final String OAUTH_REALM_KEY = "io.oauth.realm";

    protected final OAuthAccessor accessor;

    protected final String realm;

    private String username;

    public OauthAuthFilter(String username) {
        this(username, Framework.getProperty(CONSUMER_KEY_KEY),
                Framework.getProperty(CONSUMER_SECRET_KEY));
    }

    public OauthAuthFilter(String username, String consumerKey,
            String consumerSecret) {
        this.username = username;
        realm = Framework.getProperty(OAUTH_REALM_KEY, "NuxeoIO");

        OAuthConsumer consumer = new OAuthConsumer(null, consumerKey,
                consumerSecret, null);
        consumer.setProperty(OAuth.OAUTH_SIGNATURE_METHOD, OAuth.HMAC_SHA1);
        accessor = new OAuthAccessor(consumer);
    }

    @Override
    public ClientResponse handle(ClientRequest cr)
            throws ClientHandlerException {
        if (!cr.getHeaders().containsKey(HttpHeaders.AUTHORIZATION)) {
            try {
                String uri = OAuth.addParameters(cr.getURI().toString(),
                        "opensocial_viewer_id", username);
                cr.setURI(new URI(uri));

                List<Map.Entry<String, String>> params = new ArrayList<>();
                MultivaluedMap<String, String> entity = (MultivaluedMap<String, String>) cr.getEntity();
                if (entity != null) {
                    for (String key : entity.keySet()) {
                        for (String value : entity.get(key)) {
                            params.add(new AbstractMap.SimpleEntry<>(key, value));
                        }
                    }
                }

                OAuthMessage message = accessor.newRequestMessage(
                        cr.getMethod(), uri, params);

                String authentication = message.getAuthorizationHeader(realm);
                cr.getHeaders().add(HttpHeaders.AUTHORIZATION, authentication);
            } catch (IOException | URISyntaxException | OAuthException e) {
                throw new ClientHandlerException(
                        "Unable to handle Oauth filter", e);
            }
        }
        return getNext().handle(cr);
    }

}
