/*
 * (C) Copyright 2014 Nuxeo SA (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-2.1.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     nuxeo.io Team
 */

package org.nuxeo.io.listener;

import static org.nuxeo.io.Constants.BLOCK_DOMAIN_SUFFIX_KEY;
import static org.nuxeo.io.Constants.DEFAULT_DOMAIN_SUFFIX_KEY;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.apache.commons.lang.StringUtils;
import org.nuxeo.ecm.core.api.ClientException;
import org.nuxeo.ecm.core.api.CoreSession;
import org.nuxeo.ecm.core.api.DocumentModel;
import org.nuxeo.ecm.core.api.DocumentRef;
import org.nuxeo.ecm.core.api.IterableQueryResult;
import org.nuxeo.ecm.core.event.Event;
import org.nuxeo.ecm.core.event.EventContext;
import org.nuxeo.ecm.core.event.EventListener;
import org.nuxeo.ecm.core.event.impl.DocumentEventContext;
import org.nuxeo.io.Constants;
import org.nuxeo.io.NameValidationException;
import org.nuxeo.io.adapter.IoEnvironment;
import org.nuxeo.runtime.api.Framework;

/**
 * Listener validating the domain of the nuxeo.io environment.
 *
 * @since 1.0
 */
public class DomainValidationListener implements EventListener {

    public static Pattern FULL_DOMAIN_REGEXP = Pattern.compile("^[a-z0-9][a-z0-9\\-\\.]*$");

    public static Pattern RESTRICTED_DOMAIN_REGEXP = Pattern.compile("^[a-z0-9][a-z0-9\\-]*$");

    @Override
    public void handleEvent(Event event) throws ClientException {
        EventContext ctx = event.getContext();
        if (ctx instanceof DocumentEventContext) {
            DocumentEventContext docCtx = (DocumentEventContext) ctx;
            IoEnvironment environment = docCtx.getSourceDocument().getAdapter(
                    IoEnvironment.class);
            if (environment != null) {
                CoreSession session = ctx.getCoreSession();
                checkDomain(session, event, environment);

                List<String> envDomains = new ArrayList<>();
                IterableQueryResult envDomainResults = null;
                try {
                    envDomainResults = session.queryAndFetch(
                            String.format(
                                    "SELECT "
                                            + "ioenvironment:domain from ioEnvironment "
                                            + "where ecm:uuid != '%s'",
                                    environment.getId()), "NXQL");
                    for (Map<String, Serializable> record : envDomainResults) {
                        envDomains.add((String) record.get(Constants.IO_ENVIRONMENT_DOMAIN_PROPERTY));
                    }
                } finally {
                    if (envDomainResults != null) {
                        envDomainResults.close();
                    }
                }

                String domain = environment.getDomain();
                if (envDomains.contains(domain)) {
                    event.markBubbleException();
                    String message = String.format("Domain '%s' already used.",
                            domain);
                    throw new NameValidationException(message);
                }
            }
        }
    }

    protected void checkDomain(CoreSession session, Event event,
            IoEnvironment environment) throws ClientException {
        if (StringUtils.isBlank(environment.getDomain())) {
            computeDomain(session, environment);
        }
        validateDomain(event, environment);
    }

    protected void computeDomain(CoreSession session, IoEnvironment environment)
            throws ClientException {
        // compute default environment domain
        DocumentRef parentRef = session.getParentDocumentRef(environment.getDocument().getRef()); // environments
        DocumentModel client = session.getParentDocument(parentRef);

        String defaultDomainSuffix = getDefaultDomainSuffix();
        if (Framework.isBooleanPropertyTrue(BLOCK_DOMAIN_SUFFIX_KEY)) {
            environment.setDomain(String.format("%s-%s", environment.getName(),
                    client.getName()));
        } else {
            environment.setDomain(String.format("%s-%s.%s",
                    environment.getName(), client.getName(),
                    defaultDomainSuffix));
        }
    }

    protected void validateDomain(Event event, IoEnvironment environment)
            throws NameValidationException {
        String domain = environment.getDomain();
        if (Framework.isBooleanPropertyTrue(BLOCK_DOMAIN_SUFFIX_KEY)) {
            if (!RESTRICTED_DOMAIN_REGEXP.matcher(domain).matches()) {
                event.markBubbleException();
                String message = String.format(
                        "Invalid domain '%s'. Domain must start"
                                + " with a letter or number and can only contain "
                                + "lowercase letters, numbers, and dashes.",
                        domain);
                throw new NameValidationException(message);
            }

            // append the domain suffix if needed
            String defaultDomainSuffix = getDefaultDomainSuffix();
            if (!domain.endsWith(defaultDomainSuffix)) {
                if (!domain.endsWith(".")) {
                    domain += ".";
                }
                environment.setDomain(domain + defaultDomainSuffix);
            }
        } else {
            if (!FULL_DOMAIN_REGEXP.matcher(domain).matches()) {
                event.markBubbleException();
                String message = String.format(
                        "Invalid domain name '%s'. Domain must start"
                                + " with a letter or number and can only contain "
                                + "lowercase letters, numbers, dots and dashes.",
                        domain);
                throw new NameValidationException(message);
            }
        }
    }

    /**
     * Returns the default domain suffix for the cluster
     */
    private String getDefaultDomainSuffix() {
        return Framework.getProperty(DEFAULT_DOMAIN_SUFFIX_KEY,
                "trial.nuxeo.io");
    }

}
