package org.nuxeo.io.listener;

import static org.nuxeo.ecm.core.api.event.DocumentEventTypes.ABOUT_TO_CREATE;
import static org.nuxeo.io.Constants.IO_CLIENT_LIMIT_PROPERTY;
import static org.nuxeo.io.Constants.IO_ENVIRONMENT_DOCUMENT_TYPE;

import java.io.Serializable;
import java.util.Map;

import org.nuxeo.ecm.core.api.ClientException;
import org.nuxeo.ecm.core.api.CoreSession;
import org.nuxeo.ecm.core.api.DocumentModel;
import org.nuxeo.ecm.core.api.DocumentRef;
import org.nuxeo.ecm.core.api.IterableQueryResult;
import org.nuxeo.ecm.core.event.Event;
import org.nuxeo.ecm.core.event.EventContext;
import org.nuxeo.ecm.core.event.EventListener;
import org.nuxeo.ecm.core.event.impl.DocumentEventContext;
import org.nuxeo.ecm.core.query.sql.NXQL;
import org.nuxeo.io.LimitReachedException;

/**
 * @author <a href="mailto:ak@nuxeo.com">Arnaud Kervern</a>
 *
 * @since 1.0
 */
public class LimitEnvironmentListener implements EventListener {

    private static final String QUERY_FIELD = "COUNT(ecm:uuid)";

    private static final String QUERY_ENVS = "Select "
            + QUERY_FIELD
            + " from "
            + IO_ENVIRONMENT_DOCUMENT_TYPE
            + " where ecm:parentId = '%s' AND ecm:currentLifeCycleState <> 'deleted' AND ecm:isProxy = 0";

    @Override
    public void handleEvent(Event event) throws ClientException {
        if (!ABOUT_TO_CREATE.equals(event.getName())) {
            return;
        }

        EventContext ctx = event.getContext();
        if (!(ctx instanceof DocumentEventContext)) {
            return;
        }

        DocumentModel doc = ((DocumentEventContext) ctx).getSourceDocument();
        if (!doc.getType().equals(IO_ENVIRONMENT_DOCUMENT_TYPE)) {
            return;
        }

        CoreSession session = ctx.getCoreSession();
        DocumentModel envsRef = session.getDocument(doc.getParentRef());
        DocumentModel client = session.getParentDocument(envsRef.getRef());

        if (client == null) {
            return;
        }

        long clientLimit = (long) client.getPropertyValue(IO_CLIENT_LIMIT_PROPERTY);
        long envsCount = countEnvironments(session, envsRef.getRef());

        if (envsCount >= clientLimit) {
            event.markBubbleException();
            throw new LimitReachedException(String.format(
                    "You reached your environments limit (%d).", clientLimit));
        }
    }

    /**
     * @return the number of active environment from a Environments document
     */
    private long countEnvironments(CoreSession session, DocumentRef parentRef)
            throws ClientException {
        IterableQueryResult iqr = null;
        try {
            String query = String.format(QUERY_ENVS, parentRef.toString());
            iqr = session.queryAndFetch(query, NXQL.NXQL);
            Map<String, Serializable> next = iqr.iterator().next();
            return (Long) next.get(QUERY_FIELD);
        } finally {
            if (iqr != null) {
                iqr.close();
            }
        }
    }
}
