/*
 * (C) Copyright 2014 Nuxeo SA (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-2.1.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     nuxeo.io Team
 */

package org.nuxeo.io.service;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.nuxeo.connect.connector.NuxeoClientInstanceType;
import org.nuxeo.connect.connector.http.ConnectUrlConfig;
import org.nuxeo.connect.data.AbstractJSONSerializableData;
import org.nuxeo.connect.data.ConnectProject;
import org.nuxeo.connect.identity.LogicalInstanceIdentifier;
import org.nuxeo.connect.registration.RegistrationHelper;
import org.nuxeo.ecm.platform.commandline.executor.api.CmdParameters;
import org.nuxeo.ecm.platform.commandline.executor.api.CommandLineExecutorService;
import org.nuxeo.ecm.platform.commandline.executor.api.CommandNotAvailable;
import org.nuxeo.etcd.EtcdService;
import org.nuxeo.io.adapter.IoEnvironment;
import org.nuxeo.io.adapter.IoEnvironmentStatus;
import org.nuxeo.io.connect.InstanceRegistrator;
import org.nuxeo.io.connect.IoConnectClient;
import org.nuxeo.runtime.api.Framework;

import com.sun.jersey.api.client.WebResource;

/**
 * @since 1.0
 */
public class IoServiceImpl implements IoService {

    private static final int _1 = 1;

    private static final Log log = LogFactory.getLog(IoServiceImpl.class);

    public static final String ETCD_ENDPOINT_PROPERTY_NAME = "org.nuxeo.io.etcd.endpoint";

    private static final String CONTAINER_UNIT_NAME = "nxio.%s.%d.service";

    private static final String CLEANER_UNIT_NAME = "cleaner.%s.service";

    public static final String CREATE_ENV_COMMAND = "createEnvironment";

    public static final String DELETE_ENV_COMMAND = "deleteEnvironment";

    public static final String START_ENV_COMMAND = "startEnvironment";

    public static final String STOP_ENV_COMMAND = "stopEnvironment";

    public static final String START_CLEANER_COMMAND = "startCleaner";

    public static final String FLEET_CONTAINER_UNIT_TEMPLATE = "fleet_container_unit_template.service";

    public static final String FLEET_CLEANER_UNIT_TEMPLATE = "fleet_cleaner_unit_template.service";

    public static final String IO_CONTAINER_TYPE = "service";

    public static final String DOMAIN_KEY_PATTERN = "/domains/%s";

    public static final String DOMAIN_TYPE_KEY_PATTERN = DOMAIN_KEY_PATTERN
            + "/type";

    public static final String DOMAIN_VALUE_KEY_PATTERN = DOMAIN_KEY_PATTERN
            + "/value";

    public static final String SERVICE_KEY_PATTERN = "/services/%s/";

    public static final String SERVICE_DOMAIN_KEY_PATTERN = SERVICE_KEY_PATTERN
            + "/%d/domain";

    public static final String SERVICE_STATUS_KEY_PATTERN = SERVICE_KEY_PATTERN
            + "/%d/status";

    public static final String SERVICE_CURRENT_STATUS_KEY_PATTERN = SERVICE_STATUS_KEY_PATTERN
            + "/current";

    public static final String SERVICE_CONFIG_KEY_PATTERN = SERVICE_KEY_PATTERN
            + "/%d/config";

    public static final String SERVICE_CONFIG_PACKAGES = SERVICE_CONFIG_KEY_PATTERN
            + "/packages";

    public static final String SERVICE_CONFIG_CLIDS = SERVICE_CONFIG_KEY_PATTERN
            + "/instance.clid";

    public static final String SERVICE_EXPECTED_STATUS_KEY_PATTERN = SERVICE_STATUS_KEY_PATTERN
            + "/expected";

    public static final String SERVICE_ALIVE_STATUS_KEY_PATTERN = SERVICE_STATUS_KEY_PATTERN
            + "/alive";

    public static final String SERVICES_TO_CLEAN = "postgres s3 etcd";

    protected String fleetContainerUnitTemplate;

    protected String fleetCleanerUnitTemplate;

    @Override
    public void createEnvironment(IoEnvironment environment) {
        try {
            String unitTemplate = getFleetContainerUnitTemplate();
            String techId = environment.getTechId();
            unitTemplate = unitTemplate.replaceAll("\\$\\{envTechId\\}", techId);

            String unitName = String.format(CONTAINER_UNIT_NAME, techId, _1);
            File file = new File(getTempWorkingDirectory(), unitName);
            file.deleteOnExit();
            Framework.trackFile(file, this);
            FileUtils.writeStringToFile(file, unitTemplate);

            CmdParameters parameters = new CmdParameters();
            parameters.addNamedParameter("unitFilePath", file);
            parameters.addNamedParameter("etcdEndpoint",
                    Framework.getProperty(ETCD_ENDPOINT_PROPERTY_NAME));
            CommandLineExecutorService cles = Framework.getLocalService(CommandLineExecutorService.class);
            cles.execCommand(CREATE_ENV_COMMAND, parameters);
        } catch (CommandNotAvailable | IOException e) {
            log.error("Cannot create environment", e);
        }

        List<ConnectProject> projects = getAvailableConnectProjects();
        if (projects.size() > 0) {
            environment.setApplication(projects.get(0).getUuid());
        }

        createEtcdKeysFor(environment);

        setCurrentStatus(environment, Statuses.STOPPED);
        setExpectedStatus(environment, Statuses.STOPPED);
    }

    protected String getFleetContainerUnitTemplate() {
        if (fleetContainerUnitTemplate == null) {
            fleetContainerUnitTemplate = getFleetUnitTemplate(FLEET_CONTAINER_UNIT_TEMPLATE);
        }

        return fleetContainerUnitTemplate;
    }

    protected String getFleetCleanerUnitTemplate() {
        if (fleetCleanerUnitTemplate == null) {
            fleetCleanerUnitTemplate = getFleetUnitTemplate(FLEET_CLEANER_UNIT_TEMPLATE);
        }

        return fleetCleanerUnitTemplate;
    }

    protected String getFleetUnitTemplate(String name) {
        String template = null;
        try {
            URL fileUrl = Thread.currentThread().getContextClassLoader().getResource(
                    name);
            if (fileUrl != null) {
                template = IOUtils.toString(fileUrl.openStream());
            }
        } catch (IOException e) {
            log.error(e, e);
        }

        return template;
    }

    protected String getCLIDsEntry(IoEnvironment environment) {
        LogicalInstanceIdentifier CLIDs = new InstanceRegistrator(
                environment.getApplication(), environment.getName(),
                NuxeoClientInstanceType.DEV).getCLIDs();
        if (CLIDs != null) {
            return String.format("%s\\n%s\\n%s", CLIDs.getCLID1(),
                    CLIDs.getCLID2(), CLIDs.getInstanceDescription());
        }
        return null;
    }

    protected File getTempWorkingDirectory() {
        File workingDir = new File(System.getProperty("java.io.tmpdir"), "nxio");
        if (!workingDir.exists()) {
            workingDir.mkdir();
        }
        return workingDir;
    }

    protected void createEtcdKeysFor(IoEnvironment environment) {
        EtcdService etcdService = Framework.getLocalService(EtcdService.class);
        String domain = environment.getDomain();
        String techId = environment.getTechId();
        String typeKey = String.format(DOMAIN_TYPE_KEY_PATTERN, domain);
        String valueKey = String.format(DOMAIN_VALUE_KEY_PATTERN, domain);
        String domainKey = String.format(SERVICE_DOMAIN_KEY_PATTERN, techId, _1);
        String clidsKey = String.format(SERVICE_CONFIG_CLIDS, techId, _1);

        etcdService.set(typeKey, IO_CONTAINER_TYPE);
        etcdService.set(valueKey, techId);
        etcdService.set(domainKey, domain);
        etcdService.set(clidsKey, getCLIDsEntry(environment));
    }

    protected void setCurrentStatus(IoEnvironment environment, Statuses status) {
        EtcdService etcdService = Framework.getLocalService(EtcdService.class);
        String currentStatusKey = getCurrentStatusKeyFor(environment, _1);
        etcdService.set(currentStatusKey, status.toString());
    }

    protected void setExpectedStatus(IoEnvironment environment, Statuses status) {
        EtcdService etcdService = Framework.getLocalService(EtcdService.class);
        String expectedStatusKey = getExpectedStatusKeyFor(environment, _1);
        etcdService.set(expectedStatusKey, status.toString());
    }

    @Override
    public void deleteEnvironment(IoEnvironment environment) {
        CommandLineExecutorService cles = Framework.getLocalService(CommandLineExecutorService.class);

        try {
            CmdParameters parameters = computeUnitNameCmdParameters(environment);
            cles.execCommand(DELETE_ENV_COMMAND, parameters);
        } catch (CommandNotAvailable e) {
            log.error(e, e);
        }

        try {
            String techId = environment.getTechId();
            String unitName = String.format(CLEANER_UNIT_NAME, techId);
            String unitTemplate = getFleetCleanerUnitTemplate();
            unitTemplate = unitTemplate.replaceAll("\\$\\{envTechId\\}", techId);
            unitTemplate = unitTemplate.replaceAll("\\$\\{unitName\\}", unitName);
            unitTemplate = unitTemplate.replaceAll("\\$\\{services\\}", SERVICES_TO_CLEAN);

            File file = new File(getTempWorkingDirectory(), unitName);
            file.deleteOnExit();
            Framework.trackFile(file, this);
            FileUtils.writeStringToFile(file, unitTemplate);

            CmdParameters parameters = new CmdParameters();
            parameters.addNamedParameter("unitFilePath", file);
            parameters.addNamedParameter("etcdEndpoint",
                    Framework.getProperty(ETCD_ENDPOINT_PROPERTY_NAME));

            cles.execCommand(START_CLEANER_COMMAND, parameters);
        } catch (CommandNotAvailable | IOException e) {
            log.error("Cannot delete environment", e);
        }
    }

    protected CmdParameters computeUnitNameCmdParameters(
            IoEnvironment environment) {
        String techId = environment.getTechId();
        String unitName = String.format(CONTAINER_UNIT_NAME, techId, _1);
        CmdParameters parameters = new CmdParameters();
        parameters.addNamedParameter("unitName", unitName);
        parameters.addNamedParameter("etcdEndpoint",
                Framework.getProperty(ETCD_ENDPOINT_PROPERTY_NAME));
        return parameters;
    }

    protected void deleteServiceEtcdKeyFor(IoEnvironment environment) {
        EtcdService etcdService = Framework.getLocalService(EtcdService.class);
        String techId = environment.getTechId();

        String serviceKey = String.format(SERVICE_KEY_PATTERN, techId);
        etcdService.delete(serviceKey, true);
    }

    @Override
    public void startEnvironment(IoEnvironment environment) {
        try {
            CommandLineExecutorService cles = Framework.getLocalService(CommandLineExecutorService.class);
            CmdParameters parameters = computeUnitNameCmdParameters(environment);
            // Stop env while starting; to ensure that Fleet unit is ready to
            // start
            cles.execCommand(STOP_ENV_COMMAND, parameters);

            setExpectedStatus(environment, Statuses.STARTED);
            cles.execCommand(START_ENV_COMMAND, parameters);
        } catch (CommandNotAvailable e) {
            log.error(e, e);
        }
    }

    @Override
    public void stopEnvironment(IoEnvironment environment) {
        try {
            setExpectedStatus(environment, Statuses.STOPPED);

            CommandLineExecutorService cles = Framework.getLocalService(CommandLineExecutorService.class);
            CmdParameters parameters = computeUnitNameCmdParameters(environment);
            cles.execCommand(STOP_ENV_COMMAND, parameters);
        } catch (CommandNotAvailable e) {
            log.error(e, e);
        }
    }

    @Override
    public void updateEnvironment(IoEnvironment oldEnvironment,
            IoEnvironment newEnvironment) {
        EtcdService etcdService = Framework.getLocalService(EtcdService.class);

        String oldDomain = oldEnvironment.getDomain();
        String oldApplication = oldEnvironment.getApplication();
        String newDomain = newEnvironment.getDomain();
        String newApplication = newEnvironment.getApplication();
        String techId = newEnvironment.getTechId();

        if (!oldDomain.equals(newDomain)) {
            String typeKey = String.format(DOMAIN_TYPE_KEY_PATTERN, newDomain);
            String valueKey = String.format(DOMAIN_VALUE_KEY_PATTERN, newDomain);
            etcdService.set(typeKey, IO_CONTAINER_TYPE);
            etcdService.set(valueKey, techId);

            String domainKey = String.format(SERVICE_DOMAIN_KEY_PATTERN,
                    techId, _1);
            etcdService.set(domainKey, newDomain);

            deleteDomainEtcdKeyFor(oldEnvironment);
        }

        if (oldApplication == null || !oldApplication.equals(newApplication)) {
            String clidsKey = String.format(SERVICE_CONFIG_CLIDS, techId, _1);
            etcdService.set(clidsKey, getCLIDsEntry(newEnvironment));
        }
    }

    protected void deleteDomainEtcdKeyFor(IoEnvironment environment) {
        EtcdService etcdService = Framework.getLocalService(EtcdService.class);
        String oldDomain = environment.getDomain();
        String key = String.format(DOMAIN_KEY_PATTERN, oldDomain);
        etcdService.delete(key, true);
    }

    @Override
    public IoEnvironmentStatus getEnvironmentStatus(IoEnvironment environment) {
        EtcdService etcdService = Framework.getLocalService(EtcdService.class);
        String currentStatus = etcdService.getValue(getCurrentStatusKeyFor(
                environment, _1));
        String expectedStatus = etcdService.getValue(getExpectedStatusKeyFor(
                environment, _1));
        String alive = etcdService.getValue(getAliveStatusKeyFor(environment,
                _1));

        EnvironmentStatusComputer computer = new EnvironmentStatusComputer(
                expectedStatus, currentStatus, alive);
        String environmentStatus = computer.computeStatus();
        return new IoEnvironmentStatus(environment.getDocument(),
                environmentStatus);
    }

    @Override
    public List<ConnectProject> getAvailableConnectProjects() {
        if (Framework.isTestModeSet()) {
            return Collections.emptyList();
        }

        WebResource r = IoConnectClient.resource(ConnectUrlConfig.getRegistrationBaseUrl()
                + RegistrationHelper.GET_PROJECTS_SUFFIX);

        List<ConnectProject> result = new ArrayList<>();
        try {
            JSONArray array = new JSONArray(r.get(String.class));
            for (int i = 0; i < array.length(); i++) {
                JSONObject ob = (JSONObject) array.get(i);

                result.add(AbstractJSONSerializableData.loadFromJSON(
                        ConnectProject.class, ob));
            }
        } catch (JSONException e) {
            log.debug(e, e);
        }

        return result;
    }

    public static String getCurrentStatusKeyFor(IoEnvironment env, int index) {
        return String.format(SERVICE_CURRENT_STATUS_KEY_PATTERN,
                env.getTechId(), index);
    }

    public static String getExpectedStatusKeyFor(IoEnvironment env, int index) {
        return String.format(SERVICE_EXPECTED_STATUS_KEY_PATTERN,
                env.getTechId(), index);
    }

    public static String getAliveStatusKeyFor(IoEnvironment env, int index) {
        return String.format(SERVICE_ALIVE_STATUS_KEY_PATTERN, env.getTechId(),
                index);
    }

    public static String getNXDomainKeyFor(IoEnvironment env, int index) {
        return String.format(SERVICE_DOMAIN_KEY_PATTERN, env.getTechId(), index);
    }

}
