/*
 * (C) Copyright 2006-2007 Nuxeo SAS (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     Nuxeo - initial API and implementation
 *
 * $Id$
 */

package org.nuxeo.runtime.model;

import java.net.URL;

import org.nuxeo.runtime.RuntimeService;
import org.osgi.framework.Bundle;

/**
 * The runtime context.
 * <p>
 * Runtime contexts are used to create components.
 * They provides custom methods to load classes and find resources.
 * <p>
 * Runtime contexts are generally attached to a bundle context
 * (or module deployment context)
 *
 * @author  <a href="mailto:bs@nuxeo.com">Bogdan Stefanescu</a>
 *
 */
public interface RuntimeContext {

    /**
     * Gets the curent runtime service.
     *
     * @return the runtime service
     */
    RuntimeService getRuntime();

    /**
     * Gets the container bundle or null if we are not running in an OSGi environment.
     */
    Bundle getBundle();

    /**
     * Loads the class given its name.
     *
     * @param className the class name
     * @return the class object
     * @throws ClassNotFoundException if no such class were found
     * @see ClassLoader#loadClass(String)
     */
    Class loadClass(String className) throws ClassNotFoundException;

    /**
     * Finds a resource having the given name.
     *
     * @param name the resource name
     * @return an URL to the resource having that name or null if not was found
     * @see ClassLoader#getResource(String)
     */
    URL getResource(String name);

    /**
     * Finds a local resource having the given name.
     * <p>
     * Only the current bundle should be searched for the resource.
     *
     * @param name the local resource name
     * @return an URL to the resource having that name or null if not was found
     * @see ClassLoader#getResource(String)
     */
    URL getLocalResource(String name);

    /**
     * Deploys a component XML descriptor given its URL.
     * <p>
     * Do nothing if component is already deployed.
     *
     * @param url the url of the XML descriptor
     * @throws Exception if any error occurs
     */
    void deploy(URL url) throws Exception;

    /**
     * Undeploys a component XML descriptor given its URL.
     * <p>
     * Do nothing if no component was registered fro the given URL.
     *
     * @param url the url of the XML descriptor
     * @throws Exception if any error occurs
     */
    void undeploy(URL url) throws Exception;

    /**
     * Checks whether the component XML file at given URL was deployeds.
     *
     * @param url the url to check
     * @return true it deployed, false otherwise
     */
    boolean isDeployed(URL url);

    /**
     * Deploys the component whose XML descriptor is at the given location.
     * <p>
     * If the component is already deployed do nothing.
     * <p>
     * The location is interpreted as a relative path inside the bundle (jar)
     * containing the component - and will be loaded using
     * {@link RuntimeContext#getLocalResource(String)}.
     *
     * @param location the location
     * @throws Exception
     */
    void deploy(String location) throws Exception;

    /**
     * Undeploys the component at the given location if any was deployed.
     * <p>
     * If the component was not deployed do nothing.
     *
     * @param location the location of the component to undeploy
     * @throws Exception if any error occurs
     */
    void undeploy(String location) throws Exception;

    /**
     * Checks if the component at the given location is deployed.
     *
     * @param location the component location to check
     * @return true if deployed, false otherwise
     */
    boolean isDeployed(String location);

    /**
     * Destroys this context.
     */
    void destroy();

}
