/*
 * Decompiled with CFR 0.152.
 */
package org.owasp.dependencycheck.data.update;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import javax.annotation.concurrent.ThreadSafe;
import org.owasp.dependencycheck.Engine;
import org.owasp.dependencycheck.data.nvd.json.MetaProperties;
import org.owasp.dependencycheck.data.nvdcve.CveDB;
import org.owasp.dependencycheck.data.nvdcve.DatabaseException;
import org.owasp.dependencycheck.data.nvdcve.DatabaseProperties;
import org.owasp.dependencycheck.data.update.CachedWebDataSource;
import org.owasp.dependencycheck.data.update.exception.InvalidDataException;
import org.owasp.dependencycheck.data.update.exception.UpdateException;
import org.owasp.dependencycheck.data.update.nvd.DownloadTask;
import org.owasp.dependencycheck.data.update.nvd.NvdCveInfo;
import org.owasp.dependencycheck.data.update.nvd.ProcessTask;
import org.owasp.dependencycheck.utils.DateUtil;
import org.owasp.dependencycheck.utils.DownloadFailedException;
import org.owasp.dependencycheck.utils.Downloader;
import org.owasp.dependencycheck.utils.InvalidSettingException;
import org.owasp.dependencycheck.utils.ResourceNotFoundException;
import org.owasp.dependencycheck.utils.Settings;
import org.owasp.dependencycheck.utils.TooManyRequestsException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class NvdCveUpdater
implements CachedWebDataSource {
    private static final Logger LOGGER = LoggerFactory.getLogger(NvdCveUpdater.class);
    private static final int PROCESSING_THREAD_POOL_SIZE = Runtime.getRuntime().availableProcessors();
    private static final int DOWNLOAD_THREAD_POOL_SIZE = Math.round(1.5f * (float)Runtime.getRuntime().availableProcessors());
    private ExecutorService processingExecutorService = null;
    private ExecutorService downloadExecutorService = null;
    private Settings settings;
    private CveDB cveDb = null;
    private DatabaseProperties dbProperties = null;

    @Override
    public synchronized boolean update(Engine engine) throws UpdateException {
        this.settings = engine.getSettings();
        this.cveDb = engine.getDatabase();
        if (this.isUpdateConfiguredFalse()) {
            return false;
        }
        boolean updatesMade = false;
        try {
            this.dbProperties = this.cveDb.getDatabaseProperties();
            if (this.checkUpdate()) {
                List<NvdCveInfo> updateable = this.getUpdatesNeeded();
                if (!updateable.isEmpty()) {
                    this.initializeExecutorServices();
                    this.performUpdate(updateable);
                    updatesMade = true;
                }
                this.dbProperties.save("NVD CVE Checked", Long.toString(System.currentTimeMillis() / 1000L));
            }
        }
        catch (UpdateException ex) {
            String jre;
            if (ex.getCause() != null && ex.getCause() instanceof DownloadFailedException && ((jre = System.getProperty("java.version")) == null || jre.startsWith("1.4") || jre.startsWith("1.5") || jre.startsWith("1.6") || jre.startsWith("1.7"))) {
                LOGGER.error("An old JRE is being used ({} {}), and likely does not have the correct root certificates or algorithms to connect to the NVD - consider upgrading your JRE.", (Object)System.getProperty("java.vendor"), (Object)jre);
            }
            throw ex;
        }
        catch (DatabaseException ex) {
            throw new UpdateException("Database Exception, unable to update the data to use the most current data.", ex);
        }
        finally {
            this.shutdownExecutorServices();
        }
        return updatesMade;
    }

    private boolean isUpdateConfiguredFalse() {
        if (!this.settings.getBoolean("updater.nvdcve.enabled", true)) {
            return true;
        }
        boolean autoUpdate = true;
        try {
            autoUpdate = this.settings.getBoolean("odc.autoupdate");
        }
        catch (InvalidSettingException ex) {
            LOGGER.debug("Invalid setting for auto-update; using true.");
        }
        return !autoUpdate;
    }

    protected void initializeExecutorServices() {
        int max = this.settings.getInt("max.download.threads", 3);
        int downloadPoolSize = DOWNLOAD_THREAD_POOL_SIZE > max ? max : DOWNLOAD_THREAD_POOL_SIZE;
        this.downloadExecutorService = Executors.newFixedThreadPool(downloadPoolSize);
        this.processingExecutorService = Executors.newFixedThreadPool(PROCESSING_THREAD_POOL_SIZE);
        LOGGER.debug("#download   threads: {}", (Object)downloadPoolSize);
        LOGGER.debug("#processing threads: {}", (Object)PROCESSING_THREAD_POOL_SIZE);
    }

    private void shutdownExecutorServices() {
        if (this.processingExecutorService != null) {
            this.processingExecutorService.shutdownNow();
        }
        if (this.downloadExecutorService != null) {
            this.downloadExecutorService.shutdownNow();
        }
    }

    private boolean checkUpdate() throws UpdateException {
        boolean proceed = true;
        int validForHours = this.settings.getInt("cve.check.validforhours", 0);
        if (this.dataExists() && 0 < validForHours) {
            long validForSeconds = (long)validForHours * 60L * 60L;
            long lastChecked = this.getPropertyInSeconds("NVD CVE Checked");
            long now = System.currentTimeMillis() / 1000L;
            boolean bl = proceed = now - lastChecked > validForSeconds;
            if (!proceed) {
                LOGGER.info("Skipping NVD check since last check was within {} hours.", (Object)validForHours);
                LOGGER.debug("Last NVD was at {}, and now {} is within {} s.", new Object[]{lastChecked, now, validForSeconds});
            }
        }
        return proceed;
    }

    private boolean dataExists() {
        return this.cveDb.dataExists();
    }

    private void performUpdate(List<NvdCveInfo> updateable) throws UpdateException {
        if (updateable.isEmpty()) {
            return;
        }
        if (updateable.size() > 3) {
            LOGGER.info("NVD CVE requires several updates; this could take a couple of minutes.");
        }
        DownloadTask runLast = null;
        HashSet<Future<Future<ProcessTask>>> downloadFutures = new HashSet<Future<Future<ProcessTask>>>(updateable.size());
        for (NvdCveInfo nvdCveInfo : updateable) {
            DownloadTask downloadTask = new DownloadTask(nvdCveInfo, this.processingExecutorService, this.cveDb, this.settings);
            if (downloadTask.isModified()) {
                runLast = downloadTask;
                continue;
            }
            boolean added = downloadFutures.add(this.downloadExecutorService.submit(downloadTask));
            if (added) continue;
            throw new UpdateException("Unable to add the download task for " + nvdCveInfo.getId());
        }
        HashSet<Future> processFutures = new HashSet<Future>(updateable.size());
        for (Future future : downloadFutures) {
            try {
                Future task = (Future)future.get();
                if (task == null) continue;
                processFutures.add(task);
            }
            catch (InterruptedException ex) {
                LOGGER.debug("Thread was interrupted during download", (Throwable)ex);
                Thread.currentThread().interrupt();
                throw new UpdateException("The download was interrupted", ex);
            }
            catch (ExecutionException ex) {
                LOGGER.debug("Thread was interrupted during download execution", (Throwable)ex);
                throw new UpdateException("The execution of the download was interrupted", ex);
            }
        }
        for (Future future : processFutures) {
            try {
                ProcessTask task = (ProcessTask)future.get();
                if (task.getException() == null) continue;
                throw task.getException();
            }
            catch (InterruptedException ex) {
                LOGGER.debug("Thread was interrupted during processing", (Throwable)ex);
                Thread.currentThread().interrupt();
                throw new UpdateException(ex);
            }
            catch (ExecutionException ex) {
                LOGGER.debug("Execution Exception during process", (Throwable)ex);
                throw new UpdateException(ex);
            }
        }
        if (runLast != null) {
            Future<Future<ProcessTask>> future = this.downloadExecutorService.submit(runLast);
            try {
                Future<ProcessTask> future2 = future.get();
                ProcessTask last = future2.get();
                if (last.getException() != null) {
                    throw last.getException();
                }
            }
            catch (InterruptedException ex) {
                LOGGER.debug("Thread was interrupted during download", (Throwable)ex);
                Thread.currentThread().interrupt();
                throw new UpdateException("The download was interrupted", ex);
            }
            catch (ExecutionException ex) {
                LOGGER.debug("Thread was interrupted during download execution", (Throwable)ex);
                throw new UpdateException("The execution of the download was interrupted", ex);
            }
        }
        try {
            this.cveDb.cleanupDatabase();
        }
        catch (DatabaseException databaseException) {
            throw new UpdateException(databaseException.getMessage(), databaseException.getCause());
        }
    }

    protected final MetaProperties getMetaFile(String url) throws UpdateException {
        String metaUrl = url.substring(0, url.length() - 7) + "meta";
        try {
            URL u = new URL(metaUrl);
            Downloader d = new Downloader(this.settings);
            String content = d.fetchContent(u, true);
            return new MetaProperties(content);
        }
        catch (MalformedURLException ex) {
            throw new UpdateException("Meta file url is invalid: " + metaUrl, ex);
        }
        catch (InvalidDataException ex) {
            throw new UpdateException("Meta file content is invalid: " + metaUrl, ex);
        }
        catch (DownloadFailedException ex) {
            throw new UpdateException("Unable to download meta file: " + metaUrl, ex);
        }
        catch (TooManyRequestsException ex) {
            throw new UpdateException("Unable to download meta file: " + metaUrl + "; received 429 -- too many requests", ex);
        }
        catch (ResourceNotFoundException ex) {
            throw new UpdateException("Unable to download meta file: " + metaUrl + "; received 404 -- resource not found", ex);
        }
    }

    protected final List<NvdCveInfo> getUpdatesNeeded() throws UpdateException {
        LOGGER.debug("starting getUpdatesNeeded() ...");
        ArrayList<NvdCveInfo> updates = new ArrayList<NvdCveInfo>();
        if (this.dbProperties != null && !this.dbProperties.isEmpty()) {
            try {
                int startYear = this.settings.getInt("cve.startyear", 2002);
                int endYear = Calendar.getInstance().get(1);
                boolean needsFullUpdate = false;
                for (int y = startYear; y <= endYear; ++y) {
                    long val = Long.parseLong(this.dbProperties.getProperty("NVD CVE " + y, "0"));
                    if (val != 0L) continue;
                    needsFullUpdate = true;
                    break;
                }
                long lastUpdated = this.getPropertyInSeconds("NVD CVE Modified");
                long now = System.currentTimeMillis() / 1000L;
                int days = this.settings.getInt("cve.url.modified.validfordays", 7);
                String url = this.settings.getString("cve.url.modified");
                MetaProperties modified = this.getMetaFile(url);
                if (!needsFullUpdate && lastUpdated == modified.getLastModifiedDate()) {
                    return updates;
                }
                NvdCveInfo item = new NvdCveInfo("Modified", url, modified.getLastModifiedDate());
                updates.add(item);
                if (needsFullUpdate || !DateUtil.withinDateRange(lastUpdated, now, days)) {
                    int start = this.settings.getInt("cve.startyear");
                    int end = Calendar.getInstance().get(1);
                    String baseUrl = this.settings.getString("cve.url.base");
                    for (int i = start; i <= end; ++i) {
                        url = String.format(baseUrl, i);
                        MetaProperties meta = this.getMetaFile(url);
                        long currentTimestamp = this.getPropertyInSeconds("NVD CVE " + i);
                        if (currentTimestamp >= meta.getLastModifiedDate()) continue;
                        NvdCveInfo entry = new NvdCveInfo(Integer.toString(i), url, meta.getLastModifiedDate());
                        updates.add(entry);
                    }
                }
            }
            catch (NumberFormatException ex) {
                LOGGER.warn("An invalid schema version or timestamp exists in the data.properties file.");
                LOGGER.debug("", (Throwable)ex);
            }
            catch (InvalidSettingException ex) {
                throw new UpdateException("The NVD CVE start year property is set to an invalid value", ex);
            }
        }
        return updates;
    }

    private long getPropertyInSeconds(String key) {
        String value = this.dbProperties.getProperty(key, "0");
        return DateUtil.getEpochValueInSeconds(value);
    }

    protected synchronized void setSettings(Settings settings) {
        this.settings = settings;
    }

    @Override
    public boolean purge(Engine engine) {
        boolean result = true;
        try {
            File lockFile;
            File dataDir = engine.getSettings().getDataDirectory();
            File db = new File(dataDir, engine.getSettings().getString("data.file_name", "odc.mv.db"));
            if (db.exists()) {
                if (db.delete()) {
                    LOGGER.info("Database file purged; local copy of the NVD has been removed");
                } else {
                    LOGGER.error("Unable to delete '{}'; please delete the file manually", (Object)db.getAbsolutePath());
                    result = false;
                }
            } else {
                LOGGER.info("Unable to purge database; the database file does not exist: {}", (Object)db.getAbsolutePath());
                result = false;
            }
            File traceFile = new File(dataDir, "odc.trace.db");
            if (traceFile.exists() && !traceFile.delete()) {
                LOGGER.error("Unable to delete '{}'; please delete the file manually", (Object)traceFile.getAbsolutePath());
                result = false;
            }
            if ((lockFile = new File(dataDir, "odc.update.lock")).exists() && !lockFile.delete()) {
                LOGGER.error("Unable to delete '{}'; please delete the file manually", (Object)lockFile.getAbsolutePath());
                result = false;
            }
        }
        catch (IOException ex) {
            String msg = "Unable to delete the database";
            LOGGER.error("Unable to delete the database", (Throwable)ex);
            result = false;
        }
        return result;
    }
}

