/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.boot.devtools.tunnel.server;

import java.io.IOException;
import java.net.ConnectException;
import java.nio.ByteBuffer;
import java.nio.channels.ByteChannel;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.Iterator;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.boot.devtools.tunnel.payload.HttpTunnelPayload;
import org.springframework.boot.devtools.tunnel.payload.HttpTunnelPayloadForwarder;
import org.springframework.boot.devtools.tunnel.server.TargetServerConnection;
import org.springframework.http.HttpInputMessage;
import org.springframework.http.HttpOutputMessage;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.server.ServerHttpAsyncRequestControl;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.util.Assert;

public class HttpTunnelServer {
    private static final int SECONDS = 1000;
    private static final int DEFAULT_LONG_POLL_TIMEOUT = 10000;
    private static final long DEFAULT_DISCONNECT_TIMEOUT = 30000L;
    private static final MediaType DISCONNECT_MEDIA_TYPE = new MediaType("application", "x-disconnect");
    private static final Log logger = LogFactory.getLog(HttpTunnelServer.class);
    private final TargetServerConnection serverConnection;
    private int longPollTimeout = 10000;
    private long disconnectTimeout = 30000L;
    private volatile ServerThread serverThread;

    public HttpTunnelServer(TargetServerConnection serverConnection) {
        Assert.notNull((Object)serverConnection, (String)"ServerConnection must not be null");
        this.serverConnection = serverConnection;
    }

    public void handle(ServerHttpRequest request, ServerHttpResponse response) throws IOException {
        this.handle(new HttpConnection(request, response));
    }

    protected void handle(HttpConnection httpConnection) throws IOException {
        try {
            this.getServerThread().handleIncomingHttp(httpConnection);
            httpConnection.waitForResponse();
        }
        catch (ConnectException ex) {
            httpConnection.respond(HttpStatus.GONE);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected ServerThread getServerThread() throws IOException {
        HttpTunnelServer httpTunnelServer = this;
        synchronized (httpTunnelServer) {
            if (this.serverThread == null) {
                ByteChannel channel = this.serverConnection.open(this.longPollTimeout);
                this.serverThread = new ServerThread(channel);
                this.serverThread.start();
            }
            return this.serverThread;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void clearServerThread() {
        HttpTunnelServer httpTunnelServer = this;
        synchronized (httpTunnelServer) {
            this.serverThread = null;
        }
    }

    public void setLongPollTimeout(int longPollTimeout) {
        Assert.isTrue((longPollTimeout > 0 ? 1 : 0) != 0, (String)"LongPollTimeout must be a positive value");
        this.longPollTimeout = longPollTimeout;
    }

    public void setDisconnectTimeout(long disconnectTimeout) {
        Assert.isTrue((disconnectTimeout > 0L ? 1 : 0) != 0, (String)"DisconnectTimeout must be a positive value");
        this.disconnectTimeout = disconnectTimeout;
    }

    protected static class HttpConnection {
        private final long createTime = System.currentTimeMillis();
        private final ServerHttpRequest request;
        private final ServerHttpResponse response;
        private ServerHttpAsyncRequestControl async;
        private volatile boolean complete = false;

        public HttpConnection(ServerHttpRequest request, ServerHttpResponse response) {
            this.request = request;
            this.response = response;
            this.async = this.startAsync();
        }

        protected ServerHttpAsyncRequestControl startAsync() {
            try {
                ServerHttpAsyncRequestControl async = this.request.getAsyncRequestControl(this.response);
                async.start();
                return async;
            }
            catch (Exception ex) {
                return null;
            }
        }

        public final ServerHttpRequest getRequest() {
            return this.request;
        }

        protected final ServerHttpResponse getResponse() {
            return this.response;
        }

        public boolean isOlderThan(int time) {
            long runningTime = System.currentTimeMillis() - this.createTime;
            return runningTime > (long)time;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void waitForResponse() {
            if (this.async == null) {
                while (!this.complete) {
                    try {
                        HttpConnection httpConnection = this;
                        synchronized (httpConnection) {
                            this.wait(1000L);
                        }
                    }
                    catch (InterruptedException interruptedException) {
                    }
                }
            }
        }

        public boolean isDisconnectRequest() {
            return DISCONNECT_MEDIA_TYPE.equals((Object)this.request.getHeaders().getContentType());
        }

        public void respond(HttpStatus status) throws IOException {
            Assert.notNull((Object)status, (String)"Status must not be null");
            this.response.setStatusCode(status);
            this.complete();
        }

        public void respond(HttpTunnelPayload payload) throws IOException {
            Assert.notNull((Object)payload, (String)"Payload must not be null");
            this.response.setStatusCode(HttpStatus.OK);
            payload.assignTo((HttpOutputMessage)this.response);
            this.complete();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void complete() {
            if (this.async != null) {
                this.async.complete();
            } else {
                HttpConnection httpConnection = this;
                synchronized (httpConnection) {
                    this.complete = true;
                    this.notifyAll();
                }
            }
        }
    }

    protected class ServerThread
    extends Thread {
        private final ByteChannel targetServer;
        private final Deque<HttpConnection> httpConnections;
        private final HttpTunnelPayloadForwarder payloadForwarder;
        private boolean closed;
        private AtomicLong responseSeq = new AtomicLong();
        private long lastHttpRequestTime;

        public ServerThread(ByteChannel targetServer) {
            Assert.notNull((Object)targetServer, (String)"TargetServer must not be null");
            this.targetServer = targetServer;
            this.httpConnections = new ArrayDeque<HttpConnection>(2);
            this.payloadForwarder = new HttpTunnelPayloadForwarder(targetServer);
        }

        @Override
        public void run() {
            try {
                try {
                    this.readAndForwardTargetServerData();
                }
                catch (Exception ex) {
                    logger.trace((Object)"Unexpected exception from tunnel server", (Throwable)ex);
                }
            }
            finally {
                this.closed = true;
                this.closeHttpConnections();
                this.closeTargetServer();
                HttpTunnelServer.this.clearServerThread();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void readAndForwardTargetServerData() throws IOException {
            while (this.targetServer.isOpen()) {
                this.closeStaleHttpConnections();
                ByteBuffer data = HttpTunnelPayload.getPayloadData(this.targetServer);
                Deque<HttpConnection> deque = this.httpConnections;
                synchronized (deque) {
                    if (data != null) {
                        HttpTunnelPayload payload = new HttpTunnelPayload(this.responseSeq.incrementAndGet(), data);
                        payload.logIncoming();
                        HttpConnection connection = this.getOrWaitForHttpConnection();
                        connection.respond(payload);
                    }
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private HttpConnection getOrWaitForHttpConnection() {
            Deque<HttpConnection> deque = this.httpConnections;
            synchronized (deque) {
                HttpConnection httpConnection = this.httpConnections.pollFirst();
                while (httpConnection == null) {
                    try {
                        this.httpConnections.wait(HttpTunnelServer.this.longPollTimeout);
                    }
                    catch (InterruptedException ex) {
                        this.closeHttpConnections();
                    }
                    httpConnection = this.httpConnections.pollFirst();
                }
                return httpConnection;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void closeStaleHttpConnections() throws IOException {
            Deque<HttpConnection> deque = this.httpConnections;
            synchronized (deque) {
                this.checkNotDisconnected();
                Iterator<HttpConnection> iterator = this.httpConnections.iterator();
                while (iterator.hasNext()) {
                    HttpConnection httpConnection = iterator.next();
                    if (!httpConnection.isOlderThan(HttpTunnelServer.this.longPollTimeout)) continue;
                    httpConnection.respond(HttpStatus.NO_CONTENT);
                    iterator.remove();
                }
            }
        }

        private void checkNotDisconnected() {
            if (this.lastHttpRequestTime > 0L) {
                long timeout = HttpTunnelServer.this.disconnectTimeout;
                long duration = System.currentTimeMillis() - this.lastHttpRequestTime;
                Assert.state((duration < timeout ? 1 : 0) != 0, (String)("Disconnect timeout: " + timeout + " " + duration));
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void closeHttpConnections() {
            Deque<HttpConnection> deque = this.httpConnections;
            synchronized (deque) {
                while (!this.httpConnections.isEmpty()) {
                    try {
                        this.httpConnections.removeFirst().respond(HttpStatus.GONE);
                    }
                    catch (Exception ex) {
                        logger.trace((Object)"Unable to close remote HTTP connection");
                    }
                }
            }
        }

        private void closeTargetServer() {
            try {
                this.targetServer.close();
            }
            catch (IOException ex) {
                logger.trace((Object)"Unable to target server connection");
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void handleIncomingHttp(HttpConnection httpConnection) throws IOException {
            if (this.closed) {
                httpConnection.respond(HttpStatus.GONE);
            }
            Deque<HttpConnection> deque = this.httpConnections;
            synchronized (deque) {
                while (this.httpConnections.size() > 1) {
                    this.httpConnections.removeFirst().respond(HttpStatus.TOO_MANY_REQUESTS);
                }
                this.lastHttpRequestTime = System.currentTimeMillis();
                this.httpConnections.addLast(httpConnection);
                this.httpConnections.notify();
            }
            this.forwardToTargetServer(httpConnection);
        }

        private void forwardToTargetServer(HttpConnection httpConnection) throws IOException {
            ServerHttpRequest request;
            HttpTunnelPayload payload;
            if (httpConnection.isDisconnectRequest()) {
                this.targetServer.close();
                this.interrupt();
            }
            if ((payload = HttpTunnelPayload.get((HttpInputMessage)(request = httpConnection.getRequest()))) != null) {
                this.payloadForwarder.forward(payload);
            }
        }
    }
}

