/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.boot.json;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Path;
import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.Deque;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Predicate;
import org.jspecify.annotations.Nullable;
import org.springframework.boot.json.JsonWriter;
import org.springframework.boot.json.JsonWriterFiltersAndProcessors;
import org.springframework.boot.json.WritableJson;
import org.springframework.boot.util.LambdaSafe;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;
import org.springframework.util.function.ThrowingConsumer;

class JsonValueWriter {
    private static final int DEFAULT_MAX_NESTING_DEPTH = 500;
    private final Appendable out;
    private final int maxNestingDepth;
    private JsonWriter.MemberPath path = JsonWriter.MemberPath.ROOT;
    private final Deque<JsonWriterFiltersAndProcessors> filtersAndProcessors = new ArrayDeque<JsonWriterFiltersAndProcessors>();
    private final Deque<ActiveSeries> activeSeries = new ArrayDeque<ActiveSeries>();

    JsonValueWriter(Appendable out) {
        this(out, 500);
    }

    JsonValueWriter(Appendable out, int maxNestingDepth) {
        this.out = out;
        this.maxNestingDepth = maxNestingDepth;
    }

    void pushProcessors(JsonWriterFiltersAndProcessors jsonProcessors) {
        this.filtersAndProcessors.addLast(jsonProcessors);
    }

    void popProcessors() {
        this.filtersAndProcessors.removeLast();
    }

    <N, V> void write(@Nullable N name, @Nullable V value) {
        if (name != null) {
            this.writePair(name, value);
        } else {
            this.write(value);
        }
    }

    <V> void write(@Nullable V value) {
        Iterable iterable;
        if ((value = this.processValue(value)) == null) {
            this.append("null");
        } else if (value instanceof WritableJson) {
            WritableJson writableJson = (WritableJson)value;
            try {
                writableJson.to(this.out);
            }
            catch (IOException ex) {
                throw new UncheckedIOException(ex);
            }
        } else if (value instanceof Iterable && this.canWriteAsArray(iterable = (Iterable)value)) {
            this.writeArray(iterable::forEach);
        } else if (ObjectUtils.isArray(value)) {
            this.writeArray(Arrays.asList(ObjectUtils.toObjectArray(value))::forEach);
        } else if (value instanceof Map) {
            Map map = (Map)value;
            this.writeObject(map::forEach);
        } else if (value instanceof Number || value instanceof Boolean) {
            this.append(value.toString());
        } else {
            this.writeString(value);
        }
    }

    private <V> boolean canWriteAsArray(Iterable<?> iterable) {
        return !(iterable instanceof Path);
    }

    void start(@Nullable Series series) {
        if (series != null) {
            int nestingDepth = this.activeSeries.size();
            Assert.state((nestingDepth <= this.maxNestingDepth ? 1 : 0) != 0, () -> "JSON nesting depth (%s) exceeds maximum depth of %s (current path: %s)".formatted(nestingDepth, this.maxNestingDepth, this.path));
            this.activeSeries.push(new ActiveSeries(series));
            this.append(series.openChar);
        }
    }

    void end(@Nullable Series series) {
        if (series != null) {
            this.activeSeries.pop();
            this.append(series.closeChar);
        }
    }

    <E> void writeArray(Consumer<Consumer<E>> elements) {
        this.start(Series.ARRAY);
        elements.accept((Consumer<E>)ThrowingConsumer.of(this::writeElement));
        this.end(Series.ARRAY);
    }

    <E> void writeElements(Consumer<Consumer<E>> elements) {
        elements.accept((Consumer<E>)ThrowingConsumer.of(this::writeElement));
    }

    <E> void writeElement(E element) {
        ActiveSeries activeSeries = this.activeSeries.peek();
        Assert.state((activeSeries != null ? 1 : 0) != 0, (String)"No series has been started");
        this.path = activeSeries.updatePath(this.path);
        activeSeries.incrementIndexAndAddCommaIfRequired();
        this.write(element);
        this.path = activeSeries.restorePath(this.path);
    }

    <N, V> void writeObject(Consumer<BiConsumer<N, V>> pairs) {
        this.start(Series.OBJECT);
        pairs.accept(this::writePair);
        this.end(Series.OBJECT);
    }

    <N, V> void writePairs(Consumer<BiConsumer<N, V>> pairs) {
        pairs.accept(this::writePair);
    }

    private <N, V> void writePair(N name, @Nullable V value) {
        this.path = this.path.child(name.toString());
        if (!this.isFilteredPath()) {
            String processedName = this.processName(name.toString());
            ActiveSeries activeSeries = this.activeSeries.peek();
            Assert.state((activeSeries != null ? 1 : 0) != 0, (String)"No series has been started");
            activeSeries.incrementIndexAndAddCommaIfRequired();
            Assert.state((boolean)activeSeries.addName(processedName), () -> "The name '" + processedName + "' has already been written");
            this.writeString(processedName);
            this.append(":");
            this.write(value);
        }
        this.path = this.path.parent();
    }

    private void writeString(Object value) {
        try {
            this.out.append('\"');
            String string = value.toString();
            block11: for (int i = 0; i < string.length(); ++i) {
                char ch = string.charAt(i);
                switch (ch) {
                    case '\"': {
                        this.out.append("\\\"");
                        continue block11;
                    }
                    case '\\': {
                        this.out.append("\\\\");
                        continue block11;
                    }
                    case '\b': {
                        this.out.append("\\b");
                        continue block11;
                    }
                    case '\f': {
                        this.out.append("\\f");
                        continue block11;
                    }
                    case '\n': {
                        this.out.append("\\n");
                        continue block11;
                    }
                    case '\r': {
                        this.out.append("\\r");
                        continue block11;
                    }
                    case '\t': {
                        this.out.append("\\t");
                        continue block11;
                    }
                    default: {
                        if (Character.isISOControl(ch)) {
                            this.out.append("\\u");
                            this.out.append(String.format("%04X", ch));
                            continue block11;
                        }
                        this.out.append(ch);
                    }
                }
            }
            this.out.append('\"');
        }
        catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    private void append(String value) {
        try {
            this.out.append(value);
        }
        catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    private void append(char ch) {
        try {
            this.out.append(ch);
        }
        catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    private boolean isFilteredPath() {
        for (JsonWriterFiltersAndProcessors filtersAndProcessors : this.filtersAndProcessors) {
            for (Predicate<JsonWriter.MemberPath> pathFilter : filtersAndProcessors.pathFilters()) {
                if (!pathFilter.test(this.path)) continue;
                return true;
            }
        }
        return false;
    }

    private String processName(String name) {
        for (JsonWriterFiltersAndProcessors filtersAndProcessors : this.filtersAndProcessors) {
            for (JsonWriter.NameProcessor nameProcessor : filtersAndProcessors.nameProcessors()) {
                name = this.processName(name, nameProcessor);
            }
        }
        return name;
    }

    private String processName(String name, JsonWriter.NameProcessor nameProcessor) {
        name = nameProcessor.processName(this.path, name);
        Assert.state((boolean)StringUtils.hasLength((String)name), (String)("NameProcessor " + String.valueOf(nameProcessor) + " returned an empty result"));
        return name;
    }

    private <V> @Nullable V processValue(@Nullable V value) {
        for (JsonWriterFiltersAndProcessors filtersAndProcessors : this.filtersAndProcessors) {
            for (JsonWriter.ValueProcessor<?> valueProcessor : filtersAndProcessors.valueProcessors()) {
                value = this.processValue(value, valueProcessor);
            }
        }
        return value;
    }

    private <V> @Nullable V processValue(@Nullable V value, JsonWriter.ValueProcessor<?> valueProcessor) {
        return (V)LambdaSafe.callback(JsonWriter.ValueProcessor.class, valueProcessor, this.path, value).invokeAnd(call -> call.processValue(this.path, value)).get(value);
    }

    private final class ActiveSeries {
        private final Series series;
        private int index;
        private Set<String> names = new HashSet<String>();

        private ActiveSeries(Series series) {
            this.series = series;
        }

        boolean addName(String processedName) {
            return this.names.add(processedName);
        }

        JsonWriter.MemberPath updatePath(JsonWriter.MemberPath path) {
            return this.series != Series.ARRAY ? path : path.child(this.index);
        }

        JsonWriter.MemberPath restorePath(JsonWriter.MemberPath path) {
            return this.series != Series.ARRAY ? path : path.parent();
        }

        void incrementIndexAndAddCommaIfRequired() {
            if (this.index > 0) {
                JsonValueWriter.this.append(',');
            }
            ++this.index;
        }
    }

    static enum Series {
        OBJECT('{', '}'),
        ARRAY('[', ']');

        final char openChar;
        final char closeChar;

        private Series(char openChar, char closeChar) {
            this.openChar = openChar;
            this.closeChar = closeChar;
        }
    }
}

