/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.cloud.contract.verifier.file;

import java.io.File;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.contract.spec.Contract;
import org.springframework.cloud.contract.spec.ContractConverter;
import org.springframework.cloud.contract.verifier.converter.YamlContractConverter;
import org.springframework.cloud.contract.verifier.file.ContractMetadata;
import org.springframework.cloud.contract.verifier.util.ContractVerifierDslConverter;
import org.springframework.core.io.support.SpringFactoriesLoader;
import org.springframework.util.CollectionUtils;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;

public class ContractFileScanner {
    private static final Logger LOG = LoggerFactory.getLogger(ContractFileScanner.class);
    private static final String OS_NAME = System.getProperty("os.name");
    private static final String OS_NAME_WINDOWS_PREFIX = "Windows";
    protected static final boolean IS_OS_WINDOWS = ContractFileScanner.getOSMatchesName("Windows");
    private static final String MATCH_PREFIX = "glob:";
    private static final Pattern SCENARIO_STEP_FILENAME_PATTERN = Pattern.compile("[0-9]+_.*");
    private final File baseDir;
    private final Set<PathMatcher> excludeMatchers;
    private final Set<PathMatcher> ignoreMatchers;
    private final Set<PathMatcher> includeMatchers;
    private final String includeMatcher;

    public ContractFileScanner(File baseDir, Set<String> excluded, Set<String> ignored, Set<String> included, String includeMatcher) {
        this.baseDir = baseDir;
        this.excludeMatchers = this.processPatterns(excluded != null ? excluded : Collections.emptySet());
        this.ignoreMatchers = this.processPatterns(ignored != null ? ignored : Collections.emptySet());
        this.includeMatchers = this.processPatterns(included != null ? included : Collections.emptySet());
        this.includeMatcher = includeMatcher != null ? includeMatcher : "";
    }

    private Set<PathMatcher> processPatterns(Set<String> patterns) {
        FileSystem fileSystem = FileSystems.getDefault();
        HashSet<PathMatcher> pathMatchers = new HashSet<PathMatcher>();
        for (String pattern : patterns) {
            Object syntaxAndPattern = "glob:**" + File.separator + pattern;
            if (IS_OS_WINDOWS) {
                syntaxAndPattern = ((String)syntaxAndPattern).replace("\\", "\\\\");
            }
            pathMatchers.add(fileSystem.getPathMatcher((String)syntaxAndPattern));
        }
        return pathMatchers;
    }

    public MultiValueMap<Path, ContractMetadata> findContractsRecursively() {
        MultiValueMap result = CollectionUtils.toMultiValueMap(new LinkedHashMap());
        this.appendRecursively(this.baseDir, (MultiValueMap<Path, ContractMetadata>)result);
        return result;
    }

    private void appendRecursively(File baseDir, MultiValueMap<Path, ContractMetadata> result) {
        Object[] files;
        List<ContractConverter> converters = this.convertersWithYml();
        if (LOG.isTraceEnabled()) {
            LOG.trace("Found the following contract converters " + converters);
        }
        if ((files = baseDir.listFiles()) == null) {
            return;
        }
        Arrays.sort(files);
        for (int i = 0; i < files.length; ++i) {
            Object file = files[i];
            boolean excluded = this.matchesPattern((File)file, this.excludeMatchers);
            if (!excluded) {
                boolean contractFile = this.isContractFile((File)file);
                boolean included = !StringUtils.hasText((String)this.includeMatcher) || ((File)file).getAbsolutePath().matches(this.includeMatcher);
                boolean bl = included = !CollectionUtils.isEmpty(this.includeMatchers) ? this.matchesPattern((File)file, this.includeMatchers) : included;
                if (contractFile && included) {
                    this.addContractToTestGeneration(result, (File[])files, (File)file, i, ContractVerifierDslConverter.convertAsCollection(baseDir, (File)file));
                }
                if (!contractFile && included) {
                    this.addContractToTestGeneration(converters, result, (File[])files, (File)file, i);
                    continue;
                }
                this.appendRecursively((File)file, result);
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("File [" + (File)file + "] is ignored. Is a contract file? [" + contractFile + "]. Should be included by pattern? [" + included + "]");
                continue;
            }
            if (!LOG.isDebugEnabled()) continue;
            LOG.debug("File [" + (File)file + "] is ignored. Should be excluded? [" + excluded + "]");
        }
    }

    protected List<ContractConverter> convertersWithYml() {
        List<ContractConverter> converters = this.converters();
        converters.add(ContractVerifierDslConverter.INSTANCE);
        converters.add(YamlContractConverter.INSTANCE);
        return converters;
    }

    protected List<ContractConverter> converters() {
        return SpringFactoriesLoader.loadFactories(ContractConverter.class, null);
    }

    private void addContractToTestGeneration(List<ContractConverter> converters, MultiValueMap<Path, ContractMetadata> result, File[] files, File file, int index) {
        boolean converted = false;
        if (!file.isDirectory()) {
            for (ContractConverter converter : converters) {
                Collection<Contract> contracts = this.tryConvert(converter, file);
                if (contracts == null) continue;
                this.addContractToTestGeneration(result, files, file, index, contracts);
                converted = true;
                break;
            }
        }
        if (!converted) {
            this.appendRecursively(file, result);
            if (LOG.isDebugEnabled()) {
                LOG.debug("File [" + file + "] wasn't ignored but no converter was applicable. The file is a directory [" + file.isDirectory() + "]");
            }
        }
    }

    private Collection<Contract> tryConvert(ContractConverter converter, File file) {
        boolean accepted = converter.isAccepted(file);
        if (!accepted) {
            return null;
        }
        try {
            return converter.convertFrom(file);
        }
        catch (Exception e) {
            throw new IllegalStateException("Failed to convert file [" + file + "]", e);
        }
    }

    private void addContractToTestGeneration(MultiValueMap<Path, ContractMetadata> result, File[] files, File file, int index, Collection<Contract> convertedContract) {
        Path path = file.toPath();
        Integer order = null;
        if (this.hasScenarioFilenamePattern(path)) {
            order = index;
        }
        Path parent = file.getParentFile().toPath();
        ContractMetadata metadata = new ContractMetadata(path, this.matchesPattern(file, this.ignoreMatchers), files.length, order, convertedContract);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Creating a contract entry for path [" + path + "] and metadata [" + metadata + "]");
        }
        result.add((Object)parent, (Object)metadata);
    }

    private boolean hasScenarioFilenamePattern(Path path) {
        return SCENARIO_STEP_FILENAME_PATTERN.matcher(path.getFileName().toString()).matches();
    }

    private boolean matchesPattern(File file, Set<PathMatcher> matchers) {
        for (PathMatcher matcher : matchers) {
            if (!matcher.matches(file.toPath())) continue;
            return true;
        }
        return false;
    }

    private boolean isContractFile(File file) {
        return file.isFile() && ContractVerifierDslConverter.INSTANCE.isAccepted(file);
    }

    private static boolean getOSMatchesName(String osNamePrefix) {
        return ContractFileScanner.isOSNameMatch(OS_NAME, osNamePrefix);
    }

    private static boolean isOSNameMatch(String osName, String osNamePrefix) {
        if (osName == null) {
            return false;
        }
        return osName.startsWith(osNamePrefix);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private File baseDir;
        private Set<String> excluded;
        private Set<String> ignored;
        private Set<String> included = Collections.emptySet();
        private String includeMatcher = "";

        public Builder baseDir(File baseDir) {
            this.baseDir = baseDir;
            return this;
        }

        public Builder excluded(Set<String> excluded) {
            this.excluded = excluded;
            return this;
        }

        public Builder ignored(Set<String> ignored) {
            this.ignored = ignored;
            return this;
        }

        public Builder included(Set<String> included) {
            this.included = included;
            return this;
        }

        public Builder includeMatcher(String includeMatcher) {
            this.includeMatcher = includeMatcher;
            return this;
        }

        public ContractFileScanner build() {
            return new ContractFileScanner(this.baseDir, this.excluded, this.ignored, this.included, this.includeMatcher);
        }
    }
}

