/*
 * Copyright 2019-2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.aot.build.context;

import java.util.List;
import java.util.Set;
import java.util.function.Consumer;

import org.springframework.nativex.domain.init.InitializationDescriptor;
import org.springframework.nativex.domain.proxies.ProxiesDescriptor;
import org.springframework.nativex.domain.reflect.ReflectionDescriptor;
import org.springframework.nativex.domain.resources.ResourcesDescriptor;
import org.springframework.nativex.domain.serialization.SerializationDescriptor;

/**
 * Provide build context information.
 *
 * @author Brian Clozel
 * @author Sebastien Deleuze
 */
public interface BuildContext {

	@Deprecated
		// for org.springframework.nativex.type.TypeSystem
	List<String> getClasspath();

	/**
	 * @return The {@link ClassLoader} to use based on the "compile+runtime" application classpath.
	 */
	ClassLoader getClassLoader();

	/**
	 * @return The (optional) main class of the application to build if it needs to be specified.
	 */
	String getMainClass();

	/**
	 * @return The (optional) application class of the application to build if it needs to be specified.
	 * Can be different from {@link #getMainClass()} for Kotlin applications for example.
	 */
	String getApplicationClass();

	List<String> getTestClasses();

	/**
	 * @return the native-image options.
	 */
	Set<String> getOptions();

	/**
	 * @return the AOT proxies.
	 */
	ProxiesDescriptor getProxiesDescriptor();

	/**
	 * Contribute source files to the application.
	 * <p>This additional source code will be compiled and packaged with the application by a build plugin.
	 * @param sourceFile the source files to add
	 */
	void addSourceFiles(SourceFile... sourceFile);

	/**
	 * Contribute resource files to the application.
	 * <p>This additional packaged with the application by a build plugin.
	 * @param resourceFile the resource files to add
	 */
	void addResources(ResourceFile... resourceFile);

	/**
	 * Contribute reflection information to the application.
	 * @param consumer The reflection descriptor consumer
	 */
	void describeReflection(Consumer<ReflectionDescriptor> consumer);

	/**
	 * Contribute reflection information to the application, this is for types reflectively accessible from JNI.
	 * @param consumer The reflection descriptor consumer
	 */
	void describeJNIReflection(Consumer<ReflectionDescriptor> consumer);

	/**
	 * Contribute proxies information to the application.
	 * @param consumer The proxies descriptor consumer
	 */
	void describeProxies(Consumer<ProxiesDescriptor> consumer);

	/**
	 * Contribute serialization type information to the application.
	 * @param consumer The serialization descriptor consumer
	 */
	void describeSerialization(Consumer<SerializationDescriptor> consumer);

	/**
	 * Contribute resources information to the application.
	 * @param consumer The resources descriptor consumer
	 */
	void describeResources(Consumer<ResourcesDescriptor> consumer);

	/**
	 * Contribute initialization information to the application.
	 * @param consumer The initialization descriptor consumer
	 */
	void describeInitialization(Consumer<InitializationDescriptor> consumer);

}
